/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.sdk.core;

import com.databricks.sdk.core.BodyLogger;
import com.databricks.sdk.core.ConfigLoader;
import com.databricks.sdk.core.DatabricksConfig;
import com.databricks.sdk.core.DatabricksError;
import com.databricks.sdk.core.DatabricksException;
import com.databricks.sdk.core.GrpcTranscodingQueryParamsSerializer;
import com.databricks.sdk.core.UserAgent;
import com.databricks.sdk.core.error.ApiErrors;
import com.databricks.sdk.core.error.PrivateLinkInfo;
import com.databricks.sdk.core.http.HttpClient;
import com.databricks.sdk.core.http.Request;
import com.databricks.sdk.core.http.Response;
import com.databricks.sdk.core.retry.RequestBasedRetryStrategyPicker;
import com.databricks.sdk.core.retry.RetryStrategy;
import com.databricks.sdk.core.retry.RetryStrategyPicker;
import com.databricks.sdk.core.utils.SerDeUtils;
import com.databricks.sdk.core.utils.SystemTimer;
import com.databricks.sdk.core.utils.Timer;
import com.databricks.sdk.support.Header;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApiClient {
    private static final Logger LOG = LoggerFactory.getLogger(ApiClient.class);
    private final int maxAttempts;
    private final ObjectMapper mapper;
    private final Random random;
    private final HttpClient httpClient;
    private final BodyLogger bodyLogger;
    private final RetryStrategyPicker retryStrategyPicker;
    private final Timer timer;
    private final Function<Void, Map<String, String>> authenticateFunc;
    private final Function<Void, String> getHostFunc;
    private final Function<Void, String> getAuthTypeFunc;
    private final String accountId;
    private final boolean isDebugHeaders;
    private static final String RETRY_AFTER_HEADER = "retry-after";

    public ApiClient() {
        this(ConfigLoader.getDefault());
    }

    public String configuredAccountID() {
        return this.accountId;
    }

    public ApiClient(DatabricksConfig config) {
        this(config, new SystemTimer());
    }

    public ApiClient(DatabricksConfig config, Timer timer) {
        this(new Builder().withDatabricksConfig(config.resolve()).withTimer(timer));
    }

    private ApiClient(Builder builder) {
        this.timer = builder.timer != null ? builder.timer : new SystemTimer();
        this.authenticateFunc = builder.authenticateFunc != null ? builder.authenticateFunc : v -> new HashMap();
        this.getHostFunc = builder.getHostFunc != null ? builder.getHostFunc : v -> "";
        this.getAuthTypeFunc = builder.getAuthTypeFunc != null ? builder.getAuthTypeFunc : v -> "";
        this.httpClient = builder.httpClient;
        this.accountId = builder.accountId;
        this.retryStrategyPicker = builder.retryStrategyPicker != null ? builder.retryStrategyPicker : new RequestBasedRetryStrategyPicker(this.getHostFunc.apply(null));
        this.isDebugHeaders = builder.isDebugHeaders;
        Integer debugTruncateBytes = builder.debugTruncateBytes;
        if (debugTruncateBytes == null) {
            debugTruncateBytes = 96;
        }
        this.maxAttempts = 4;
        this.mapper = SerDeUtils.createMapper();
        this.random = new Random();
        this.bodyLogger = new BodyLogger(this.mapper, 1024, debugTruncateBytes);
    }

    public static <I> void setQuery(Request in, I entity) {
        if (entity == null) {
            return;
        }
        for (GrpcTranscodingQueryParamsSerializer.QueryParamPair e : GrpcTranscodingQueryParamsSerializer.serialize(entity)) {
            in.withQueryParam(e.getKey(), e.getValue());
        }
    }

    public <O> Collection<O> getCollection(Request req, Class<O> element) {
        return (Collection)this.withJavaType(req, (JavaType)this.mapper.getTypeFactory().constructCollectionType(Collection.class, element));
    }

    public Map<String, String> getStringMap(Request req) {
        return (Map)this.withJavaType(req, (JavaType)this.mapper.getTypeFactory().constructMapType(Map.class, String.class, String.class));
    }

    protected <I, O> O withJavaType(Request request, JavaType javaType) {
        try {
            Response response = this.getResponse(request);
            return (O)this.deserialize(response.getBody(), javaType);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <T> T execute(Request in, Class<T> target) throws IOException {
        Response out = this.getResponse(in);
        if (target == Void.class) {
            return null;
        }
        return this.deserialize(out, target);
    }

    private Response getResponse(Request in) {
        return this.executeInner(in, in.getUrl());
    }

    private Response executeInner(Request in, String path) {
        RetryStrategy retryStrategy = this.retryStrategyPicker.getRetryStrategy(in);
        int attemptNumber = 0;
        while (true) {
            ++attemptNumber;
            IOException err = null;
            Response out = null;
            in.withHeaders(this.authenticateFunc.apply(null));
            in.withUrl(this.getHostFunc.apply(null) + path);
            String userAgent = UserAgent.asString();
            String authType = this.getAuthTypeFunc.apply(null);
            if (authType != "") {
                userAgent = userAgent + String.format(" auth/%s", authType);
            }
            in.withHeader("User-Agent", userAgent);
            try {
                out = this.httpClient.execute(in);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.makeLogRecord(in, out));
                }
            }
            catch (IOException e) {
                err = e;
                LOG.debug("Request {} failed", (Object)in, (Object)e);
            }
            if (this.isRequestSuccessful(out, err)) {
                return out;
            }
            DatabricksError databricksError = ApiErrors.getDatabricksError(out, err);
            if (!retryStrategy.isRetriable(databricksError)) {
                throw databricksError;
            }
            if (attemptNumber == this.maxAttempts) {
                throw new DatabricksException(String.format("Request %s failed after %d retries", in, this.maxAttempts), err);
            }
            long sleepMillis = this.getBackoffMillis(out, attemptNumber);
            LOG.debug(String.format("Retry %s in %dms", in.getRequestLine(), sleepMillis));
            try {
                this.timer.sleep(sleepMillis);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw new DatabricksException("Current thread was interrupted", ex);
            }
        }
    }

    private boolean isRequestSuccessful(Response response, Exception e) {
        return e == null && response.getStatusCode() >= 200 && response.getStatusCode() < 300 && !PrivateLinkInfo.isPrivateLinkRedirect(response);
    }

    public long getBackoffMillis(Response response, int attemptNumber) {
        Optional<Long> backoffMillisInResponse = ApiClient.getBackoffFromRetryAfterHeader(response);
        if (backoffMillisInResponse.isPresent()) {
            return backoffMillisInResponse.get();
        }
        int minWait = 1000;
        int maxWait = 60000;
        int minJitter = 50;
        int maxJitter = 750;
        int wait = Math.min(maxWait, minWait * (1 << attemptNumber - 1));
        int jitter = this.random.nextInt(maxJitter - minJitter + 1) + minJitter;
        return wait + jitter;
    }

    public static Optional<Long> getBackoffFromRetryAfterHeader(Response response) {
        if (response == null) {
            return Optional.empty();
        }
        List<String> retryAfterHeader = response.getHeaders(RETRY_AFTER_HEADER);
        if (retryAfterHeader == null) {
            return Optional.empty();
        }
        long waitTime = 0L;
        for (String retryAfter : retryAfterHeader) {
            try {
                ZonedDateTime retryAfterDate = ZonedDateTime.parse(retryAfter, DateTimeFormatter.RFC_1123_DATE_TIME);
                ZonedDateTime now = ZonedDateTime.now();
                waitTime = Duration.between(now, retryAfterDate).getSeconds();
            }
            catch (Exception e) {
                try {
                    waitTime = Long.parseLong(retryAfter);
                }
                catch (NumberFormatException nfe) {
                    return Optional.empty();
                }
            }
        }
        return Optional.of(waitTime * 1000L);
    }

    private String makeLogRecord(Request in, Response out) {
        StringBuilder sb = new StringBuilder();
        sb.append("> ");
        sb.append(in.getRequestLine());
        if (this.isDebugHeaders) {
            sb.append("\n * Host: ");
            sb.append(this.getHostFunc.apply(null));
            in.getHeaders().forEach((header, value) -> sb.append(String.format("\n * %s: %s", header, value)));
        }
        if (in.isBodyStreaming()) {
            sb.append("\n> (streamed body)");
        } else {
            String requestBody = in.getBodyString();
            if (requestBody != null && !requestBody.isEmpty()) {
                String[] stringArray = this.bodyLogger.redactedDump(requestBody).split("\n");
                int n = stringArray.length;
                for (int i = 0; i < n; ++i) {
                    String line = stringArray[i];
                    sb.append("\n> ");
                    sb.append(line);
                }
            }
        }
        sb.append("\n< ");
        sb.append(out.toString());
        for (String line : this.bodyLogger.redactedDump(out.getDebugBody()).split("\n")) {
            sb.append("\n< ");
            sb.append(line);
        }
        return sb.toString();
    }

    public <T> T deserialize(Response response, Class<T> target) throws IOException {
        T object;
        if (target == InputStream.class) {
            return (T)response.getBody();
        }
        try {
            object = target.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Exception e) {
            throw new DatabricksException("Unable to initialize an instance of type " + target.getName());
        }
        this.deserialize(response, object);
        return object;
    }

    public <T> T deserialize(InputStream body, JavaType target) throws IOException {
        if (target == this.mapper.constructType(InputStream.class)) {
            return (T)body;
        }
        return (T)this.mapper.readValue(body, target);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> void fillInHeaders(T target, Response response) {
        for (Field field : target.getClass().getDeclaredFields()) {
            String firstHeader;
            Header headerAnnotation = field.getAnnotation(Header.class);
            if (headerAnnotation == null || (firstHeader = response.getFirstHeader(headerAnnotation.value())) == null) continue;
            Field field2 = field;
            synchronized (field2) {
                try {
                    field.setAccessible(true);
                    if (field.getType() == String.class) {
                        field.set(target, firstHeader);
                    } else if (field.getType() == Long.class) {
                        field.set(target, Long.parseLong(firstHeader));
                    } else {
                        LOG.warn("Unsupported header type: " + field.getType());
                    }
                }
                catch (IllegalAccessException e) {
                    throw new DatabricksException("Failed to unmarshal headers: " + e.getMessage(), e);
                }
                finally {
                    field.setAccessible(false);
                }
            }
        }
    }

    private <T> Optional<Field> getContentsField(T target) {
        for (Field field : target.getClass().getDeclaredFields()) {
            if (!field.getName().equals("contents") || field.getType() != InputStream.class) continue;
            return Optional.of(field);
        }
        return Optional.empty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <T> void deserialize(Response response, T object) throws IOException {
        this.fillInHeaders(object, response);
        Optional<Field> contentsField = this.getContentsField(object);
        if (contentsField.isPresent()) {
            Field field;
            Field field2 = field = contentsField.get();
            synchronized (field2) {
                try {
                    field.setAccessible(true);
                    field.set(object, response.getBody());
                }
                catch (IllegalAccessException e) {
                    throw new DatabricksException("Failed to unmarshal headers: " + e.getMessage(), e);
                }
                finally {
                    field.setAccessible(false);
                }
            }
        }
        if (response.getBody() != null) {
            this.mapper.readerForUpdating(object).readValue(response.getBody());
        }
    }

    public String serialize(Object body) throws JsonProcessingException {
        if (body == null) {
            return null;
        }
        return this.mapper.writeValueAsString(body);
    }

    public static class Builder {
        private Timer timer;
        private Function<Void, Map<String, String>> authenticateFunc;
        private Function<Void, String> getHostFunc;
        private Function<Void, String> getAuthTypeFunc;
        private Integer debugTruncateBytes;
        private HttpClient httpClient;
        private String accountId;
        private RetryStrategyPicker retryStrategyPicker;
        private boolean isDebugHeaders;

        public Builder withDatabricksConfig(DatabricksConfig config) {
            this.authenticateFunc = v -> config.authenticate();
            this.getHostFunc = v -> config.getHost();
            this.getAuthTypeFunc = v -> config.getAuthType();
            this.httpClient = config.getHttpClient();
            this.debugTruncateBytes = config.getDebugTruncateBytes();
            this.accountId = config.getAccountId();
            this.retryStrategyPicker = new RequestBasedRetryStrategyPicker(config.getHost());
            this.isDebugHeaders = config.isDebugHeaders();
            return this;
        }

        public Builder withTimer(Timer timer) {
            this.timer = timer;
            return this;
        }

        public Builder withAuthenticateFunc(Function<Void, Map<String, String>> authenticateFunc) {
            this.authenticateFunc = authenticateFunc;
            return this;
        }

        public Builder withGetHostFunc(Function<Void, String> getHostFunc) {
            this.getHostFunc = getHostFunc;
            return this;
        }

        public Builder withGetAuthTypeFunc(Function<Void, String> getAuthTypeFunc) {
            this.getAuthTypeFunc = getAuthTypeFunc;
            return this;
        }

        public Builder withHttpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }

        public Builder withRetryStrategyPicker(RetryStrategyPicker retryStrategyPicker) {
            this.retryStrategyPicker = retryStrategyPicker;
            return this;
        }

        public ApiClient build() {
            return new ApiClient(this);
        }
    }
}

