/*
 * Decompiled with CFR 0.152.
 */
package com.github.robtimus.filesystems.memory;

import com.github.robtimus.filesystems.AbstractDirectoryStream;
import com.github.robtimus.filesystems.Messages;
import com.github.robtimus.filesystems.memory.CopyOptions;
import com.github.robtimus.filesystems.memory.MemoryFileAttributeView;
import com.github.robtimus.filesystems.memory.MemoryFileAttributes;
import com.github.robtimus.filesystems.memory.MemoryMessages;
import com.github.robtimus.filesystems.memory.MemoryPath;
import com.github.robtimus.filesystems.memory.OpenOptions;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.NonReadableChannelException;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.OverlappingFileLockException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.AccessMode;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileSystemException;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.NotLinkException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileStoreAttributeView;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;

class MemoryFileStore
extends FileStore {
    static final MemoryFileStore INSTANCE = new MemoryFileStore();
    final Directory rootNode = new Directory();
    private static final Set<String> BASIC_ATTRIBUTES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("basic:lastModifiedTime", "basic:lastAccessTime", "basic:creationTime", "basic:size", "basic:isRegularFile", "basic:isDirectory", "basic:isSymbolicLink", "basic:isOther", "basic:fileKey")));
    private static final Set<String> MEMORY_ATTRIBUTES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("memory:lastModifiedTime", "memory:lastAccessTime", "memory:creationTime", "memory:size", "memory:isRegularFile", "memory:isDirectory", "memory:isSymbolicLink", "memory:isOther", "memory:fileKey", "memory:readOnly", "memory:hidden")));

    MemoryFileStore() {
    }

    @Override
    public String name() {
        return "/";
    }

    @Override
    public String type() {
        return "memory";
    }

    @Override
    public boolean isReadOnly() {
        return false;
    }

    @Override
    public synchronized long getTotalSpace() throws IOException {
        return Runtime.getRuntime().maxMemory();
    }

    @Override
    public long getUsableSpace() throws IOException {
        return Runtime.getRuntime().freeMemory();
    }

    @Override
    public long getUnallocatedSpace() throws IOException {
        return Runtime.getRuntime().freeMemory();
    }

    @Override
    public boolean supportsFileAttributeView(Class<? extends FileAttributeView> type) {
        return type == BasicFileAttributeView.class || type == MemoryFileAttributeView.class;
    }

    @Override
    public boolean supportsFileAttributeView(String name) {
        return "basic".equals(name) || "memory".equals(name);
    }

    @Override
    public <V extends FileStoreAttributeView> V getFileStoreAttributeView(Class<V> type) {
        Objects.requireNonNull(type);
        return null;
    }

    @Override
    public Object getAttribute(String attribute) throws IOException {
        if ("totalSpace".equals(attribute)) {
            return this.getTotalSpace();
        }
        if ("usableSpace".equals(attribute)) {
            return this.getUsableSpace();
        }
        if ("unallocatedSpace".equals(attribute)) {
            return this.getUnallocatedSpace();
        }
        throw Messages.fileStore().unsupportedAttribute(attribute);
    }

    private MemoryPath normalize(MemoryPath path) {
        return path.toAbsolutePath().normalize();
    }

    private Directory findParentNode(MemoryPath path) throws FileSystemException {
        assert (path.isAbsolute()) : "path must be absolute";
        int nameCount = path.getNameCount();
        if (nameCount == 0) {
            return null;
        }
        Directory parent = this.rootNode;
        for (int i = 0; i < nameCount - 1; ++i) {
            Node node = parent.get(path.nameAt(i));
            if (node instanceof Link) {
                MemoryPath resolvedPath = this.resolveLink((Link)node, path.subpath(0, i + 1));
                Directory resolvedParent = this.findParentNode(resolvedPath);
                node = this.findNode(resolvedParent, resolvedPath);
            }
            if (!(node instanceof Directory)) {
                return null;
            }
            parent = (Directory)node;
        }
        return parent;
    }

    private Directory getExistingParentNode(MemoryPath path) throws FileSystemException {
        Directory parent = this.findParentNode(path);
        if (parent == null) {
            throw new NoSuchFileException(path.parentPath());
        }
        return parent;
    }

    private Node findNode(Directory parent, MemoryPath path) {
        assert (path.isAbsolute()) : "path must be absolute";
        if (parent == null) {
            return path.getNameCount() == 0 ? this.rootNode : null;
        }
        return parent.get(path.fileName());
    }

    private Node getExistingNode(MemoryPath path) throws FileSystemException {
        Node node;
        assert (path.isAbsolute()) : "path must be absolute";
        if (path.getNameCount() == 0) {
            return this.rootNode;
        }
        Directory parent = this.findParentNode(path);
        Node node2 = node = parent != null ? parent.get(path.fileName()) : null;
        if (node == null) {
            throw new NoSuchFileException(path.path());
        }
        return node;
    }

    private Node getExistingNode(MemoryPath path, boolean followLinks) throws FileSystemException {
        Node node = this.getExistingNode(path);
        if (followLinks && node instanceof Link) {
            MemoryPath resolvedPath = this.resolveLink((Link)node, path);
            node = this.getExistingNode(resolvedPath);
        }
        return node;
    }

    private MemoryPath resolveLink(Link link, MemoryPath path) throws FileSystemException {
        MemoryPath currentPath = path;
        Link currentLink = link;
        for (int depth = 0; depth < 100; ++depth) {
            MemoryPath targetPath = this.normalize(currentPath.resolveSibling(currentLink.target));
            Directory parent = this.findParentNode(targetPath);
            Node node = this.findNode(parent, targetPath);
            if (!(node instanceof Link)) {
                return targetPath;
            }
            currentPath = targetPath;
            currentLink = (Link)node;
        }
        throw new FileSystemException(path.path(), null, MemoryMessages.maximumLinkDepthExceeded());
    }

    private Link resolveLastLink(Link link, MemoryPath path) throws FileSystemException {
        MemoryPath currentPath = path;
        Link currentLink = link;
        for (int depth = 0; depth < 100; ++depth) {
            MemoryPath targetPath = this.normalize(currentPath.resolveSibling(currentLink.target));
            Directory parent = this.findParentNode(targetPath);
            Node node = this.findNode(parent, targetPath);
            if (!(node instanceof Link)) {
                return currentLink;
            }
            currentPath = targetPath;
            currentLink = (Link)node;
        }
        throw new FileSystemException(path.path(), null, MemoryMessages.maximumLinkDepthExceeded());
    }

    MemoryPath toRealPath(MemoryPath path, boolean followLinks) throws IOException {
        MemoryPath currentPath = this.normalize(path);
        Node node = this.getExistingNode(currentPath);
        if (followLinks && node instanceof Link) {
            Link currentLink = (Link)node;
            for (int depth = 0; depth < 100; ++depth) {
                MemoryPath targetPath = this.normalize(currentPath.resolveSibling(currentLink.target));
                node = this.getExistingNode(targetPath);
                if (!(node instanceof Link)) {
                    return targetPath;
                }
                currentPath = targetPath;
                currentLink = (Link)node;
            }
            throw new FileSystemException(path.path(), null, MemoryMessages.maximumLinkDepthExceeded());
        }
        return currentPath;
    }

    synchronized byte[] getContent(MemoryPath path) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, true);
        if (node instanceof Directory) {
            throw Messages.fileSystemProvider().isDirectory(normalizedPath.path());
        }
        File file = (File)node;
        return file.getContent();
    }

    synchronized void setContent(MemoryPath path, byte[] content) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Directory parent = this.getExistingParentNode(normalizedPath);
        Node node = this.findNode(parent, normalizedPath);
        MemoryPath fileToSave = normalizedPath;
        if (node instanceof Link) {
            MemoryPath resolvedPath = this.resolveLink((Link)node, normalizedPath);
            parent = this.getExistingParentNode(resolvedPath);
            node = this.findNode(parent, resolvedPath);
            fileToSave = resolvedPath;
        }
        if (node instanceof Directory) {
            throw Messages.fileSystemProvider().isDirectory(normalizedPath.path());
        }
        File file = (File)node;
        if (file == null) {
            this.validateTarget(parent, fileToSave, normalizedPath, true);
            file = new File();
            parent.add(fileToSave.fileName(), file);
        }
        if (file.isReadOnly()) {
            throw new AccessDeniedException(normalizedPath.path());
        }
        file.setContent(content);
    }

    synchronized InputStream newInputStream(MemoryPath path, OpenOption ... options) throws IOException {
        OpenOptions openOptions = OpenOptions.forNewInputStream(options);
        assert (openOptions.read);
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, true);
        if (node instanceof Directory) {
            throw Messages.fileSystemProvider().isDirectory(normalizedPath.path());
        }
        File file = (File)node;
        OnCloseAction onClose = openOptions.deleteOnClose ? () -> this.delete(normalizedPath) : null;
        return file.newInputStream(onClose);
    }

    synchronized OutputStream newOutputStream(MemoryPath path, OpenOption ... options) throws IOException {
        OnCloseAction onClose;
        OpenOptions openOptions = OpenOptions.forNewOutputStream(options);
        assert (openOptions.write);
        MemoryPath normalizedPath = this.normalize(path);
        Directory parent = this.getExistingParentNode(normalizedPath);
        Node node = this.findNode(parent, normalizedPath);
        MemoryPath fileToSave = normalizedPath;
        if (node instanceof Link) {
            MemoryPath resolvedPath = this.resolveLink((Link)node, normalizedPath);
            parent = this.getExistingParentNode(resolvedPath);
            node = this.findNode(parent, resolvedPath);
            fileToSave = resolvedPath;
        }
        if (node instanceof Directory) {
            throw Messages.fileSystemProvider().isDirectory(normalizedPath.path());
        }
        File file = (File)node;
        OnCloseAction onCloseAction = onClose = openOptions.deleteOnClose ? () -> this.delete(normalizedPath) : null;
        if (file == null) {
            if (!openOptions.create && !openOptions.createNew) {
                throw new NoSuchFileException(normalizedPath.path());
            }
            this.validateTarget(parent, fileToSave, normalizedPath, openOptions.create && !openOptions.createNew);
            file = new File();
            parent.add(fileToSave.fileName(), file);
        } else if (openOptions.createNew) {
            throw new FileAlreadyExistsException(normalizedPath.path());
        }
        if (file.isReadOnly()) {
            throw new AccessDeniedException(normalizedPath.path());
        }
        return file.newOutputStream(openOptions.append, onClose);
    }

    synchronized FileChannel newFileChannel(MemoryPath path, Set<? extends OpenOption> options, FileAttribute<?> ... attrs) throws IOException {
        OnCloseAction onClose;
        OpenOptions openOptions = OpenOptions.forNewFileChannel(options);
        MemoryPath normalizedPath = this.normalize(path);
        Directory parent = this.getExistingParentNode(normalizedPath);
        Node node = this.findNode(parent, normalizedPath);
        MemoryPath fileToSave = normalizedPath;
        if (node instanceof Link) {
            MemoryPath resolvedPath = this.resolveLink((Link)node, normalizedPath);
            parent = this.getExistingParentNode(resolvedPath);
            node = this.findNode(parent, resolvedPath);
            fileToSave = resolvedPath;
        }
        if (node instanceof Directory) {
            throw Messages.fileSystemProvider().isDirectory(normalizedPath.path());
        }
        File file = (File)node;
        OnCloseAction onCloseAction = onClose = openOptions.deleteOnClose ? () -> this.delete(normalizedPath) : null;
        if (openOptions.read && !openOptions.write) {
            if (file == null) {
                throw new NoSuchFileException(normalizedPath.path());
            }
            return file.newFileChannel(true, false, false, onClose);
        }
        if (file == null) {
            if (!openOptions.create && !openOptions.createNew) {
                throw new NoSuchFileException(normalizedPath.path());
            }
            this.validateTarget(parent, fileToSave, normalizedPath, openOptions.create && !openOptions.createNew);
            file = new File();
            FileChannel channel = file.newFileChannel(openOptions.read, openOptions.write, openOptions.append, onClose);
            for (FileAttribute<?> attribute : attrs) {
                try {
                    this.setAttribute(file, attribute.name(), attribute.value());
                }
                catch (IllegalArgumentException e) {
                    channel.close();
                    throw new UnsupportedOperationException(e.getMessage());
                }
            }
            parent.add(fileToSave.fileName(), file);
            if (file.isReadOnly()) {
                throw new AccessDeniedException(normalizedPath.path());
            }
            return channel;
        }
        if (openOptions.createNew) {
            throw new FileAlreadyExistsException(normalizedPath.path());
        }
        if (file.isReadOnly()) {
            throw new AccessDeniedException(normalizedPath.path());
        }
        return file.newFileChannel(openOptions.read, openOptions.write, openOptions.append, onClose);
    }

    synchronized SeekableByteChannel newByteChannel(MemoryPath path, Set<? extends OpenOption> options, FileAttribute<?> ... attrs) throws IOException {
        return this.newFileChannel(path, options, attrs);
    }

    synchronized DirectoryStream<Path> newDirectoryStream(MemoryPath path, DirectoryStream.Filter<? super Path> filter) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, true);
        if (!(node instanceof Directory)) {
            throw new NotDirectoryException(normalizedPath.path());
        }
        Objects.requireNonNull(filter);
        return new MemoryPathDirectoryStream(normalizedPath, filter);
    }

    synchronized void createDirectory(MemoryPath dir, FileAttribute<?> ... attrs) throws IOException {
        MemoryPath normalizedDir = this.normalize(dir);
        if (normalizedDir.getNameCount() == 0) {
            throw new FileAlreadyExistsException(normalizedDir.path());
        }
        Directory parent = this.getExistingParentNode(normalizedDir);
        this.validateTarget(parent, normalizedDir, false);
        Directory directory = new Directory();
        for (FileAttribute<?> attribute : attrs) {
            try {
                this.setAttribute(directory, attribute.name(), attribute.value());
            }
            catch (IllegalArgumentException e) {
                throw new UnsupportedOperationException(e.getMessage());
            }
        }
        parent.add(normalizedDir.fileName(), directory);
    }

    synchronized void createSymbolicLink(MemoryPath link, MemoryPath target, FileAttribute<?> ... attrs) throws IOException {
        MemoryPath normalizedLink = this.normalize(link);
        Directory parent = this.getExistingParentNode(normalizedLink);
        this.validateTarget(parent, normalizedLink, false);
        Link node = new Link(target.path());
        for (FileAttribute<?> attribute : attrs) {
            try {
                this.setAttribute(node, attribute.name(), attribute.value());
            }
            catch (IllegalArgumentException e) {
                throw new UnsupportedOperationException(e.getMessage());
            }
        }
        parent.add(normalizedLink.fileName(), node);
    }

    synchronized void createLink(MemoryPath link, MemoryPath existing) throws IOException {
        MemoryPath normalizedLink = this.normalize(link);
        MemoryPath normalizedExisting = this.normalize(existing);
        Directory parent = this.getExistingParentNode(normalizedLink);
        this.validateTarget(parent, normalizedLink, false);
        Node node = this.getExistingNode(normalizedExisting);
        if (node instanceof Directory) {
            throw Messages.fileSystemProvider().isDirectory(normalizedExisting.path());
        }
        parent.add(normalizedLink.fileName(), node);
    }

    synchronized void delete(MemoryPath path) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath);
        this.deleteNode(node, normalizedPath);
    }

    synchronized boolean deleteIfExists(MemoryPath path) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Directory parent = this.findParentNode(normalizedPath);
        Node node = this.findNode(parent, normalizedPath);
        if (node != null) {
            this.deleteNode(node, normalizedPath);
            return true;
        }
        return false;
    }

    private void deleteNode(Node node, MemoryPath path) throws IOException {
        if (this.isNonEmptyDirectory(node)) {
            throw new DirectoryNotEmptyException(path.path());
        }
        if (node == this.rootNode) {
            throw new AccessDeniedException(path.path());
        }
        if (node.parent.isReadOnly()) {
            throw new AccessDeniedException(path.parentPath());
        }
        node.parent.remove(path.fileName());
    }

    synchronized MemoryPath readSymbolicLink(MemoryPath link) throws IOException {
        MemoryPath normalizedLink = this.normalize(link);
        Node node = this.getExistingNode(normalizedLink);
        if (!(node instanceof Link)) {
            throw new NotLinkException(normalizedLink.path());
        }
        return new MemoryPath(link.getFileSystem(), ((Link)node).target);
    }

    synchronized void copy(MemoryPath source, MemoryPath target, CopyOption ... options) throws IOException {
        Directory targetDirectory;
        Node targetNode;
        CopyOptions copyOptions = CopyOptions.forCopy(options);
        MemoryPath normalizedSource = this.normalize(source);
        MemoryPath normalizedTarget = this.normalize(target);
        Node sourceNode = this.getExistingNode(normalizedSource, copyOptions.followLinks);
        if (sourceNode == (targetNode = this.findNode(targetDirectory = this.findParentNode(normalizedTarget), normalizedTarget))) {
            return;
        }
        if (sourceNode instanceof Link) {
            sourceNode = this.resolveLastLink((Link)sourceNode, normalizedSource);
        }
        this.validateTarget(targetDirectory, normalizedTarget, copyOptions.replaceExisting);
        targetNode = sourceNode.copy(copyOptions.copyAttributes);
        targetDirectory.add(normalizedTarget.fileName(), targetNode);
    }

    synchronized void move(MemoryPath source, MemoryPath target, CopyOption ... options) throws IOException {
        Directory targetDirectory;
        Node targetNode;
        CopyOptions copyOptions = CopyOptions.forMove(options);
        MemoryPath normalizedSource = this.normalize(source);
        MemoryPath normalizedTarget = this.normalize(target);
        Node sourceNode = this.getExistingNode(normalizedSource);
        if (sourceNode == (targetNode = this.findNode(targetDirectory = this.findParentNode(normalizedTarget), normalizedTarget))) {
            return;
        }
        if (sourceNode == this.rootNode) {
            throw new DirectoryNotEmptyException(normalizedSource.path());
        }
        Directory sourceDirectory = sourceNode.parent;
        if (sourceDirectory.isReadOnly()) {
            throw new AccessDeniedException(normalizedSource.parentPath());
        }
        this.validateTarget(targetDirectory, normalizedTarget, copyOptions.replaceExisting);
        sourceDirectory.remove(normalizedSource.fileName());
        targetDirectory.add(normalizedTarget.fileName(), sourceNode);
    }

    private void validateTarget(Directory targetDirectory, MemoryPath target, boolean replaceExisting) throws IOException {
        this.validateTarget(targetDirectory, target, target, replaceExisting);
    }

    private void validateTarget(Directory targetDirectory, MemoryPath target, MemoryPath originalTarget, boolean replaceExisting) throws IOException {
        if (targetDirectory == null) {
            throw new NoSuchFileException(originalTarget.parentPath());
        }
        if (targetDirectory.isReadOnly()) {
            throw new AccessDeniedException(originalTarget.parentPath());
        }
        Node targetNode = targetDirectory.get(target.fileName());
        if (targetNode != null && !replaceExisting) {
            throw new FileAlreadyExistsException(originalTarget.path());
        }
        if (this.isNonEmptyDirectory(targetNode)) {
            throw new DirectoryNotEmptyException(originalTarget.path());
        }
    }

    private boolean isNonEmptyDirectory(Node node) {
        return node instanceof Directory && !((Directory)node).isEmpty();
    }

    synchronized boolean isSameFile(MemoryPath path, MemoryPath path2) throws IOException {
        MemoryPath normalizedPath2;
        MemoryPath normalizedPath = this.normalize(path);
        if (normalizedPath.equals((Object)(normalizedPath2 = this.normalize(path2)))) {
            return true;
        }
        return this.getExistingNode(normalizedPath, true) == this.getExistingNode(normalizedPath2, true);
    }

    synchronized boolean isHidden(MemoryPath path) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath);
        return node.isHidden();
    }

    synchronized FileStore getFileStore(MemoryPath path) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        this.getExistingNode(normalizedPath);
        return this;
    }

    synchronized void checkAccess(MemoryPath path, AccessMode ... modes) throws IOException {
        boolean w = false;
        boolean x = false;
        for (AccessMode mode : modes) {
            if (mode == AccessMode.WRITE) {
                w = true;
                continue;
            }
            if (mode != AccessMode.EXECUTE) continue;
            x = true;
        }
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, true);
        boolean isReadOnly = node.isReadOnly();
        if (w && isReadOnly) {
            throw new AccessDeniedException(normalizedPath.path());
        }
        if (x && !node.isDirectory()) {
            throw new AccessDeniedException(normalizedPath.path());
        }
    }

    synchronized void setTimes(MemoryPath path, FileTime lastModifiedTime, FileTime lastAccessTime, FileTime createTime, boolean followLinks) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, followLinks);
        if (lastModifiedTime != null) {
            node.setLastModifiedTime(lastModifiedTime);
        }
        if (lastAccessTime != null) {
            node.setLastAccessTime(lastAccessTime);
        }
        if (createTime != null) {
            node.setCreationTime(createTime);
        }
    }

    synchronized void setReadOnly(MemoryPath path, boolean value, boolean followLinks) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, followLinks);
        node.setReadOnly(value);
    }

    synchronized void setHidden(MemoryPath path, boolean value, boolean followLinks) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, followLinks);
        node.setHidden(value);
    }

    synchronized MemoryFileAttributes readAttributes(MemoryPath path, boolean followLinks) throws IOException {
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, followLinks);
        return node.getAttributes();
    }

    synchronized Map<String, Object> readAttributes(MemoryPath path, String attributes, boolean followLinks) throws IOException {
        Set<String> allowedAttributes;
        String view;
        int pos = attributes.indexOf(58);
        if (pos == -1) {
            view = "basic";
            attributes = "basic:" + attributes;
        } else {
            view = attributes.substring(0, pos);
        }
        if (attributes.startsWith("basic:")) {
            allowedAttributes = BASIC_ATTRIBUTES;
        } else if (attributes.startsWith("memory:")) {
            allowedAttributes = MEMORY_ATTRIBUTES;
        } else {
            throw Messages.fileSystemProvider().unsupportedFileAttributeView(view);
        }
        Map<String, Object> result = this.getAttributeMap(attributes, allowedAttributes);
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, followLinks);
        block35: for (Map.Entry<String, Object> entry : result.entrySet()) {
            switch (entry.getKey()) {
                case "basic:lastModifiedTime": 
                case "memory:lastModifiedTime": {
                    entry.setValue(node.getLastModifiedTime());
                    continue block35;
                }
                case "basic:lastAccessTime": 
                case "memory:lastAccessTime": {
                    entry.setValue(node.getLastAccessTime());
                    continue block35;
                }
                case "basic:creationTime": 
                case "memory:creationTime": {
                    entry.setValue(node.getCreationTime());
                    continue block35;
                }
                case "basic:size": 
                case "memory:size": {
                    entry.setValue(node.getSize());
                    continue block35;
                }
                case "basic:isRegularFile": 
                case "memory:isRegularFile": {
                    entry.setValue(node.isRegularFile());
                    continue block35;
                }
                case "basic:isDirectory": 
                case "memory:isDirectory": {
                    entry.setValue(node.isDirectory());
                    continue block35;
                }
                case "basic:isSymbolicLink": 
                case "memory:isSymbolicLink": {
                    entry.setValue(false);
                    continue block35;
                }
                case "basic:isOther": 
                case "memory:isOther": {
                    entry.setValue(false);
                    continue block35;
                }
                case "basic:fileKey": 
                case "memory:fileKey": {
                    entry.setValue(null);
                    continue block35;
                }
                case "memory:readOnly": {
                    entry.setValue(node.isReadOnly());
                    continue block35;
                }
                case "memory:hidden": {
                    entry.setValue(node.isHidden());
                    continue block35;
                }
            }
            throw new IllegalStateException("unexpected attribute name: " + entry.getKey());
        }
        return result;
    }

    private Map<String, Object> getAttributeMap(String attributes, Set<String> allowedAttributes) {
        int indexOfColon = attributes.indexOf(58);
        String prefix = attributes.substring(0, indexOfColon + 1);
        attributes = attributes.substring(indexOfColon + 1);
        String[] attributeList = attributes.split(",");
        HashMap<String, Object> result = new HashMap<String, Object>(allowedAttributes.size());
        for (String attribute : attributeList) {
            String prefixedAttribute = prefix + attribute;
            if (allowedAttributes.contains(prefixedAttribute)) {
                result.put(prefixedAttribute, null);
                continue;
            }
            if ("*".equals(attribute)) {
                for (String s : allowedAttributes) {
                    result.put(s, null);
                }
                continue;
            }
            throw Messages.fileSystemProvider().unsupportedFileAttribute(attribute);
        }
        return result;
    }

    synchronized void setAttribute(MemoryPath path, String attribute, Object value, boolean followLinks) throws IOException {
        String view;
        int pos = attribute.indexOf(58);
        if (pos == -1) {
            view = "basic";
            attribute = "basic:" + attribute;
        } else {
            view = attribute.substring(0, pos);
        }
        if (!"basic".equals(view) && !"memory".equals(view)) {
            throw Messages.fileSystemProvider().unsupportedFileAttributeView(view);
        }
        MemoryPath normalizedPath = this.normalize(path);
        Node node = this.getExistingNode(normalizedPath, followLinks);
        this.setAttribute(node, attribute, value);
    }

    private void setAttribute(Node node, String attribute, Object value) {
        switch (attribute) {
            case "basic:lastModifiedTime": 
            case "memory:lastModifiedTime": {
                node.setLastModifiedTime((FileTime)value);
                break;
            }
            case "basic:lastAccessTime": 
            case "memory:lastAccessTime": {
                node.setLastAccessTime((FileTime)value);
                break;
            }
            case "basic:creationTime": 
            case "memory:creationTime": {
                node.setCreationTime((FileTime)value);
                break;
            }
            case "memory:readOnly": {
                node.setReadOnly((Boolean)value);
                break;
            }
            case "memory:hidden": {
                node.setHidden((Boolean)value);
                break;
            }
            default: {
                throw Messages.fileSystemProvider().unsupportedFileAttribute(attribute);
            }
        }
    }

    void clear() {
        this.rootNode.clear();
    }

    static interface OnCloseAction {
        public void run() throws IOException;
    }

    static final class Link
    extends Node {
        private static final int MAX_DEPTH = 100;
        final String target;

        Link(String target) {
            this.target = Objects.requireNonNull(target);
        }

        @Override
        boolean isRegularFile() {
            return false;
        }

        @Override
        boolean isDirectory() {
            return false;
        }

        @Override
        boolean isSymbolicLink() {
            return true;
        }

        @Override
        long getSize() {
            return 0L;
        }

        @Override
        Node copy(boolean copyAttributes) {
            Link copy = new Link(this.target);
            if (copyAttributes) {
                this.copyAttributes(copy);
            }
            return copy;
        }
    }

    static final class File
    extends Node {
        private final List<Byte> content = new ArrayList<Byte>();
        private LockTable lockTable;

        File() {
        }

        @Override
        boolean isRegularFile() {
            return true;
        }

        @Override
        boolean isDirectory() {
            return false;
        }

        @Override
        boolean isSymbolicLink() {
            return false;
        }

        @Override
        synchronized long getSize() {
            return this.content.size();
        }

        @Override
        synchronized Node copy(boolean copyAttributes) {
            File copy = new File();
            copy.content.addAll(this.content);
            if (copyAttributes) {
                this.copyAttributes(copy);
            }
            return copy;
        }

        synchronized InputStream newInputStream(OnCloseAction onClose) {
            this.updateLastAccessTime();
            return new ContentInputStream(this, onClose);
        }

        synchronized OutputStream newOutputStream(boolean append, OnCloseAction onClose) {
            if (!append) {
                this.content.clear();
            }
            this.updateLastModifiedAndAccessTimes();
            return new ContentOutputStream(this, onClose);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        synchronized FileChannel newFileChannel(boolean readable, boolean writable, boolean append, OnCloseAction onClose) {
            assert (!append || writable) : "append is only allowed if writable is true";
            ContentFileChannel channel = new ContentFileChannel(this, readable, writable, onClose);
            if (append) {
                ContentFileChannel contentFileChannel = channel;
                synchronized (contentFileChannel) {
                    channel.position = this.content.size();
                }
                this.updateLastModifiedAndAccessTimes();
            } else if (writable) {
                this.content.clear();
                this.updateLastModifiedAndAccessTimes();
            } else {
                this.updateLastAccessTime();
            }
            return channel;
        }

        synchronized LockTable lockTable() {
            if (this.lockTable == null) {
                this.lockTable = new LockTable();
            }
            return this.lockTable;
        }

        synchronized byte[] getContent() {
            byte[] result = new byte[this.content.size()];
            int i = 0;
            for (Byte b : this.content) {
                result[i++] = b;
            }
            this.updateLastAccessTime();
            return result;
        }

        synchronized void setContent(byte ... newContent) {
            this.content.clear();
            for (byte b : newContent) {
                this.content.add(b);
            }
            this.updateLastModifiedAndAccessTimes();
        }

        private static final class LockTable {
            private final List<Lock> locks = new ArrayList<Lock>();

            private LockTable() {
            }

            private void checkLocks(long position, long size) {
                for (Lock lock : this.locks) {
                    if (!lock.overlaps(position, size)) continue;
                    throw new OverlappingFileLockException();
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void add(Lock lock) {
                List<Lock> list = this.locks;
                synchronized (list) {
                    this.checkLocks(lock.position(), lock.size());
                    this.locks.add(lock);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void remove(Lock lock) {
                List<Lock> list = this.locks;
                synchronized (list) {
                    this.locks.remove(lock);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void invalidateAndRemoveAll(FileChannel channel) {
                List<Lock> list = this.locks;
                synchronized (list) {
                    Iterator<Lock> i = this.locks.iterator();
                    while (i.hasNext()) {
                        Lock lock = i.next();
                        if (lock.channel() != channel) continue;
                        lock.invalidate();
                        i.remove();
                    }
                }
            }
        }

        private static final class Lock
        extends FileLock {
            private final LockTable lockTable;
            private boolean isValid = true;

            private Lock(ContentFileChannel channel, long position, long size, LockTable lockTable) {
                super(channel, position, size, false);
                this.lockTable = lockTable;
            }

            @Override
            public synchronized boolean isValid() {
                return this.isValid;
            }

            private synchronized void invalidate() {
                this.isValid = false;
            }

            @Override
            public synchronized void release() throws IOException {
                Channel channel = this.acquiredBy();
                if (!channel.isOpen()) {
                    throw new ClosedChannelException();
                }
                if (this.isValid) {
                    this.lockTable.remove(this);
                    this.isValid = false;
                }
            }
        }

        private static final class ContentFileChannel
        extends FileChannel {
            private final File file;
            private final boolean readable;
            private final boolean writeable;
            private final OnCloseAction onClose;
            private int position = 0;

            private ContentFileChannel(File file, boolean readable, boolean writable, OnCloseAction onClose) {
                this.file = file;
                this.readable = readable;
                this.writeable = writable;
                this.onClose = onClose;
            }

            private void checkReadable() {
                if (!this.readable) {
                    throw new NonReadableChannelException();
                }
            }

            private void checkWritable() {
                if (!this.writeable) {
                    throw new NonWritableChannelException();
                }
            }

            private void checkOpen() throws ClosedChannelException {
                if (!this.isOpen()) {
                    throw new ClosedChannelException();
                }
            }

            @Override
            protected void implCloseChannel() throws IOException {
                this.file.lockTable().invalidateAndRemoveAll(this);
                if (this.onClose != null) {
                    this.onClose.run();
                }
            }

            @Override
            public synchronized int read(ByteBuffer dst) throws IOException {
                int read = this.read(dst, this.position);
                if (read > 0) {
                    this.position += read;
                }
                return read;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized int read(ByteBuffer dst, long pos) throws IOException {
                this.checkOpen();
                this.checkReadable();
                File file = this.file;
                synchronized (file) {
                    int totalRead;
                    int bytesToRead;
                    int size = this.file.content.size();
                    if (pos >= (long)size) {
                        return -1;
                    }
                    int available = size - (int)pos;
                    int len = dst.remaining();
                    if (len > available) {
                        len = available;
                    }
                    if (len <= 0) {
                        return 0;
                    }
                    byte[] buffer = new byte[8192];
                    for (totalRead = 0; totalRead < len; totalRead += bytesToRead) {
                        bytesToRead = Math.min(len - totalRead, buffer.length);
                        for (int i = 0; i < bytesToRead && pos <= Integer.MAX_VALUE; ++i, ++pos) {
                            buffer[i] = (Byte)this.file.content.get((int)pos);
                        }
                        dst.put(buffer, 0, bytesToRead);
                    }
                    this.file.updateLastAccessTime();
                    return totalRead;
                }
            }

            @Override
            public synchronized long read(ByteBuffer[] dsts, int offset, int length) throws IOException {
                int totalRead = 0;
                for (int i = 0; i < length; ++i) {
                    int read = this.read(dsts[offset + i]);
                    if (read == -1) {
                        return totalRead == 0 ? -1L : (long)totalRead;
                    }
                    totalRead += read;
                }
                return totalRead;
            }

            @Override
            public synchronized int write(ByteBuffer src) throws IOException {
                int written = this.write(src, this.position);
                if (written > 0) {
                    this.position += written;
                }
                return written;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized int write(ByteBuffer src, long pos) throws IOException {
                this.checkOpen();
                this.checkWritable();
                File file = this.file;
                synchronized (file) {
                    int bytesToWrite;
                    int totalWritten;
                    if (pos > Integer.MAX_VALUE) {
                        return 0;
                    }
                    while ((long)this.file.content.size() < pos) {
                        this.file.content.add((byte)0);
                    }
                    int len = src.remaining();
                    if (len == 0) {
                        return totalWritten;
                    }
                    byte[] buffer = new byte[8192];
                    for (totalWritten = 0; totalWritten < len; totalWritten += bytesToWrite) {
                        bytesToWrite = Math.min(len - totalWritten, buffer.length);
                        src.get(buffer, 0, bytesToWrite);
                        for (int i = 0; i < bytesToWrite && pos <= Integer.MAX_VALUE; ++i, ++pos) {
                            if (pos < (long)this.file.content.size()) {
                                this.file.content.set((int)pos, buffer[i]);
                                continue;
                            }
                            assert (pos == (long)this.file.content.size());
                            this.file.content.add(buffer[i]);
                        }
                    }
                    this.file.updateLastModifiedAndAccessTimes();
                    return totalWritten;
                }
            }

            @Override
            public synchronized long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
                int totalWritten = 0;
                for (int i = 0; i < length; ++i) {
                    int written = this.write(srcs[offset + i]);
                    totalWritten += written;
                }
                return totalWritten;
            }

            @Override
            public synchronized long position() throws IOException {
                this.checkOpen();
                return this.position;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized FileChannel position(long newPosition) throws IOException {
                if (newPosition < 0L) {
                    throw Messages.byteChannel().negativePosition(newPosition);
                }
                this.checkOpen();
                this.position = (int)Math.min(newPosition, Integer.MAX_VALUE);
                File file = this.file;
                synchronized (file) {
                    this.file.updateLastAccessTime();
                }
                return this;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized long size() throws IOException {
                this.checkOpen();
                File file = this.file;
                synchronized (file) {
                    return this.file.content.size();
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized FileChannel truncate(long size) throws IOException {
                this.checkOpen();
                this.checkWritable();
                if (size < 0L) {
                    throw Messages.byteChannel().negativeSize(size);
                }
                File file = this.file;
                synchronized (file) {
                    int oldSize = this.file.content.size();
                    if (size < (long)oldSize) {
                        this.file.content.subList((int)size, oldSize).clear();
                    }
                    this.file.updateLastModifiedAndAccessTimes();
                }
                return this;
            }

            @Override
            public void force(boolean metaData) {
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized long transferTo(long pos, long count, WritableByteChannel target) throws IOException {
                if (pos < 0L) {
                    throw Messages.fileChannel().negativePosition(pos);
                }
                if (count < 0L) {
                    throw Messages.fileChannel().negativeCount(count);
                }
                this.checkOpen();
                this.checkReadable();
                if (count == 0L) {
                    return 0L;
                }
                File file = this.file;
                synchronized (file) {
                    if (pos > (long)this.file.content.size()) {
                        return 0L;
                    }
                    if (target instanceof ContentFileChannel) {
                        return this.transferTo((int)pos, count, (ContentFileChannel)target);
                    }
                    return this.transferToGeneric((int)pos, count, target);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private long transferTo(int pos, long count, ContentFileChannel target) throws IOException {
                File file = target.file;
                synchronized (file) {
                    target.checkOpen();
                    target.checkWritable();
                    int toTransfer = (int)Math.min(count, (long)(this.file.content.size() - pos));
                    toTransfer = Math.min(toTransfer, Integer.MAX_VALUE - target.file.content.size());
                    if (toTransfer == 0) {
                        return 0L;
                    }
                    target.file.content.addAll(this.file.content.subList(pos, pos + toTransfer));
                    target.position += toTransfer;
                    this.file.updateLastAccessTime();
                    target.file.updateLastModifiedAndAccessTimes();
                    return toTransfer;
                }
            }

            private long transferToGeneric(int pos, long count, WritableByteChannel target) throws IOException {
                int toTransfer;
                int bytesToRead;
                ByteBuffer buffer = ByteBuffer.allocate(8192);
                int transferred = 0;
                for (int remaining = toTransfer = (int)Math.min(count, (long)(this.file.content.size() - pos)); remaining > 0; remaining -= bytesToRead) {
                    bytesToRead = Math.min(buffer.capacity(), remaining);
                    int i = 0;
                    while (i < bytesToRead) {
                        buffer.put((Byte)this.file.content.get(pos));
                        ++i;
                        ++pos;
                    }
                    buffer.flip();
                    target.write(buffer);
                    buffer.clear();
                    transferred += bytesToRead;
                }
                this.file.updateLastAccessTime();
                return transferred;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public synchronized long transferFrom(ReadableByteChannel src, long pos, long count) throws IOException {
                if (pos < 0L) {
                    throw Messages.fileChannel().negativePosition(pos);
                }
                if (count < 0L) {
                    throw Messages.fileChannel().negativeCount(count);
                }
                this.checkOpen();
                this.checkWritable();
                if (count == 0L) {
                    return 0L;
                }
                File file = this.file;
                synchronized (file) {
                    if (pos > (long)this.file.content.size()) {
                        return 0L;
                    }
                    if (src instanceof ContentFileChannel) {
                        return this.transferFrom((ContentFileChannel)src, (int)pos, count);
                    }
                    return this.transferFromGeneric(src, (int)pos, count);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private long transferFrom(ContentFileChannel src, int pos, long count) throws IOException {
                File file = src.file;
                synchronized (file) {
                    src.checkOpen();
                    src.checkReadable();
                    int toTransfer = (int)Math.min(count, (long)(src.file.content.size() - src.position));
                    toTransfer = Math.min(toTransfer, Integer.MAX_VALUE - pos);
                    if (toTransfer == 0) {
                        return 0L;
                    }
                    int toOverwrite = Math.min(toTransfer, this.file.content.size() - pos);
                    int toAdd = toTransfer - toOverwrite;
                    for (int i = 0; i < toOverwrite; ++i) {
                        this.file.content.set(pos + i, src.file.content.get(src.position++));
                    }
                    if (toAdd > 0) {
                        this.file.content.addAll(src.file.content.subList(src.position, src.position + toAdd));
                        src.position += toAdd;
                    }
                    this.file.updateLastModifiedAndAccessTimes();
                    src.file.updateLastAccessTime();
                    return toTransfer;
                }
            }

            private long transferFromGeneric(ReadableByteChannel src, int pos, long count) throws IOException {
                int toTransfer;
                int bytesRead;
                ByteBuffer buffer = ByteBuffer.allocate(8192);
                int transferred = 0;
                for (int remaining = toTransfer = (int)Math.min(count, (long)(Integer.MAX_VALUE - pos)); remaining > 0; remaining -= bytesRead) {
                    int bytesToRead = Math.min(buffer.capacity(), remaining);
                    buffer.limit(bytesToRead);
                    bytesRead = src.read(buffer);
                    if (bytesRead == -1) break;
                    buffer.flip();
                    int i = 0;
                    while (i < bytesRead) {
                        if (pos < this.file.content.size()) {
                            this.file.content.set(pos, buffer.get(i));
                        } else {
                            assert (pos == this.file.content.size());
                            this.file.content.add(buffer.get(i));
                        }
                        ++i;
                        ++pos;
                    }
                    buffer.clear();
                    transferred += bytesRead;
                }
                this.file.updateLastModifiedAndAccessTimes();
                return transferred;
            }

            @Override
            public MappedByteBuffer map(FileChannel.MapMode mode, long position, long size) {
                throw Messages.unsupportedOperation(FileChannel.class, (String)"map");
            }

            @Override
            public synchronized FileLock lock(long position, long size, boolean shared) throws IOException {
                this.checkOpen();
                if (shared && !this.readable) {
                    throw new NonReadableChannelException();
                }
                if (!shared && !this.writeable) {
                    throw new NonWritableChannelException();
                }
                LockTable lockTable = this.file.lockTable();
                Lock lock = new Lock(this, position, size, lockTable);
                lockTable.add(lock);
                return lock;
            }

            @Override
            public FileLock tryLock(long position, long size, boolean shared) throws IOException {
                return this.lock(position, size, shared);
            }
        }

        private static final class ContentOutputStream
        extends OutputStream {
            private final File file;
            private final OnCloseAction onClose;
            private boolean open = true;

            private ContentOutputStream(File file, OnCloseAction onClose) {
                this.file = file;
                this.onClose = onClose;
            }

            @Override
            public synchronized void close() throws IOException {
                if (this.open) {
                    this.open = false;
                    if (this.onClose != null) {
                        this.onClose.run();
                    }
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void write(int b) throws IOException {
                File file = this.file;
                synchronized (file) {
                    this.file.content.add((byte)b);
                    this.file.updateLastModifiedAndAccessTimes();
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void write(byte[] b, int off, int len) throws IOException {
                if (off < 0 || off > b.length || len < 0 || off + len - b.length > 0) {
                    throw new IndexOutOfBoundsException();
                }
                File file = this.file;
                synchronized (file) {
                    for (int i = 0; i < len; ++i) {
                        this.file.content.add(b[off + i]);
                    }
                    this.file.updateLastModifiedAndAccessTimes();
                }
            }
        }

        private static final class ContentInputStream
        extends InputStream {
            private final File file;
            private final OnCloseAction onClose;
            private int pos = 0;
            private int mark = 0;
            private boolean open = true;

            private ContentInputStream(File file, OnCloseAction onClose) {
                this.file = file;
                this.onClose = onClose;
            }

            @Override
            public synchronized void close() throws IOException {
                if (this.open) {
                    this.open = false;
                    if (this.onClose != null) {
                        this.onClose.run();
                    }
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int read() throws IOException {
                File file = this.file;
                synchronized (file) {
                    int result = this.pos < this.file.content.size() ? (Byte)this.file.content.get(this.pos++) & 0xFF : -1;
                    this.file.updateLastAccessTime();
                    return result;
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                Objects.requireNonNull(b);
                if (off < 0 || len < 0 || len > b.length - off) {
                    throw new IndexOutOfBoundsException();
                }
                File file = this.file;
                synchronized (file) {
                    int size = this.file.content.size();
                    if (this.pos >= size) {
                        return -1;
                    }
                    int available = size - this.pos;
                    if (len > available) {
                        len = available;
                    }
                    if (len <= 0) {
                        return 0;
                    }
                    int i = 0;
                    while (i < len) {
                        b[off + i] = (Byte)this.file.content.get(this.pos);
                        ++i;
                        ++this.pos;
                    }
                    this.file.updateLastAccessTime();
                    return len;
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public long skip(long n) throws IOException {
                File file = this.file;
                synchronized (file) {
                    long k = this.file.content.size() - this.pos;
                    if (n < k) {
                        k = n < 0L ? 0L : n;
                    }
                    this.pos = (int)((long)this.pos + k);
                    this.file.updateLastAccessTime();
                    return k;
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int available() throws IOException {
                File file = this.file;
                synchronized (file) {
                    int result = Math.max(0, this.file.content.size() - this.pos);
                    this.file.updateLastAccessTime();
                    return result;
                }
            }

            @Override
            public synchronized void mark(int readlimit) {
                this.mark = this.pos;
            }

            @Override
            public synchronized void reset() throws IOException {
                this.pos = this.mark;
            }

            @Override
            public boolean markSupported() {
                return true;
            }
        }
    }

    static final class Directory
    extends Node {
        private final Map<String, Node> children = new TreeMap<String, Node>();

        Directory() {
        }

        @Override
        boolean isRegularFile() {
            return false;
        }

        @Override
        boolean isDirectory() {
            return true;
        }

        @Override
        boolean isSymbolicLink() {
            return false;
        }

        @Override
        long getSize() {
            return 0L;
        }

        @Override
        synchronized Node copy(boolean copyAttributes) {
            Directory copy = new Directory();
            if (copyAttributes) {
                this.copyAttributes(copy);
            }
            return copy;
        }

        synchronized boolean isEmpty() {
            return this.children.isEmpty();
        }

        synchronized Node get(String name) {
            Node node = this.children.get(name);
            if (node != null) {
                this.updateLastAccessTime();
            }
            return node;
        }

        synchronized Node add(String name, Node node) {
            Node oldNode = this.children.put(name, node);
            if (oldNode != null) {
                oldNode.parent = null;
            }
            node.parent = this;
            this.updateLastModifiedAndAccessTimes();
            return node;
        }

        synchronized Node remove(String name) {
            Node node = this.children.remove(name);
            if (node != null) {
                node.parent = null;
                this.updateLastModifiedAndAccessTimes();
            }
            return node;
        }

        synchronized void clear() {
            if (!this.children.isEmpty()) {
                this.children.clear();
                this.updateLastModifiedAndAccessTimes();
            }
        }
    }

    static abstract class Node {
        private Directory parent;
        private long lastModifiedTimestamp;
        private long lastAccessTimestamp;
        private long creationTimestamp;
        private FileTime lastModifiedFileTime;
        private FileTime lastAccessFileTime;
        private FileTime creationFileTime;
        private boolean readOnly;
        private boolean hidden;
        private MemoryFileAttributes attributes;

        Node() {
            long now;
            this.creationTimestamp = now = System.currentTimeMillis();
            this.lastModifiedTimestamp = now;
            this.lastAccessTimestamp = now;
        }

        abstract boolean isRegularFile();

        abstract boolean isDirectory();

        abstract boolean isSymbolicLink();

        abstract long getSize();

        abstract Node copy(boolean var1);

        synchronized void copyAttributes(Node target) {
            target.lastModifiedTimestamp = this.lastModifiedTimestamp;
            target.lastAccessTimestamp = this.lastAccessTimestamp;
            target.creationTimestamp = this.creationTimestamp;
            target.lastModifiedFileTime = this.lastModifiedFileTime;
            target.lastAccessFileTime = this.lastAccessFileTime;
            target.creationFileTime = this.creationFileTime;
            target.readOnly = this.readOnly;
            target.hidden = this.hidden;
        }

        synchronized void updateLastAccessTime() {
            long now = System.currentTimeMillis();
            this.setLastAccessTime(now);
        }

        synchronized void updateLastModifiedAndAccessTimes() {
            long now = System.currentTimeMillis();
            this.setLastModifiedTime(now);
            this.setLastAccessTime(now);
        }

        synchronized FileTime getLastModifiedTime() {
            if (this.lastModifiedFileTime == null) {
                this.lastModifiedFileTime = FileTime.fromMillis(this.lastModifiedTimestamp);
            }
            return this.lastModifiedFileTime;
        }

        synchronized void setLastModifiedTime(long lastModifiedTimestamp) {
            this.lastModifiedTimestamp = lastModifiedTimestamp;
            this.lastModifiedFileTime = null;
        }

        synchronized void setLastModifiedTime(FileTime lastModifiedFileTime) {
            this.lastModifiedTimestamp = lastModifiedFileTime.toMillis();
            this.lastModifiedFileTime = lastModifiedFileTime;
        }

        synchronized FileTime getLastAccessTime() {
            if (this.lastAccessFileTime == null) {
                this.lastAccessFileTime = FileTime.fromMillis(this.lastAccessTimestamp);
            }
            return this.lastAccessFileTime;
        }

        synchronized void setLastAccessTime(long lastAccessTimestamp) {
            this.lastAccessTimestamp = lastAccessTimestamp;
            this.lastAccessFileTime = null;
        }

        synchronized void setLastAccessTime(FileTime lastAccessFileTime) {
            this.lastAccessTimestamp = lastAccessFileTime.toMillis();
            this.lastAccessFileTime = lastAccessFileTime;
        }

        synchronized FileTime getCreationTime() {
            if (this.creationFileTime == null) {
                this.creationFileTime = FileTime.fromMillis(this.creationTimestamp);
            }
            return this.creationFileTime;
        }

        synchronized void setCreationTime(long creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            this.creationFileTime = null;
        }

        synchronized void setCreationTime(FileTime creationFileTime) {
            this.creationTimestamp = creationFileTime.toMillis();
            this.creationFileTime = creationFileTime;
        }

        synchronized boolean isReadOnly() {
            return this.readOnly;
        }

        synchronized void setReadOnly(boolean readOnly) {
            this.readOnly = readOnly;
        }

        synchronized boolean isHidden() {
            return this.hidden;
        }

        synchronized void setHidden(boolean hidden) {
            this.hidden = hidden;
        }

        synchronized MemoryFileAttributes getAttributes() {
            if (this.attributes == null) {
                this.attributes = new FileAttributes();
            }
            return this.attributes;
        }

        private final class FileAttributes
        implements MemoryFileAttributes {
            private FileAttributes() {
            }

            @Override
            public FileTime lastModifiedTime() {
                return Node.this.getLastModifiedTime();
            }

            @Override
            public FileTime lastAccessTime() {
                return Node.this.getLastAccessTime();
            }

            @Override
            public FileTime creationTime() {
                return Node.this.getCreationTime();
            }

            @Override
            public boolean isRegularFile() {
                return Node.this.isRegularFile();
            }

            @Override
            public boolean isDirectory() {
                return Node.this.isDirectory();
            }

            @Override
            public boolean isSymbolicLink() {
                return Node.this.isSymbolicLink();
            }

            @Override
            public boolean isOther() {
                return false;
            }

            @Override
            public long size() {
                return Node.this.getSize();
            }

            @Override
            public Object fileKey() {
                return null;
            }

            @Override
            public boolean isReadOnly() {
                return Node.this.isReadOnly();
            }

            @Override
            public boolean isHidden() {
                return Node.this.isHidden();
            }
        }
    }

    private final class MemoryPathDirectoryStream
    extends AbstractDirectoryStream<Path> {
        private final MemoryPath path;
        private Iterator<String> names;

        private MemoryPathDirectoryStream(MemoryPath path, DirectoryStream.Filter<? super Path> filter) {
            super(filter);
            this.path = Objects.requireNonNull(path);
        }

        protected void setupIteration() {
            Node node = null;
            try {
                Directory parent = MemoryFileStore.this.findParentNode(this.path);
                node = MemoryFileStore.this.findNode(parent, this.path);
                if (node instanceof Link) {
                    MemoryPath resolvedPath = MemoryFileStore.this.resolveLink((Link)node, this.path);
                    parent = MemoryFileStore.this.findParentNode(resolvedPath);
                    node = MemoryFileStore.this.findNode(parent, resolvedPath);
                }
            }
            catch (FileSystemException parent) {
                // empty catch block
            }
            if (node instanceof Directory) {
                Directory directory = (Directory)node;
                this.names = new ArrayList(directory.children.keySet()).iterator();
            } else {
                this.names = Collections.emptyIterator();
            }
        }

        protected Path getNext() {
            if (this.names.hasNext()) {
                String name = this.names.next();
                return this.path.resolve(name);
            }
            return null;
        }
    }
}

