/*
 * Decompiled with CFR 0.152.
 */
package com.github.signaflo.timeseries.model.arima;

import com.github.signaflo.math.operations.DoubleFunctions;
import com.github.signaflo.math.stats.Statistics;
import com.github.signaflo.timeseries.model.arima.Arima;
import com.github.signaflo.timeseries.model.arima.ArimaOrder;
import java.text.DecimalFormat;
import java.util.Arrays;

public class ArimaCoefficients {
    private static final double EPSILON = Math.ulp(1.0);
    private final double[] arCoeffs;
    private final double[] maCoeffs;
    private final double[] sarCoeffs;
    private final double[] smaCoeffs;
    private final int d;
    private final int D;
    private final int seasonalFrequency;
    private final double mean;
    private final double intercept;
    private final double drift;

    ArimaCoefficients(double[] arCoeffs, double[] maCoeffs, double[] sarCoeffs, double[] smaCoeffs, int d, int D, double mean, double drift, int seasonalFrequency) {
        this.arCoeffs = (double[])arCoeffs.clone();
        this.maCoeffs = (double[])maCoeffs.clone();
        this.sarCoeffs = (double[])sarCoeffs.clone();
        this.smaCoeffs = (double[])smaCoeffs.clone();
        this.d = d;
        this.D = D;
        this.seasonalFrequency = seasonalFrequency;
        this.mean = mean;
        this.intercept = ArimaCoefficients.meanToIntercept(ArimaCoefficients.expandArCoefficients(arCoeffs, sarCoeffs, seasonalFrequency), mean);
        this.drift = drift;
    }

    private ArimaCoefficients(Builder builder) {
        this.arCoeffs = (double[])builder.arCoeffs.clone();
        this.maCoeffs = (double[])builder.maCoeffs.clone();
        this.sarCoeffs = (double[])builder.sarCoeffs.clone();
        this.smaCoeffs = (double[])builder.smaCoeffs.clone();
        this.d = builder.d;
        this.D = builder.D;
        this.seasonalFrequency = builder.seasonalFrequency;
        this.mean = builder.mean;
        this.intercept = ArimaCoefficients.meanToIntercept(ArimaCoefficients.expandArCoefficients(this.arCoeffs, this.sarCoeffs, builder.seasonalFrequency), this.mean);
        this.drift = builder.drift;
    }

    static double[] expandArCoefficients(double[] arCoeffs, double[] sarCoeffs, int seasonalFrequency) {
        double[] arSarCoeffs = new double[arCoeffs.length + sarCoeffs.length * seasonalFrequency];
        System.arraycopy(arCoeffs, 0, arSarCoeffs, 0, arCoeffs.length);
        for (int i = 0; i < sarCoeffs.length; ++i) {
            arSarCoeffs[(i + 1) * seasonalFrequency - 1] = sarCoeffs[i];
            for (int j = 0; j < arCoeffs.length; ++j) {
                arSarCoeffs[(i + 1) * seasonalFrequency + j] = -sarCoeffs[i] * arCoeffs[j];
            }
        }
        return arSarCoeffs;
    }

    static double[] expandMaCoefficients(double[] maCoeffs, double[] smaCoeffs, int seasonalFrequency) {
        double[] maSmaCoeffs = new double[maCoeffs.length + smaCoeffs.length * seasonalFrequency];
        System.arraycopy(maCoeffs, 0, maSmaCoeffs, 0, maCoeffs.length);
        for (int i = 0; i < smaCoeffs.length; ++i) {
            maSmaCoeffs[(i + 1) * seasonalFrequency - 1] = smaCoeffs[i];
            for (int j = 0; j < maCoeffs.length; ++j) {
                maSmaCoeffs[(i + 1) * seasonalFrequency + j] = smaCoeffs[i] * maCoeffs[j];
            }
        }
        return maSmaCoeffs;
    }

    private static double meanToIntercept(double[] autoRegressiveCoefficients, double mean) {
        return mean * (1.0 - Statistics.sumOf((double[])autoRegressiveCoefficients));
    }

    static double interceptToMean(double[] autoRegressiveCoefficients, double intercept) {
        return intercept / (1.0 - Statistics.sumOf((double[])autoRegressiveCoefficients));
    }

    double[] getAllMovingAverageCoefficients() {
        return ArimaCoefficients.expandMaCoefficients(this.maCoeffs(), this.seasonalMACoeffs(), this.seasonalFrequency);
    }

    double[] getAllAutoRegressiveCoefficients() {
        return ArimaCoefficients.expandArCoefficients(this.arCoeffs(), this.seasonalARCoeffs(), this.seasonalFrequency);
    }

    final double[] arCoeffs() {
        return (double[])this.arCoeffs.clone();
    }

    final double[] maCoeffs() {
        return (double[])this.maCoeffs.clone();
    }

    final double[] seasonalARCoeffs() {
        return (double[])this.sarCoeffs.clone();
    }

    final double[] seasonalMACoeffs() {
        return (double[])this.smaCoeffs.clone();
    }

    final int d() {
        return this.d;
    }

    final int D() {
        return this.D;
    }

    final int seasonalFrequency() {
        return this.seasonalFrequency;
    }

    final double mean() {
        return this.mean;
    }

    final double drift() {
        return this.drift;
    }

    final double intercept() {
        return this.intercept;
    }

    final double[] getAllCoeffs() {
        if (Math.abs(this.mean) < EPSILON && Math.abs(this.drift) < EPSILON) {
            return DoubleFunctions.combine((double[][])new double[][]{this.arCoeffs, this.maCoeffs, this.sarCoeffs, this.smaCoeffs});
        }
        return DoubleFunctions.append((double[])DoubleFunctions.append((double[])DoubleFunctions.combine((double[][])new double[][]{this.arCoeffs, this.maCoeffs, this.sarCoeffs, this.smaCoeffs}), (double)this.mean), (double)this.drift);
    }

    final boolean isSeasonal() {
        return this.D > 0 || this.sarCoeffs.length > 0 || this.smaCoeffs.length > 0;
    }

    ArimaOrder extractModelOrder() {
        Arima.Constant constant = Math.abs(this.mean) > EPSILON ? Arima.Constant.INCLUDE : Arima.Constant.EXCLUDE;
        Arima.Drift drift = Math.abs(this.drift) > EPSILON ? Arima.Drift.INCLUDE : Arima.Drift.EXCLUDE;
        return new ArimaOrder(this.arCoeffs.length, this.d, this.maCoeffs.length, this.sarCoeffs.length, this.D, this.smaCoeffs.length, constant, drift);
    }

    double[] getRegressors(ArimaOrder order) {
        double[] regressors = new double[order.npar() - order.sumARMA()];
        if (order.constant().include()) {
            regressors[0] = this.mean;
        }
        if (order.drift().include()) {
            regressors[order.constant().asInt()] = this.drift;
        }
        return regressors;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String toString() {
        String newLine = System.lineSeparator();
        DecimalFormat numFormatter = new DecimalFormat("#0.0000");
        StringBuilder sb = new StringBuilder();
        if (this.arCoeffs.length > 0) {
            sb.append(newLine).append("autoregressive:");
            for (double d : this.arCoeffs) {
                sb.append(" ").append(numFormatter.format(d));
            }
        }
        if (this.maCoeffs.length > 0) {
            sb.append(newLine).append("moving-average:");
            for (double d : this.maCoeffs) {
                sb.append(" ").append(numFormatter.format(d));
            }
        }
        if (this.sarCoeffs.length > 0) {
            sb.append(newLine).append("seasonal autoregressive:");
            for (double d : this.sarCoeffs) {
                sb.append(" ").append(numFormatter.format(d));
            }
        }
        if (this.smaCoeffs.length > 0) {
            sb.append(newLine).append("seasonal moving-average:");
            for (double d : this.smaCoeffs) {
                sb.append(" ").append(numFormatter.format(d));
            }
        }
        if (Math.abs(this.mean) > EPSILON) {
            sb.append(newLine).append("mean: ").append(numFormatter.format(this.mean));
            sb.append(newLine).append("intercept: ").append(numFormatter.format(this.intercept));
        } else {
            sb.append(newLine).append("zero mean");
        }
        if (Math.abs(this.drift) > EPSILON) {
            sb.append(newLine).append("drift: ").append(numFormatter.format(this.drift));
        }
        if (this.d > 0) {
            sb.append(newLine).append(this.d).append(" non-seasonal difference").append(this.d > 1 ? "s" : "");
        }
        if (this.D > 0) {
            sb.append(newLine).append(this.D).append(" seasonal difference").append(this.D > 1 ? "s" : "");
        }
        return sb.toString();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ArimaCoefficients that = (ArimaCoefficients)o;
        if (this.d != that.d) {
            return false;
        }
        if (this.D != that.D) {
            return false;
        }
        if (Double.compare(that.mean, this.mean) != 0) {
            return false;
        }
        if (Double.compare(that.intercept, this.intercept) != 0) {
            return false;
        }
        if (Double.compare(that.drift, this.drift) != 0) {
            return false;
        }
        if (!Arrays.equals(this.arCoeffs, that.arCoeffs)) {
            return false;
        }
        if (!Arrays.equals(this.maCoeffs, that.maCoeffs)) {
            return false;
        }
        if (!Arrays.equals(this.sarCoeffs, that.sarCoeffs)) {
            return false;
        }
        return Arrays.equals(this.smaCoeffs, that.smaCoeffs);
    }

    public int hashCode() {
        int result = Arrays.hashCode(this.arCoeffs);
        result = 31 * result + Arrays.hashCode(this.maCoeffs);
        result = 31 * result + Arrays.hashCode(this.sarCoeffs);
        result = 31 * result + Arrays.hashCode(this.smaCoeffs);
        result = 31 * result + this.d;
        result = 31 * result + this.D;
        long temp = Double.doubleToLongBits(this.mean);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.intercept);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.drift);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        return result;
    }

    public static class Builder {
        private double[] arCoeffs = new double[0];
        private double[] maCoeffs = new double[0];
        private double[] sarCoeffs = new double[0];
        private double[] smaCoeffs = new double[0];
        private int d = 0;
        private int D = 0;
        private int seasonalFrequency = 1;
        private double mean = 0.0;
        private double drift = 0.0;

        private Builder() {
        }

        public Builder setARCoeffs(double ... arCoeffs) {
            this.arCoeffs = (double[])arCoeffs.clone();
            return this;
        }

        public Builder setSeasonalARCoeffs(double ... sarCoeffs) {
            this.sarCoeffs = (double[])sarCoeffs.clone();
            return this;
        }

        public Builder setMACoeffs(double ... maCoeffs) {
            this.maCoeffs = (double[])maCoeffs.clone();
            return this;
        }

        public Builder setSeasonalMACoeffs(double ... smaCoeffs) {
            this.smaCoeffs = (double[])smaCoeffs.clone();
            return this;
        }

        public Builder setDifferences(int d) {
            this.d = d;
            return this;
        }

        public Builder setSeasonalDifferences(int D) {
            this.D = D;
            return this;
        }

        public Builder setSeasonalFrequency(int seasonalFrequency) {
            this.seasonalFrequency = seasonalFrequency;
            return this;
        }

        public Builder setMean(double mean) {
            this.mean = mean;
            return this;
        }

        public Builder setDrift(double drift) {
            this.drift = drift;
            return this;
        }

        public ArimaCoefficients build() {
            this.verifyState(this);
            return new ArimaCoefficients(this);
        }

        private void verifyState(Builder builder) {
            if (builder.d + builder.D > 0 && builder.mean != 0.0) {
                String message = "An ARIMA model cannot have both differencing and a mean, though it might possibly include a drift term.";
                throw new IllegalStateException(message);
            }
            if (builder.d + builder.D > 1 && builder.drift != 0.0) {
                String message = "An ARIMA model with more than one degree of differencing may not include a drift term.";
                throw new IllegalStateException(message);
            }
        }
    }
}

