/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ShortToLongFunction;
import com.gs.collections.api.block.function.primitive.LongFunction;
import com.gs.collections.api.block.function.primitive.LongFunction0;
import com.gs.collections.api.block.function.primitive.LongToLongFunction;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortLongPredicate;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.procedure.Procedure;
import com.gs.collections.api.block.procedure.Procedure2;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortLongProcedure;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.block.procedure.primitive.ObjectIntProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableLongCollection;
import com.gs.collections.api.collection.primitive.MutableLongCollection;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.map.primitive.ShortLongMap;
import com.gs.collections.api.map.primitive.ImmutableShortLongMap;
import com.gs.collections.api.map.primitive.MutableShortLongMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ShortSet;
import com.gs.collections.api.set.primitive.LongSet;
import com.gs.collections.api.set.primitive.ImmutableShortSet;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.api.tuple.primitive.ShortLongPair;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.block.factory.primitive.ShortPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedLongCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableLongCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.LongLists;
import com.gs.collections.impl.factory.primitive.ShortLongMaps;
import com.gs.collections.impl.factory.primitive.ShortSets;
import com.gs.collections.impl.lazy.AbstractLazyIterable;
import com.gs.collections.impl.lazy.primitive.AbstractLazyShortIterable;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedShortSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableShortSet;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;
import com.gs.collections.impl.tuple.primitive.PrimitiveTuples;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class ShortLongHashMap implements MutableShortLongMap, Externalizable
{
    static final long EMPTY_VALUE = 0L;
    private static final long serialVersionUID = 1L;
    private static final short EMPTY_KEY = (short) 0;
    private static final short REMOVED_KEY = (short) 1;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private short[] keys;
    private long[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public ShortLongHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ShortLongHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public ShortLongHashMap(ShortLongMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static ShortLongHashMap newWithKeysValues(short key1, long value1)
    {
        return new ShortLongHashMap(1).withKeyValue(key1, value1);
    }

    public static ShortLongHashMap newWithKeysValues(short key1, long value1, short key2, long value2)
    {
        return new ShortLongHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static ShortLongHashMap newWithKeysValues(short key1, long value1, short key2, long value2, short key3, long value3)
    {
        return new ShortLongHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static ShortLongHashMap newWithKeysValues(short key1, long value1, short key2, long value2, short key3, long value3, short key4, long value4)
    {
        return new ShortLongHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ShortLongMap))
        {
            return false;
        }

        ShortLongMap other = (ShortLongMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            short key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) EMPTY_KEY ^ (int) (this.sentinelValues.zeroValue ^ this.sentinelValues.zeroValue >>> 32);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) REMOVED_KEY ^ (int) (this.sentinelValues.oneValue ^ this.sentinelValues.oneValue >>> 32);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) this.keys[i] ^ (int) (this.values[i] ^ this.values[i] >>> 32);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        StringBuilder appendable = new StringBuilder();

        appendable.append("{");

        boolean first = true;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                first = false;
            }
            if (this.sentinelValues.containsOneKey)
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                first = false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            short key = this.keys[i];
            if (isNonSentinel(key))
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                first = false;
            }
        }
        appendable.append("}");

        return appendable.toString();
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                short key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public LongIterator longIterator()
    {
        return new InternalLongIterator();
    }

    public long[] toArray()
    {
        long[] array = new long[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(long value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(long... source)
    {
        for (long each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(LongIterable source)
    {
        return source.allSatisfy(new LongPredicate()
        {
            public boolean accept(long value)
            {
                return ShortLongHashMap.this.contains(value);
            }
        });
    }

    public void forEach(LongProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableLongCollection select(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableLongCollection reject(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public long detectIfNone(LongPredicate predicate, long value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(LongPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectLongToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableLongList toList()
    {
        return LongArrayList.newList(this);
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSet(this);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBag(this);
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(short key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.addEmptyKeyValue(value);
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.addRemovedKeyValue(value);
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        this.addKeyValueAtIndex(key, value, index);
    }

    public void putAll(ShortLongMap map)
    {
        map.forEachKeyValue(new ShortLongProcedure()
        {
            public void value(short key, long value)
            {
                ShortLongHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(short key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            this.removeEmptyKey();
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            this.removeRemovedKey();
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(short key)
    {
        this.removeKey(key);
    }

    public long removeKeyIfAbsent(short key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            long oldValue = this.sentinelValues.zeroValue;
            this.removeEmptyKey();
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            long oldValue = this.sentinelValues.oneValue;
            this.removeRemovedKey();
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            long oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public long getIfAbsentPut(short key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long getIfAbsentPut(short key, LongFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.value();
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.value();
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.value();
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.value();
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        long value = function.value();
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public <P> long getIfAbsentPutWith(short key, LongFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.longValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.longValueOf(parameter);
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.longValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.longValueOf(parameter);
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        long value = function.longValueOf(parameter);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long getIfAbsentPutWithKey(short key, ShortToLongFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.valueOf(key);
            this.addEmptyKeyValue(value);
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(value);
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.valueOf(key);
            this.addRemovedKeyValue(value);
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        long value = function.valueOf(key);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long addToValue(short key, long toBeAdded)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(toBeAdded);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue += toBeAdded;
            }
            else
            {
                this.addEmptyKeyValue(toBeAdded);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(toBeAdded);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue += toBeAdded;
            }
            else
            {
                this.addRemovedKeyValue(toBeAdded);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] += toBeAdded;
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, toBeAdded, index);
        return this.values[index];
    }

    private void addKeyValueAtIndex(short key, long value, int index)
    {
        if (this.keys[index] == REMOVED_KEY)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        this.values[index] = value;
        this.occupiedWithData++;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    private void addEmptyKeyValue(long value)
    {
        this.sentinelValues.containsZeroKey = true;
        this.sentinelValues.zeroValue = value;
    }

    private void removeEmptyKey()
    {
        if (this.sentinelValues.containsOneKey)
        {
            this.sentinelValues.containsZeroKey = false;
            this.sentinelValues.zeroValue = EMPTY_VALUE;
        }
        else
        {
            this.sentinelValues = null;
        }
    }

    private void addRemovedKeyValue(long value)
    {
        this.sentinelValues.containsOneKey = true;
        this.sentinelValues.oneValue = value;
    }

    private void removeRemovedKey()
    {
        if (this.sentinelValues.containsZeroKey)
        {
            this.sentinelValues.containsOneKey = false;
            this.sentinelValues.oneValue = EMPTY_VALUE;
        }
        else
        {
            this.sentinelValues = null;
        }
    }

    public long updateValue(short key, long initialValueIfAbsent, LongToLongFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addEmptyKeyValue(function.valueOf(initialValueIfAbsent));
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.addEmptyKeyValue(function.valueOf(initialValueIfAbsent));
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.addRemovedKeyValue(function.valueOf(initialValueIfAbsent));
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.addRemovedKeyValue(function.valueOf(initialValueIfAbsent));
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        long value = function.valueOf(initialValueIfAbsent);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public ShortLongHashMap withKeyValue(short key1, long value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ShortLongHashMap withKeysValues(short key1, long value1, short key2, long value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ShortLongHashMap withKeysValues(short key1, long value1, short key2, long value2, short key3, long value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ShortLongHashMap withKeysValues(short key1, long value1, short key2, long value2, short key3, long value3, short key4, long value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ShortLongHashMap withoutKey(short key)
    {
        this.removeKey(key);
        return this;
    }

    public ShortLongHashMap withoutAllKeys(ShortIterable keys)
    {
        keys.forEach(new ShortProcedure()
        {
            public void value(short key)
            {
                ShortLongHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableShortLongMap asUnmodifiable()
    {
        return new UnmodifiableShortLongMap(this);
    }

    public MutableShortLongMap asSynchronized()
    {
        return new SynchronizedShortLongMap(this);
    }

    public ImmutableShortLongMap toImmutable()
    {
        return ShortLongMaps.immutable.ofAll(this);
    }

    public long get(short key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public long getIfAbsent(short key, long ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public long getOrThrow(short key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(short key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(long value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(LongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(ShortProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (short key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(ShortLongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyShortIterable keysView()
    {
        return new KeysView();
    }

    public RichIterable<ShortLongPair> keyValuesView()
    {
        return new KeyValuesView();
    }

    public ShortLongHashMap select(ShortLongPredicate predicate)
    {
        ShortLongHashMap result = new ShortLongHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public ShortLongHashMap reject(ShortLongPredicate predicate)
    {
        ShortLongHashMap result = new ShortLongHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public long max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        LongIterator iterator = this.longIterator();
        long max = iterator.next();
        while (iterator.hasNext())
        {
            long value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public long maxIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public long min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        LongIterator iterator = this.longIterator();
        long min = iterator.next();
        while (iterator.hasNext())
        {
            long value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public long minIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        long[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            long first = sortedArray[middleIndex];
            long second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public long[] toSortedArray()
    {
        long[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableLongList toSortedList()
    {
        return LongArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeShort(EMPTY_KEY);
                out.writeLong(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeShort(REMOVED_KEY);
                out.writeLong(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeShort(this.keys[i]);
                out.writeLong(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readShort(), in.readLong());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        short[] old = this.keys;
        long[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(short element)
    {
        int index = this.spread(element);
        short keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(short element)
    {
        int code = (int) element;
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new short[sizeToAllocate];
        this.values = new long[sizeToAllocate];
    }

    private static boolean isEmptyKey(short key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(short key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(short key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private long zeroValue;
        private long oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(long value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalLongIterator implements LongIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ShortLongHashMap.this.size();
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ShortLongHashMap.this.containsKey(EMPTY_KEY))
                {
                    return ShortLongHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ShortLongHashMap.this.containsKey(REMOVED_KEY))
                {
                    return ShortLongHashMap.this.get(REMOVED_KEY);
                }
            }

            short[] keys = ShortLongHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            long result = ShortLongHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView extends AbstractLazyShortIterable
    {
        public ShortIterator shortIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ShortProcedure procedure)
        {
            ShortLongHashMap.this.forEachKey(procedure);
        }
    }

    private class KeySetIterator implements ShortIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ShortLongHashMap.this.size();
        }

        public short next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ShortLongHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ShortLongHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            short[] keys = ShortLongHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            short result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableShortSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableShortSet
    {
        public ShortIterator shortIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ShortProcedure procedure)
        {
            ShortLongHashMap.this.forEachKey(procedure);
        }

        public int count(ShortPredicate predicate)
        {
            int count = 0;
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(ShortPredicate predicate)
        {
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(ShortPredicate predicate)
        {
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(ShortPredicate predicate)
        {
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(short element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(short... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(ShortIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(short key)
        {
            int oldSize = ShortLongHashMap.this.size();
            ShortLongHashMap.this.removeKey(key);
            return oldSize != ShortLongHashMap.this.size();
        }

        public boolean removeAll(ShortIterable source)
        {
            int oldSize = ShortLongHashMap.this.size();
            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                ShortLongHashMap.this.removeKey(iterator.next());
            }
            return oldSize != ShortLongHashMap.this.size();
        }

        public boolean removeAll(short... source)
        {
            int oldSize = ShortLongHashMap.this.size();
            for (short item : source)
            {
                ShortLongHashMap.this.removeKey(item);
            }
            return oldSize != ShortLongHashMap.this.size();
        }

        public boolean retainAll(ShortIterable source)
        {
            int oldSize = this.size();
            final ShortSet sourceSet = source instanceof ShortSet ? (ShortSet) source : source.toSet();
            ShortLongHashMap retained = ShortLongHashMap.this.select(new ShortLongPredicate()
            {
                public boolean accept(short key, long value)
                {
                    return sourceSet.contains(key);
                }
            });
            if (retained.size() != oldSize)
            {
                ShortLongHashMap.this.keys = retained.keys;
                ShortLongHashMap.this.values = retained.values;
                ShortLongHashMap.this.sentinelValues = retained.sentinelValues;
                ShortLongHashMap.this.occupiedWithData = retained.occupiedWithData;
                ShortLongHashMap.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(short... source)
        {
            return this.retainAll(ShortHashSet.newSetWith(source));
        }

        public void clear()
        {
            ShortLongHashMap.this.clear();
        }

        public MutableShortSet select(ShortPredicate predicate)
        {
            MutableShortSet result = new ShortHashSet();
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableShortSet reject(ShortPredicate predicate)
        {
            MutableShortSet result = new ShortHashSet();
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableShortSet with(short element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableShortSet without(short element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableShortSet withAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableShortSet withoutAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public short detectIfNone(ShortPredicate predicate, short ifNone)
        {
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(ShortToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableShortSet asUnmodifiable()
        {
            return UnmodifiableShortSet.of(this);
        }

        public MutableShortSet asSynchronized()
        {
            return SynchronizedShortSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (short key : ShortLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public short max()
        {
            if (ShortLongHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            short max = 0;
            boolean isMaxSet = false;

            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]) && (!isMaxSet || max < ShortLongHashMap.this.keys[i]))
                {
                    max = ShortLongHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public short maxIfEmpty(short defaultValue)
        {
            if (ShortLongHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public short min()
        {
            if (ShortLongHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            short min = 0;
            boolean isMinSet = false;

            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]) && (!isMinSet || ShortLongHashMap.this.keys[i] < min))
                {
                    min = ShortLongHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public short minIfEmpty(short defaultValue)
        {
            if (ShortLongHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            short[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                short first = sortedArray[middleIndex];
                short second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public short[] toSortedArray()
        {
            short[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableShortList toSortedList()
        {
            return ShortArrayList.newList(this).sortThis();
        }

        public short[] toArray()
        {
            int size = ShortLongHashMap.this.size();
            final short[] result = new short[size];
            ShortLongHashMap.this.forEachKey(new ShortProcedure()
            {
                private int index;

                public void value(short each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(short value)
        {
            return ShortLongHashMap.this.containsKey(value);
        }

        public boolean containsAll(short... source)
        {
            for (short item : source)
            {
                if (!ShortLongHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ShortIterable source)
        {
            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                if (!ShortLongHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableShortList toList()
        {
            return ShortArrayList.newList(this);
        }

        public MutableShortSet toSet()
        {
            return ShortHashSet.newSet(this);
        }

        public MutableShortBag toBag()
        {
            return ShortHashBag.newBag(this);
        }

        public LazyShortIterable asLazy()
        {
            return new LazyShortIterableAdapter(this);
        }

        public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, ShortLongHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public ShortSet freeze()
        {
            throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".freeze() not implemented yet");
        }

        public ImmutableShortSet toImmutable()
        {
            return ShortSets.immutable.withAll(this);
        }

        public int size()
        {
            return ShortLongHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ShortLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ShortLongHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof ShortSet))
            {
                return false;
            }

            ShortSet other = (ShortSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) EMPTY_KEY;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) REMOVED_KEY;
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]))
                {
                    result += (int) ShortLongHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ShortLongHashMap.this.sentinelValues != null)
                {
                    if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (short key : ShortLongHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableLongCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableLongCollection
    {
        public void clear()
        {
            ShortLongHashMap.this.clear();
        }

        public MutableLongCollection select(LongPredicate predicate)
        {
            return ShortLongHashMap.this.select(predicate);
        }

        public MutableLongCollection reject(LongPredicate predicate)
        {
            return ShortLongHashMap.this.reject(predicate);
        }

        public long detectIfNone(LongPredicate predicate, long ifNone)
        {
            return ShortLongHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
        {
            return ShortLongHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
        {
            return ShortLongHashMap.this.injectInto(injectedValue, function);
        }

        public long sum()
        {
            return ShortLongHashMap.this.sum();
        }

        public long max()
        {
            return ShortLongHashMap.this.max();
        }

        public long maxIfEmpty(long defaultValue)
        {
            return ShortLongHashMap.this.maxIfEmpty(defaultValue);
        }

        public long min()
        {
            return ShortLongHashMap.this.min();
        }

        public long minIfEmpty(long defaultValue)
        {
            return ShortLongHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return ShortLongHashMap.this.average();
        }

        public double median()
        {
            return ShortLongHashMap.this.median();
        }

        public long[] toSortedArray()
        {
            return ShortLongHashMap.this.toSortedArray();
        }

        public MutableLongList toSortedList()
        {
            return ShortLongHashMap.this.toSortedList();
        }

        public MutableLongCollection with(long element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection without(long element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection withAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection withoutAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection asUnmodifiable()
        {
            return UnmodifiableLongCollection.of(this);
        }

        public MutableLongCollection asSynchronized()
        {
            return SynchronizedLongCollection.of(this);
        }

        public ImmutableLongCollection toImmutable()
        {
            return LongLists.immutable.withAll(this);
        }

        public boolean contains(long value)
        {
            return ShortLongHashMap.this.containsValue(value);
        }

        public boolean containsAll(long... source)
        {
            return ShortLongHashMap.this.containsAll(source);
        }

        public boolean containsAll(LongIterable source)
        {
            return ShortLongHashMap.this.containsAll(source);
        }

        public MutableLongList toList()
        {
            return ShortLongHashMap.this.toList();
        }

        public MutableLongSet toSet()
        {
            return ShortLongHashMap.this.toSet();
        }

        public MutableLongBag toBag()
        {
            return ShortLongHashMap.this.toBag();
        }

        public LazyLongIterable asLazy()
        {
            return new LazyLongIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return ShortLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ShortLongHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (ShortLongHashMap.this.sentinelValues != null)
                {
                    if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(ShortLongHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ShortLongHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
                {
                    short key = ShortLongHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ShortLongHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public LongIterator longIterator()
        {
            return ShortLongHashMap.this.longIterator();
        }

        public void forEach(LongProcedure procedure)
        {
            ShortLongHashMap.this.forEach(procedure);
        }

        public int count(LongPredicate predicate)
        {
            return ShortLongHashMap.this.count(predicate);
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            return ShortLongHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            return ShortLongHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            return ShortLongHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(long element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(long... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(LongIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(long item)
        {
            int oldSize = ShortLongHashMap.this.size();

            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey && item == ShortLongHashMap.this.sentinelValues.zeroValue)
                {
                    ShortLongHashMap.this.removeKey(EMPTY_KEY);
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey && item == ShortLongHashMap.this.sentinelValues.oneValue)
                {
                    ShortLongHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]) && item == ShortLongHashMap.this.values[i])
                {
                    ShortLongHashMap.this.removeKey(ShortLongHashMap.this.keys[i]);
                }
            }
            return oldSize != ShortLongHashMap.this.size();
        }

        public boolean removeAll(LongIterable source)
        {
            int oldSize = ShortLongHashMap.this.size();

            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ShortLongHashMap.this.size();
        }

        public boolean removeAll(long... source)
        {
            int oldSize = ShortLongHashMap.this.size();

            for (long item : source)
            {
                this.remove(item);
            }
            return oldSize != ShortLongHashMap.this.size();
        }

        public boolean retainAll(LongIterable source)
        {
            int oldSize = this.size();
            final LongSet sourceSet = source instanceof LongSet ? (LongSet) source : source.toSet();
            ShortLongHashMap retained = ShortLongHashMap.this.select(new ShortLongPredicate()
            {
                public boolean accept(short key, long value)
                {
                    return sourceSet.contains(value);
                }
            });
            if (retained.size() != oldSize)
            {
                ShortLongHashMap.this.keys = retained.keys;
                ShortLongHashMap.this.values = retained.values;
                ShortLongHashMap.this.sentinelValues = retained.sentinelValues;
                ShortLongHashMap.this.occupiedWithData = retained.occupiedWithData;
                ShortLongHashMap.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(long... source)
        {
            return this.retainAll(LongHashSet.newSetWith(source));
        }

        public int size()
        {
            return ShortLongHashMap.this.size();
        }

        public long[] toArray()
        {
            return ShortLongHashMap.this.toArray();
        }
    }

    private class KeyValuesView extends AbstractLazyIterable<ShortLongPair>
    {
        public void forEach(Procedure<? super ShortLongPair> procedure)
        {
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    procedure.value(PrimitiveTuples.pair(EMPTY_KEY, ShortLongHashMap.this.sentinelValues.zeroValue));
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    procedure.value(PrimitiveTuples.pair(REMOVED_KEY, ShortLongHashMap.this.sentinelValues.oneValue));
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ShortLongHashMap.this.keys[i], ShortLongHashMap.this.values[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super ShortLongPair> objectIntProcedure)
        {
            int index = 0;
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(EMPTY_KEY, ShortLongHashMap.this.sentinelValues.zeroValue), index);
                    index++;
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(REMOVED_KEY, ShortLongHashMap.this.sentinelValues.oneValue), index);
                    index++;
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]))
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(ShortLongHashMap.this.keys[i], ShortLongHashMap.this.values[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super ShortLongPair, ? super P> procedure, P parameter)
        {
            if (ShortLongHashMap.this.sentinelValues != null)
            {
                if (ShortLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    procedure.value(PrimitiveTuples.pair(EMPTY_KEY, ShortLongHashMap.this.sentinelValues.zeroValue), parameter);
                }
                if (ShortLongHashMap.this.sentinelValues.containsOneKey)
                {
                    procedure.value(PrimitiveTuples.pair(REMOVED_KEY, ShortLongHashMap.this.sentinelValues.oneValue), parameter);
                }
            }
            for (int i = 0; i < ShortLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ShortLongHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ShortLongHashMap.this.keys[i], ShortLongHashMap.this.values[i]), parameter);
                }
            }
        }

        public Iterator<ShortLongPair> iterator()
        {
            return new InternalKeyValuesIterator();
        }

        public class InternalKeyValuesIterator implements Iterator<ShortLongPair>
        {
            private int count;
            private int position;
            private boolean handledZero;
            private boolean handledOne;

            public ShortLongPair next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException("next() called, but the iterator is exhausted");
                }
                this.count++;

                if (!this.handledZero)
                {
                    this.handledZero = true;
                    if (ShortLongHashMap.this.containsKey(EMPTY_KEY))
                    {
                        return PrimitiveTuples.pair(EMPTY_KEY, ShortLongHashMap.this.sentinelValues.zeroValue);
                    }
                }
                if (!this.handledOne)
                {
                    this.handledOne = true;
                    if (ShortLongHashMap.this.containsKey(REMOVED_KEY))
                    {
                        return PrimitiveTuples.pair(REMOVED_KEY, ShortLongHashMap.this.sentinelValues.oneValue);
                    }
                }

                short[] keys = ShortLongHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                ShortLongPair result = PrimitiveTuples.pair(keys[this.position], ShortLongHashMap.this.values[this.position]);
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ShortLongHashMap.this.size();
            }
        }
    }
}
