/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.primitive.ImmutableDoubleSet;
import com.gs.collections.api.set.primitive.DoubleSet;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleIntProcedure;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.set.mutable.UnifiedSet;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;
import com.gs.collections.impl.factory.primitive.DoubleSets;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import net.jcip.annotations.Immutable;

/**
 * ImmutableDoubleSingletonSet is an optimization for {@link ImmutableDoubleSet} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonSet.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableDoubleSingletonSet implements ImmutableDoubleSet, Serializable
{
    private static final long serialVersionUID = 1L;

    private final double element;

    ImmutableDoubleSingletonSet (double element)
    {
        this.element = element;
    }

    public ImmutableDoubleSet newWith(double element)
    {
        return DoubleSets.immutable.with(this.element, element);
    }

    public ImmutableDoubleSet newWithout(double element)
    {
        return Double.compare(this.element, element) == 0 ? DoubleSets.immutable.with() : this;
    }

    public ImmutableDoubleSet newWithAll(DoubleIterable elements)
    {
        return DoubleHashSet.newSet(elements).with(this.element).toImmutable();
    }

    public ImmutableDoubleSet newWithoutAll(DoubleIterable elements)
    {
        return elements.contains(this.element) ? DoubleSets.immutable.with() : this;
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public boolean contains(double value)
    {
        return Double.compare(this.element, value) == 0;
    }

    public boolean containsAll(DoubleIterable source)
    {
        for (DoubleIterator iterator = source.doubleIterator(); iterator.hasNext(); )
        {
            if (Double.compare(this.element, iterator.next()) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(double... source)
    {
        for (double value : source)
        {
            if (Double.compare(this.element, value) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(DoubleProcedure procedure)
    {
        procedure.value(this.element);
    }

    public ImmutableDoubleSet select(DoublePredicate predicate)
    {
        return predicate.accept(this.element) ? DoubleHashSet.newSetWith(this.element).toImmutable()
                : new DoubleHashSet().toImmutable();
    }

    public ImmutableDoubleSet reject(DoublePredicate predicate)
    {
        return predicate.accept(this.element) ? new DoubleHashSet().toImmutable()
                : DoubleHashSet.newSetWith(this.element).toImmutable();
    }

    public <V> ImmutableSet<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        return UnifiedSet.newSetWith(function.valueOf(this.element)).toImmutable();
    }

    public MutableDoubleList toList()
    {
        return DoubleArrayList.newListWith(this.element);
    }

    public int sizeDistinct()
    {
        return 1;
    }

    public int occurrencesOf(double item)
    {
        return Double.compare(this.element, item) == 0 ? 1 : 0;
    }

    public void forEachWithOccurrences(DoubleIntProcedure doubleIntProcedure)
    {
        doubleIntProcedure.value(this.element, 1);
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        return predicate.accept(this.element) ? this.element : ifNone;
    }

    public int count(DoublePredicate predicate)
    {
        return predicate.accept(this.element) ? 1 : 0;
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public double sum()
    {
        return this.element;
    }

    public double min()
    {
        return this.element;
    }

    public double max()
    {
        return this.element;
    }

    public double maxIfEmpty(double defaultValue)
    {
        return this.element;
    }

    public double minIfEmpty(double defaultValue)
    {
        return this.element;
    }

    public double average()
    {
        return this.element;
    }

    public double median()
    {
        return this.element;
    }

    public double[] toSortedArray()
    {
        return new double[]{this.element};
    }

    public MutableDoubleList toSortedList()
    {
        return DoubleArrayList.newListWith(this.element);
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        return !predicate.accept(this.element);
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof DoubleSet))
        {
            return false;
        }
        DoubleSet set = (DoubleSet) obj;
        if (set.size() != 1)
        {
            return false;
        }
        return set.contains(this.element);
    }

    @Override
    public int hashCode()
    {
        return (int) (Double.doubleToLongBits(this.element) ^ Double.doubleToLongBits(this.element) >>> 32) ^ 1;
    }

    public MutableDoubleSet toSet()
    {
        return DoubleHashSet.newSetWith(this.element);
    }

    public MutableDoubleBag toBag()
    {
        return DoubleHashBag.newBagWith(this.element);
    }

    public DoubleSet freeze()
    {
        return this;
    }

    public ImmutableDoubleSet toImmutable()
    {
        return this;
    }

    public LazyDoubleIterable asLazy()
    {
        return new LazyDoubleIterableAdapter(this);
    }

    public double[] toArray()
    {
        return new double[]{this.element};
    }

    @Override
    public String toString()
    {
        return '[' + this.makeString() + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public DoubleIterator doubleIterator()
    {
        return DoubleHashSet.newSetWith(this.element).doubleIterator();
    }
}

