/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.common.data.LabelType;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Branch;
import com.google.gerrit.reviewdb.client.PatchSetApproval;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.ApprovalsUtil;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.git.CodeReviewCommit;
import com.google.gerrit.server.git.CommitMergeStatus;
import com.google.gerrit.server.git.MergeException;
import com.google.gerrit.server.git.MergeSorter;
import com.google.gerrit.server.project.ProjectState;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Provider;
import com.google.inject.assistedinject.Assisted;
import com.google.inject.assistedinject.AssistedInject;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;
import org.eclipse.jgit.errors.IncorrectObjectTypeException;
import org.eclipse.jgit.errors.LargeObjectException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.errors.NoMergeBaseException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.CommitBuilder;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectInserter;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.merge.MergeStrategy;
import org.eclipse.jgit.merge.Merger;
import org.eclipse.jgit.merge.ThreeWayMerger;
import org.eclipse.jgit.revwalk.FooterKey;
import org.eclipse.jgit.revwalk.FooterLine;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevFlag;
import org.eclipse.jgit.revwalk.RevSort;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.transport.PackParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MergeUtil {
    public static final FooterKey CHANGE_ID = new FooterKey("Change-Id");
    private static final FooterKey REVIEWED_ON = new FooterKey("Reviewed-on");
    private static final Logger log = LoggerFactory.getLogger(MergeUtil.class);
    private static final String R_HEADS_MASTER = "refs/heads/master";
    private final Provider<ReviewDb> db;
    private final IdentifiedUser.GenericFactory identifiedUserFactory;
    private final Provider<String> urlProvider;
    private final ApprovalsUtil approvalsUtil;
    private final ProjectState project;
    private final boolean useContentMerge;
    private final boolean useRecursiveMerge;

    public static boolean useRecursiveMerge(Config cfg) {
        return cfg.getBoolean("core", null, "useRecursiveMerge", true);
    }

    @AssistedInject
    MergeUtil(@GerritServerConfig Config serverConfig, Provider<ReviewDb> db, IdentifiedUser.GenericFactory identifiedUserFactory, @CanonicalWebUrl @Nullable Provider<String> urlProvider, ApprovalsUtil approvalsUtil, @Assisted ProjectState project) {
        this(serverConfig, db, identifiedUserFactory, urlProvider, approvalsUtil, project, project.isUseContentMerge());
    }

    @AssistedInject
    MergeUtil(@GerritServerConfig Config serverConfig, Provider<ReviewDb> db, IdentifiedUser.GenericFactory identifiedUserFactory, @CanonicalWebUrl @Nullable Provider<String> urlProvider, ApprovalsUtil approvalsUtil, @Assisted ProjectState project, @Assisted boolean useContentMerge) {
        this.db = db;
        this.identifiedUserFactory = identifiedUserFactory;
        this.urlProvider = urlProvider;
        this.approvalsUtil = approvalsUtil;
        this.project = project;
        this.useContentMerge = useContentMerge;
        this.useRecursiveMerge = MergeUtil.useRecursiveMerge(serverConfig);
    }

    public CodeReviewCommit getFirstFastForward(CodeReviewCommit mergeTip, RevWalk rw, List<CodeReviewCommit> toMerge) throws MergeException {
        Iterator<CodeReviewCommit> i = toMerge.iterator();
        while (i.hasNext()) {
            try {
                CodeReviewCommit n = i.next();
                if (mergeTip != null && !rw.isMergedInto(mergeTip, n)) continue;
                i.remove();
                return n;
            }
            catch (IOException e) {
                throw new MergeException("Cannot fast-forward test during merge", e);
            }
        }
        return mergeTip;
    }

    public void reduceToMinimalMerge(MergeSorter mergeSorter, List<CodeReviewCommit> toSort) throws MergeException {
        Collection<CodeReviewCommit> heads;
        try {
            heads = mergeSorter.sort(toSort);
        }
        catch (IOException e) {
            throw new MergeException("Branch head sorting failed", e);
        }
        toSort.clear();
        toSort.addAll(heads);
        Collections.sort(toSort, new Comparator<CodeReviewCommit>(){

            @Override
            public int compare(CodeReviewCommit a, CodeReviewCommit b) {
                return a.originalOrder - b.originalOrder;
            }
        });
    }

    public PatchSetApproval getSubmitter(CodeReviewCommit c) {
        return this.approvalsUtil.getSubmitter(this.db.get(), c.notes(), c.getPatchsetId());
    }

    public RevCommit createCherryPickFromCommit(Repository repo, ObjectInserter inserter, RevCommit mergeTip, RevCommit originalCommit, PersonIdent cherryPickCommitterIdent, String commitMsg, RevWalk rw) throws MissingObjectException, IncorrectObjectTypeException, IOException {
        ThreeWayMerger m = this.newThreeWayMerger(repo, inserter);
        m.setBase(originalCommit.getParent(0));
        if (m.merge(mergeTip, originalCommit)) {
            ObjectId tree = m.getResultTreeId();
            if (tree.equals(mergeTip.getTree())) {
                return null;
            }
            CommitBuilder mergeCommit = new CommitBuilder();
            mergeCommit.setTreeId(tree);
            mergeCommit.setParentId(mergeTip);
            mergeCommit.setAuthor(originalCommit.getAuthorIdent());
            mergeCommit.setCommitter(cherryPickCommitterIdent);
            mergeCommit.setMessage(commitMsg);
            return rw.parseCommit(this.commit(inserter, mergeCommit));
        }
        return null;
    }

    public String createCherryPickCommitMessage(CodeReviewCommit n) {
        String url;
        String siteUrl;
        List<FooterLine> footers = n.getFooterLines();
        StringBuilder msgbuf = new StringBuilder();
        msgbuf.append(n.getFullMessage());
        if (msgbuf.length() == 0) {
            msgbuf.append("<no commit message provided>");
        }
        if (msgbuf.charAt(msgbuf.length() - 1) != '\n') {
            msgbuf.append('\n');
        }
        if (footers.isEmpty()) {
            msgbuf.append('\n');
        }
        if (!MergeUtil.contains(footers, CHANGE_ID, n.change().getKey().get())) {
            msgbuf.append(CHANGE_ID.getName());
            msgbuf.append(": ");
            msgbuf.append(n.change().getKey().get());
            msgbuf.append('\n');
        }
        if ((siteUrl = this.urlProvider.get()) != null && !MergeUtil.contains(footers, REVIEWED_ON, url = siteUrl + n.getPatchsetId().getParentKey().get())) {
            msgbuf.append(REVIEWED_ON.getName());
            msgbuf.append(": ");
            msgbuf.append(url);
            msgbuf.append('\n');
        }
        PatchSetApproval submitAudit = null;
        for (PatchSetApproval a : this.safeGetApprovals(n)) {
            String tag;
            if (a.getValue() <= 0) continue;
            if (a.isSubmit()) {
                if (submitAudit != null && a.getGranted().compareTo(submitAudit.getGranted()) <= 0) continue;
                submitAudit = a;
                continue;
            }
            Account acc = this.identifiedUserFactory.create(a.getAccountId()).getAccount();
            StringBuilder identbuf = new StringBuilder();
            if (acc.getFullName() != null && acc.getFullName().length() > 0) {
                if (identbuf.length() > 0) {
                    identbuf.append(' ');
                }
                identbuf.append(acc.getFullName());
            }
            if (acc.getPreferredEmail() != null && acc.getPreferredEmail().length() > 0) {
                if (MergeUtil.isSignedOffBy(footers, acc.getPreferredEmail())) continue;
                if (identbuf.length() > 0) {
                    identbuf.append(' ');
                }
                identbuf.append('<');
                identbuf.append(acc.getPreferredEmail());
                identbuf.append('>');
            }
            if (identbuf.length() == 0) continue;
            if (MergeUtil.isCodeReview(a.getLabelId())) {
                tag = "Reviewed-by";
            } else if (MergeUtil.isVerified(a.getLabelId())) {
                tag = "Tested-by";
            } else {
                LabelType lt = this.project.getLabelTypes().byLabel(a.getLabelId());
                if (lt == null) continue;
                tag = lt.getName();
            }
            if (MergeUtil.contains(footers, new FooterKey(tag), identbuf.toString())) continue;
            msgbuf.append(tag);
            msgbuf.append(": ");
            msgbuf.append((CharSequence)identbuf);
            msgbuf.append('\n');
        }
        return msgbuf.toString();
    }

    private static boolean isCodeReview(PatchSetApproval.LabelId id) {
        return "Code-Review".equalsIgnoreCase(id.get());
    }

    private static boolean isVerified(PatchSetApproval.LabelId id) {
        return "Verified".equalsIgnoreCase(id.get());
    }

    private Iterable<PatchSetApproval> safeGetApprovals(CodeReviewCommit n) {
        try {
            return this.approvalsUtil.byPatchSet(this.db.get(), n.getControl(), n.getPatchsetId());
        }
        catch (OrmException e) {
            log.error("Can't read approval records for " + n.getPatchsetId(), e);
            return Collections.emptyList();
        }
    }

    private static boolean contains(List<FooterLine> footers, FooterKey key, String val) {
        for (FooterLine line : footers) {
            if (!line.matches(key) || !val.equals(line.getValue())) continue;
            return true;
        }
        return false;
    }

    private static boolean isSignedOffBy(List<FooterLine> footers, String email) {
        for (FooterLine line : footers) {
            if (!line.matches(FooterKey.SIGNED_OFF_BY) || !email.equals(line.getEmailAddress())) continue;
            return true;
        }
        return false;
    }

    public PersonIdent computeMergeCommitAuthor(PersonIdent myIdent, RevWalk rw, List<CodeReviewCommit> codeReviewCommits) {
        PersonIdent authorIdent;
        PatchSetApproval submitter = null;
        for (CodeReviewCommit c : codeReviewCommits) {
            PatchSetApproval s = this.getSubmitter(c);
            if (submitter != null && (s == null || s.getGranted().compareTo(submitter.getGranted()) <= 0)) continue;
            submitter = s;
        }
        if (submitter != null) {
            IdentifiedUser who = this.identifiedUserFactory.create(submitter.getAccountId());
            HashSet<String> emails = new HashSet<String>();
            for (RevCommit revCommit : codeReviewCommits) {
                try {
                    rw.parseBody(revCommit);
                }
                catch (IOException e) {
                    log.warn("Cannot parse commit " + revCommit.name(), e);
                    continue;
                }
                emails.add(revCommit.getAuthorIdent().getEmailAddress());
            }
            Timestamp dt = submitter.getGranted();
            TimeZone timeZone = myIdent.getTimeZone();
            authorIdent = emails.size() == 1 && who.getEmailAddresses().contains(emails.iterator().next()) ? new PersonIdent(codeReviewCommits.get(0).getAuthorIdent(), dt, timeZone) : who.newCommitterIdent(dt, timeZone);
        } else {
            authorIdent = myIdent;
        }
        return authorIdent;
    }

    public boolean canMerge(MergeSorter mergeSorter, Repository repo, CodeReviewCommit mergeTip, CodeReviewCommit toMerge) throws MergeException {
        if (this.hasMissingDependencies(mergeSorter, toMerge)) {
            return false;
        }
        ThreeWayMerger m = this.newThreeWayMerger(repo, MergeUtil.createDryRunInserter(repo));
        try {
            return m.merge(mergeTip, toMerge);
        }
        catch (LargeObjectException e) {
            log.warn("Cannot merge due to LargeObjectException: " + toMerge.name());
            return false;
        }
        catch (NoMergeBaseException e) {
            return false;
        }
        catch (IOException e) {
            throw new MergeException("Cannot merge " + toMerge.name(), e);
        }
    }

    public boolean canFastForward(MergeSorter mergeSorter, CodeReviewCommit mergeTip, RevWalk rw, CodeReviewCommit toMerge) throws MergeException {
        if (this.hasMissingDependencies(mergeSorter, toMerge)) {
            return false;
        }
        try {
            return mergeTip == null || rw.isMergedInto(mergeTip, toMerge);
        }
        catch (IOException e) {
            throw new MergeException("Cannot fast-forward test during merge", e);
        }
    }

    public boolean canCherryPick(MergeSorter mergeSorter, Repository repo, CodeReviewCommit mergeTip, RevWalk rw, CodeReviewCommit toMerge) throws MergeException {
        if (mergeTip == null) {
            return true;
        }
        if (toMerge.getParentCount() == 0) {
            return false;
        }
        if (toMerge.getParentCount() == 1) {
            try {
                ThreeWayMerger m = this.newThreeWayMerger(repo, MergeUtil.createDryRunInserter(repo));
                m.setBase(toMerge.getParent(0));
                return m.merge(mergeTip, toMerge);
            }
            catch (IOException e) {
                throw new MergeException("Cannot merge " + toMerge.name(), e);
            }
        }
        return this.canFastForward(mergeSorter, mergeTip, rw, toMerge) || this.canMerge(mergeSorter, repo, mergeTip, toMerge);
    }

    public boolean hasMissingDependencies(MergeSorter mergeSorter, CodeReviewCommit toMerge) throws MergeException {
        try {
            return !mergeSorter.sort(Collections.singleton(toMerge)).contains(toMerge);
        }
        catch (IOException e) {
            throw new MergeException("Branch head sorting failed", e);
        }
    }

    public static ObjectInserter createDryRunInserter(Repository db) {
        final ObjectInserter delegate = db.newObjectInserter();
        return new ObjectInserter.Filter(){

            @Override
            protected ObjectInserter delegate() {
                return delegate;
            }

            @Override
            public PackParser newPackParser(InputStream in) throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override
            public void flush() throws IOException {
            }
        };
    }

    public CodeReviewCommit mergeOneCommit(PersonIdent myIdent, Repository repo, RevWalk rw, ObjectInserter inserter, RevFlag canMergeFlag, Branch.NameKey destBranch, CodeReviewCommit mergeTip, CodeReviewCommit n) throws MergeException {
        ThreeWayMerger m = this.newThreeWayMerger(repo, inserter);
        try {
            if (m.merge(mergeTip, n)) {
                return this.writeMergeCommit(myIdent, rw, inserter, canMergeFlag, destBranch, mergeTip, m.getResultTreeId(), n);
            }
            MergeUtil.failed(rw, canMergeFlag, mergeTip, n, CommitMergeStatus.PATH_CONFLICT);
        }
        catch (NoMergeBaseException e) {
            try {
                MergeUtil.failed(rw, canMergeFlag, mergeTip, n, MergeUtil.getCommitMergeStatus(e.getReason()));
            }
            catch (IOException e2) {
                throw new MergeException("Cannot merge " + n.name(), e);
            }
        }
        catch (IOException e) {
            throw new MergeException("Cannot merge " + n.name(), e);
        }
        return mergeTip;
    }

    private static CommitMergeStatus getCommitMergeStatus(NoMergeBaseException.MergeBaseFailureReason reason) {
        switch (reason) {
            default: {
                return CommitMergeStatus.MANUAL_RECURSIVE_MERGE;
            }
            case CONFLICTS_DURING_MERGE_BASE_CALCULATION: 
        }
        return CommitMergeStatus.PATH_CONFLICT;
    }

    private static CodeReviewCommit failed(RevWalk rw, RevFlag canMergeFlag, CodeReviewCommit mergeTip, CodeReviewCommit n, CommitMergeStatus failure) throws MissingObjectException, IncorrectObjectTypeException, IOException {
        CodeReviewCommit failed;
        rw.resetRetain(canMergeFlag);
        rw.markStart(n);
        rw.markUninteresting(mergeTip);
        while ((failed = (CodeReviewCommit)rw.next()) != null) {
            failed.setStatusCode(failure);
        }
        return failed;
    }

    public CodeReviewCommit writeMergeCommit(PersonIdent myIdent, RevWalk rw, ObjectInserter inserter, RevFlag canMergeFlag, Branch.NameKey destBranch, CodeReviewCommit mergeTip, ObjectId treeId, CodeReviewCommit n) throws IOException, MissingObjectException, IncorrectObjectTypeException {
        ArrayList<CodeReviewCommit> merged = new ArrayList<CodeReviewCommit>();
        rw.resetRetain(canMergeFlag);
        rw.markStart(n);
        rw.markUninteresting(mergeTip);
        for (RevCommit c : rw) {
            CodeReviewCommit crc = (CodeReviewCommit)c;
            if (crc.getPatchsetId() == null) continue;
            merged.add(crc);
        }
        StringBuilder msgbuf = new StringBuilder().append(this.summarize(rw, merged));
        if (!R_HEADS_MASTER.equals(destBranch.get())) {
            msgbuf.append(" into ");
            msgbuf.append(destBranch.getShortName());
        }
        if (merged.size() > 1) {
            msgbuf.append("\n\n* changes:\n");
            for (CodeReviewCommit c : merged) {
                rw.parseBody(c);
                msgbuf.append("  ");
                msgbuf.append(c.getShortMessage());
                msgbuf.append("\n");
            }
        }
        PersonIdent authorIdent = this.computeMergeCommitAuthor(myIdent, rw, merged);
        CommitBuilder mergeCommit = new CommitBuilder();
        mergeCommit.setTreeId(treeId);
        mergeCommit.setParentIds((AnyObjectId)mergeTip, (AnyObjectId)n);
        mergeCommit.setAuthor(authorIdent);
        mergeCommit.setCommitter(myIdent);
        mergeCommit.setMessage(msgbuf.toString());
        CodeReviewCommit mergeResult = (CodeReviewCommit)rw.parseCommit(this.commit(inserter, mergeCommit));
        mergeResult.setControl(n.getControl());
        return mergeResult;
    }

    private String summarize(RevWalk rw, List<CodeReviewCommit> merged) throws IOException {
        if (merged.size() == 1) {
            CodeReviewCommit c = merged.get(0);
            rw.parseBody(c);
            return String.format("Merge \"%s\"", c.getShortMessage());
        }
        LinkedHashSet<String> topics = new LinkedHashSet<String>(4);
        for (CodeReviewCommit c : merged) {
            if (Strings.isNullOrEmpty(c.change().getTopic())) continue;
            topics.add(c.change().getTopic());
        }
        if (topics.size() == 1) {
            return String.format("Merge topic '%s'", Iterables.getFirst(topics, null));
        }
        if (topics.size() > 1) {
            return String.format("Merge topics '%s'", Joiner.on("', '").join(topics));
        }
        return String.format("Merge changes %s%s", Joiner.on(',').join(Iterables.transform(Iterables.limit(merged, 5), new Function<CodeReviewCommit, String>(){

            @Override
            public String apply(CodeReviewCommit in) {
                return in.change().getKey().abbreviate();
            }
        })), merged.size() > 5 ? ", ..." : "");
    }

    public ThreeWayMerger newThreeWayMerger(Repository repo, ObjectInserter inserter) {
        return MergeUtil.newThreeWayMerger(repo, inserter, MergeUtil.mergeStrategyName(this.useContentMerge, this.useRecursiveMerge));
    }

    public static String mergeStrategyName(boolean useContentMerge, boolean useRecursiveMerge) {
        if (useContentMerge) {
            if (useRecursiveMerge) {
                return MergeStrategy.RECURSIVE.getName();
            }
            return MergeStrategy.RESOLVE.getName();
        }
        return MergeStrategy.SIMPLE_TWO_WAY_IN_CORE.getName();
    }

    public static ThreeWayMerger newThreeWayMerger(Repository repo, final ObjectInserter inserter, String strategyName) {
        MergeStrategy strategy = MergeStrategy.get(strategyName);
        Preconditions.checkArgument(strategy != null, "invalid merge strategy: %s", strategyName);
        Merger m = strategy.newMerger(repo, true);
        Preconditions.checkArgument(m instanceof ThreeWayMerger, "merge strategy %s does not support three-way merging", strategyName);
        m.setObjectInserter(new ObjectInserter.Filter(){

            @Override
            protected ObjectInserter delegate() {
                return inserter;
            }

            @Override
            public void flush() {
            }

            @Override
            public void release() {
            }
        });
        return (ThreeWayMerger)m;
    }

    public ObjectId commit(ObjectInserter inserter, CommitBuilder mergeCommit) throws IOException, UnsupportedEncodingException {
        ObjectId id = inserter.insert(mergeCommit);
        inserter.flush();
        return id;
    }

    public PatchSetApproval markCleanMerges(RevWalk rw, RevFlag canMergeFlag, CodeReviewCommit mergeTip, Set<RevCommit> alreadyAccepted) throws MergeException {
        if (mergeTip == null) {
            return null;
        }
        try {
            CodeReviewCommit c;
            PatchSetApproval submitApproval = null;
            rw.resetRetain(canMergeFlag);
            rw.sort(RevSort.TOPO);
            rw.sort(RevSort.REVERSE, true);
            rw.markStart(mergeTip);
            for (RevCommit c2 : alreadyAccepted) {
                rw.markUninteresting(c2);
            }
            while ((c = (CodeReviewCommit)rw.next()) != null) {
                if (c.getPatchsetId() == null) continue;
                c.setStatusCode(CommitMergeStatus.CLEAN_MERGE);
                if (submitApproval != null) continue;
                submitApproval = this.getSubmitter(c);
            }
            return submitApproval;
        }
        catch (IOException e) {
            throw new MergeException("Cannot mark clean merges", e);
        }
    }

    public static interface Factory {
        public MergeUtil create(ProjectState var1);

        public MergeUtil create(ProjectState var1, boolean var2);
    }
}

