/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.plugins;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;
import com.google.gerrit.extensions.annotations.PluginName;
import com.google.gerrit.extensions.events.LifecycleListener;
import com.google.gerrit.extensions.systemstatus.ServerInformation;
import com.google.gerrit.extensions.webui.JavaScriptPlugin;
import com.google.gerrit.server.PluginUser;
import com.google.gerrit.server.cache.PersistentCacheFactory;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.config.ConfigUtil;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.config.SitePaths;
import com.google.gerrit.server.plugins.CleanupHandle;
import com.google.gerrit.server.plugins.InvalidPluginException;
import com.google.gerrit.server.plugins.JsPlugin;
import com.google.gerrit.server.plugins.Plugin;
import com.google.gerrit.server.plugins.PluginCleanerTask;
import com.google.gerrit.server.plugins.PluginGuiceEnvironment;
import com.google.gerrit.server.plugins.PluginInstallException;
import com.google.gerrit.server.plugins.PluginScannerThread;
import com.google.gerrit.server.plugins.ServerInformationImpl;
import com.google.gerrit.server.plugins.ServerPlugin;
import com.google.gerrit.server.plugins.ServerPluginProvider;
import com.google.gerrit.server.plugins.UniversalServerPluginProvider;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.eclipse.jgit.internal.storage.file.FileSnapshot;
import org.eclipse.jgit.lib.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class PluginLoader
implements LifecycleListener {
    static final String PLUGIN_TMP_PREFIX = "plugin_";
    static final Logger log = LoggerFactory.getLogger(PluginLoader.class);
    private final File pluginsDir;
    private final File dataDir;
    private final PluginGuiceEnvironment env;
    private final ServerInformationImpl srvInfoImpl;
    private final PluginUser.Factory pluginUserFactory;
    private final ConcurrentMap<String, Plugin> running;
    private final ConcurrentMap<String, Plugin> disabled;
    private final Map<String, FileSnapshot> broken;
    private final Map<Plugin, CleanupHandle> cleanupHandles;
    private final Queue<Plugin> toCleanup;
    private final Provider<PluginCleanerTask> cleaner;
    private final PluginScannerThread scanner;
    private final Provider<String> urlProvider;
    private final PersistentCacheFactory persistentCacheFactory;
    private final boolean remoteAdmin;
    private final UniversalServerPluginProvider serverPluginFactory;

    public String getPluginName(File srcFile) throws IOException {
        return Objects.firstNonNull(this.getGerritPluginName(srcFile), PluginLoader.nameOf(srcFile));
    }

    @Inject
    public PluginLoader(SitePaths sitePaths, PluginGuiceEnvironment pe, ServerInformationImpl sii, PluginUser.Factory puf, Provider<PluginCleanerTask> pct, @GerritServerConfig Config cfg, @CanonicalWebUrl Provider<String> provider, PersistentCacheFactory cacheFactory, UniversalServerPluginProvider pluginFactory) {
        this.pluginsDir = sitePaths.plugins_dir;
        this.dataDir = sitePaths.data_dir;
        this.env = pe;
        this.srvInfoImpl = sii;
        this.pluginUserFactory = puf;
        this.running = Maps.newConcurrentMap();
        this.disabled = Maps.newConcurrentMap();
        this.broken = Maps.newHashMap();
        this.toCleanup = Queues.newArrayDeque();
        this.cleanupHandles = Maps.newConcurrentMap();
        this.cleaner = pct;
        this.urlProvider = provider;
        this.persistentCacheFactory = cacheFactory;
        this.serverPluginFactory = pluginFactory;
        this.remoteAdmin = cfg.getBoolean("plugins", null, "allowRemoteAdmin", false);
        long checkFrequency = ConfigUtil.getTimeUnit(cfg, "plugins", null, "checkFrequency", TimeUnit.MINUTES.toMillis(1L), TimeUnit.MILLISECONDS);
        this.scanner = checkFrequency > 0L ? new PluginScannerThread(this, checkFrequency) : null;
    }

    public boolean isRemoteAdminEnabled() {
        return this.remoteAdmin;
    }

    public Plugin get(String name) {
        Plugin p = (Plugin)this.running.get(name);
        if (p != null) {
            return p;
        }
        return (Plugin)this.disabled.get(name);
    }

    public Iterable<Plugin> getPlugins(boolean all) {
        if (!all) {
            return this.running.values();
        }
        ArrayList<Plugin> plugins = new ArrayList<Plugin>(this.running.values());
        plugins.addAll(this.disabled.values());
        return plugins;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void installPluginFromStream(String originalName, InputStream in) throws IOException, PluginInstallException {
        this.checkRemoteInstall();
        String fileName = originalName;
        File tmp = PluginLoader.asTemp(in, ".next_" + fileName + "_", ".tmp", this.pluginsDir);
        String name = Objects.firstNonNull(this.getGerritPluginName(tmp), PluginLoader.nameOf(fileName));
        if (!originalName.equals(name)) {
            log.warn(String.format("Plugin provides its own name: <%s>, use it instead of the input name: <%s>", name, originalName));
        }
        String fileExtension = PluginLoader.getExtension(fileName);
        File dst = new File(this.pluginsDir, name + fileExtension);
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            Plugin active = (Plugin)this.running.get(name);
            if (active != null) {
                fileName = active.getSrcFile().getName();
                log.info(String.format("Replacing plugin %s", active.getName()));
                File old = new File(this.pluginsDir, ".last_" + fileName);
                old.delete();
                active.getSrcFile().renameTo(old);
            }
            new File(this.pluginsDir, fileName + ".disabled").delete();
            tmp.renameTo(dst);
            try {
                Plugin plugin = this.runPlugin(name, dst, active);
                if (active == null) {
                    log.info(String.format("Installed plugin %s", plugin.getName()));
                }
            }
            catch (PluginInstallException e) {
                dst.delete();
                throw e;
            }
            this.cleanInBackground();
        }
    }

    /*
     * Loose catch block
     */
    static File asTemp(InputStream in, String prefix, String suffix, File dir) throws IOException {
        File tmp = File.createTempFile(prefix, suffix, dir);
        boolean keep = false;
        try {
            try (FileOutputStream out = new FileOutputStream(tmp);){
                int n;
                byte[] data = new byte[8192];
                while ((n = in.read(data)) > 0) {
                    out.write(data, 0, n);
                }
                keep = true;
                File file = tmp;
                return file;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            if (!keep) {
                tmp.delete();
            }
        }
    }

    private synchronized void unloadPlugin(Plugin plugin) {
        this.persistentCacheFactory.onStop(plugin);
        String name = plugin.getName();
        log.info(String.format("Unloading plugin %s", name));
        plugin.stop(this.env);
        this.env.onStopPlugin(plugin);
        this.running.remove(name);
        this.disabled.remove(name);
        this.toCleanup.add(plugin);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void disablePlugins(Set<String> names) {
        if (!this.isRemoteAdminEnabled()) {
            log.warn("Remote plugin administration is disabled, ignoring disablePlugins(" + names + ")");
            return;
        }
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            for (String name : names) {
                Plugin active = (Plugin)this.running.get(name);
                if (active == null) continue;
                log.info(String.format("Disabling plugin %s", active.getName()));
                File off = new File(active.getSrcFile() + ".disabled");
                active.getSrcFile().renameTo(off);
                this.unloadPlugin(active);
                try {
                    FileSnapshot snapshot = FileSnapshot.save(off);
                    Plugin offPlugin = this.loadPlugin(name, off, snapshot);
                    this.disabled.put(name, offPlugin);
                }
                catch (Throwable e) {
                    log.warn(String.format("Cannot load disabled plugin %s", active.getName()), e.getCause());
                }
            }
            this.cleanInBackground();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void enablePlugins(Set<String> names) throws PluginInstallException {
        if (!this.isRemoteAdminEnabled()) {
            log.warn("Remote plugin administration is disabled, ignoring enablePlugins(" + names + ")");
            return;
        }
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            for (String name : names) {
                Plugin off = (Plugin)this.disabled.get(name);
                if (off == null) continue;
                log.info(String.format("Enabling plugin %s", name));
                String n = off.getSrcFile().getName();
                if (n.endsWith(".disabled")) {
                    n = n.substring(0, n.lastIndexOf(46));
                }
                File on = new File(this.pluginsDir, n);
                off.getSrcFile().renameTo(on);
                this.disabled.remove(name);
                this.runPlugin(name, on, null);
            }
            this.cleanInBackground();
        }
    }

    @Override
    public synchronized void start() {
        log.info("Loading plugins from " + this.pluginsDir.getAbsolutePath());
        this.srvInfoImpl.state = ServerInformation.State.STARTUP;
        this.rescan();
        this.srvInfoImpl.state = ServerInformation.State.RUNNING;
        if (this.scanner != null) {
            this.scanner.start();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void stop() {
        if (this.scanner != null) {
            this.scanner.end();
        }
        this.srvInfoImpl.state = ServerInformation.State.SHUTDOWN;
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            for (Plugin p : this.running.values()) {
                this.unloadPlugin(p);
            }
            this.running.clear();
            this.disabled.clear();
            this.broken.clear();
            if (!this.toCleanup.isEmpty()) {
                System.gc();
                this.processPendingCleanups();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reload(List<String> names) throws InvalidPluginException, PluginInstallException {
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            ArrayList<Plugin> reload = Lists.newArrayListWithCapacity(names.size());
            ArrayList<String> bad = Lists.newArrayListWithExpectedSize(4);
            for (String name : names) {
                Plugin active = (Plugin)this.running.get(name);
                if (active != null) {
                    reload.add(active);
                    continue;
                }
                bad.add(name);
            }
            if (!bad.isEmpty()) {
                throw new InvalidPluginException(String.format("Plugin(s) \"%s\" not running", Joiner.on("\", \"").join(bad)));
            }
            for (Plugin active : reload) {
                String name = active.getName();
                try {
                    log.info(String.format("Reloading plugin %s", name));
                    this.runPlugin(name, active.getSrcFile(), active);
                }
                catch (PluginInstallException e) {
                    log.warn(String.format("Cannot reload plugin %s", name), e.getCause());
                    throw e;
                }
            }
            this.cleanInBackground();
        }
    }

    public synchronized void rescan() {
        Multimap<String, File> pluginsFiles = this.prunePlugins(this.pluginsDir);
        if (pluginsFiles.isEmpty()) {
            return;
        }
        this.syncDisabledPlugins(pluginsFiles);
        Map<String, File> activePlugins = PluginLoader.filterDisabled(pluginsFiles);
        for (Map.Entry<String, File> entry : this.jarsFirstSortedPluginsSet(activePlugins)) {
            Plugin active;
            String name = entry.getKey();
            File file = entry.getValue();
            String fileName = file.getName();
            if (!PluginLoader.isJsPlugin(fileName) && !this.serverPluginFactory.handles(file)) {
                log.warn("No Plugin provider was found that handles this file format: {}", (Object)fileName);
                continue;
            }
            FileSnapshot brokenTime = this.broken.get(name);
            if (brokenTime != null && !brokenTime.isModified(file) || (active = (Plugin)this.running.get(name)) != null && !active.isModified(file)) continue;
            if (active != null) {
                log.info(String.format("Reloading plugin %s, version %s", active.getName(), active.getVersion()));
            }
            try {
                Plugin loadedPlugin = this.runPlugin(name, file, active);
                if (active != null || loadedPlugin.isDisabled()) continue;
                log.info(String.format("Loaded plugin %s, version %s", loadedPlugin.getName(), loadedPlugin.getVersion()));
            }
            catch (PluginInstallException e) {
                log.warn(String.format("Cannot load plugin %s", name), e.getCause());
            }
        }
        this.cleanInBackground();
    }

    private void addAllEntries(Map<String, File> from, TreeSet<Map.Entry<String, File>> to) {
        for (Map.Entry<String, File> entry : from.entrySet()) {
            to.add(new AbstractMap.SimpleImmutableEntry<String, File>(entry.getKey(), entry.getValue()));
        }
    }

    private TreeSet<Map.Entry<String, File>> jarsFirstSortedPluginsSet(Map<String, File> activePlugins) {
        TreeSet<Map.Entry<String, File>> sortedPlugins = Sets.newTreeSet(new Comparator<Map.Entry<String, File>>(){

            @Override
            public int compare(Map.Entry<String, File> entry1, Map.Entry<String, File> entry2) {
                String file1 = entry1.getValue().getName();
                String file2 = entry2.getValue().getName();
                int cmp = file1.compareTo(file2);
                if (file1.endsWith(".jar")) {
                    return file2.endsWith(".jar") ? cmp : -1;
                }
                return file2.endsWith(".jar") ? 1 : cmp;
            }
        });
        this.addAllEntries(activePlugins, sortedPlugins);
        return sortedPlugins;
    }

    private void syncDisabledPlugins(Multimap<String, File> jars) {
        this.stopRemovedPlugins(jars);
        this.dropRemovedDisabledPlugins(jars);
    }

    private Plugin runPlugin(String name, File plugin, Plugin oldPlugin) throws PluginInstallException {
        FileSnapshot snapshot = FileSnapshot.save(plugin);
        try {
            boolean reload;
            Plugin newPlugin = this.loadPlugin(name, plugin, snapshot);
            if (newPlugin.getCleanupHandle() != null) {
                this.cleanupHandles.put(newPlugin, newPlugin.getCleanupHandle());
            }
            name = newPlugin.getName();
            boolean bl = reload = oldPlugin != null && oldPlugin.canReload() && newPlugin.canReload();
            if (!reload && oldPlugin != null) {
                this.unloadPlugin(oldPlugin);
            }
            if (!newPlugin.isDisabled()) {
                newPlugin.start(this.env);
            }
            if (reload) {
                this.env.onReloadPlugin(oldPlugin, newPlugin);
                this.unloadPlugin(oldPlugin);
            } else if (!newPlugin.isDisabled()) {
                this.env.onStartPlugin(newPlugin);
            }
            if (!newPlugin.isDisabled()) {
                this.running.put(name, newPlugin);
            } else {
                this.disabled.put(name, newPlugin);
            }
            this.broken.remove(name);
            return newPlugin;
        }
        catch (Throwable err) {
            this.broken.put(name, snapshot);
            throw new PluginInstallException(err);
        }
    }

    private void stopRemovedPlugins(Multimap<String, File> jars) {
        HashSet<String> unload = Sets.newHashSet(this.running.keySet());
        for (Map.Entry<String, Collection<File>> entry : jars.asMap().entrySet()) {
            for (File file : entry.getValue()) {
                if (file.getName().endsWith(".disabled")) continue;
                unload.remove(entry.getKey());
            }
        }
        for (String name : unload) {
            this.unloadPlugin((Plugin)this.running.get(name));
        }
    }

    private void dropRemovedDisabledPlugins(Multimap<String, File> jars) {
        HashSet<String> unload = Sets.newHashSet(this.disabled.keySet());
        for (Map.Entry<String, Collection<File>> entry : jars.asMap().entrySet()) {
            for (File file : entry.getValue()) {
                if (!file.getName().endsWith(".disabled")) continue;
                unload.remove(entry.getKey());
            }
        }
        for (String name : unload) {
            this.disabled.remove(name);
        }
    }

    synchronized int processPendingCleanups() {
        Iterator iterator = this.toCleanup.iterator();
        while (iterator.hasNext()) {
            Plugin plugin = (Plugin)iterator.next();
            iterator.remove();
            CleanupHandle cleanupHandle = this.cleanupHandles.remove(plugin);
            if (cleanupHandle == null) continue;
            cleanupHandle.cleanup();
        }
        return this.toCleanup.size();
    }

    private void cleanInBackground() {
        int cnt = this.toCleanup.size();
        if (0 < cnt) {
            this.cleaner.get().clean(cnt);
        }
    }

    public static String nameOf(File plugin) {
        return PluginLoader.nameOf(plugin.getName());
    }

    private static String nameOf(String name) {
        int ext;
        if (name.endsWith(".disabled")) {
            name = name.substring(0, name.lastIndexOf(46));
        }
        return 0 < (ext = name.lastIndexOf(46)) ? name.substring(0, ext) : name;
    }

    private static String getExtension(String name) {
        int ext = name.lastIndexOf(46);
        return 0 < ext ? name.substring(ext) : "";
    }

    private Plugin loadPlugin(String name, File srcPlugin, FileSnapshot snapshot) throws IOException, ClassNotFoundException, InvalidPluginException {
        String pluginName = srcPlugin.getName();
        if (PluginLoader.isJsPlugin(pluginName)) {
            return this.loadJsPlugin(name, srcPlugin, snapshot);
        }
        if (this.serverPluginFactory.handles(srcPlugin)) {
            return this.loadServerPlugin(srcPlugin, snapshot);
        }
        throw new InvalidPluginException(String.format("Unsupported plugin type: %s", srcPlugin.getName()));
    }

    private File getPluginDataDir(String name) {
        return new File(this.dataDir, name);
    }

    private String getPluginCanonicalWebUrl(String name) {
        String url = String.format("%s/plugins/%s/", CharMatcher.is('/').trimTrailingFrom(this.urlProvider.get()), name);
        return url;
    }

    private Plugin loadJsPlugin(String name, File srcJar, FileSnapshot snapshot) {
        return new JsPlugin(name, srcJar, this.pluginUserFactory.create(name), snapshot);
    }

    private ServerPlugin loadServerPlugin(File scriptFile, FileSnapshot snapshot) throws InvalidPluginException {
        String name = this.serverPluginFactory.getPluginName(scriptFile);
        return this.serverPluginFactory.get(scriptFile, snapshot, new ServerPluginProvider.PluginDescription(this.pluginUserFactory.create(name), this.getPluginCanonicalWebUrl(name), this.getPluginDataDir(name)));
    }

    static ClassLoader parentFor(Plugin.ApiType type) throws InvalidPluginException {
        switch (type) {
            case EXTENSION: {
                return PluginName.class.getClassLoader();
            }
            case PLUGIN: {
                return PluginLoader.class.getClassLoader();
            }
            case JS: {
                return JavaScriptPlugin.class.getClassLoader();
            }
        }
        throw new InvalidPluginException("Unsupported ApiType " + (Object)((Object)type));
    }

    private static Map<String, File> filterDisabled(Multimap<String, File> pluginFiles) {
        HashMap<String, File> activePlugins = Maps.newHashMapWithExpectedSize(pluginFiles.keys().size());
        for (String name : pluginFiles.keys()) {
            for (File pluginFile : pluginFiles.asMap().get(name)) {
                if (pluginFile.getName().endsWith(".disabled")) continue;
                assert (!activePlugins.containsKey(name));
                activePlugins.put(name, pluginFile);
            }
        }
        return activePlugins;
    }

    public Multimap<String, File> prunePlugins(File pluginsDir) {
        Multimap<String, File> map;
        List<File> pluginFiles = this.scanFilesInPluginsDirectory(pluginsDir);
        try {
            map = this.asMultimap(pluginFiles);
            for (String plugin : map.keySet()) {
                Iterable<File> enabled;
                Collection<File> files = map.asMap().get(plugin);
                if (files.size() == 1 || !Iterables.skip(enabled = PluginLoader.filterDisabledPlugins(files), 1).iterator().hasNext()) continue;
                File winner = Iterables.getFirst(enabled, null);
                assert (winner != null);
                ArrayList<File> elementsToRemove = Lists.newArrayList();
                ArrayList<File> elementsToAdd = Lists.newArrayList();
                for (File loser : Iterables.skip(enabled, 1)) {
                    log.warn(String.format("Plugin <%s> was disabled, because another plugin <%s> with the same name <%s> already exists", loser, winner, plugin));
                    File disabledPlugin = new File(loser + ".disabled");
                    elementsToAdd.add(disabledPlugin);
                    elementsToRemove.add(loser);
                    loser.renameTo(disabledPlugin);
                }
                Iterables.removeAll(files, elementsToRemove);
                Iterables.addAll(files, elementsToAdd);
            }
        }
        catch (IOException e) {
            log.warn("Cannot prune plugin list", e.getCause());
            return LinkedHashMultimap.create();
        }
        return map;
    }

    private List<File> scanFilesInPluginsDirectory(File pluginsDir) {
        if (pluginsDir == null || !pluginsDir.exists()) {
            return Collections.emptyList();
        }
        File[] matches = pluginsDir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                String n = pathname.getName();
                return !n.startsWith(".last_") && !n.startsWith(".next_");
            }
        });
        if (matches == null) {
            log.error("Cannot list " + pluginsDir.getAbsolutePath());
            return Collections.emptyList();
        }
        return Arrays.asList(matches);
    }

    private static Iterable<File> filterDisabledPlugins(Collection<File> files) {
        return Iterables.filter(files, new Predicate<File>(){

            @Override
            public boolean apply(File file) {
                return !file.getName().endsWith(".disabled");
            }
        });
    }

    public String getGerritPluginName(File srcFile) throws IOException {
        String fileName = srcFile.getName();
        if (PluginLoader.isJsPlugin(fileName)) {
            return fileName.substring(0, fileName.length() - 3);
        }
        if (this.serverPluginFactory.handles(srcFile)) {
            return this.serverPluginFactory.getPluginName(srcFile);
        }
        return null;
    }

    private Multimap<String, File> asMultimap(List<File> plugins) throws IOException {
        LinkedHashMultimap<String, File> map = LinkedHashMultimap.create();
        for (File srcFile : plugins) {
            map.put(this.getPluginName(srcFile), srcFile);
        }
        return map;
    }

    private static boolean isJsPlugin(String name) {
        return PluginLoader.isPlugin(name, "js");
    }

    private static boolean isPlugin(String fileName, String ext) {
        String fullExt = "." + ext;
        return fileName.endsWith(fullExt) || fileName.endsWith(fullExt + ".disabled");
    }

    private void checkRemoteInstall() throws PluginInstallException {
        if (!this.isRemoteAdminEnabled()) {
            throw new PluginInstallException("remote installation is disabled");
        }
    }
}

