/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.account;

import com.google.gerrit.common.data.Permission;
import com.google.gerrit.common.errors.InvalidUserNameException;
import com.google.gerrit.common.errors.NameAlreadyUsedException;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.AccountExternalId;
import com.google.gerrit.reviewdb.client.AccountGroup;
import com.google.gerrit.reviewdb.client.AccountGroupMember;
import com.google.gerrit.reviewdb.client.AccountGroupMemberAudit;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.account.AccountByEmailCache;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.account.AccountException;
import com.google.gerrit.server.account.AccountUserNameException;
import com.google.gerrit.server.account.AuthRequest;
import com.google.gerrit.server.account.AuthResult;
import com.google.gerrit.server.account.ChangeUserName;
import com.google.gerrit.server.account.Realm;
import com.google.gerrit.server.project.ProjectCache;
import com.google.gerrit.server.util.TimeUtil;
import com.google.gwtorm.server.OrmException;
import com.google.gwtorm.server.SchemaFactory;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class AccountManager {
    private static final Logger log = LoggerFactory.getLogger(AccountManager.class);
    private final SchemaFactory<ReviewDb> schema;
    private final AccountCache byIdCache;
    private final AccountByEmailCache byEmailCache;
    private final Realm realm;
    private final IdentifiedUser.GenericFactory userFactory;
    private final ChangeUserName.Factory changeUserNameFactory;
    private final ProjectCache projectCache;
    private final AtomicBoolean awaitsFirstAccountCheck;

    @Inject
    AccountManager(SchemaFactory<ReviewDb> schema, AccountCache byIdCache, AccountByEmailCache byEmailCache, Realm accountMapper, IdentifiedUser.GenericFactory userFactory, ChangeUserName.Factory changeUserNameFactory, ProjectCache projectCache) throws OrmException {
        this.schema = schema;
        this.byIdCache = byIdCache;
        this.byEmailCache = byEmailCache;
        this.realm = accountMapper;
        this.userFactory = userFactory;
        this.changeUserNameFactory = changeUserNameFactory;
        this.projectCache = projectCache;
        this.awaitsFirstAccountCheck = new AtomicBoolean(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Account.Id lookup(String externalId) throws AccountException {
        Account.Id id;
        ReviewDb db = this.schema.open();
        try {
            AccountExternalId ext = db.accountExternalIds().get(new AccountExternalId.Key(externalId));
            id = ext != null ? ext.getAccountId() : null;
        }
        catch (Throwable throwable) {
            try {
                db.close();
                throw throwable;
            }
            catch (OrmException e) {
                throw new AccountException("Cannot lookup account " + externalId, e);
            }
        }
        db.close();
        return id;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public AuthResult authenticate(AuthRequest who) throws AccountException {
        who = this.realm.authenticate(who);
        try (ReviewDb db = this.schema.open();){
            AccountExternalId.Key key = AccountManager.id(who);
            AccountExternalId id = db.accountExternalIds().get(key);
            if (id == null) {
                AuthResult authResult = this.create(db, who);
                return authResult;
            }
            Account act = db.accounts().get(id.getAccountId());
            if (act == null) throw new AccountException("Authentication error, account inactive");
            if (!act.isActive()) {
                throw new AccountException("Authentication error, account inactive");
            }
            this.update(db, who, id);
            AuthResult authResult = new AuthResult(id.getAccountId(), key, false);
            return authResult;
        }
        catch (OrmException e) {
            throw new AccountException("Authentication error", e);
        }
    }

    private void update(ReviewDb db, AuthRequest who, AccountExternalId extId) throws OrmException {
        IdentifiedUser user = this.userFactory.create(extId.getAccountId());
        Account toUpdate = null;
        String newEmail = who.getEmailAddress();
        String oldEmail = extId.getEmailAddress();
        if (newEmail != null && !newEmail.equals(oldEmail)) {
            if (oldEmail != null && oldEmail.equals(user.getAccount().getPreferredEmail())) {
                toUpdate = this.load(toUpdate, user.getAccountId(), db);
                toUpdate.setPreferredEmail(newEmail);
            }
            extId.setEmailAddress(newEmail);
            db.accountExternalIds().update(Collections.singleton(extId));
        }
        if (!this.realm.allowsEdit(Account.FieldName.FULL_NAME) && !AccountManager.eq(user.getAccount().getFullName(), who.getDisplayName())) {
            toUpdate = this.load(toUpdate, user.getAccountId(), db);
            toUpdate.setFullName(who.getDisplayName());
        }
        if (!this.realm.allowsEdit(Account.FieldName.USER_NAME) && !AccountManager.eq(user.getUserName(), who.getUserName())) {
            this.changeUserNameFactory.create(db, user, who.getUserName());
        }
        if (toUpdate != null) {
            db.accounts().update(Collections.singleton(toUpdate));
        }
        if (newEmail != null && !newEmail.equals(oldEmail)) {
            this.byEmailCache.evict(oldEmail);
            this.byEmailCache.evict(newEmail);
        }
        if (toUpdate != null) {
            this.byIdCache.evict(toUpdate.getId());
        }
    }

    private Account load(Account toUpdate, Account.Id accountId, ReviewDb db) throws OrmException {
        if (toUpdate == null && (toUpdate = db.accounts().get(accountId)) == null) {
            throw new OrmException("Account " + accountId + " has been deleted");
        }
        return toUpdate;
    }

    private static boolean eq(String a, String b) {
        return a == null && b == null || a != null && a.equals(b);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private AuthResult create(ReviewDb db, AuthRequest who) throws OrmException, AccountException {
        Account.Id newId = new Account.Id(db.nextAccountId());
        Account account = new Account(newId, TimeUtil.nowTs());
        AccountExternalId extId = AccountManager.createId(newId, who);
        extId.setEmailAddress(who.getEmailAddress());
        account.setFullName(who.getDisplayName());
        account.setPreferredEmail(extId.getEmailAddress());
        boolean isFirstAccount = this.awaitsFirstAccountCheck.getAndSet(false) && db.accounts().anyAccounts().toList().isEmpty();
        try {
            db.accounts().upsert(Collections.singleton(account));
            db.accountExternalIds().upsert(Collections.singleton(extId));
        }
        finally {
            this.awaitsFirstAccountCheck.set(isFirstAccount);
        }
        if (isFirstAccount) {
            Permission admin = this.projectCache.getAllProjects().getConfig().getAccessSection("GLOBAL_CAPABILITIES").getPermission("administrateServer");
            AccountGroup.UUID uuid = admin.getRules().get(0).getGroup().getUUID();
            AccountGroup g = db.accountGroups().byUUID(uuid).iterator().next();
            AccountGroup.Id adminId = g.getId();
            AccountGroupMember m = new AccountGroupMember(new AccountGroupMember.Key(newId, adminId));
            db.accountGroupMembersAudit().insert(Collections.singleton(new AccountGroupMemberAudit(m, newId, TimeUtil.nowTs())));
            db.accountGroupMembers().insert(Collections.singleton(m));
        }
        if (who.getUserName() != null) {
            String message;
            IdentifiedUser user = this.userFactory.create(newId);
            try {
                this.changeUserNameFactory.create(db, user, who.getUserName()).call();
            }
            catch (NameAlreadyUsedException e) {
                message = "Cannot assign user name \"" + who.getUserName() + "\" to account " + newId + "; name already in use.";
                this.handleSettingUserNameFailure(db, account, extId, message, e, false);
            }
            catch (InvalidUserNameException e) {
                message = "Cannot assign user name \"" + who.getUserName() + "\" to account " + newId + "; name does not conform.";
                this.handleSettingUserNameFailure(db, account, extId, message, e, false);
            }
            catch (OrmException e) {
                message = "Cannot assign user name";
                this.handleSettingUserNameFailure(db, account, extId, "Cannot assign user name", e, true);
            }
        }
        this.byEmailCache.evict(account.getPreferredEmail());
        this.realm.onCreateAccount(who, account);
        return new AuthResult(newId, extId.getKey(), true);
    }

    private void handleSettingUserNameFailure(ReviewDb db, Account account, AccountExternalId extId, String errorMessage, Exception e, boolean logException) throws AccountUserNameException, OrmException {
        if (logException) {
            log.error(errorMessage, e);
        } else {
            log.error(errorMessage);
        }
        if (!this.realm.allowsEdit(Account.FieldName.USER_NAME)) {
            db.accounts().delete(Collections.singleton(account));
            db.accountExternalIds().delete(Collections.singleton(extId));
            throw new AccountUserNameException(errorMessage, e);
        }
    }

    private static AccountExternalId createId(Account.Id newId, AuthRequest who) {
        String ext = who.getExternalId();
        return new AccountExternalId(newId, new AccountExternalId.Key(ext));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AuthResult link(Account.Id to, AuthRequest who) throws AccountException, OrmException {
        try (ReviewDb db = this.schema.open();){
            who = this.realm.link(db, to, who);
            AccountExternalId.Key key = AccountManager.id(who);
            AccountExternalId extId = db.accountExternalIds().get(key);
            if (extId != null) {
                if (!extId.getAccountId().equals(to)) {
                    throw new AccountException("Identity in use by another account");
                }
                this.update(db, who, extId);
            } else {
                Account a;
                extId = AccountManager.createId(to, who);
                extId.setEmailAddress(who.getEmailAddress());
                db.accountExternalIds().insert(Collections.singleton(extId));
                if (who.getEmailAddress() != null && (a = db.accounts().get(to)).getPreferredEmail() == null) {
                    a.setPreferredEmail(who.getEmailAddress());
                    db.accounts().update(Collections.singleton(a));
                }
                if (who.getEmailAddress() != null) {
                    this.byEmailCache.evict(who.getEmailAddress());
                    this.byIdCache.evict(to);
                }
            }
            AuthResult authResult = new AuthResult(to, key, false);
            return authResult;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AuthResult unlink(Account.Id from, AuthRequest who) throws AccountException, OrmException {
        try (ReviewDb db = this.schema.open();){
            who = this.realm.unlink(db, from, who);
            AccountExternalId.Key key = AccountManager.id(who);
            AccountExternalId extId = db.accountExternalIds().get(key);
            if (extId != null) {
                if (!extId.getAccountId().equals(from)) {
                    throw new AccountException("Identity in use by another account");
                }
                db.accountExternalIds().delete(Collections.singleton(extId));
                if (who.getEmailAddress() != null) {
                    Account a = db.accounts().get(from);
                    if (a.getPreferredEmail() != null && a.getPreferredEmail().equals(who.getEmailAddress())) {
                        a.setPreferredEmail(null);
                        db.accounts().update(Collections.singleton(a));
                    }
                    this.byEmailCache.evict(who.getEmailAddress());
                    this.byIdCache.evict(from);
                }
            } else {
                throw new AccountException("Identity not found");
            }
            AuthResult authResult = new AuthResult(from, key, false);
            return authResult;
        }
    }

    private static AccountExternalId.Key id(AuthRequest who) {
        return new AccountExternalId.Key(who.getExternalId());
    }
}

