/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.notedb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.primitives.Ints;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.CommentRange;
import com.google.gerrit.reviewdb.client.Patch;
import com.google.gerrit.reviewdb.client.PatchLineComment;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.RevId;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.notedb.ChangeNotes;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Inject;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectInserter;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.notes.Note;
import org.eclipse.jgit.notes.NoteMap;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.util.GitDateFormatter;
import org.eclipse.jgit.util.GitDateParser;
import org.eclipse.jgit.util.MutableInteger;
import org.eclipse.jgit.util.QuotedString;
import org.eclipse.jgit.util.RawParseUtils;

public class CommentsInNotesUtil {
    private static final String AUTHOR = "Author";
    private static final String BASE_PATCH_SET = "Base-for-patch-set";
    private static final String COMMENT_RANGE = "Comment-range";
    private static final String FILE = "File";
    private static final String LENGTH = "Bytes";
    private static final String PARENT = "Parent";
    private static final String PATCH_SET = "Patch-set";
    private static final String REVISION = "Revision";
    private static final String UUID = "UUID";
    private final AccountCache accountCache;
    private final PersonIdent serverIdent;

    public static NoteMap parseCommentsFromNotes(Repository repo, String refName, RevWalk walk, Change.Id changeId, Multimap<PatchSet.Id, PatchLineComment> commentsForBase, Multimap<PatchSet.Id, PatchLineComment> commentsForPs, PatchLineComment.Status status) throws IOException, ConfigInvalidException {
        Ref ref = repo.getRef(refName);
        if (ref == null) {
            return null;
        }
        RevCommit commit = walk.parseCommit(ref.getObjectId());
        NoteMap noteMap = NoteMap.read(walk.getObjectReader(), commit);
        for (Note note : noteMap) {
            byte[] bytes = walk.getObjectReader().open(note.getData(), 3).getBytes();
            List<PatchLineComment> result = CommentsInNotesUtil.parseNote(bytes, changeId, status);
            if (result == null || result.isEmpty()) continue;
            PatchSet.Id psId = result.get(0).getKey().getParentKey().getParentKey();
            short side = result.get(0).getSide();
            if (side == 0) {
                commentsForBase.putAll(psId, result);
                continue;
            }
            commentsForPs.putAll(psId, result);
        }
        return noteMap;
    }

    public static List<PatchLineComment> parseNote(byte[] note, Change.Id changeId, PatchLineComment.Status status) throws ConfigInvalidException {
        ArrayList<PatchLineComment> result = Lists.newArrayList();
        int sizeOfNote = note.length;
        Charset enc = RawParseUtils.parseEncoding(note);
        MutableInteger curr = new MutableInteger();
        curr.value = 0;
        boolean isForBase = RawParseUtils.match(note, curr.value, PATCH_SET.getBytes(StandardCharsets.UTF_8)) < 0;
        PatchSet.Id psId = CommentsInNotesUtil.parsePsId(note, curr, changeId, enc, isForBase ? BASE_PATCH_SET : PATCH_SET);
        RevId revId = new RevId(CommentsInNotesUtil.parseStringField(note, curr, changeId, enc, REVISION));
        PatchLineComment c = null;
        while (curr.value < sizeOfNote) {
            String previousFileName = c == null ? null : c.getKey().getParentKey().getFileName();
            c = CommentsInNotesUtil.parseComment(note, curr, previousFileName, psId, revId, isForBase, enc, status);
            result.add(c);
        }
        return result;
    }

    public static String formatTime(PersonIdent ident, Timestamp t) {
        GitDateFormatter dateFormatter = new GitDateFormatter(GitDateFormatter.Format.DEFAULT);
        PersonIdent newIdent = new PersonIdent(ident, t);
        return dateFormatter.formatDate(newIdent);
    }

    public static PatchSet.Id getCommentPsId(PatchLineComment plc) {
        return plc.getKey().getParentKey().getParentKey();
    }

    private static PatchLineComment parseComment(byte[] note, MutableInteger curr, String currentFileName, PatchSet.Id psId, RevId revId, boolean isForBase, Charset enc, PatchLineComment.Status status) throws ConfigInvalidException {
        boolean newFile;
        Change.Id changeId = psId.getParentKey();
        boolean bl = newFile = RawParseUtils.match(note, curr.value, FILE.getBytes(StandardCharsets.UTF_8)) != -1;
        if (newFile) {
            currentFileName = CommentsInNotesUtil.parseFilename(note, curr, changeId, enc);
        } else if (currentFileName == null) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", FILE);
        }
        CommentRange range = CommentsInNotesUtil.parseCommentRange(note, curr, changeId);
        if (range == null) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", COMMENT_RANGE);
        }
        Timestamp commentTime = CommentsInNotesUtil.parseTimestamp(note, curr, changeId, enc);
        Account.Id aId = CommentsInNotesUtil.parseAuthor(note, curr, changeId, enc);
        boolean hasParent = RawParseUtils.match(note, curr.value, PARENT.getBytes(enc)) != -1;
        String parentUUID = null;
        if (hasParent) {
            parentUUID = CommentsInNotesUtil.parseStringField(note, curr, changeId, enc, PARENT);
        }
        String uuid = CommentsInNotesUtil.parseStringField(note, curr, changeId, enc, UUID);
        int commentLength = CommentsInNotesUtil.parseCommentLength(note, curr, changeId, enc);
        String message = RawParseUtils.decode(enc, note, curr.value, curr.value + commentLength);
        CommentsInNotesUtil.checkResult(message, "message contents", changeId);
        PatchLineComment plc = new PatchLineComment(new PatchLineComment.Key(new Patch.Key(psId, currentFileName), uuid), range.getEndLine(), aId, parentUUID, commentTime);
        plc.setMessage(message);
        plc.setSide((short)(!isForBase ? 1 : 0));
        if (range.getStartCharacter() != -1) {
            plc.setRange(range);
        }
        plc.setRevId(revId);
        plc.setStatus(status);
        curr.value = RawParseUtils.nextLF(note, curr.value + commentLength);
        curr.value = RawParseUtils.nextLF(note, curr.value);
        return plc;
    }

    private static String parseStringField(byte[] note, MutableInteger curr, Change.Id changeId, Charset enc, String fieldName) throws ConfigInvalidException {
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        CommentsInNotesUtil.checkHeaderLineFormat(note, curr, fieldName, enc, changeId);
        int startOfField = RawParseUtils.endOfFooterLineKey(note, curr.value) + 2;
        curr.value = endOfLine;
        return RawParseUtils.decode(enc, note, startOfField, endOfLine - 1);
    }

    private static CommentRange parseCommentRange(byte[] note, MutableInteger ptr, Change.Id changeId) throws ConfigInvalidException {
        CommentRange range = new CommentRange(-1, -1, -1, -1);
        int startLine = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (startLine == 0) {
            return null;
        }
        if (note[ptr.value] == 10) {
            range.setEndLine(startLine);
            return range;
        }
        if (note[ptr.value] == 58) {
            range.setStartLine(startLine);
            ++ptr.value;
        } else {
            return null;
        }
        int startChar = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (startChar == 0) {
            return null;
        }
        if (note[ptr.value] == 45) {
            range.setStartCharacter(startChar);
            ++ptr.value;
        } else {
            return null;
        }
        int endLine = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (endLine == 0) {
            return null;
        }
        if (note[ptr.value] == 58) {
            range.setEndLine(endLine);
            ++ptr.value;
        } else {
            return null;
        }
        int endChar = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (endChar == 0) {
            return null;
        }
        if (note[ptr.value] == 10) {
            range.setEndCharacter(endChar);
            ++ptr.value;
        } else {
            return null;
        }
        return range;
    }

    private static PatchSet.Id parsePsId(byte[] note, MutableInteger curr, Change.Id changeId, Charset enc, String fieldName) throws ConfigInvalidException {
        CommentsInNotesUtil.checkHeaderLineFormat(note, curr, fieldName, enc, changeId);
        int startOfPsId = RawParseUtils.endOfFooterLineKey(note, curr.value) + 1;
        MutableInteger i = new MutableInteger();
        int patchSetId = RawParseUtils.parseBase10(note, startOfPsId, i);
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        if (i.value != endOfLine - 1) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
        CommentsInNotesUtil.checkResult(patchSetId, "patchset id", changeId);
        curr.value = endOfLine;
        return new PatchSet.Id(changeId, patchSetId);
    }

    private static String parseFilename(byte[] note, MutableInteger curr, Change.Id changeId, Charset enc) throws ConfigInvalidException {
        int endOfLine;
        CommentsInNotesUtil.checkHeaderLineFormat(note, curr, FILE, enc, changeId);
        int startOfFileName = RawParseUtils.endOfFooterLineKey(note, curr.value) + 2;
        curr.value = endOfLine = RawParseUtils.nextLF(note, curr.value);
        curr.value = RawParseUtils.nextLF(note, curr.value);
        return QuotedString.GIT_PATH.dequote(RawParseUtils.decode(enc, note, startOfFileName, endOfLine - 1));
    }

    private static Timestamp parseTimestamp(byte[] note, MutableInteger curr, Change.Id changeId, Charset enc) throws ConfigInvalidException {
        Timestamp commentTime;
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        String dateString = RawParseUtils.decode(enc, note, curr.value, endOfLine - 1);
        try {
            commentTime = new Timestamp(GitDateParser.parse(dateString, null).getTime());
        }
        catch (ParseException e) {
            throw new ConfigInvalidException("could not parse comment timestamp", e);
        }
        curr.value = endOfLine;
        return CommentsInNotesUtil.checkResult(commentTime, "comment timestamp", changeId);
    }

    private static Account.Id parseAuthor(byte[] note, MutableInteger curr, Change.Id changeId, Charset enc) throws ConfigInvalidException {
        CommentsInNotesUtil.checkHeaderLineFormat(note, curr, AUTHOR, enc, changeId);
        int startOfAccountId = RawParseUtils.endOfFooterLineKey(note, curr.value) + 2;
        PersonIdent ident = RawParseUtils.parsePersonIdent(note, startOfAccountId);
        Account.Id aId = CommentsInNotesUtil.parseIdent(ident, changeId);
        curr.value = RawParseUtils.nextLF(note, curr.value);
        return CommentsInNotesUtil.checkResult(aId, "comment author", changeId);
    }

    private static int parseCommentLength(byte[] note, MutableInteger curr, Change.Id changeId, Charset enc) throws ConfigInvalidException {
        CommentsInNotesUtil.checkHeaderLineFormat(note, curr, LENGTH, enc, changeId);
        int startOfLength = RawParseUtils.endOfFooterLineKey(note, curr.value) + 1;
        MutableInteger i = new MutableInteger();
        int commentLength = RawParseUtils.parseBase10(note, startOfLength, i);
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        if (i.value != endOfLine - 1) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", PATCH_SET);
        }
        curr.value = endOfLine;
        return CommentsInNotesUtil.checkResult(commentLength, "comment length", changeId);
    }

    private static <T> T checkResult(T o, String fieldName, Change.Id changeId) throws ConfigInvalidException {
        if (o == null) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
        return o;
    }

    private static int checkResult(int i, String fieldName, Change.Id changeId) throws ConfigInvalidException {
        if (i <= 0) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
        return i;
    }

    private PersonIdent newIdent(Account author, Date when) {
        return new PersonIdent(author.getFullName(), author.getId().get() + "@" + "gerrit", when, this.serverIdent.getTimeZone());
    }

    private static Account.Id parseIdent(PersonIdent ident, Change.Id changeId) throws ConfigInvalidException {
        String email = ident.getEmailAddress();
        int at = email.indexOf(64);
        if (at >= 0) {
            String host = email.substring(at + 1, email.length());
            Integer id = Ints.tryParse(email.substring(0, at));
            if (id != null && host.equals("gerrit")) {
                return new Account.Id(id);
            }
        }
        throw ChangeNotes.parseException(changeId, "invalid identity, expected <id>@%s: %s", "gerrit", email);
    }

    private void appendHeaderField(PrintWriter writer, String field, String value) throws IOException {
        writer.print(field);
        writer.print(": ");
        writer.print(value);
        writer.print('\n');
    }

    private static void checkHeaderLineFormat(byte[] note, MutableInteger curr, String fieldName, Charset enc, Change.Id changeId) throws ConfigInvalidException {
        boolean correct = RawParseUtils.match(note, curr.value, fieldName.getBytes(enc)) != -1;
        correct &= note[curr.value + fieldName.length()] == 58;
        if (!(correct &= note[curr.value + fieldName.length() + 1] == 32)) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
    }

    @Inject
    @VisibleForTesting
    public CommentsInNotesUtil(AccountCache accountCache, @GerritPersonIdent PersonIdent serverIdent) {
        this.accountCache = accountCache;
        this.serverIdent = serverIdent;
    }

    public byte[] buildNote(List<PatchLineComment> comments) throws OrmException, IOException {
        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        OutputStreamWriter streamWriter = new OutputStreamWriter((OutputStream)buf, StandardCharsets.UTF_8);
        PrintWriter writer = new PrintWriter(streamWriter);
        PatchLineComment first = comments.get(0);
        short side = first.getSide();
        PatchSet.Id psId = CommentsInNotesUtil.getCommentPsId(first);
        this.appendHeaderField(writer, side == 0 ? BASE_PATCH_SET : PATCH_SET, Integer.toString(psId.get()));
        this.appendHeaderField(writer, REVISION, first.getRevId().get());
        String currentFilename = null;
        for (PatchLineComment c : comments) {
            CommentRange range;
            PatchSet.Id currentPsId = CommentsInNotesUtil.getCommentPsId(c);
            Preconditions.checkArgument(psId.equals(currentPsId), "All comments being added must all have the same PatchSet.Id. Thecomment below does not have the same PatchSet.Id as the others (%d).\n%s", psId.toString(), c.toString());
            Preconditions.checkArgument(side == c.getSide(), "All comments being added must all have the same side. Thecomment below does not have the same side as the others (%d).\n%s", side, c.toString());
            String commentFilename = QuotedString.GIT_PATH.quote(c.getKey().getParentKey().getFileName());
            if (!commentFilename.equals(currentFilename)) {
                currentFilename = commentFilename;
                writer.print("File: ");
                writer.print(commentFilename);
                writer.print("\n\n");
            }
            if ((range = c.getRange()) != null) {
                writer.print(range.getStartLine());
                writer.print(':');
                writer.print(range.getStartCharacter());
                writer.print('-');
                writer.print(range.getEndLine());
                writer.print(':');
                writer.print(range.getEndCharacter());
            } else {
                writer.print(c.getLine());
            }
            writer.print("\n");
            writer.print(CommentsInNotesUtil.formatTime(this.serverIdent, c.getWrittenOn()));
            writer.print("\n");
            PersonIdent ident = this.newIdent(this.accountCache.get(c.getAuthor()).getAccount(), c.getWrittenOn());
            String nameString = ident.getName() + " <" + ident.getEmailAddress() + ">";
            this.appendHeaderField(writer, AUTHOR, nameString);
            String parent = c.getParentUuid();
            if (parent != null) {
                this.appendHeaderField(writer, PARENT, parent);
            }
            this.appendHeaderField(writer, UUID, c.getKey().get());
            byte[] messageBytes = c.getMessage().getBytes(StandardCharsets.UTF_8);
            this.appendHeaderField(writer, LENGTH, Integer.toString(messageBytes.length));
            writer.print(c.getMessage());
            writer.print("\n\n");
        }
        writer.close();
        return buf.toByteArray();
    }

    public void writeCommentsToNoteMap(NoteMap noteMap, List<PatchLineComment> allComments, ObjectInserter inserter) throws OrmException, IOException {
        Preconditions.checkArgument(!allComments.isEmpty(), "No comments to write; to delete, use removeNoteFromNoteMap().");
        ObjectId commitOID = ObjectId.fromString(allComments.get(0).getRevId().get());
        Collections.sort(allComments, ChangeNotes.PatchLineCommentComparator);
        byte[] note = this.buildNote(allComments);
        ObjectId noteId = inserter.insert(3, note, 0, note.length);
        noteMap.set(commitOID, noteId);
    }

    public void removeNote(NoteMap noteMap, RevId commitId) throws IOException {
        noteMap.remove(ObjectId.fromString(commitId.get()));
    }
}

