/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.notedb;

import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.common.TimeUtil;
import com.google.gerrit.extensions.restapi.RestApiException;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.reviewdb.client.RefNames;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.reviewdb.server.ReviewDbUtil;
import com.google.gerrit.server.InternalUser;
import com.google.gerrit.server.config.AllUsersName;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.gerrit.server.git.RepoRefCache;
import com.google.gerrit.server.index.change.ChangeField;
import com.google.gerrit.server.notedb.ChangeUpdate;
import com.google.gerrit.server.notedb.NoteDbChangeState;
import com.google.gerrit.server.notedb.NoteDbUpdateManager;
import com.google.gerrit.server.notedb.rebuild.ChangeRebuilder;
import com.google.gerrit.server.project.ChangeControl;
import com.google.gerrit.server.query.change.ChangeData;
import com.google.gerrit.server.query.change.InternalChangeQuery;
import com.google.gerrit.server.update.BatchUpdate;
import com.google.gerrit.server.update.BatchUpdateOp;
import com.google.gerrit.server.update.ChangeContext;
import com.google.gerrit.server.update.UpdateException;
import com.google.gwtorm.server.AtomicUpdate;
import com.google.gwtorm.server.OrmException;
import com.google.gwtorm.server.OrmRuntimeException;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.eclipse.jgit.errors.RepositoryNotFoundException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class PrimaryStorageMigrator {
    private static final Logger log = LoggerFactory.getLogger(PrimaryStorageMigrator.class);
    private final AllUsersName allUsers;
    private final BatchUpdate.Factory batchUpdateFactory;
    private final ChangeControl.GenericFactory changeControlFactory;
    private final ChangeRebuilder rebuilder;
    private final ChangeUpdate.Factory updateFactory;
    private final GitRepositoryManager repoManager;
    private final InternalUser.Factory internalUserFactory;
    private final Provider<InternalChangeQuery> queryProvider;
    private final Provider<ReviewDb> db;
    private final long skewMs;
    private final long timeoutMs;
    private final Retryer<NoteDbChangeState> testEnsureRebuiltRetryer;

    @Inject
    PrimaryStorageMigrator(@GerritServerConfig Config cfg, Provider<ReviewDb> db, GitRepositoryManager repoManager, AllUsersName allUsers, ChangeRebuilder rebuilder, ChangeControl.GenericFactory changeControlFactory, Provider<InternalChangeQuery> queryProvider, ChangeUpdate.Factory updateFactory, InternalUser.Factory internalUserFactory, BatchUpdate.Factory batchUpdateFactory) {
        this(cfg, db, repoManager, allUsers, rebuilder, null, changeControlFactory, queryProvider, updateFactory, internalUserFactory, batchUpdateFactory);
    }

    @VisibleForTesting
    public PrimaryStorageMigrator(Config cfg, Provider<ReviewDb> db, GitRepositoryManager repoManager, AllUsersName allUsers, ChangeRebuilder rebuilder, @Nullable Retryer<NoteDbChangeState> testEnsureRebuiltRetryer, ChangeControl.GenericFactory changeControlFactory, Provider<InternalChangeQuery> queryProvider, ChangeUpdate.Factory updateFactory, InternalUser.Factory internalUserFactory, BatchUpdate.Factory batchUpdateFactory) {
        this.db = db;
        this.repoManager = repoManager;
        this.allUsers = allUsers;
        this.rebuilder = rebuilder;
        this.testEnsureRebuiltRetryer = testEnsureRebuiltRetryer;
        this.changeControlFactory = changeControlFactory;
        this.queryProvider = queryProvider;
        this.updateFactory = updateFactory;
        this.internalUserFactory = internalUserFactory;
        this.batchUpdateFactory = batchUpdateFactory;
        this.skewMs = NoteDbChangeState.getReadOnlySkew(cfg);
        String s = "notedb";
        this.timeoutMs = cfg.getTimeUnit(s, null, "primaryStorageMigrationTimeout", TimeUnit.MILLISECONDS.convert(60L, TimeUnit.SECONDS), TimeUnit.MILLISECONDS);
    }

    public void migrateToNoteDbPrimary(Change.Id id) throws OrmException, IOException {
        NoteDbChangeState rebuiltState;
        Stopwatch sw = Stopwatch.createStarted();
        Change readOnlyChange = this.setReadOnlyInReviewDb(id);
        if (readOnlyChange == null) {
            return;
        }
        try {
            rebuiltState = this.ensureRebuiltRetryer(sw).call(() -> this.ensureRebuilt(readOnlyChange.getProject(), id, NoteDbChangeState.parse(readOnlyChange)));
        }
        catch (RetryException | ExecutionException e) {
            throw new OrmException(e);
        }
        this.setPrimaryStorageNoteDb(id, rebuiltState);
        log.info("Migrated change {} to NoteDb primary in {}ms", (Object)id, (Object)sw.elapsed(TimeUnit.MILLISECONDS));
    }

    private Change setReadOnlyInReviewDb(final Change.Id id) throws OrmException {
        final AtomicBoolean alreadyMigrated = new AtomicBoolean(false);
        Change result = this.db().changes().atomicUpdate(id, new AtomicUpdate<Change>(){

            @Override
            public Change update(Change change) {
                NoteDbChangeState state = NoteDbChangeState.parse(change);
                if (state == null) {
                    throw new OrmRuntimeException("change " + id + " has no note_db_state; rebuild it first");
                }
                NoteDbChangeState.checkNotReadOnly(change, PrimaryStorageMigrator.this.skewMs);
                if (state.getPrimaryStorage() != NoteDbChangeState.PrimaryStorage.NOTE_DB) {
                    Timestamp now = TimeUtil.nowTs();
                    Timestamp until = new Timestamp(now.getTime() + PrimaryStorageMigrator.this.timeoutMs);
                    change.setNoteDbState(state.withReadOnlyUntil(until).toString());
                } else {
                    alreadyMigrated.set(true);
                }
                return change;
            }
        });
        return alreadyMigrated.get() ? null : result;
    }

    private Retryer<NoteDbChangeState> ensureRebuiltRetryer(Stopwatch sw) {
        if (this.testEnsureRebuiltRetryer != null) {
            return this.testEnsureRebuiltRetryer;
        }
        long remainingNanos = TimeUnit.MILLISECONDS.toNanos(this.timeoutMs) / 2L - sw.elapsed(TimeUnit.NANOSECONDS);
        remainingNanos = Math.max(remainingNanos, 0L);
        return RetryerBuilder.newBuilder().retryIfException(e -> e instanceof IOException || e instanceof OrmException).withWaitStrategy(WaitStrategies.join(WaitStrategies.exponentialWait(250L, TimeUnit.MILLISECONDS), WaitStrategies.randomWait(50L, TimeUnit.MILLISECONDS))).withStopStrategy(StopStrategies.stopAfterDelay(remainingNanos, TimeUnit.NANOSECONDS)).build();
    }

    private NoteDbChangeState ensureRebuilt(Project.NameKey project, Change.Id id, NoteDbChangeState readOnlyState) throws IOException, OrmException, RepositoryNotFoundException {
        try (Repository changeRepo = this.repoManager.openRepository(project);
             Repository allUsersRepo = this.repoManager.openRepository(this.allUsers);){
            if (!readOnlyState.isUpToDate(new RepoRefCache(changeRepo), new RepoRefCache(allUsersRepo))) {
                NoteDbUpdateManager.Result r = this.rebuilder.rebuildEvenIfReadOnly(this.db(), id);
                Preconditions.checkState(r.newState().getReadOnlyUntil().equals(readOnlyState.getReadOnlyUntil()), "state after rebuilding has different read-only lease: %s != %s", (Object)r.newState(), (Object)readOnlyState);
                readOnlyState = r.newState();
            }
        }
        return readOnlyState;
    }

    private void setPrimaryStorageNoteDb(final Change.Id id, final NoteDbChangeState expectedState) throws OrmException {
        this.db().changes().atomicUpdate(id, new AtomicUpdate<Change>(){

            @Override
            public Change update(Change change) {
                NoteDbChangeState state = NoteDbChangeState.parse(change);
                if (!Objects.equals(state, expectedState)) {
                    throw new OrmRuntimeException(PrimaryStorageMigrator.this.badState(state, expectedState));
                }
                Timestamp until = state.getReadOnlyUntil().get();
                if (TimeUtil.nowTs().after(until)) {
                    throw new OrmRuntimeException("read-only lease on change " + id + " expired at " + until);
                }
                change.setNoteDbState("N");
                return change;
            }
        });
    }

    private ReviewDb db() {
        return ReviewDbUtil.unwrapDb(this.db.get());
    }

    private String badState(NoteDbChangeState actual, NoteDbChangeState expected) {
        return "state changed unexpectedly: " + actual + " != " + expected;
    }

    public void migrateToReviewDbPrimary(Change.Id id, @Nullable Project.NameKey project) throws OrmException, IOException {
        Stopwatch sw = Stopwatch.createStarted();
        if (project == null) {
            project = this.getProject(id);
        }
        ObjectId newMetaId = this.setReadOnlyInNoteDb(project, id);
        this.rebuilder.rebuildReviewDb(this.db(), project, id);
        this.setPrimaryStorageReviewDb(id, newMetaId);
        this.releaseReadOnlyLeaseInNoteDb(project, id);
        log.info("Migrated change {} to ReviewDb primary in {}ms", (Object)id, (Object)sw.elapsed(TimeUnit.MILLISECONDS));
    }

    private ObjectId setReadOnlyInNoteDb(Project.NameKey project, Change.Id id) throws OrmException, IOException {
        Timestamp now = TimeUtil.nowTs();
        Timestamp until = new Timestamp(now.getTime() + this.timeoutMs);
        ChangeUpdate update = this.updateFactory.create(this.changeControlFactory.controlFor(this.db.get(), project, id, this.internalUserFactory.create()));
        update.setReadOnlyUntil(until);
        return update.commit();
    }

    private void setPrimaryStorageReviewDb(final Change.Id id, ObjectId newMetaId) throws OrmException, IOException {
        ImmutableMap.Builder<Account.Id, ObjectId> draftIds = ImmutableMap.builder();
        try (Repository repo = this.repoManager.openRepository(this.allUsers);){
            for (Ref draftRef : repo.getRefDatabase().getRefs(RefNames.refsDraftCommentsPrefix(id)).values()) {
                Account.Id accountId = Account.Id.fromRef(draftRef.getName());
                if (accountId == null) continue;
                draftIds.put(accountId, draftRef.getObjectId().copy());
            }
        }
        final NoteDbChangeState newState = new NoteDbChangeState(id, NoteDbChangeState.PrimaryStorage.REVIEW_DB, Optional.of(NoteDbChangeState.RefState.create(newMetaId, draftIds.build())), Optional.empty());
        this.db().changes().atomicUpdate(id, new AtomicUpdate<Change>(){

            @Override
            public Change update(Change change) {
                if (NoteDbChangeState.PrimaryStorage.of(change) != NoteDbChangeState.PrimaryStorage.NOTE_DB) {
                    throw new OrmRuntimeException("change " + id + " is not NoteDb primary: " + change.getNoteDbState());
                }
                change.setNoteDbState(newState.toString());
                return change;
            }
        });
    }

    private void releaseReadOnlyLeaseInNoteDb(Project.NameKey project, Change.Id id) throws OrmException {
        try (BatchUpdate bu = this.batchUpdateFactory.create(this.db.get(), project, this.internalUserFactory.create(), TimeUtil.nowTs());){
            bu.addOp(id, new BatchUpdateOp(){

                @Override
                public boolean updateChange(ChangeContext ctx) {
                    ctx.getUpdate(ctx.getChange().currentPatchSetId()).setReadOnlyUntil(new Timestamp(0L));
                    return true;
                }
            });
            bu.execute();
        }
        catch (RestApiException | UpdateException e) {
            throw new OrmException(e);
        }
    }

    private Project.NameKey getProject(Change.Id id) throws OrmException {
        List<ChangeData> cds = ((InternalChangeQuery)this.queryProvider.get().setRequestedFields(ImmutableSet.of(ChangeField.PROJECT.getName()))).byLegacyChangeId(id);
        TreeSet<Project.NameKey> projects = new TreeSet<Project.NameKey>();
        for (ChangeData cd : cds) {
            projects.add(cd.project());
        }
        if (projects.size() != 1) {
            throw new OrmException("zero or multiple projects found for change " + id + ", must specify project explicitly: " + projects);
        }
        return (Project.NameKey)projects.iterator().next();
    }
}

