/*
 * Decompiled with CFR 0.152.
 */
package com.google.template.soy.soytree;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.template.soy.base.SourceLocation;
import com.google.template.soy.basetree.CopyState;
import com.google.template.soy.soytree.AbstractSoyNode;
import com.google.template.soy.soytree.HtmlContext;
import com.google.template.soy.soytree.SoyNode;
import java.util.Arrays;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class RawTextNode
extends AbstractSoyNode
implements SoyNode.StandaloneNode {
    private static final Pattern SPECIAL_CHARS_TO_ESCAPE = Pattern.compile("[\n\r\t{}]");
    private static final Map<String, String> SPECIAL_CHAR_TO_TAG = ImmutableMap.builder().put("\n", "{\\n}").put("\r", "{\\r}").put("\t", "{\\t}").put("{", "{lb}").put("}", "{rb}").build();
    private final String rawText;
    @Nullable
    private final SourceOffsets offsets;
    @Nullable
    private HtmlContext htmlContext;

    public RawTextNode(int id, String rawText, SourceLocation sourceLocation) {
        this(id, rawText, sourceLocation, SourceOffsets.fromLocation(sourceLocation, rawText.length()));
    }

    public RawTextNode(int id, String rawText, SourceLocation sourceLocation, HtmlContext htmlContext) {
        super(id, sourceLocation);
        Preconditions.checkArgument(!rawText.isEmpty(), "you can't create empty RawTextNodes");
        this.rawText = rawText;
        this.htmlContext = htmlContext;
        this.offsets = SourceOffsets.fromLocation(sourceLocation, rawText.length());
    }

    public RawTextNode(int id, String rawText, SourceLocation sourceLocation, SourceOffsets offsets) {
        super(id, sourceLocation);
        Preconditions.checkArgument(!rawText.isEmpty(), "you can't create empty RawTextNodes");
        this.rawText = rawText;
        this.offsets = offsets;
    }

    private RawTextNode(RawTextNode orig, CopyState copyState) {
        super(orig, copyState);
        this.rawText = orig.rawText;
        this.htmlContext = orig.htmlContext;
        this.offsets = orig.offsets;
    }

    public HtmlContext getHtmlContext() {
        return Preconditions.checkNotNull(this.htmlContext, "Cannot access HtmlContext before HtmlTransformVisitor");
    }

    @Override
    public SoyNode.Kind getKind() {
        return SoyNode.Kind.RAW_TEXT_NODE;
    }

    public void setHtmlContext(HtmlContext value) {
        this.htmlContext = value;
    }

    public String getRawText() {
        return this.rawText;
    }

    public SourceLocation.Point locationOf(int i) {
        Preconditions.checkElementIndex(i, this.rawText.length(), "index");
        if (this.offsets == null) {
            return SourceLocation.Point.UNKNOWN_POINT;
        }
        return this.offsets.getPoint(this.rawText, i);
    }

    public SourceLocation substringLocation(int start, int end) {
        Preconditions.checkElementIndex(start, this.rawText.length(), "start");
        Preconditions.checkArgument(start < end);
        Preconditions.checkArgument(end <= this.rawText.length());
        if (this.offsets == null) {
            return this.getSourceLocation();
        }
        return new SourceLocation(this.getSourceLocation().getFilePath(), this.offsets.getPoint(this.rawText, start), this.offsets.getPoint(this.rawText, end - 1));
    }

    public RawTextNode substring(int newId, int start, int end) {
        Preconditions.checkArgument(start >= 0);
        Preconditions.checkArgument(start < end);
        Preconditions.checkArgument(end <= this.rawText.length());
        if (start == 0 && end == this.rawText.length()) {
            return this;
        }
        String newText = this.rawText.substring(start, end);
        SourceOffsets newOffsets = null;
        SourceLocation newLocation = this.getSourceLocation();
        if (this.offsets != null) {
            newOffsets = this.offsets.substring(start, end, this.rawText);
            newLocation = newOffsets.getSourceLocation(this.getSourceLocation().getFilePath());
        }
        return new RawTextNode(newId, newText, newLocation, newOffsets);
    }

    public RawTextNode concat(int newId, RawTextNode node) {
        Preconditions.checkNotNull(node);
        String newText = this.rawText.concat(node.getRawText());
        SourceOffsets newOffsets = null;
        SourceLocation newLocation = this.getSourceLocation().extend(node.getSourceLocation());
        if (this.offsets != null && node.offsets != null) {
            newOffsets = this.offsets.concat(node.offsets);
        }
        return new RawTextNode(newId, newText, newLocation, newOffsets);
    }

    @Override
    public String toSourceString() {
        StringBuffer sb = new StringBuffer();
        Matcher matcher = SPECIAL_CHARS_TO_ESCAPE.matcher(this.rawText);
        while (matcher.find()) {
            String specialCharTag = SPECIAL_CHAR_TO_TAG.get(matcher.group());
            matcher.appendReplacement(sb, Matcher.quoteReplacement(specialCharTag));
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    @Override
    public SoyNode.BlockNode getParent() {
        return (SoyNode.BlockNode)super.getParent();
    }

    @Override
    public RawTextNode copy(CopyState copyState) {
        return new RawTextNode(this, copyState);
    }

    public static final class SourceOffsets {
        private final int[] indexes;
        private final int[] columns;
        private final int[] lines;

        @Nullable
        static SourceOffsets fromLocation(SourceLocation location, int length) {
            if (!location.isKnown()) {
                return null;
            }
            return new Builder().add(0, location.getBeginLine(), location.getBeginColumn(), location.getEndLine(), location.getEndColumn()).build(length);
        }

        private SourceOffsets(int[] indexes, int[] lines, int[] columns) {
            this.indexes = Preconditions.checkNotNull(indexes);
            this.lines = Preconditions.checkNotNull(lines);
            this.columns = Preconditions.checkNotNull(columns);
        }

        SourceLocation.Point getPoint(String text, int textIndex) {
            int location = Arrays.binarySearch(this.indexes, textIndex);
            if (location < 0) {
                location = -(location + 1);
            }
            if (this.indexes[location] == textIndex) {
                return SourceLocation.Point.create(this.lines[location], this.columns[location]);
            }
            return this.getLocationOf(text, location - 1, textIndex);
        }

        private SourceLocation.Point getLocationOf(String text, int startLocation, int textIndex) {
            int start;
            int line = this.lines[startLocation];
            int column = this.columns[startLocation];
            for (int i = start = this.indexes[startLocation]; i < textIndex; ++i) {
                char c = text.charAt(i);
                if (c == '\n') {
                    ++line;
                    column = 1;
                    continue;
                }
                if (c == '\r') {
                    if (i + 1 < text.length() && text.charAt(i + 1) == '\n') {
                        ++i;
                    }
                    ++line;
                    column = 1;
                    continue;
                }
                ++column;
            }
            return SourceLocation.Point.create(line, column);
        }

        SourceOffsets substring(int startTextIndex, int endTextIndex, String text) {
            int startColumn;
            int startLine;
            Preconditions.checkArgument(startTextIndex >= 0);
            Preconditions.checkArgument(startTextIndex < endTextIndex);
            Preconditions.checkArgument(endTextIndex <= text.length());
            int substringLength = endTextIndex - startTextIndex;
            --endTextIndex;
            int startLocation = Arrays.binarySearch(this.indexes, startTextIndex);
            if (startLocation < 0) {
                startLocation = -(startLocation + 1);
            }
            Builder builder = new Builder();
            if (this.indexes[startLocation] == startTextIndex) {
                startLine = this.lines[startLocation];
                startColumn = this.columns[startLocation];
                builder.doAdd(0, this.lines[startLocation], this.columns[startLocation]);
            } else {
                SourceLocation.Point startPoint = this.getLocationOf(text, --startLocation, startTextIndex);
                startLine = startPoint.line();
                startColumn = startPoint.column();
            }
            builder.doAdd(0, startLine, startColumn);
            if (startTextIndex == endTextIndex) {
                builder.setEndLocation(startLine, startColumn);
                return builder.build(substringLength);
            }
            int i = startLocation + 1;
            while (true) {
                int index;
                if ((index = this.indexes[i]) < endTextIndex) {
                    builder.doAdd(index - startTextIndex, this.lines[i], this.columns[i]);
                } else {
                    if (index == endTextIndex) {
                        builder.setEndLocation(this.lines[i], this.columns[i]);
                        break;
                    }
                    if (index > endTextIndex) {
                        SourceLocation.Point endPoint = this.getLocationOf(text, i - 1, endTextIndex);
                        builder.setEndLocation(endPoint.line(), endPoint.column());
                        break;
                    }
                }
                ++i;
            }
            return builder.build(substringLength);
        }

        SourceOffsets concat(SourceOffsets other) {
            int sizeToPreserve = this.indexes.length - 1;
            int lengthOfThis = this.indexes[this.indexes.length - 1];
            int newSize = sizeToPreserve + other.indexes.length;
            int[] newIndexes = Arrays.copyOf(this.indexes, newSize);
            int[] newLines = Arrays.copyOf(this.lines, newSize);
            int[] newColumns = Arrays.copyOf(this.columns, newSize);
            System.arraycopy(other.lines, 0, newLines, sizeToPreserve, other.lines.length);
            System.arraycopy(other.columns, 0, newColumns, sizeToPreserve, other.columns.length);
            for (int i = 0; i < other.indexes.length; ++i) {
                newIndexes[i + sizeToPreserve] = other.indexes[i] + lengthOfThis;
            }
            return new SourceOffsets(newIndexes, newLines, newColumns);
        }

        public SourceLocation getSourceLocation(String filePath) {
            return new SourceLocation(filePath, this.lines[0], this.columns[0], this.lines[this.lines.length - 1], this.columns[this.columns.length - 1]);
        }

        public String toString() {
            return String.format("SourceOffsets{\n  index:\t%s\n  lines:\t%s\n   cols:\t%s\n}", Arrays.toString(this.indexes), Arrays.toString(this.lines), Arrays.toString(this.columns));
        }

        public static final class Builder {
            private int size;
            private int[] indexes = new int[16];
            private int[] lines = new int[16];
            private int[] columns = new int[16];
            private int endLine = -1;
            private int endCol = -1;

            public Builder add(int index, int startLine, int startCol, int endLine, int endCol) {
                Preconditions.checkArgument(index >= 0, "expected index to be non-negative: %s", index);
                Preconditions.checkArgument(startLine > 0, "expected startLine to be positive: %s", startLine);
                Preconditions.checkArgument(startCol > 0, "expected startCol to be positive: %s", startCol);
                Preconditions.checkArgument(endLine > 0, "expected endLine to be positive: %s", endLine);
                Preconditions.checkArgument(endCol > 0, "expected endCol to be positive: %s", endCol);
                if (this.size != 0 && index <= this.indexes[this.size - 1]) {
                    throw new IllegalArgumentException("expected indexes to be added in increasing order");
                }
                this.doAdd(index, startLine, startCol);
                this.endLine = endLine;
                this.endCol = endCol;
                return this;
            }

            public Builder setEndLocation(int endLine, int endCol) {
                Preconditions.checkArgument(endLine > 0, "expected endLine to be positive: %s", endLine);
                Preconditions.checkArgument(endCol > 0, "expected endCol to be positive: %s", endCol);
                this.endLine = endLine;
                this.endCol = endCol;
                return this;
            }

            public Builder delete(int from) {
                int location = Arrays.binarySearch(this.indexes, 0, this.size, from);
                if (location < 0) {
                    location = -(location + 1);
                }
                this.size = location;
                return this;
            }

            public boolean isEmpty() {
                return this.size == 0;
            }

            public int endLine() {
                return this.endLine;
            }

            public int endColumn() {
                return this.endCol;
            }

            private void doAdd(int index, int line, int col) {
                if (this.size == this.indexes.length) {
                    int newCapacity = this.size + (this.size >> 1);
                    this.indexes = Arrays.copyOf(this.indexes, newCapacity);
                    this.lines = Arrays.copyOf(this.lines, newCapacity);
                    this.columns = Arrays.copyOf(this.columns, newCapacity);
                }
                this.indexes[this.size] = index;
                this.lines[this.size] = line;
                this.columns[this.size] = col;
                ++this.size;
            }

            public SourceOffsets build(int length) {
                this.doAdd(length, this.endLine, this.endCol);
                Preconditions.checkArgument(this.size > 1, "The builder should have size >= 2, got %s", this.size);
                Preconditions.checkArgument(this.indexes[0] == 0, "expected first index to be zero, got: %s", this.indexes[0]);
                SourceOffsets built = new SourceOffsets(Arrays.copyOf(this.indexes, this.size), Arrays.copyOf(this.lines, this.size), Arrays.copyOf(this.columns, this.size));
                --this.size;
                return built;
            }
        }
    }
}

