/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import com.google.common.primitives.Shorts;
import com.google.gerrit.common.data.LabelType;
import com.google.gerrit.common.data.LabelTypes;
import com.google.gerrit.common.data.Permission;
import com.google.gerrit.common.data.PermissionRange;
import com.google.gerrit.extensions.restapi.AuthException;
import com.google.gerrit.extensions.restapi.BadRequestException;
import com.google.gerrit.extensions.restapi.RestApiException;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.LabelId;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.PatchSetApproval;
import com.google.gerrit.reviewdb.client.PatchSetInfo;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.ApprovalCopier;
import com.google.gerrit.server.CurrentUser;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.ReviewerSet;
import com.google.gerrit.server.ReviewerStatusUpdate;
import com.google.gerrit.server.notedb.ChangeNotes;
import com.google.gerrit.server.notedb.ChangeUpdate;
import com.google.gerrit.server.notedb.NotesMigration;
import com.google.gerrit.server.notedb.ReviewerStateInternal;
import com.google.gerrit.server.project.ChangeControl;
import com.google.gerrit.server.util.LabelVote;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class ApprovalsUtil {
    private static final Logger log = LoggerFactory.getLogger(ApprovalsUtil.class);
    private static final Ordering<PatchSetApproval> SORT_APPROVALS = Ordering.from(Comparator.comparing(PatchSetApproval::getGranted));
    private final NotesMigration migration;
    private final IdentifiedUser.GenericFactory userFactory;
    private final ChangeControl.GenericFactory changeControlFactory;
    private final ApprovalCopier copier;

    public static List<PatchSetApproval> sortApprovals(Iterable<PatchSetApproval> approvals) {
        return SORT_APPROVALS.sortedCopy(approvals);
    }

    public static PatchSetApproval newApproval(PatchSet.Id psId, CurrentUser user, LabelId labelId, int value, Date when) {
        PatchSetApproval psa = new PatchSetApproval(new PatchSetApproval.Key(psId, user.getAccountId(), labelId), Shorts.checkedCast(value), when);
        user.updateRealAccountId(psa::setRealAccountId);
        return psa;
    }

    private static Iterable<PatchSetApproval> filterApprovals(Iterable<PatchSetApproval> psas, Account.Id accountId) {
        return Iterables.filter(psas, a -> Objects.equals(a.getAccountId(), accountId));
    }

    @Inject
    @VisibleForTesting
    public ApprovalsUtil(NotesMigration migration, IdentifiedUser.GenericFactory userFactory, ChangeControl.GenericFactory changeControlFactory, ApprovalCopier copier) {
        this.migration = migration;
        this.userFactory = userFactory;
        this.changeControlFactory = changeControlFactory;
        this.copier = copier;
    }

    public ReviewerSet getReviewers(ReviewDb db, ChangeNotes notes) throws OrmException {
        if (!this.migration.readChanges()) {
            return ReviewerSet.fromApprovals(db.patchSetApprovals().byChange(notes.getChangeId()));
        }
        return ((ChangeNotes)notes.load()).getReviewers();
    }

    public ReviewerSet getReviewers(ChangeNotes notes, Iterable<PatchSetApproval> allApprovals) throws OrmException {
        if (!this.migration.readChanges()) {
            return ReviewerSet.fromApprovals(allApprovals);
        }
        return ((ChangeNotes)notes.load()).getReviewers();
    }

    public List<ReviewerStatusUpdate> getReviewerUpdates(ChangeNotes notes) throws OrmException {
        if (!this.migration.readChanges()) {
            return ImmutableList.of();
        }
        return ((ChangeNotes)notes.load()).getReviewerUpdates();
    }

    public List<PatchSetApproval> addReviewers(ReviewDb db, ChangeUpdate update, LabelTypes labelTypes, Change change, PatchSet ps, PatchSetInfo info, Iterable<Account.Id> wantReviewers, Collection<Account.Id> existingReviewers) throws OrmException {
        return this.addReviewers(db, update, labelTypes, change, ps.getId(), info.getAuthor().getAccount(), info.getCommitter().getAccount(), wantReviewers, existingReviewers);
    }

    public List<PatchSetApproval> addReviewers(ReviewDb db, ChangeNotes notes, ChangeUpdate update, LabelTypes labelTypes, Change change, Iterable<Account.Id> wantReviewers) throws OrmException {
        PatchSet.Id psId = change.currentPatchSetId();
        AbstractCollection existingReviewers = this.migration.readChanges() ? ((ChangeNotes)notes.load()).getReviewers().byState(ReviewerStateInternal.REVIEWER) : this.getReviewers(db, notes).all();
        existingReviewers = Lists.newArrayList(existingReviewers);
        for (Map.Entry<Account.Id, ReviewerStateInternal> entry : update.getReviewers().entrySet()) {
            if (entry.getValue() != ReviewerStateInternal.REVIEWER) continue;
            existingReviewers.add((Account.Id)entry.getKey());
        }
        return this.addReviewers(db, update, labelTypes, change, psId, null, null, wantReviewers, existingReviewers);
    }

    private List<PatchSetApproval> addReviewers(ReviewDb db, ChangeUpdate update, LabelTypes labelTypes, Change change, PatchSet.Id psId, Account.Id authorId, Account.Id committerId, Iterable<Account.Id> wantReviewers, Collection<Account.Id> existingReviewers) throws OrmException {
        List<LabelType> allTypes = labelTypes.getLabelTypes();
        if (allTypes.isEmpty()) {
            return ImmutableList.of();
        }
        LinkedHashSet<Account.Id> need = Sets.newLinkedHashSet(wantReviewers);
        if (authorId != null && this.canSee(db, update.getNotes(), authorId)) {
            need.add(authorId);
        }
        if (committerId != null && this.canSee(db, update.getNotes(), committerId)) {
            need.add(committerId);
        }
        need.remove(change.getOwner());
        need.removeAll(existingReviewers);
        if (need.isEmpty()) {
            return ImmutableList.of();
        }
        ArrayList<PatchSetApproval> cells = Lists.newArrayListWithCapacity(need.size());
        LabelId labelId = Iterables.getLast(allTypes).getLabelId();
        for (Account.Id account : need) {
            cells.add(new PatchSetApproval(new PatchSetApproval.Key(psId, account, labelId), 0, update.getWhen()));
            update.putReviewer(account, ReviewerStateInternal.REVIEWER);
        }
        db.patchSetApprovals().upsert(cells);
        return Collections.unmodifiableList(cells);
    }

    private boolean canSee(ReviewDb db, ChangeNotes notes, Account.Id accountId) {
        try {
            IdentifiedUser user = this.userFactory.create(accountId);
            return this.changeControlFactory.controlFor(notes, user).isVisible(db);
        }
        catch (OrmException e) {
            log.warn("Failed to check if account {} can see change {}", accountId.get(), notes.getChangeId().get(), e);
            return false;
        }
    }

    public Collection<Account.Id> addCcs(ChangeNotes notes, ChangeUpdate update, Collection<Account.Id> wantCCs) throws OrmException {
        return this.addCcs(update, wantCCs, ((ChangeNotes)notes.load()).getReviewers());
    }

    private Collection<Account.Id> addCcs(ChangeUpdate update, Collection<Account.Id> wantCCs, ReviewerSet existingReviewers) {
        LinkedHashSet<Account.Id> need = new LinkedHashSet<Account.Id>(wantCCs);
        need.removeAll(existingReviewers.all());
        need.removeAll(update.getReviewers().keySet());
        for (Account.Id account : need) {
            update.putReviewer(account, ReviewerStateInternal.CC);
        }
        return need;
    }

    public Iterable<PatchSetApproval> addApprovalsForNewPatchSet(ReviewDb db, ChangeUpdate update, LabelTypes labelTypes, PatchSet ps, ChangeControl changeCtl, Map<String, Short> approvals) throws RestApiException, OrmException {
        Account.Id accountId = changeCtl.getUser().getAccountId();
        Preconditions.checkArgument(accountId.equals(ps.getUploader()), "expected user %s to match patch set uploader %s", (Object)accountId, (Object)ps.getUploader());
        if (approvals.isEmpty()) {
            return Collections.emptyList();
        }
        ApprovalsUtil.checkApprovals(approvals, changeCtl);
        ArrayList<PatchSetApproval> cells = new ArrayList<PatchSetApproval>(approvals.size());
        Date ts = update.getWhen();
        for (Map.Entry<String, Short> vote : approvals.entrySet()) {
            LabelType lt = labelTypes.byLabel(vote.getKey());
            cells.add(ApprovalsUtil.newApproval(ps.getId(), changeCtl.getUser(), lt.getLabelId(), vote.getValue().shortValue(), ts));
        }
        for (PatchSetApproval psa : cells) {
            update.putApproval(psa.getLabel(), psa.getValue());
        }
        db.patchSetApprovals().insert(cells);
        return cells;
    }

    public static void checkLabel(LabelTypes labelTypes, String name, Short value) throws BadRequestException {
        LabelType label = labelTypes.byLabel(name);
        if (label == null) {
            throw new BadRequestException(String.format("label \"%s\" is not a configured label", name));
        }
        if (label.getValue(value) == null) {
            throw new BadRequestException(String.format("label \"%s\": %d is not a valid value", name, value));
        }
    }

    private static void checkApprovals(Map<String, Short> approvals, ChangeControl changeCtl) throws AuthException {
        for (Map.Entry<String, Short> vote : approvals.entrySet()) {
            String name = vote.getKey();
            Short value = vote.getValue();
            PermissionRange range = changeCtl.getRange(Permission.forLabel(name));
            if (range != null && range.contains(value.shortValue())) continue;
            throw new AuthException(String.format("applying label \"%s\": %d is restricted", name, value));
        }
    }

    public ListMultimap<PatchSet.Id, PatchSetApproval> byChange(ReviewDb db, ChangeNotes notes) throws OrmException {
        if (!this.migration.readChanges()) {
            ImmutableListMultimap.Builder result = ImmutableListMultimap.builder();
            for (PatchSetApproval psa : db.patchSetApprovals().byChange(notes.getChangeId())) {
                result.put(psa.getPatchSetId(), psa);
            }
            return result.build();
        }
        return ((ChangeNotes)notes.load()).getApprovals();
    }

    public Iterable<PatchSetApproval> byPatchSet(ReviewDb db, ChangeControl ctl, PatchSet.Id psId) throws OrmException {
        if (!this.migration.readChanges()) {
            return ApprovalsUtil.sortApprovals(db.patchSetApprovals().byPatchSet(psId));
        }
        return this.copier.getForPatchSet(db, ctl, psId);
    }

    public Iterable<PatchSetApproval> byPatchSetUser(ReviewDb db, ChangeControl ctl, PatchSet.Id psId, Account.Id accountId) throws OrmException {
        if (!this.migration.readChanges()) {
            return ApprovalsUtil.sortApprovals(db.patchSetApprovals().byPatchSetUser(psId, accountId));
        }
        return ApprovalsUtil.filterApprovals(this.byPatchSet(db, ctl, psId), accountId);
    }

    public PatchSetApproval getSubmitter(ReviewDb db, ChangeNotes notes, PatchSet.Id c) {
        if (c == null) {
            return null;
        }
        try {
            return ApprovalsUtil.getSubmitter(c, this.byChange(db, notes).get((Object)c));
        }
        catch (OrmException e) {
            return null;
        }
    }

    public static PatchSetApproval getSubmitter(PatchSet.Id c, Iterable<PatchSetApproval> approvals) {
        if (c == null) {
            return null;
        }
        PatchSetApproval submitter = null;
        for (PatchSetApproval a : approvals) {
            if (!a.getPatchSetId().equals(c) || a.getValue() <= 0 || !a.isLegacySubmit() || submitter != null && a.getGranted().compareTo(submitter.getGranted()) <= 0) continue;
            submitter = a;
        }
        return submitter;
    }

    public static String renderMessageWithApprovals(int patchSetId, Map<String, Short> n, Map<String, PatchSetApproval> c) {
        StringBuilder msgs = new StringBuilder("Uploaded patch set " + patchSetId);
        if (!n.isEmpty()) {
            boolean first = true;
            for (Map.Entry<String, Short> e : n.entrySet()) {
                if (c.containsKey(e.getKey()) && c.get(e.getKey()).getValue() == e.getValue().shortValue()) continue;
                if (first) {
                    msgs.append(":");
                    first = false;
                }
                msgs.append(" ").append(LabelVote.create(e.getKey(), e.getValue()).format());
            }
        }
        return msgs.toString();
    }
}

