/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.notedb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ListMultimap;
import com.google.common.primitives.Ints;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.Comment;
import com.google.gerrit.reviewdb.client.CommentRange;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.RevId;
import com.google.gerrit.server.CommentsUtil;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.config.AnonymousCowardName;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.config.GerritServerId;
import com.google.gerrit.server.notedb.ChangeNotes;
import com.google.gerrit.server.notedb.CommentTimestampAdapter;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.inject.Inject;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.revwalk.FooterKey;
import org.eclipse.jgit.util.GitDateFormatter;
import org.eclipse.jgit.util.GitDateParser;
import org.eclipse.jgit.util.MutableInteger;
import org.eclipse.jgit.util.QuotedString;
import org.eclipse.jgit.util.RawParseUtils;

public class ChangeNoteUtil {
    public static final FooterKey FOOTER_ASSIGNEE = new FooterKey("Assignee");
    public static final FooterKey FOOTER_BRANCH = new FooterKey("Branch");
    public static final FooterKey FOOTER_CHANGE_ID = new FooterKey("Change-id");
    public static final FooterKey FOOTER_COMMIT = new FooterKey("Commit");
    public static final FooterKey FOOTER_CURRENT = new FooterKey("Current");
    public static final FooterKey FOOTER_GROUPS = new FooterKey("Groups");
    public static final FooterKey FOOTER_HASHTAGS = new FooterKey("Hashtags");
    public static final FooterKey FOOTER_LABEL = new FooterKey("Label");
    public static final FooterKey FOOTER_PATCH_SET = new FooterKey("Patch-set");
    public static final FooterKey FOOTER_PATCH_SET_DESCRIPTION = new FooterKey("Patch-set-description");
    public static final FooterKey FOOTER_READ_ONLY_UNTIL = new FooterKey("Read-only-until");
    public static final FooterKey FOOTER_REAL_USER = new FooterKey("Real-user");
    public static final FooterKey FOOTER_STATUS = new FooterKey("Status");
    public static final FooterKey FOOTER_SUBJECT = new FooterKey("Subject");
    public static final FooterKey FOOTER_SUBMISSION_ID = new FooterKey("Submission-id");
    public static final FooterKey FOOTER_SUBMITTED_WITH = new FooterKey("Submitted-with");
    public static final FooterKey FOOTER_TOPIC = new FooterKey("Topic");
    public static final FooterKey FOOTER_TAG = new FooterKey("Tag");
    private static final String AUTHOR = "Author";
    private static final String BASE_PATCH_SET = "Base-for-patch-set";
    private static final String COMMENT_RANGE = "Comment-range";
    private static final String FILE = "File";
    private static final String LENGTH = "Bytes";
    private static final String PARENT = "Parent";
    private static final String PARENT_NUMBER = "Parent-number";
    private static final String PATCH_SET = "Patch-set";
    private static final String REAL_AUTHOR = "Real-author";
    private static final String REVISION = "Revision";
    private static final String UUID = "UUID";
    private static final String UNRESOLVED = "Unresolved";
    private static final String TAG = FOOTER_TAG.getName();
    private final AccountCache accountCache;
    private final PersonIdent serverIdent;
    private final String anonymousCowardName;
    private final String serverId;
    private final Gson gson = ChangeNoteUtil.newGson();
    private final boolean writeJson;
    private static final CharMatcher INVALID_FOOTER_CHARS = CharMatcher.anyOf("\r\n\u0000");

    public static String formatTime(PersonIdent ident, Timestamp t) {
        GitDateFormatter dateFormatter = new GitDateFormatter(GitDateFormatter.Format.DEFAULT);
        PersonIdent newIdent = new PersonIdent(ident, t);
        return dateFormatter.formatDate(newIdent);
    }

    static Gson newGson() {
        return new GsonBuilder().registerTypeAdapter((Type)((Object)Timestamp.class), new CommentTimestampAdapter().nullSafe()).setPrettyPrinting().create();
    }

    @Inject
    public ChangeNoteUtil(AccountCache accountCache, @GerritPersonIdent PersonIdent serverIdent, @AnonymousCowardName String anonymousCowardName, @GerritServerId String serverId, @GerritServerConfig Config config) {
        this.accountCache = accountCache;
        this.serverIdent = serverIdent;
        this.anonymousCowardName = anonymousCowardName;
        this.serverId = serverId;
        this.writeJson = config.getBoolean("notedb", "writeJson", false);
    }

    @VisibleForTesting
    public PersonIdent newIdent(Account author, Date when, PersonIdent serverIdent, String anonymousCowardName) {
        return new PersonIdent(author.getName(anonymousCowardName), author.getId().get() + "@" + this.serverId, when, serverIdent.getTimeZone());
    }

    public boolean getWriteJson() {
        return this.writeJson;
    }

    public Gson getGson() {
        return this.gson;
    }

    public String getServerId() {
        return this.serverId;
    }

    public Account.Id parseIdent(PersonIdent ident, Change.Id changeId) throws ConfigInvalidException {
        Integer id;
        String host;
        String email = ident.getEmailAddress();
        int at = email.indexOf(64);
        if (at >= 0 && (host = email.substring(at + 1, email.length())).equals(this.serverId) && (id = Ints.tryParse(email.substring(0, at))) != null) {
            return new Account.Id(id);
        }
        throw ChangeNotes.parseException(changeId, "invalid identity, expected <id>@%s: %s", this.serverId, email);
    }

    private static boolean match(byte[] note, MutableInteger p, byte[] expected) {
        int m = RawParseUtils.match(note, p.value, expected);
        return m == p.value + expected.length;
    }

    public List<Comment> parseNote(byte[] note, MutableInteger p, Change.Id changeId) throws ConfigInvalidException {
        if (p.value >= note.length) {
            return ImmutableList.of();
        }
        HashSet<Comment.Key> seen = new HashSet<Comment.Key>();
        ArrayList<Comment> result = new ArrayList<Comment>();
        int sizeOfNote = note.length;
        byte[] psb = PATCH_SET.getBytes(StandardCharsets.UTF_8);
        byte[] bpsb = BASE_PATCH_SET.getBytes(StandardCharsets.UTF_8);
        byte[] bpn = PARENT_NUMBER.getBytes(StandardCharsets.UTF_8);
        RevId revId = new RevId(ChangeNoteUtil.parseStringField(note, p, changeId, REVISION));
        String fileName = null;
        PatchSet.Id psId = null;
        boolean isForBase = false;
        Integer parentNumber = null;
        while (p.value < sizeOfNote) {
            boolean matchPs = ChangeNoteUtil.match(note, p, psb);
            boolean matchBase = ChangeNoteUtil.match(note, p, bpsb);
            if (matchPs) {
                fileName = null;
                psId = ChangeNoteUtil.parsePsId(note, p, changeId, PATCH_SET);
                isForBase = false;
            } else if (matchBase) {
                fileName = null;
                psId = ChangeNoteUtil.parsePsId(note, p, changeId, BASE_PATCH_SET);
                isForBase = true;
                if (ChangeNoteUtil.match(note, p, bpn)) {
                    parentNumber = ChangeNoteUtil.parseParentNumber(note, p, changeId);
                }
            } else if (psId == null) {
                throw ChangeNotes.parseException(changeId, "missing %s or %s header", PATCH_SET, BASE_PATCH_SET);
            }
            Comment c = this.parseComment(note, p, fileName, psId, revId, isForBase, parentNumber);
            fileName = c.key.filename;
            if (!seen.add(c.key)) {
                throw ChangeNotes.parseException(changeId, "multiple comments for %s in note", c.key);
            }
            result.add(c);
        }
        return result;
    }

    private Comment parseComment(byte[] note, MutableInteger curr, String currentFileName, PatchSet.Id psId, RevId revId, boolean isForBase, Integer parentNumber) throws ConfigInvalidException {
        boolean hasUnresolved;
        boolean newFile;
        Change.Id changeId = psId.getParentKey();
        boolean bl = newFile = RawParseUtils.match(note, curr.value, FILE.getBytes(StandardCharsets.UTF_8)) != -1;
        if (newFile) {
            currentFileName = ChangeNoteUtil.parseFilename(note, curr, changeId);
        } else if (currentFileName == null) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", FILE);
        }
        CommentRange range = ChangeNoteUtil.parseCommentRange(note, curr);
        if (range == null) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", COMMENT_RANGE);
        }
        Timestamp commentTime = ChangeNoteUtil.parseTimestamp(note, curr, changeId);
        Account.Id aId = this.parseAuthor(note, curr, changeId, AUTHOR);
        boolean hasRealAuthor = RawParseUtils.match(note, curr.value, REAL_AUTHOR.getBytes(StandardCharsets.UTF_8)) != -1;
        Account.Id raId = null;
        if (hasRealAuthor) {
            raId = this.parseAuthor(note, curr, changeId, REAL_AUTHOR);
        }
        boolean hasParent = RawParseUtils.match(note, curr.value, PARENT.getBytes(StandardCharsets.UTF_8)) != -1;
        String parentUUID = null;
        boolean unresolved = false;
        if (hasParent) {
            parentUUID = ChangeNoteUtil.parseStringField(note, curr, changeId, PARENT);
        }
        boolean bl2 = hasUnresolved = RawParseUtils.match(note, curr.value, UNRESOLVED.getBytes(StandardCharsets.UTF_8)) != -1;
        if (hasUnresolved) {
            unresolved = this.parseBooleanField(note, curr, changeId, UNRESOLVED);
        }
        String uuid = ChangeNoteUtil.parseStringField(note, curr, changeId, UUID);
        boolean hasTag = RawParseUtils.match(note, curr.value, TAG.getBytes(StandardCharsets.UTF_8)) != -1;
        String tag = null;
        if (hasTag) {
            tag = ChangeNoteUtil.parseStringField(note, curr, changeId, TAG);
        }
        int commentLength = ChangeNoteUtil.parseCommentLength(note, curr, changeId);
        String message = RawParseUtils.decode(StandardCharsets.UTF_8, note, curr.value, curr.value + commentLength);
        ChangeNoteUtil.checkResult(message, "message contents", changeId);
        Comment c = new Comment(new Comment.Key(uuid, currentFileName, psId.get()), aId, commentTime, isForBase ? (short)(parentNumber == null ? 0 : -parentNumber.intValue()) : (short)1, message, this.serverId, unresolved);
        c.lineNbr = range.getEndLine();
        c.parentUuid = parentUUID;
        c.tag = tag;
        c.setRevId(revId);
        if (raId != null) {
            c.setRealAuthor(raId);
        }
        if (range.getStartCharacter() != -1) {
            c.setRange(range);
        }
        curr.value = RawParseUtils.nextLF(note, curr.value + commentLength);
        curr.value = RawParseUtils.nextLF(note, curr.value);
        return c;
    }

    private static String parseStringField(byte[] note, MutableInteger curr, Change.Id changeId, String fieldName) throws ConfigInvalidException {
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        ChangeNoteUtil.checkHeaderLineFormat(note, curr, fieldName, changeId);
        int startOfField = RawParseUtils.endOfFooterLineKey(note, curr.value) + 2;
        curr.value = endOfLine;
        return RawParseUtils.decode(StandardCharsets.UTF_8, note, startOfField, endOfLine - 1);
    }

    private static CommentRange parseCommentRange(byte[] note, MutableInteger ptr) {
        CommentRange range = new CommentRange(-1, -1, -1, -1);
        int last = ptr.value;
        int startLine = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (ptr.value == last) {
            return null;
        }
        if (note[ptr.value] == 10) {
            range.setEndLine(startLine);
            ++ptr.value;
            return range;
        }
        if (note[ptr.value] == 58) {
            range.setStartLine(startLine);
            ++ptr.value;
        } else {
            return null;
        }
        last = ptr.value;
        int startChar = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (ptr.value == last) {
            return null;
        }
        if (note[ptr.value] == 45) {
            range.setStartCharacter(startChar);
            ++ptr.value;
        } else {
            return null;
        }
        last = ptr.value;
        int endLine = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (ptr.value == last) {
            return null;
        }
        if (note[ptr.value] == 58) {
            range.setEndLine(endLine);
            ++ptr.value;
        } else {
            return null;
        }
        last = ptr.value;
        int endChar = RawParseUtils.parseBase10(note, ptr.value, ptr);
        if (ptr.value == last) {
            return null;
        }
        if (note[ptr.value] == 10) {
            range.setEndCharacter(endChar);
            ++ptr.value;
        } else {
            return null;
        }
        return range;
    }

    private static PatchSet.Id parsePsId(byte[] note, MutableInteger curr, Change.Id changeId, String fieldName) throws ConfigInvalidException {
        ChangeNoteUtil.checkHeaderLineFormat(note, curr, fieldName, changeId);
        int startOfPsId = RawParseUtils.endOfFooterLineKey(note, curr.value) + 1;
        MutableInteger i = new MutableInteger();
        int patchSetId = RawParseUtils.parseBase10(note, startOfPsId, i);
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        if (i.value != endOfLine - 1) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
        ChangeNoteUtil.checkResult(patchSetId, "patchset id", changeId);
        curr.value = endOfLine;
        return new PatchSet.Id(changeId, patchSetId);
    }

    private static Integer parseParentNumber(byte[] note, MutableInteger curr, Change.Id changeId) throws ConfigInvalidException {
        ChangeNoteUtil.checkHeaderLineFormat(note, curr, PARENT_NUMBER, changeId);
        int start = RawParseUtils.endOfFooterLineKey(note, curr.value) + 1;
        MutableInteger i = new MutableInteger();
        int parentNumber = RawParseUtils.parseBase10(note, start, i);
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        if (i.value != endOfLine - 1) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", PARENT_NUMBER);
        }
        ChangeNoteUtil.checkResult(parentNumber, "parent number", changeId);
        curr.value = endOfLine;
        return parentNumber;
    }

    private static String parseFilename(byte[] note, MutableInteger curr, Change.Id changeId) throws ConfigInvalidException {
        int endOfLine;
        ChangeNoteUtil.checkHeaderLineFormat(note, curr, FILE, changeId);
        int startOfFileName = RawParseUtils.endOfFooterLineKey(note, curr.value) + 2;
        curr.value = endOfLine = RawParseUtils.nextLF(note, curr.value);
        curr.value = RawParseUtils.nextLF(note, curr.value);
        return QuotedString.GIT_PATH.dequote(RawParseUtils.decode(StandardCharsets.UTF_8, note, startOfFileName, endOfLine - 1));
    }

    private static Timestamp parseTimestamp(byte[] note, MutableInteger curr, Change.Id changeId) throws ConfigInvalidException {
        Timestamp commentTime;
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        String dateString = RawParseUtils.decode(StandardCharsets.UTF_8, note, curr.value, endOfLine - 1);
        try {
            commentTime = new Timestamp(GitDateParser.parse(dateString, null, Locale.US).getTime());
        }
        catch (ParseException e) {
            throw new ConfigInvalidException("could not parse comment timestamp", e);
        }
        curr.value = endOfLine;
        return ChangeNoteUtil.checkResult(commentTime, "comment timestamp", changeId);
    }

    private Account.Id parseAuthor(byte[] note, MutableInteger curr, Change.Id changeId, String fieldName) throws ConfigInvalidException {
        ChangeNoteUtil.checkHeaderLineFormat(note, curr, fieldName, changeId);
        int startOfAccountId = RawParseUtils.endOfFooterLineKey(note, curr.value) + 2;
        PersonIdent ident = RawParseUtils.parsePersonIdent(note, startOfAccountId);
        Account.Id aId = this.parseIdent(ident, changeId);
        curr.value = RawParseUtils.nextLF(note, curr.value);
        return ChangeNoteUtil.checkResult(aId, fieldName, changeId);
    }

    private static int parseCommentLength(byte[] note, MutableInteger curr, Change.Id changeId) throws ConfigInvalidException {
        ChangeNoteUtil.checkHeaderLineFormat(note, curr, LENGTH, changeId);
        int startOfLength = RawParseUtils.endOfFooterLineKey(note, curr.value) + 1;
        MutableInteger i = new MutableInteger();
        i.value = startOfLength;
        int commentLength = RawParseUtils.parseBase10(note, startOfLength, i);
        if (i.value == startOfLength) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", LENGTH);
        }
        int endOfLine = RawParseUtils.nextLF(note, curr.value);
        if (i.value != endOfLine - 1) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", LENGTH);
        }
        curr.value = endOfLine;
        return ChangeNoteUtil.checkResult(commentLength, "comment length", changeId);
    }

    private boolean parseBooleanField(byte[] note, MutableInteger curr, Change.Id changeId, String fieldName) throws ConfigInvalidException {
        String str = ChangeNoteUtil.parseStringField(note, curr, changeId, fieldName);
        if ("true".equalsIgnoreCase(str)) {
            return true;
        }
        if ("false".equalsIgnoreCase(str)) {
            return false;
        }
        throw ChangeNotes.parseException(changeId, "invalid boolean for %s: %s", fieldName, str);
    }

    private static <T> T checkResult(T o, String fieldName, Change.Id changeId) throws ConfigInvalidException {
        if (o == null) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
        return o;
    }

    private static int checkResult(int i, String fieldName, Change.Id changeId) throws ConfigInvalidException {
        if (i <= 0) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
        return i;
    }

    private void appendHeaderField(PrintWriter writer, String field, String value) {
        writer.print(field);
        writer.print(": ");
        writer.print(value);
        writer.print('\n');
    }

    private static void checkHeaderLineFormat(byte[] note, MutableInteger curr, String fieldName, Change.Id changeId) throws ConfigInvalidException {
        boolean correct = RawParseUtils.match(note, curr.value, fieldName.getBytes(StandardCharsets.UTF_8)) != -1;
        int p = curr.value + fieldName.length();
        correct &= p < note.length && note[p] == 58;
        if (!(correct &= ++p < note.length && note[p] == 32)) {
            throw ChangeNotes.parseException(changeId, "could not parse %s", fieldName);
        }
    }

    void buildNote(ListMultimap<Integer, Comment> comments, OutputStream out) {
        if (comments.isEmpty()) {
            return;
        }
        ArrayList psIds = new ArrayList(comments.keySet());
        Collections.sort(psIds);
        OutputStreamWriter streamWriter = new OutputStreamWriter(out, StandardCharsets.UTF_8);
        try (PrintWriter writer = new PrintWriter(streamWriter);){
            String revId = ((Comment)comments.values().iterator().next()).revId;
            this.appendHeaderField(writer, REVISION, revId);
            Iterator iterator = psIds.iterator();
            while (iterator.hasNext()) {
                int psId = (Integer)iterator.next();
                List psComments = CommentsUtil.COMMENT_ORDER.sortedCopy(comments.get((Object)psId));
                Comment first = (Comment)psComments.get(0);
                short side = first.side;
                this.appendHeaderField(writer, side <= 0 ? BASE_PATCH_SET : PATCH_SET, Integer.toString(psId));
                if (side < 0) {
                    this.appendHeaderField(writer, PARENT_NUMBER, Integer.toString(-side));
                }
                String currentFilename = null;
                for (Comment c : psComments) {
                    Preconditions.checkArgument(revId.equals(c.revId), "All comments being added must have all the same RevId. The comment below does not have the same RevId as the others (%s).\n%s", (Object)revId, (Object)c);
                    Preconditions.checkArgument(side == c.side, "All comments being added must all have the same side. The comment below does not have the same side as the others (%s).\n%s", (int)side, (Object)c);
                    String commentFilename = QuotedString.GIT_PATH.quote(c.key.filename);
                    if (!commentFilename.equals(currentFilename)) {
                        currentFilename = commentFilename;
                        writer.print("File: ");
                        writer.print(commentFilename);
                        writer.print("\n\n");
                    }
                    this.appendOneComment(writer, c);
                }
            }
        }
    }

    private void appendOneComment(PrintWriter writer, Comment c) {
        String parent;
        Comment.Range range = c.range;
        if (range != null) {
            writer.print(range.startLine);
            writer.print(':');
            writer.print(range.startChar);
            writer.print('-');
            writer.print(range.endLine);
            writer.print(':');
            writer.print(range.endChar);
        } else {
            writer.print(c.lineNbr);
        }
        writer.print("\n");
        writer.print(ChangeNoteUtil.formatTime(this.serverIdent, c.writtenOn));
        writer.print("\n");
        this.appendIdent(writer, AUTHOR, c.author.getId(), c.writtenOn);
        if (!c.getRealAuthor().equals(c.author)) {
            this.appendIdent(writer, REAL_AUTHOR, c.getRealAuthor().getId(), c.writtenOn);
        }
        if ((parent = c.parentUuid) != null) {
            this.appendHeaderField(writer, PARENT, parent);
        }
        this.appendHeaderField(writer, UNRESOLVED, Boolean.toString(c.unresolved));
        this.appendHeaderField(writer, UUID, c.key.uuid);
        if (c.tag != null) {
            this.appendHeaderField(writer, TAG, c.tag);
        }
        byte[] messageBytes = c.message.getBytes(StandardCharsets.UTF_8);
        this.appendHeaderField(writer, LENGTH, Integer.toString(messageBytes.length));
        writer.print(c.message);
        writer.print("\n\n");
    }

    private void appendIdent(PrintWriter writer, String header, Account.Id id, Timestamp ts) {
        PersonIdent ident = this.newIdent(this.accountCache.get(id).getAccount(), ts, this.serverIdent, this.anonymousCowardName);
        StringBuilder name = new StringBuilder();
        PersonIdent.appendSanitized(name, ident.getName());
        name.append(" <");
        PersonIdent.appendSanitized(name, ident.getEmailAddress());
        name.append('>');
        this.appendHeaderField(writer, header, name.toString());
    }

    static String sanitizeFooter(String value) {
        return INVALID_FOOTER_CHARS.trimAndCollapseFrom(value, ' ');
    }
}

