/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.httpd.raw;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.collect.ImmutableMap;
import com.google.common.hash.Hashing;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.httpd.HtmlDomUtil;
import com.google.gwtexpui.server.CacheHeaders;
import com.google.gwtjsonrpc.server.RPCServletUtils;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPOutputStream;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ResourceServlet
extends HttpServlet {
    private static final long serialVersionUID = 1L;
    private static final Logger log = LoggerFactory.getLogger(ResourceServlet.class);
    private static final int CACHE_FILE_SIZE_LIMIT_BYTES = 102400;
    private static final String JS = "application/x-javascript";
    private static final ImmutableMap<String, String> MIME_TYPES = ImmutableMap.builder().put("css", "text/css").put("gif", "image/gif").put("htm", "text/html").put("html", "text/html").put("ico", "image/x-icon").put("jpeg", "image/jpeg").put("jpg", "image/jpeg").put("js", "application/x-javascript").put("pdf", "application/pdf").put("png", "image/png").put("rtf", "text/rtf").put("svg", "image/svg+xml").put("text", "text/plain").put("tif", "image/tiff").put("tiff", "image/tiff").put("txt", "text/plain").put("woff", "font/woff").put("woff2", "font/woff2").build();
    private final Cache<Path, Resource> cache;
    private final boolean refresh;
    private final boolean cacheOnClient;
    private final int cacheFileSizeLimitBytes;

    protected static String contentType(String name) {
        int dot = name.lastIndexOf(46);
        String ext = 0 < dot ? name.substring(dot + 1) : "";
        String type = MIME_TYPES.get(ext);
        return type != null ? type : "application/octet-stream";
    }

    protected ResourceServlet(Cache<Path, Resource> cache, boolean refresh) {
        this(cache, refresh, true, 102400);
    }

    protected ResourceServlet(Cache<Path, Resource> cache, boolean refresh, boolean cacheOnClient) {
        this(cache, refresh, cacheOnClient, 102400);
    }

    @VisibleForTesting
    ResourceServlet(Cache<Path, Resource> cache, boolean refresh, boolean cacheOnClient, int cacheFileSizeLimitBytes) {
        this.cache = Preconditions.checkNotNull(cache, "cache");
        this.refresh = refresh;
        this.cacheOnClient = cacheOnClient;
        this.cacheFileSizeLimitBytes = cacheFileSizeLimitBytes;
    }

    protected abstract Path getResourcePath(String var1) throws IOException;

    protected FileTime getLastModifiedTime(Path p) throws IOException {
        return Files.getLastModifiedTime(p, new LinkOption[0]);
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse rsp) throws IOException {
        byte[] gz;
        String name = req.getPathInfo() == null ? "/" : CharMatcher.is('/').trimFrom(req.getPathInfo());
        if (ResourceServlet.isUnreasonableName(name)) {
            ResourceServlet.notFound(rsp);
            return;
        }
        Path p = this.getResourcePath(name);
        if (p == null) {
            ResourceServlet.notFound(rsp);
            return;
        }
        Resource r = this.cache.getIfPresent(p);
        try {
            if (r == null) {
                if (this.maybeStream(p, req, rsp)) {
                    return;
                }
                r = this.cache.get(p, this.newLoader(p));
            }
            if (this.refresh && r.isStale(p, this)) {
                this.cache.invalidate(p);
                r = this.cache.get(p, this.newLoader(p));
            }
        }
        catch (ExecutionException e) {
            log.warn("Cannot load static resource {}", (Object)req.getPathInfo(), (Object)e);
            CacheHeaders.setNotCacheable(rsp);
            rsp.setStatus(500);
            return;
        }
        if (r == Resource.NOT_FOUND) {
            ResourceServlet.notFound(rsp);
            return;
        }
        String e = req.getParameter("e");
        if (e != null && !r.etag.equals(e)) {
            CacheHeaders.setNotCacheable(rsp);
            rsp.setStatus(404);
            return;
        }
        if (this.cacheOnClient && r.etag.equals(req.getHeader("If-None-Match"))) {
            rsp.setStatus(304);
            return;
        }
        byte[] tosend = r.raw;
        if (!r.contentType.equals(JS) && RPCServletUtils.acceptsGzipEncoding(req) && (gz = HtmlDomUtil.compress(tosend)).length + 24 < tosend.length) {
            rsp.setHeader("Content-Encoding", "gzip");
            tosend = gz;
        }
        if (this.cacheOnClient) {
            rsp.setHeader("ETag", r.etag);
        } else {
            CacheHeaders.setNotCacheable(rsp);
        }
        if (!CacheHeaders.hasCacheHeader(rsp)) {
            if (e != null && r.etag.equals(e)) {
                CacheHeaders.setCacheable(req, rsp, 360L, TimeUnit.DAYS, false);
            } else {
                CacheHeaders.setCacheable(req, rsp, 15L, TimeUnit.MINUTES, this.refresh);
            }
        }
        rsp.setContentType(r.contentType);
        rsp.setContentLength(tosend.length);
        try (ServletOutputStream out = rsp.getOutputStream();){
            out.write(tosend);
        }
    }

    @Nullable
    Resource getResource(String name) {
        try {
            Path p = this.getResourcePath(name);
            if (p == null) {
                log.warn("Path doesn't exist {}", (Object)name);
                return null;
            }
            return this.cache.get(p, this.newLoader(p));
        }
        catch (IOException | ExecutionException e) {
            log.warn("Cannot load static resource {}", (Object)name, (Object)e);
            return null;
        }
    }

    private static void notFound(HttpServletResponse rsp) {
        rsp.setStatus(404);
        CacheHeaders.setNotCacheable(rsp);
    }

    private boolean maybeStream(Path p, HttpServletRequest req, HttpServletResponse rsp) throws IOException {
        try {
            if (Files.size(p) < (long)this.cacheFileSizeLimitBytes) {
                return false;
            }
        }
        catch (NoSuchFileException e) {
            this.cache.put(p, Resource.NOT_FOUND);
            ResourceServlet.notFound(rsp);
            return true;
        }
        long lastModified = this.getLastModifiedTime(p).toMillis();
        if (req.getDateHeader("If-Modified-Since") >= lastModified) {
            rsp.setStatus(304);
            return true;
        }
        if (lastModified > 0L) {
            rsp.setDateHeader("Last-Modified", lastModified);
        }
        if (!CacheHeaders.hasCacheHeader(rsp)) {
            CacheHeaders.setCacheable(req, rsp, 15L, TimeUnit.MINUTES, this.refresh);
        }
        rsp.setContentType(ResourceServlet.contentType(p.toString()));
        OutputStream out = rsp.getOutputStream();
        GZIPOutputStream gz = null;
        if (RPCServletUtils.acceptsGzipEncoding(req)) {
            rsp.setHeader("Content-Encoding", "gzip");
            gz = new GZIPOutputStream(out);
            out = gz;
        }
        Files.copy(p, out);
        if (gz != null) {
            gz.finish();
        }
        return true;
    }

    private static boolean isUnreasonableName(String name) {
        return name.length() < 1 || name.contains("\\") || name.startsWith("../") || name.contains("/../") || name.contains("/./") || name.contains("//");
    }

    private Callable<Resource> newLoader(Path p) {
        return () -> {
            try {
                return new Resource(this.getLastModifiedTime(p), ResourceServlet.contentType(p.toString()), Files.readAllBytes(p));
            }
            catch (NoSuchFileException e) {
                return Resource.NOT_FOUND;
            }
        };
    }

    public static class Weigher
    implements com.google.common.cache.Weigher<Path, Resource> {
        @Override
        public int weigh(Path p, Resource r) {
            return 2 * p.toString().length() + r.raw.length;
        }
    }

    public static class Resource {
        static final Resource NOT_FOUND = new Resource(FileTime.fromMillis(0L), "", new byte[0]);
        final FileTime lastModified;
        final String contentType;
        final String etag;
        final byte[] raw;

        Resource(FileTime lastModified, String contentType, byte[] raw) {
            this.lastModified = Preconditions.checkNotNull(lastModified, "lastModified");
            this.contentType = Preconditions.checkNotNull(contentType, "contentType");
            this.raw = Preconditions.checkNotNull(raw, "raw");
            this.etag = Hashing.md5().hashBytes(raw).toString();
        }

        boolean isStale(Path p, ResourceServlet rs) throws IOException {
            FileTime t;
            try {
                t = rs.getLastModifiedTime(p);
            }
            catch (NoSuchFileException e) {
                return this != NOT_FOUND;
            }
            return t.toMillis() == 0L || this.lastModified.toMillis() == 0L || !this.lastModified.equals(t);
        }
    }
}

