/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.account;

import com.google.common.base.Strings;
import com.google.gerrit.audit.AuditService;
import com.google.gerrit.common.TimeUtil;
import com.google.gerrit.common.data.Permission;
import com.google.gerrit.common.errors.NameAlreadyUsedException;
import com.google.gerrit.extensions.client.AccountFieldName;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.AccountGroup;
import com.google.gerrit.reviewdb.client.AccountGroupMember;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.account.AccountByEmailCache;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.account.AccountException;
import com.google.gerrit.server.account.AccountUserNameException;
import com.google.gerrit.server.account.AuthRequest;
import com.google.gerrit.server.account.AuthResult;
import com.google.gerrit.server.account.ChangeUserName;
import com.google.gerrit.server.account.ExternalId;
import com.google.gerrit.server.account.ExternalIdsUpdate;
import com.google.gerrit.server.account.InvalidUserNameException;
import com.google.gerrit.server.account.Realm;
import com.google.gerrit.server.project.ProjectCache;
import com.google.gwtorm.server.OrmException;
import com.google.gwtorm.server.SchemaFactory;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class AccountManager {
    private static final Logger log = LoggerFactory.getLogger(AccountManager.class);
    private final SchemaFactory<ReviewDb> schema;
    private final AccountCache byIdCache;
    private final AccountByEmailCache byEmailCache;
    private final Realm realm;
    private final IdentifiedUser.GenericFactory userFactory;
    private final ChangeUserName.Factory changeUserNameFactory;
    private final ProjectCache projectCache;
    private final AtomicBoolean awaitsFirstAccountCheck;
    private final AuditService auditService;
    private final ExternalIdsUpdate.Server externalIdsUpdateFactory;

    @Inject
    AccountManager(SchemaFactory<ReviewDb> schema, AccountCache byIdCache, AccountByEmailCache byEmailCache, Realm accountMapper, IdentifiedUser.GenericFactory userFactory, ChangeUserName.Factory changeUserNameFactory, ProjectCache projectCache, AuditService auditService, ExternalIdsUpdate.Server externalIdsUpdateFactory) {
        this.schema = schema;
        this.byIdCache = byIdCache;
        this.byEmailCache = byEmailCache;
        this.realm = accountMapper;
        this.userFactory = userFactory;
        this.changeUserNameFactory = changeUserNameFactory;
        this.projectCache = projectCache;
        this.awaitsFirstAccountCheck = new AtomicBoolean(true);
        this.auditService = auditService;
        this.externalIdsUpdateFactory = externalIdsUpdateFactory;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Optional<Account.Id> lookup(String externalId) throws AccountException {
        try (ReviewDb db = this.schema.open();){
            ExternalId extId = this.findExternalId(db, ExternalId.Key.parse(externalId));
            Optional<Account.Id> optional = extId != null ? Optional.of(extId.accountId()) : Optional.empty();
            return optional;
        }
        catch (OrmException e) {
            throw new AccountException("Cannot lookup account " + externalId, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public AuthResult authenticate(AuthRequest who) throws AccountException, IOException {
        who = this.realm.authenticate(who);
        try (ReviewDb db = this.schema.open();){
            ExternalId id = this.findExternalId(db, who.getExternalIdKey());
            if (id == null) {
                Object key;
                ExternalId existingId;
                if (who.getUserName() != null && (existingId = this.findExternalId(db, (ExternalId.Key)(key = ExternalId.Key.create("username", who.getUserName())))) != null) {
                    log.warn("User {} already has an account; link new identity to the existing account.", (Object)who.getUserName());
                    AuthResult authResult = this.link(existingId.accountId(), who);
                    return authResult;
                }
                log.debug("External ID not found. Attempting to create new account.");
                key = this.create(db, who);
                return key;
            }
            Account act = this.byIdCache.get(id.accountId()).getAccount();
            if (!act.isActive()) {
                throw new AccountException("Authentication error, account inactive");
            }
            this.update(db, who, id);
            AuthResult authResult = new AuthResult(id.accountId(), who.getExternalIdKey(), false);
            return authResult;
        }
        catch (OrmException | ConfigInvalidException e) {
            throw new AccountException("Authentication error", e);
        }
    }

    private ExternalId findExternalId(ReviewDb db, ExternalId.Key key) throws OrmException {
        return ExternalId.from(db.accountExternalIds().get(key.asAccountExternalIdKey()));
    }

    private void update(ReviewDb db, AuthRequest who, ExternalId extId) throws OrmException, IOException {
        IdentifiedUser user = this.userFactory.create(extId.accountId());
        Account toUpdate = null;
        String newEmail = who.getEmailAddress();
        String oldEmail = extId.email();
        if (newEmail != null && !newEmail.equals(oldEmail)) {
            if (oldEmail != null && oldEmail.equals(user.getAccount().getPreferredEmail())) {
                toUpdate = this.load(toUpdate, user.getAccountId(), db);
                toUpdate.setPreferredEmail(newEmail);
            }
            this.externalIdsUpdateFactory.create().replace(db, extId, ExternalId.create(extId.key(), extId.accountId(), newEmail, extId.password()));
        }
        if (!(this.realm.allowsEdit(AccountFieldName.FULL_NAME) || Strings.isNullOrEmpty(who.getDisplayName()) || AccountManager.eq(user.getAccount().getFullName(), who.getDisplayName()))) {
            toUpdate = this.load(toUpdate, user.getAccountId(), db);
            toUpdate.setFullName(who.getDisplayName());
        }
        if (!this.realm.allowsEdit(AccountFieldName.USER_NAME) && who.getUserName() != null && !AccountManager.eq(user.getUserName(), who.getUserName())) {
            log.warn("Not changing already set username {} to {}", (Object)user.getUserName(), (Object)who.getUserName());
        }
        if (toUpdate != null) {
            db.accounts().update(Collections.singleton(toUpdate));
        }
        if (newEmail != null && !newEmail.equals(oldEmail)) {
            this.byEmailCache.evict(oldEmail);
            this.byEmailCache.evict(newEmail);
        }
        if (toUpdate != null) {
            this.byIdCache.evict(toUpdate.getId());
        }
    }

    private Account load(Account toUpdate, Account.Id accountId, ReviewDb db) throws OrmException {
        if (toUpdate == null && (toUpdate = db.accounts().get(accountId)) == null) {
            throw new OrmException("Account " + accountId + " has been deleted");
        }
        return toUpdate;
    }

    private static boolean eq(String a, String b) {
        return a == null && b == null || a != null && a.equals(b);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private AuthResult create(ReviewDb db, AuthRequest who) throws OrmException, AccountException, IOException, ConfigInvalidException {
        Account.Id newId = new Account.Id(db.nextAccountId());
        log.debug("Assigning new Id {} to account", (Object)newId);
        Account account = new Account(newId, TimeUtil.nowTs());
        ExternalId extId = ExternalId.createWithEmail(who.getExternalIdKey(), newId, who.getEmailAddress());
        log.debug("Created external Id: {}", (Object)extId);
        account.setFullName(who.getDisplayName());
        account.setPreferredEmail(extId.email());
        boolean isFirstAccount = this.awaitsFirstAccountCheck.getAndSet(false) && db.accounts().anyAccounts().toList().isEmpty();
        try {
            db.accounts().upsert(Collections.singleton(account));
            ExternalId existingExtId = ExternalId.from(db.accountExternalIds().get(extId.key().asAccountExternalIdKey()));
            if (existingExtId != null && !existingExtId.accountId().equals(extId.accountId())) {
                db.accounts().delete(Collections.singleton(account));
                throw new AccountException("Cannot assign external ID \"" + extId.key().get() + "\" to account " + newId + "; external ID already in use.");
            }
            this.externalIdsUpdateFactory.create().upsert(db, extId);
        }
        finally {
            this.awaitsFirstAccountCheck.set(isFirstAccount);
        }
        if (isFirstAccount) {
            Permission admin = this.projectCache.getAllProjects().getConfig().getAccessSection("GLOBAL_CAPABILITIES").getPermission("administrateServer");
            AccountGroup.UUID uuid = admin.getRules().get(0).getGroup().getUUID();
            Iterator<AccountGroup> adminGroupIt = db.accountGroups().byUUID(uuid).iterator();
            if (!adminGroupIt.hasNext()) {
                throw new OrmException("Administrator group's UUID is misaligned in backend and All-Projects repository");
            }
            AccountGroup g = adminGroupIt.next();
            AccountGroup.Id adminId = g.getId();
            AccountGroupMember m = new AccountGroupMember(new AccountGroupMember.Key(newId, adminId));
            this.auditService.dispatchAddAccountsToGroup(newId, Collections.singleton(m));
            db.accountGroupMembers().insert(Collections.singleton(m));
        }
        log.debug("Username from AuthRequest: {}", (Object)who.getUserName());
        if (who.getUserName() != null) {
            String message;
            log.debug("Setting username for: {}", (Object)who.getUserName());
            IdentifiedUser user = this.userFactory.create(newId);
            log.debug("Identified user {} was created from {}", (Object)user, (Object)who.getUserName());
            try {
                this.changeUserNameFactory.create(db, user, who.getUserName()).call();
            }
            catch (NameAlreadyUsedException e) {
                message = "Cannot assign user name \"" + who.getUserName() + "\" to account " + newId + "; name already in use.";
                this.handleSettingUserNameFailure(db, account, extId, message, e, false);
            }
            catch (InvalidUserNameException e) {
                message = "Cannot assign user name \"" + who.getUserName() + "\" to account " + newId + "; name does not conform.";
                this.handleSettingUserNameFailure(db, account, extId, message, e, false);
            }
            catch (OrmException e) {
                message = "Cannot assign user name";
                this.handleSettingUserNameFailure(db, account, extId, message, e, true);
            }
        }
        this.byEmailCache.evict(account.getPreferredEmail());
        this.byIdCache.evict(account.getId());
        this.realm.onCreateAccount(who, account);
        return new AuthResult(newId, extId.key(), true);
    }

    private void handleSettingUserNameFailure(ReviewDb db, Account account, ExternalId extId, String errorMessage, Exception e, boolean logException) throws AccountUserNameException, OrmException, IOException {
        if (logException) {
            log.error(errorMessage, e);
        } else {
            log.error(errorMessage);
        }
        if (!this.realm.allowsEdit(AccountFieldName.USER_NAME)) {
            db.accounts().delete(Collections.singleton(account));
            this.externalIdsUpdateFactory.create().delete(db, extId);
            throw new AccountUserNameException(errorMessage, e);
        }
    }

    public AuthResult link(Account.Id to, AuthRequest who) throws AccountException, OrmException, IOException {
        try (ReviewDb db = this.schema.open();){
            log.debug("Link another authentication identity to an existing account");
            ExternalId extId = this.findExternalId(db, who.getExternalIdKey());
            if (extId != null) {
                if (!extId.accountId().equals(to)) {
                    throw new AccountException("Identity in use by another account");
                }
                log.debug("Updating existing external ID data");
                this.update(db, who, extId);
            } else {
                log.debug("Linking new external ID to the existing account");
                this.externalIdsUpdateFactory.create().insert(db, ExternalId.createWithEmail(who.getExternalIdKey(), to, who.getEmailAddress()));
                if (who.getEmailAddress() != null) {
                    Account a = db.accounts().get(to);
                    if (a.getPreferredEmail() == null) {
                        a.setPreferredEmail(who.getEmailAddress());
                        db.accounts().update(Collections.singleton(a));
                        this.byIdCache.evict(to);
                    }
                    this.byEmailCache.evict(who.getEmailAddress());
                }
            }
            AuthResult authResult = new AuthResult(to, who.getExternalIdKey(), false);
            return authResult;
        }
    }

    public AuthResult updateLink(Account.Id to, AuthRequest who) throws OrmException, AccountException, IOException {
        try (ReviewDb db = this.schema.open();){
            Collection filteredExtIdsByScheme = ExternalId.from(db.accountExternalIds().byAccount(to).toList()).stream().filter(e -> e.isScheme(who.getExternalIdKey().scheme())).collect(Collectors.toSet());
            if (!(filteredExtIdsByScheme.isEmpty() || filteredExtIdsByScheme.size() <= 1 && filteredExtIdsByScheme.stream().filter(e -> e.key().equals(who.getExternalIdKey())).findAny().isPresent())) {
                this.externalIdsUpdateFactory.create().delete(db, filteredExtIdsByScheme);
            }
            this.byIdCache.evict(to);
            AuthResult authResult = this.link(to, who);
            return authResult;
        }
    }

    public AuthResult unlink(Account.Id from, AuthRequest who) throws AccountException, OrmException, IOException {
        try (ReviewDb db = this.schema.open();){
            ExternalId extId = this.findExternalId(db, who.getExternalIdKey());
            if (extId != null) {
                if (!extId.accountId().equals(from)) {
                    throw new AccountException("Identity '" + who.getExternalIdKey().get() + "' in use by another account");
                }
                this.externalIdsUpdateFactory.create().delete(db, extId);
                if (who.getEmailAddress() != null) {
                    Account a = db.accounts().get(from);
                    if (a.getPreferredEmail() != null && a.getPreferredEmail().equals(who.getEmailAddress())) {
                        a.setPreferredEmail(null);
                        db.accounts().update(Collections.singleton(a));
                        this.byIdCache.evict(from);
                    }
                    this.byEmailCache.evict(who.getEmailAddress());
                }
            } else {
                throw new AccountException("Identity '" + who.getExternalIdKey().get() + "' not found");
            }
            AuthResult authResult = new AuthResult(from, who.getExternalIdKey(), false);
            return authResult;
        }
    }
}

