/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git;

import com.google.common.base.CaseFormat;
import com.google.common.base.Supplier;
import com.google.gerrit.extensions.events.LifecycleListener;
import com.google.gerrit.lifecycle.LifecycleModule;
import com.google.gerrit.metrics.Description;
import com.google.gerrit.metrics.MetricMaker;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.git.ProjectRunnable;
import com.google.gerrit.server.git.TaskInfoFactory;
import com.google.gerrit.server.util.IdGenerator;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.RunnableScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jgit.lib.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class WorkQueue {
    private static final Logger log = LoggerFactory.getLogger(WorkQueue.class);
    private static final Thread.UncaughtExceptionHandler LOG_UNCAUGHT_EXCEPTION = new Thread.UncaughtExceptionHandler(){

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            log.error("WorkQueue thread " + t.getName() + " threw exception", e);
        }
    };
    private final Executor defaultQueue;
    private final IdGenerator idGenerator;
    private final MetricMaker metrics;
    private final CopyOnWriteArrayList<Executor> queues;

    @Inject
    WorkQueue(IdGenerator idGenerator, @GerritServerConfig Config cfg, MetricMaker metrics) {
        this(idGenerator, cfg.getInt("execution", "defaultThreadPoolSize", 1), metrics);
    }

    public WorkQueue(IdGenerator idGenerator, int defaultThreadPoolSize, MetricMaker metrics) {
        this.idGenerator = idGenerator;
        this.metrics = metrics;
        this.queues = new CopyOnWriteArrayList();
        this.defaultQueue = this.createQueue(defaultThreadPoolSize, "WorkQueue", true);
    }

    public Executor getDefaultQueue() {
        return this.defaultQueue;
    }

    public Executor createQueue(int poolsize, String queueName) {
        return this.createQueue(poolsize, queueName, false);
    }

    public Executor createQueue(int poolsize, String queueName, boolean withMetrics) {
        Executor r = new Executor(poolsize, queueName);
        if (withMetrics) {
            log.info("Adding metrics for '{}' queue", (Object)queueName);
            r.buildMetrics(queueName);
        }
        r.setContinueExistingPeriodicTasksAfterShutdownPolicy(false);
        r.setExecuteExistingDelayedTasksAfterShutdownPolicy(true);
        this.queues.add(r);
        return r;
    }

    public List<Task<?>> getTasks() {
        ArrayList r = new ArrayList();
        for (Executor e : this.queues) {
            e.addAllTo(r);
        }
        return r;
    }

    public <T> List<T> getTaskInfos(TaskInfoFactory<T> factory) {
        ArrayList<T> taskInfos = new ArrayList<T>();
        for (Executor exe : this.queues) {
            for (Task<?> task : exe.getTasks()) {
                taskInfos.add(factory.getTaskInfo(task));
            }
        }
        return taskInfos;
    }

    public Task<?> getTask(int id) {
        Task<?> result = null;
        for (Executor e : this.queues) {
            Task<?> t = e.getTask(id);
            if (t == null) continue;
            if (result != null) {
                return null;
            }
            result = t;
        }
        return result;
    }

    public Executor getExecutor(String queueName) {
        for (Executor e : this.queues) {
            if (!e.queueName.equals(queueName)) continue;
            return e;
        }
        return null;
    }

    private void stop() {
        for (Executor p : this.queues) {
            boolean isTerminated;
            p.shutdown();
            do {
                try {
                    isTerminated = p.awaitTermination(10L, TimeUnit.SECONDS);
                }
                catch (InterruptedException ie) {
                    isTerminated = false;
                }
            } while (!isTerminated);
        }
        this.queues.clear();
    }

    public static class ProjectTask<V>
    extends Task<V>
    implements ProjectRunnable {
        private final ProjectRunnable runnable;

        ProjectTask(ProjectRunnable runnable, RunnableScheduledFuture<V> task, Executor executor, int taskId) {
            super(runnable, task, executor, taskId);
            this.runnable = runnable;
        }

        @Override
        public Project.NameKey getProjectNameKey() {
            return this.runnable.getProjectNameKey();
        }

        @Override
        public String getRemoteName() {
            return this.runnable.getRemoteName();
        }

        @Override
        public boolean hasCustomizedPrint() {
            return this.runnable.hasCustomizedPrint();
        }
    }

    public static class Task<V>
    implements RunnableScheduledFuture<V> {
        private final Runnable runnable;
        private final RunnableScheduledFuture<V> task;
        private final Executor executor;
        private final int taskId;
        private final AtomicBoolean running;
        private final Date startTime;

        Task(Runnable runnable, RunnableScheduledFuture<V> task, Executor executor, int taskId) {
            this.runnable = runnable;
            this.task = task;
            this.executor = executor;
            this.taskId = taskId;
            this.running = new AtomicBoolean();
            this.startTime = new Date();
        }

        public int getTaskId() {
            return this.taskId;
        }

        public State getState() {
            if (this.isCancelled()) {
                return State.CANCELLED;
            }
            if (this.isDone() && !this.isPeriodic()) {
                return State.DONE;
            }
            if (this.running.get()) {
                return State.RUNNING;
            }
            long delay = this.getDelay(TimeUnit.MILLISECONDS);
            if (delay <= 0L) {
                return State.READY;
            }
            return State.SLEEPING;
        }

        public Date getStartTime() {
            return this.startTime;
        }

        public String getQueueName() {
            return this.executor.queueName;
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            if (this.task.cancel(mayInterruptIfRunning)) {
                if (this.runnable instanceof CancelableRunnable) {
                    if (this.running.compareAndSet(false, true)) {
                        ((CancelableRunnable)this.runnable).cancel();
                    } else if (this.runnable instanceof CanceledWhileRunning) {
                        ((CanceledWhileRunning)this.runnable).setCanceledWhileRunning();
                    }
                }
                if (this.runnable instanceof Future) {
                    ((Future)((Object)this.runnable)).cancel(mayInterruptIfRunning);
                }
                this.executor.remove(this);
                this.executor.purge();
                return true;
            }
            return false;
        }

        @Override
        public int compareTo(Delayed o) {
            return this.task.compareTo(o);
        }

        @Override
        public V get() throws InterruptedException, ExecutionException {
            return this.task.get();
        }

        @Override
        public V get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            return this.task.get(timeout, unit);
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return this.task.getDelay(unit);
        }

        @Override
        public boolean isCancelled() {
            return this.task.isCancelled();
        }

        @Override
        public boolean isDone() {
            return this.task.isDone();
        }

        @Override
        public boolean isPeriodic() {
            return this.task.isPeriodic();
        }

        @Override
        public void run() {
            if (this.running.compareAndSet(false, true)) {
                try {
                    this.task.run();
                }
                finally {
                    if (this.isPeriodic()) {
                        this.running.set(false);
                    } else {
                        this.executor.remove(this);
                    }
                }
            }
        }

        public String toString() {
            try {
                if (this.runnable.getClass().isAssignableFrom(Class.forName("com.google.common.util.concurrent.TrustedListenableFutureTask"))) {
                    Class<?> trustedFutureInterruptibleTask = Class.forName("com.google.common.util.concurrent.TrustedListenableFutureTask$TrustedFutureInterruptibleTask");
                    for (Field field : this.runnable.getClass().getDeclaredFields()) {
                        if (!field.getType().isAssignableFrom(trustedFutureInterruptibleTask)) continue;
                        field.setAccessible(true);
                        Object innerObj = field.get(this.runnable);
                        if (innerObj == null) continue;
                        for (Field innerField : innerObj.getClass().getDeclaredFields()) {
                            if (!innerField.getType().isAssignableFrom(Callable.class)) continue;
                            innerField.setAccessible(true);
                            return ((Callable)innerField.get(innerObj)).toString();
                        }
                    }
                }
            }
            catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException e) {
                log.debug("Cannot get a proper name for TrustedListenableFutureTask: {}", (Object)e.getMessage());
            }
            return this.runnable.toString();
        }

        public static enum State {
            DONE,
            CANCELLED,
            RUNNING,
            READY,
            SLEEPING,
            OTHER;

        }
    }

    public static interface CanceledWhileRunning
    extends CancelableRunnable {
        public void setCanceledWhileRunning();
    }

    public static interface CancelableRunnable
    extends Runnable {
        public void cancel();
    }

    public class Executor
    extends ScheduledThreadPoolExecutor {
        private final ConcurrentHashMap<Integer, Task<?>> all;
        private final String queueName;

        Executor(int corePoolSize, final String queueName) {
            super(corePoolSize, new ThreadFactory(){
                private final ThreadFactory parent = Executors.defaultThreadFactory();
                private final AtomicInteger tid = new AtomicInteger(1);

                @Override
                public Thread newThread(Runnable task) {
                    Thread t = this.parent.newThread(task);
                    t.setName(queueName + "-" + this.tid.getAndIncrement());
                    t.setUncaughtExceptionHandler(LOG_UNCAUGHT_EXCEPTION);
                    return t;
                }
            });
            this.all = new ConcurrentHashMap(corePoolSize << 1, 0.75f, corePoolSize + 4);
            this.queueName = queueName;
        }

        private void buildMetrics(String queueName) {
            WorkQueue.this.metrics.newCallbackMetric(this.getMetricName(queueName, "max_pool_size"), Long.class, new Description("Maximum allowed number of threads in the pool").setGauge().setUnit("threads"), new Supplier<Long>(){

                @Override
                public Long get() {
                    return Executor.this.getMaximumPoolSize();
                }
            });
            WorkQueue.this.metrics.newCallbackMetric(this.getMetricName(queueName, "pool_size"), Long.class, new Description("Current number of threads in the pool").setGauge().setUnit("threads"), new Supplier<Long>(){

                @Override
                public Long get() {
                    return Executor.this.getPoolSize();
                }
            });
            WorkQueue.this.metrics.newCallbackMetric(this.getMetricName(queueName, "active_threads"), Long.class, new Description("Number number of threads that are actively executing tasks").setGauge().setUnit("threads"), new Supplier<Long>(){

                @Override
                public Long get() {
                    return Executor.this.getActiveCount();
                }
            });
            WorkQueue.this.metrics.newCallbackMetric(this.getMetricName(queueName, "scheduled_tasks"), Integer.class, new Description("Number of scheduled tasks in the queue").setGauge().setUnit("tasks"), new Supplier<Integer>(){

                @Override
                public Integer get() {
                    return Executor.this.getQueue().size();
                }
            });
            WorkQueue.this.metrics.newCallbackMetric(this.getMetricName(queueName, "total_scheduled_tasks_count"), Long.class, new Description("Total number of tasks that have been scheduled for execution").setCumulative().setUnit("tasks"), new Supplier<Long>(){

                @Override
                public Long get() {
                    return Executor.this.getTaskCount();
                }
            });
            WorkQueue.this.metrics.newCallbackMetric(this.getMetricName(queueName, "total_completed_tasks_count"), Long.class, new Description("Total number of tasks that have completed execution").setCumulative().setUnit("tasks"), new Supplier<Long>(){

                @Override
                public Long get() {
                    return Executor.this.getCompletedTaskCount();
                }
            });
        }

        private String getMetricName(String queueName, String metricName) {
            String name = CaseFormat.UPPER_CAMEL.to(CaseFormat.LOWER_UNDERSCORE, queueName.replaceFirst("SSH", "Ssh").replaceAll("-", ""));
            return WorkQueue.this.metrics.sanitizeMetricName(String.format("queue/%s/%s", name, metricName));
        }

        public void unregisterWorkQueue() {
            WorkQueue.this.queues.remove(this);
        }

        @Override
        protected <V> RunnableScheduledFuture<V> decorateTask(Runnable runnable, RunnableScheduledFuture<V> r) {
            int id;
            Task task;
            r = super.decorateTask(runnable, r);
            do {
                id = WorkQueue.this.idGenerator.next();
            } while (this.all.putIfAbsent((task = runnable instanceof ProjectRunnable ? new ProjectTask<V>((ProjectRunnable)runnable, r, this, id) : new Task(runnable, r, this, id)).getTaskId(), task) != null);
            return task;
        }

        @Override
        protected <V> RunnableScheduledFuture<V> decorateTask(Callable<V> callable, RunnableScheduledFuture<V> task) {
            throw new UnsupportedOperationException("Callable not implemented");
        }

        void remove(Task<?> task) {
            this.all.remove(task.getTaskId(), task);
        }

        Task<?> getTask(int id) {
            return this.all.get(id);
        }

        void addAllTo(List<Task<?>> list) {
            list.addAll(this.all.values());
        }

        Collection<Task<?>> getTasks() {
            return this.all.values();
        }
    }

    public static class Module
    extends LifecycleModule {
        @Override
        protected void configure() {
            this.bind(WorkQueue.class);
            this.listener().to(Lifecycle.class);
        }
    }

    public static class Lifecycle
    implements LifecycleListener {
        private final WorkQueue workQueue;

        @Inject
        Lifecycle(WorkQueue workQeueue) {
            this.workQueue = workQeueue;
        }

        @Override
        public void start() {
        }

        @Override
        public void stop() {
            this.workQueue.stop();
        }
    }
}

