/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.edit;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.gerrit.common.TimeUtil;
import com.google.gerrit.extensions.restapi.AuthException;
import com.google.gerrit.extensions.restapi.MergeConflictException;
import com.google.gerrit.extensions.restapi.RawInput;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.RefNames;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.CurrentUser;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.PatchSetUtil;
import com.google.gerrit.server.edit.ChangeEdit;
import com.google.gerrit.server.edit.ChangeEditUtil;
import com.google.gerrit.server.edit.UnchangedCommitMessageException;
import com.google.gerrit.server.edit.tree.ChangeFileContentModification;
import com.google.gerrit.server.edit.tree.DeleteFileModification;
import com.google.gerrit.server.edit.tree.RenameFileModification;
import com.google.gerrit.server.edit.tree.RestoreFileModification;
import com.google.gerrit.server.edit.tree.TreeCreator;
import com.google.gerrit.server.edit.tree.TreeModification;
import com.google.gerrit.server.index.change.ChangeIndexer;
import com.google.gerrit.server.project.ChangeControl;
import com.google.gerrit.server.project.InvalidChangeOperationException;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.Optional;
import java.util.TimeZone;
import org.eclipse.jgit.lib.BatchRefUpdate;
import org.eclipse.jgit.lib.CommitBuilder;
import org.eclipse.jgit.lib.NullProgressMonitor;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectInserter;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.RefUpdate;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.merge.MergeStrategy;
import org.eclipse.jgit.merge.ThreeWayMerger;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.transport.ReceiveCommand;

@Singleton
public class ChangeEditModifier {
    private final TimeZone tz;
    private final ChangeIndexer indexer;
    private final Provider<ReviewDb> reviewDb;
    private final Provider<CurrentUser> currentUser;
    private final ChangeEditUtil changeEditUtil;
    private final PatchSetUtil patchSetUtil;

    @Inject
    ChangeEditModifier(@GerritPersonIdent PersonIdent gerritIdent, ChangeIndexer indexer, Provider<ReviewDb> reviewDb, Provider<CurrentUser> currentUser, ChangeEditUtil changeEditUtil, PatchSetUtil patchSetUtil) {
        this.indexer = indexer;
        this.reviewDb = reviewDb;
        this.currentUser = currentUser;
        this.tz = gerritIdent.getTimeZone();
        this.changeEditUtil = changeEditUtil;
        this.patchSetUtil = patchSetUtil;
    }

    public void createEdit(Repository repository, ChangeControl changeControl) throws AuthException, IOException, InvalidChangeOperationException, OrmException {
        this.ensureAuthenticatedAndPermitted(changeControl);
        Optional<ChangeEdit> changeEdit = this.lookupChangeEdit(changeControl);
        if (changeEdit.isPresent()) {
            throw new InvalidChangeOperationException(String.format("A change edit already exists for change %s", changeControl.getId()));
        }
        PatchSet currentPatchSet = this.lookupCurrentPatchSet(changeControl);
        ObjectId patchSetCommitId = ChangeEditModifier.getPatchSetCommitId(currentPatchSet);
        this.createEditReference(repository, changeControl, currentPatchSet, patchSetCommitId, TimeUtil.nowTs());
    }

    public void rebaseEdit(Repository repository, ChangeControl changeControl) throws AuthException, InvalidChangeOperationException, IOException, OrmException, MergeConflictException {
        PatchSet currentPatchSet;
        this.ensureAuthenticatedAndPermitted(changeControl);
        Optional<ChangeEdit> optionalChangeEdit = this.lookupChangeEdit(changeControl);
        if (!optionalChangeEdit.isPresent()) {
            throw new InvalidChangeOperationException(String.format("No change edit exists for change %s", changeControl.getId()));
        }
        ChangeEdit changeEdit = optionalChangeEdit.get();
        if (ChangeEditModifier.isBasedOn(changeEdit, currentPatchSet = this.lookupCurrentPatchSet(changeControl))) {
            throw new InvalidChangeOperationException(String.format("Change edit for change %s is already based on latest patch set %s", changeControl.getId(), currentPatchSet.getId()));
        }
        this.rebase(repository, changeEdit, currentPatchSet);
    }

    private void rebase(Repository repository, ChangeEdit changeEdit, PatchSet currentPatchSet) throws IOException, MergeConflictException, InvalidChangeOperationException, OrmException {
        RevCommit currentEditCommit = changeEdit.getEditCommit();
        if (currentEditCommit.getParentCount() == 0) {
            throw new InvalidChangeOperationException("Rebase change edit against root commit not supported");
        }
        Change change = changeEdit.getChange();
        RevCommit basePatchSetCommit = ChangeEditModifier.lookupCommit(repository, currentPatchSet);
        RevTree basePatchSetTree = basePatchSetCommit.getTree();
        ObjectId newTreeId = this.merge(repository, changeEdit, basePatchSetTree);
        Timestamp nowTimestamp = TimeUtil.nowTs();
        String commitMessage = currentEditCommit.getFullMessage();
        ObjectId newEditCommitId = this.createCommit(repository, basePatchSetCommit, newTreeId, commitMessage, nowTimestamp);
        String newEditRefName = this.getEditRefName(change, currentPatchSet);
        this.updateReferenceWithNameChange(repository, changeEdit.getRefName(), currentEditCommit, newEditRefName, newEditCommitId, nowTimestamp);
        this.reindex(change);
    }

    public void modifyMessage(Repository repository, ChangeControl changeControl, String newCommitMessage) throws AuthException, IOException, UnchangedCommitMessageException, OrmException {
        this.ensureAuthenticatedAndPermitted(changeControl);
        newCommitMessage = this.getWellFormedCommitMessage(newCommitMessage);
        Optional<ChangeEdit> optionalChangeEdit = this.lookupChangeEdit(changeControl);
        PatchSet basePatchSet = this.getBasePatchSet(optionalChangeEdit, changeControl);
        RevCommit basePatchSetCommit = ChangeEditModifier.lookupCommit(repository, basePatchSet);
        RevCommit baseCommit = optionalChangeEdit.map(ChangeEdit::getEditCommit).orElse(basePatchSetCommit);
        String currentCommitMessage = baseCommit.getFullMessage();
        if (newCommitMessage.equals(currentCommitMessage)) {
            throw new UnchangedCommitMessageException();
        }
        RevTree baseTree = baseCommit.getTree();
        Timestamp nowTimestamp = TimeUtil.nowTs();
        ObjectId newEditCommit = this.createCommit(repository, basePatchSetCommit, baseTree, newCommitMessage, nowTimestamp);
        if (optionalChangeEdit.isPresent()) {
            this.updateEditReference(repository, optionalChangeEdit.get(), newEditCommit, nowTimestamp);
        } else {
            this.createEditReference(repository, changeControl, basePatchSet, newEditCommit, nowTimestamp);
        }
    }

    public void modifyFile(Repository repository, ChangeControl changeControl, String filePath, RawInput newContent) throws AuthException, InvalidChangeOperationException, IOException, OrmException {
        this.modifyTree(repository, changeControl, new ChangeFileContentModification(filePath, newContent));
    }

    public void deleteFile(Repository repository, ChangeControl changeControl, String file) throws AuthException, InvalidChangeOperationException, IOException, OrmException {
        this.modifyTree(repository, changeControl, new DeleteFileModification(file));
    }

    public void renameFile(Repository repository, ChangeControl changeControl, String currentFilePath, String newFilePath) throws AuthException, InvalidChangeOperationException, IOException, OrmException {
        this.modifyTree(repository, changeControl, new RenameFileModification(currentFilePath, newFilePath));
    }

    public void restoreFile(Repository repository, ChangeControl changeControl, String file) throws AuthException, InvalidChangeOperationException, IOException, OrmException {
        this.modifyTree(repository, changeControl, new RestoreFileModification(file));
    }

    private void modifyTree(Repository repository, ChangeControl changeControl, TreeModification treeModification) throws AuthException, IOException, OrmException, InvalidChangeOperationException {
        this.ensureAuthenticatedAndPermitted(changeControl);
        Optional<ChangeEdit> optionalChangeEdit = this.lookupChangeEdit(changeControl);
        PatchSet basePatchSet = this.getBasePatchSet(optionalChangeEdit, changeControl);
        RevCommit basePatchSetCommit = ChangeEditModifier.lookupCommit(repository, basePatchSet);
        RevCommit baseCommit = optionalChangeEdit.map(ChangeEdit::getEditCommit).orElse(basePatchSetCommit);
        ObjectId newTreeId = ChangeEditModifier.createNewTree(repository, baseCommit, treeModification);
        String commitMessage = baseCommit.getFullMessage();
        Timestamp nowTimestamp = TimeUtil.nowTs();
        ObjectId newEditCommit = this.createCommit(repository, basePatchSetCommit, newTreeId, commitMessage, nowTimestamp);
        if (optionalChangeEdit.isPresent()) {
            this.updateEditReference(repository, optionalChangeEdit.get(), newEditCommit, nowTimestamp);
        } else {
            this.createEditReference(repository, changeControl, basePatchSet, newEditCommit, nowTimestamp);
        }
    }

    private void ensureAuthenticatedAndPermitted(ChangeControl changeControl) throws AuthException, OrmException {
        this.ensureAuthenticated();
        this.ensurePermitted(changeControl);
    }

    private void ensureAuthenticated() throws AuthException {
        if (!this.currentUser.get().isIdentifiedUser()) {
            throw new AuthException("Authentication required");
        }
    }

    private void ensurePermitted(ChangeControl changeControl) throws OrmException, AuthException {
        if (!changeControl.canAddPatchSet(this.reviewDb.get())) {
            throw new AuthException("Not allowed to edit a change.");
        }
    }

    private String getWellFormedCommitMessage(String commitMessage) {
        String wellFormedMessage = Strings.nullToEmpty(commitMessage).trim();
        Preconditions.checkState(!wellFormedMessage.isEmpty(), "Commit message cannot be null or empty");
        wellFormedMessage = wellFormedMessage + "\n";
        return wellFormedMessage;
    }

    private Optional<ChangeEdit> lookupChangeEdit(ChangeControl changeControl) throws AuthException, IOException {
        return this.changeEditUtil.byChange(changeControl);
    }

    private PatchSet getBasePatchSet(Optional<ChangeEdit> optionalChangeEdit, ChangeControl changeControl) throws OrmException {
        Optional<PatchSet> editBasePatchSet = optionalChangeEdit.map(ChangeEdit::getBasePatchSet);
        return editBasePatchSet.isPresent() ? editBasePatchSet.get() : this.lookupCurrentPatchSet(changeControl);
    }

    private PatchSet lookupCurrentPatchSet(ChangeControl changeControl) throws OrmException {
        return this.patchSetUtil.current(this.reviewDb.get(), changeControl.getNotes());
    }

    private static boolean isBasedOn(ChangeEdit changeEdit, PatchSet patchSet) {
        PatchSet editBasePatchSet = changeEdit.getBasePatchSet();
        return editBasePatchSet.getId().equals(patchSet.getId());
    }

    private static RevCommit lookupCommit(Repository repository, PatchSet patchSet) throws IOException {
        ObjectId patchSetCommitId = ChangeEditModifier.getPatchSetCommitId(patchSet);
        try (RevWalk revWalk = new RevWalk(repository);){
            RevCommit revCommit = revWalk.parseCommit(patchSetCommitId);
            return revCommit;
        }
    }

    private static ObjectId createNewTree(Repository repository, RevCommit baseCommit, TreeModification treeModification) throws IOException, InvalidChangeOperationException {
        TreeCreator treeCreator = new TreeCreator(baseCommit);
        treeCreator.addTreeModification(treeModification);
        ObjectId newTreeId = treeCreator.createNewTreeAndGetId(repository);
        if (ObjectId.equals(newTreeId, baseCommit.getTree())) {
            throw new InvalidChangeOperationException("no changes were made");
        }
        return newTreeId;
    }

    private ObjectId merge(Repository repository, ChangeEdit changeEdit, ObjectId newTreeId) throws IOException, MergeConflictException {
        PatchSet basePatchSet = changeEdit.getBasePatchSet();
        ObjectId basePatchSetCommitId = ChangeEditModifier.getPatchSetCommitId(basePatchSet);
        RevCommit editCommitId = changeEdit.getEditCommit();
        ThreeWayMerger threeWayMerger = MergeStrategy.RESOLVE.newMerger(repository, true);
        threeWayMerger.setBase(basePatchSetCommitId);
        boolean successful = threeWayMerger.merge(newTreeId, editCommitId);
        if (!successful) {
            throw new MergeConflictException("The existing change edit could not be merged with another tree.");
        }
        return threeWayMerger.getResultTreeId();
    }

    private ObjectId createCommit(Repository repository, RevCommit basePatchSetCommit, ObjectId tree, String commitMessage, Timestamp timestamp) throws IOException {
        try (ObjectInserter objectInserter = repository.newObjectInserter();){
            CommitBuilder builder = new CommitBuilder();
            builder.setTreeId(tree);
            builder.setParentIds(basePatchSetCommit.getParents());
            builder.setAuthor(basePatchSetCommit.getAuthorIdent());
            builder.setCommitter(this.getCommitterIdent(timestamp));
            builder.setMessage(commitMessage);
            ObjectId newCommitId = objectInserter.insert(builder);
            objectInserter.flush();
            ObjectId objectId = newCommitId;
            return objectId;
        }
    }

    private PersonIdent getCommitterIdent(Timestamp commitTimestamp) {
        IdentifiedUser user = this.currentUser.get().asIdentifiedUser();
        return user.newCommitterIdent(commitTimestamp, this.tz);
    }

    private static ObjectId getPatchSetCommitId(PatchSet patchSet) {
        return ObjectId.fromString(patchSet.getRevision().get());
    }

    private void createEditReference(Repository repository, ChangeControl changeControl, PatchSet basePatchSet, ObjectId newEditCommit, Timestamp timestamp) throws IOException, OrmException {
        Change change = changeControl.getChange();
        String editRefName = this.getEditRefName(change, basePatchSet);
        this.updateReference(repository, editRefName, ObjectId.zeroId(), newEditCommit, timestamp);
        this.reindex(change);
    }

    private String getEditRefName(Change change, PatchSet basePatchSet) {
        IdentifiedUser me = this.currentUser.get().asIdentifiedUser();
        return RefNames.refsEdit(me.getAccountId(), change.getId(), basePatchSet.getId());
    }

    private void updateEditReference(Repository repository, ChangeEdit changeEdit, ObjectId newEditCommit, Timestamp timestamp) throws IOException, OrmException {
        String editRefName = changeEdit.getRefName();
        RevCommit currentEditCommit = changeEdit.getEditCommit();
        this.updateReference(repository, editRefName, currentEditCommit, newEditCommit, timestamp);
        this.reindex(changeEdit.getChange());
    }

    private void updateReference(Repository repository, String refName, ObjectId currentObjectId, ObjectId targetObjectId, Timestamp timestamp) throws IOException {
        RefUpdate ru = repository.updateRef(refName);
        ru.setExpectedOldObjectId(currentObjectId);
        ru.setNewObjectId(targetObjectId);
        ru.setRefLogIdent(this.getRefLogIdent(timestamp));
        ru.setRefLogMessage("inline edit (amend)", false);
        ru.setForceUpdate(true);
        try (RevWalk revWalk = new RevWalk(repository);){
            RefUpdate.Result res = ru.update(revWalk);
            if (res != RefUpdate.Result.NEW && res != RefUpdate.Result.FORCED) {
                throw new IOException("update failed: " + ru);
            }
        }
    }

    private void updateReferenceWithNameChange(Repository repository, String currentRefName, ObjectId currentObjectId, String newRefName, ObjectId targetObjectId, Timestamp timestamp) throws IOException {
        BatchRefUpdate batchRefUpdate = repository.getRefDatabase().newBatchUpdate();
        batchRefUpdate.addCommand(new ReceiveCommand(ObjectId.zeroId(), targetObjectId, newRefName));
        batchRefUpdate.addCommand(new ReceiveCommand(currentObjectId, ObjectId.zeroId(), currentRefName));
        batchRefUpdate.setRefLogMessage("rebase edit", false);
        batchRefUpdate.setRefLogIdent(this.getRefLogIdent(timestamp));
        try (RevWalk revWalk = new RevWalk(repository);){
            batchRefUpdate.execute(revWalk, NullProgressMonitor.INSTANCE);
        }
        for (ReceiveCommand cmd : batchRefUpdate.getCommands()) {
            if (cmd.getResult() == ReceiveCommand.Result.OK) continue;
            throw new IOException("failed: " + cmd);
        }
    }

    private PersonIdent getRefLogIdent(Timestamp timestamp) {
        IdentifiedUser user = this.currentUser.get().asIdentifiedUser();
        return user.newRefLogIdent(timestamp, this.tz);
    }

    private void reindex(Change change) throws IOException, OrmException {
        this.indexer.index(this.reviewDb.get(), change);
    }
}

