/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git.validators;

import com.google.common.base.CharMatcher;
import com.google.common.collect.ImmutableList;
import com.google.gerrit.common.FooterConstants;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.extensions.registration.DynamicSet;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.RefNames;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.account.WatchConfig;
import com.google.gerrit.server.config.AllUsersName;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.events.CommitReceivedEvent;
import com.google.gerrit.server.git.BanCommit;
import com.google.gerrit.server.git.ProjectConfig;
import com.google.gerrit.server.git.ReceiveCommits;
import com.google.gerrit.server.git.ValidationError;
import com.google.gerrit.server.git.validators.CommitValidationException;
import com.google.gerrit.server.git.validators.CommitValidationListener;
import com.google.gerrit.server.git.validators.CommitValidationMessage;
import com.google.gerrit.server.project.ProjectControl;
import com.google.gerrit.server.project.RefControl;
import com.google.gerrit.server.ssh.SshInfo;
import com.google.gerrit.server.util.MagicBranch;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.jcraft.jsch.HostKey;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.notes.NoteMap;
import org.eclipse.jgit.revwalk.FooterKey;
import org.eclipse.jgit.revwalk.FooterLine;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.util.SystemReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitValidators {
    private static final Logger log = LoggerFactory.getLogger(CommitValidators.class);
    private final List<CommitValidationListener> validators;

    CommitValidators(List<CommitValidationListener> validators) {
        this.validators = validators;
    }

    public List<CommitValidationMessage> validate(CommitReceivedEvent receiveEvent) throws CommitValidationException {
        ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
        try {
            for (CommitValidationListener commitValidator : this.validators) {
                messages.addAll(commitValidator.onCommitReceived(receiveEvent));
            }
        }
        catch (CommitValidationException e) {
            log.debug("CommitValidationException occurred: {}", (Object)e.getFullMessage(), (Object)e);
            messages.addAll(e.getMessages());
            throw new CommitValidationException(e.getMessage(), messages);
        }
        return messages;
    }

    private static CommitValidationMessage getInvalidEmailError(RevCommit c, String type, PersonIdent who, IdentifiedUser currentUser, String canonicalWebUrl) {
        StringBuilder sb = new StringBuilder();
        sb.append("\n");
        sb.append("ERROR:  In commit ").append(c.name()).append("\n");
        sb.append("ERROR:  ").append(type).append(" email address ").append(who.getEmailAddress()).append("\n");
        sb.append("ERROR:  does not match your user account.\n");
        sb.append("ERROR:\n");
        if (currentUser.getEmailAddresses().isEmpty()) {
            sb.append("ERROR:  You have not registered any email addresses.\n");
        } else {
            sb.append("ERROR:  The following addresses are currently registered:\n");
            for (String address : currentUser.getEmailAddresses()) {
                sb.append("ERROR:    ").append(address).append("\n");
            }
        }
        sb.append("ERROR:\n");
        if (canonicalWebUrl != null) {
            sb.append("ERROR:  To register an email address, please visit:\n");
            sb.append("ERROR:  ").append(canonicalWebUrl).append("#").append("/settings/contact").append("\n");
        }
        sb.append("\n");
        return new CommitValidationMessage(sb.toString(), false);
    }

    private static String getGerritUrl(String canonicalWebUrl) {
        if (canonicalWebUrl != null) {
            return CharMatcher.is('/').trimTrailingFrom(canonicalWebUrl);
        }
        return "http://" + CommitValidators.getGerritHost(canonicalWebUrl);
    }

    private static String getGerritHost(String canonicalWebUrl) {
        String host;
        if (canonicalWebUrl != null) {
            try {
                host = new URL(canonicalWebUrl).getHost();
            }
            catch (MalformedURLException e) {
                host = SystemReader.getInstance().getHostname();
            }
        } else {
            host = SystemReader.getInstance().getHostname();
        }
        return host;
    }

    private static void addError(String error, List<CommitValidationMessage> messages) {
        messages.add(new CommitValidationMessage(error, true));
    }

    public static class BlockExternalIdUpdateListener
    implements CommitValidationListener {
        private final AllUsersName allUsers;

        public BlockExternalIdUpdateListener(AllUsersName allUsers) {
            this.allUsers = allUsers;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (this.allUsers.equals(receiveEvent.project.getNameKey()) && "refs/meta/external-ids".equals(receiveEvent.refName)) {
                throw new CommitValidationException("not allowed to update refs/meta/external-ids");
            }
            return Collections.emptyList();
        }
    }

    public static class BannedCommitsValidator
    implements CommitValidationListener {
        private final NoteMap rejectCommits;

        public BannedCommitsValidator(NoteMap rejectCommits) {
            this.rejectCommits = rejectCommits;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            try {
                if (this.rejectCommits.contains(receiveEvent.commit)) {
                    throw new CommitValidationException("contains banned commit " + receiveEvent.commit.getName());
                }
                return Collections.emptyList();
            }
            catch (IOException e) {
                String m = "error checking banned commits";
                log.warn(m, e);
                throw new CommitValidationException(m, e);
            }
        }
    }

    public static class AmendedGerritMergeCommitValidationListener
    implements CommitValidationListener {
        private final PersonIdent gerritIdent;
        private final RefControl refControl;

        public AmendedGerritMergeCommitValidationListener(RefControl refControl, PersonIdent gerritIdent) {
            this.refControl = refControl;
            this.gerritIdent = gerritIdent;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            if (receiveEvent.commit.getParentCount() > 1 && author.getName().equals(this.gerritIdent.getName()) && author.getEmailAddress().equals(this.gerritIdent.getEmailAddress()) && !this.refControl.canForgeGerritServerIdentity()) {
                throw new CommitValidationException("do not amend merges not made by you");
            }
            return Collections.emptyList();
        }
    }

    public static class CommitterUploaderValidator
    implements CommitValidationListener {
        private final RefControl refControl;
        private final String canonicalWebUrl;

        public CommitterUploaderValidator(RefControl refControl, String canonicalWebUrl) {
            this.refControl = refControl;
            this.canonicalWebUrl = canonicalWebUrl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent committer;
            IdentifiedUser currentUser = this.refControl.getUser().asIdentifiedUser();
            if (!currentUser.hasEmailAddress((committer = receiveEvent.commit.getCommitterIdent()).getEmailAddress()) && !this.refControl.canForgeCommitter()) {
                ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
                messages.add(CommitValidators.getInvalidEmailError(receiveEvent.commit, "committer", committer, currentUser, this.canonicalWebUrl));
                throw new CommitValidationException("invalid committer", messages);
            }
            return Collections.emptyList();
        }
    }

    public static class AuthorUploaderValidator
    implements CommitValidationListener {
        private final RefControl refControl;
        private final String canonicalWebUrl;

        public AuthorUploaderValidator(RefControl refControl, String canonicalWebUrl) {
            this.refControl = refControl;
            this.canonicalWebUrl = canonicalWebUrl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent author;
            IdentifiedUser currentUser = this.refControl.getUser().asIdentifiedUser();
            if (!currentUser.hasEmailAddress((author = receiveEvent.commit.getAuthorIdent()).getEmailAddress()) && !this.refControl.canForgeAuthor()) {
                ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
                messages.add(CommitValidators.getInvalidEmailError(receiveEvent.commit, "author", author, currentUser, this.canonicalWebUrl));
                throw new CommitValidationException("invalid author", messages);
            }
            return Collections.emptyList();
        }
    }

    public static class SignedOffByValidator
    implements CommitValidationListener {
        private final RefControl refControl;

        public SignedOffByValidator(RefControl refControl) {
            this.refControl = refControl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            IdentifiedUser currentUser = this.refControl.getUser().asIdentifiedUser();
            PersonIdent committer = receiveEvent.commit.getCommitterIdent();
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            ProjectControl projectControl = this.refControl.getProjectControl();
            if (projectControl.getProjectState().isUseSignedOffBy()) {
                boolean sboAuthor = false;
                boolean sboCommitter = false;
                boolean sboMe = false;
                for (FooterLine footer : receiveEvent.commit.getFooterLines()) {
                    String e;
                    if (!footer.matches(FooterKey.SIGNED_OFF_BY) || (e = footer.getEmailAddress()) == null) continue;
                    sboAuthor |= author.getEmailAddress().equals(e);
                    sboCommitter |= committer.getEmailAddress().equals(e);
                    sboMe |= currentUser.hasEmailAddress(e);
                }
                if (!(sboAuthor || sboCommitter || sboMe || this.refControl.canForgeCommitter())) {
                    throw new CommitValidationException("not Signed-off-by author/committer/uploader in commit message footer");
                }
            }
            return Collections.emptyList();
        }
    }

    public static class PluginCommitValidationListener
    implements CommitValidationListener {
        private final DynamicSet<CommitValidationListener> commitValidationListeners;

        public PluginCommitValidationListener(DynamicSet<CommitValidationListener> commitValidationListeners) {
            this.commitValidationListeners = commitValidationListeners;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
            for (CommitValidationListener validator : this.commitValidationListeners) {
                try {
                    messages.addAll(validator.onCommitReceived(receiveEvent));
                }
                catch (CommitValidationException e) {
                    messages.addAll(e.getMessages());
                    throw new CommitValidationException(e.getMessage(), messages);
                }
            }
            return messages;
        }
    }

    public static class UploadMergesPermissionValidator
    implements CommitValidationListener {
        private final RefControl refControl;

        public UploadMergesPermissionValidator(RefControl refControl) {
            this.refControl = refControl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (receiveEvent.commit.getParentCount() > 1 && !this.refControl.canUploadMerges()) {
                throw new CommitValidationException("you are not allowed to upload merges");
            }
            return Collections.emptyList();
        }
    }

    public static class ConfigValidator
    implements CommitValidationListener {
        private final RefControl refControl;
        private final Repository repo;
        private final AllUsersName allUsers;

        public ConfigValidator(RefControl refControl, Repository repo, AllUsersName allUsers) {
            this.refControl = refControl;
            this.repo = repo;
            this.allUsers = allUsers;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            ArrayList<CommitValidationMessage> messages;
            IdentifiedUser currentUser = this.refControl.getUser().asIdentifiedUser();
            if ("refs/meta/config".equals(this.refControl.getRefName())) {
                messages = new ArrayList<CommitValidationMessage>();
                try {
                    ProjectConfig cfg = new ProjectConfig(receiveEvent.project.getNameKey());
                    cfg.load(this.repo, receiveEvent.command.getNewId());
                    if (!cfg.getValidationErrors().isEmpty()) {
                        CommitValidators.addError("Invalid project configuration:", messages);
                        for (ValidationError err : cfg.getValidationErrors()) {
                            CommitValidators.addError("  " + err.getMessage(), messages);
                        }
                        throw new ConfigInvalidException("invalid project configuration");
                    }
                }
                catch (IOException | ConfigInvalidException e) {
                    log.error("User " + currentUser.getUserName() + " tried to push an invalid project configuration " + receiveEvent.command.getNewId().name() + " for project " + receiveEvent.project.getName(), e);
                    throw new CommitValidationException("invalid project configuration", messages);
                }
            }
            if (this.allUsers.equals(this.refControl.getProjectControl().getProject().getNameKey()) && RefNames.isRefsUsers(this.refControl.getRefName())) {
                messages = new ArrayList();
                Account.Id accountId = Account.Id.fromRef(this.refControl.getRefName());
                if (accountId != null) {
                    try {
                        WatchConfig wc = new WatchConfig(accountId);
                        wc.load(this.repo, receiveEvent.command.getNewId());
                        if (!wc.getValidationErrors().isEmpty()) {
                            CommitValidators.addError("Invalid project configuration:", messages);
                            for (ValidationError err : wc.getValidationErrors()) {
                                CommitValidators.addError("  " + err.getMessage(), messages);
                            }
                            throw new ConfigInvalidException("invalid watch configuration");
                        }
                    }
                    catch (IOException | ConfigInvalidException e) {
                        log.error("User " + currentUser.getUserName() + " tried to push an invalid watch configuration " + receiveEvent.command.getNewId().name() + " for account " + accountId.get(), e);
                        throw new CommitValidationException("invalid watch configuration", messages);
                    }
                }
            }
            return Collections.emptyList();
        }
    }

    public static class ChangeIdValidator
    implements CommitValidationListener {
        private static final String CHANGE_ID_PREFIX = FooterConstants.CHANGE_ID.getName() + ":";
        private static final String MISSING_CHANGE_ID_MSG = "[%s] missing " + FooterConstants.CHANGE_ID.getName() + " in commit message footer";
        private static final String MISSING_SUBJECT_MSG = "[%s] missing subject; " + FooterConstants.CHANGE_ID.getName() + " must be in commit message footer";
        private static final String MULTIPLE_CHANGE_ID_MSG = "[%s] multiple " + FooterConstants.CHANGE_ID.getName() + " lines in commit message footer";
        private static final String INVALID_CHANGE_ID_MSG = "[%s] invalid " + FooterConstants.CHANGE_ID.getName() + " line format in commit message footer";
        private static final Pattern CHANGE_ID = Pattern.compile("^[iI][0-9a-f]{4,}.*$");
        private final ProjectControl projectControl;
        private final String canonicalWebUrl;
        private final String installCommitMsgHookCommand;
        private final SshInfo sshInfo;
        private final IdentifiedUser user;

        public ChangeIdValidator(RefControl refControl, String canonicalWebUrl, String installCommitMsgHookCommand, SshInfo sshInfo) {
            this.projectControl = refControl.getProjectControl();
            this.canonicalWebUrl = canonicalWebUrl;
            this.installCommitMsgHookCommand = installCommitMsgHookCommand;
            this.sshInfo = sshInfo;
            this.user = this.projectControl.getUser().asIdentifiedUser();
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (!ChangeIdValidator.shouldValidateChangeId(receiveEvent)) {
                return Collections.emptyList();
            }
            RevCommit commit = receiveEvent.commit;
            ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
            List<String> idList = commit.getFooterLines(FooterConstants.CHANGE_ID);
            String sha1 = commit.abbreviate(7).name();
            if (idList.isEmpty()) {
                String shortMsg = commit.getShortMessage();
                if (shortMsg.startsWith(CHANGE_ID_PREFIX) && CHANGE_ID.matcher(shortMsg.substring(CHANGE_ID_PREFIX.length()).trim()).matches()) {
                    String errMsg = String.format(MISSING_SUBJECT_MSG, sha1);
                    throw new CommitValidationException(errMsg);
                }
                if (this.projectControl.getProjectState().isRequireChangeID()) {
                    String errMsg = String.format(MISSING_CHANGE_ID_MSG, sha1);
                    messages.add(this.getMissingChangeIdErrorMsg(errMsg, commit));
                    throw new CommitValidationException(errMsg, messages);
                }
            } else {
                if (idList.size() > 1) {
                    String errMsg = String.format(MULTIPLE_CHANGE_ID_MSG, sha1);
                    throw new CommitValidationException(errMsg, messages);
                }
                String v = idList.get(idList.size() - 1).trim();
                if (!CHANGE_ID.matcher(v).matches() || v.matches("^I00*$")) {
                    String errMsg = String.format(INVALID_CHANGE_ID_MSG, sha1);
                    messages.add(this.getMissingChangeIdErrorMsg(errMsg, receiveEvent.commit));
                    throw new CommitValidationException(errMsg, messages);
                }
            }
            return Collections.emptyList();
        }

        private static boolean shouldValidateChangeId(CommitReceivedEvent event) {
            return MagicBranch.isMagicBranch(event.command.getRefName()) || ReceiveCommits.NEW_PATCHSET.matcher(event.command.getRefName()).matches();
        }

        private CommitValidationMessage getMissingChangeIdErrorMsg(String errMsg, RevCommit c) {
            StringBuilder sb = new StringBuilder();
            sb.append("ERROR: ").append(errMsg);
            if (c.getFullMessage().indexOf(CHANGE_ID_PREFIX) >= 0) {
                String lastLine;
                String[] lines = c.getFullMessage().trim().split("\n");
                String string = lastLine = lines.length > 0 ? lines[lines.length - 1] : "";
                if (lastLine.indexOf(CHANGE_ID_PREFIX) == -1) {
                    sb.append('\n');
                    sb.append('\n');
                    sb.append("Hint: A potential ");
                    sb.append(FooterConstants.CHANGE_ID.getName());
                    sb.append(" was found, but it was not in the ");
                    sb.append("footer (last paragraph) of the commit message.");
                }
            }
            sb.append('\n');
            sb.append('\n');
            sb.append("Hint: To automatically insert ");
            sb.append(FooterConstants.CHANGE_ID.getName());
            sb.append(", install the hook:\n");
            sb.append(this.getCommitMessageHookInstallationHint());
            sb.append('\n');
            sb.append("And then amend the commit:\n");
            sb.append("  git commit --amend\n");
            return new CommitValidationMessage(sb.toString(), false);
        }

        private String getCommitMessageHookInstallationHint() {
            int sshPort;
            String sshHost;
            if (this.installCommitMsgHookCommand != null) {
                return this.installCommitMsgHookCommand;
            }
            List<HostKey> hostKeys = this.sshInfo.getHostKeys();
            if (hostKeys.isEmpty()) {
                String p = "${gitdir}/hooks/commit-msg";
                return String.format("  gitdir=$(git rev-parse --git-dir); curl -o %s %s/tools/hooks/commit-msg ; chmod +x %s", p, CommitValidators.getGerritUrl(this.canonicalWebUrl), p);
            }
            String host = hostKeys.get(0).getHost();
            int c = host.lastIndexOf(58);
            if (0 <= c) {
                sshHost = host.startsWith("*:") ? CommitValidators.getGerritHost(this.canonicalWebUrl) : host.substring(0, c);
                sshPort = Integer.parseInt(host.substring(c + 1));
            } else {
                sshHost = host;
                sshPort = 22;
            }
            return String.format("  gitdir=$(git rev-parse --git-dir); scp -p -P %d %s@%s:hooks/commit-msg ${gitdir}/hooks/", sshPort, this.user.getUserName(), sshHost);
        }
    }

    @Singleton
    public static class Factory {
        private final PersonIdent gerritIdent;
        private final String canonicalWebUrl;
        private final DynamicSet<CommitValidationListener> pluginValidators;
        private final AllUsersName allUsers;
        private final String installCommitMsgHookCommand;

        @Inject
        Factory(@GerritPersonIdent PersonIdent gerritIdent, @CanonicalWebUrl @Nullable String canonicalWebUrl, @GerritServerConfig Config cfg, DynamicSet<CommitValidationListener> pluginValidators, AllUsersName allUsers) {
            this.gerritIdent = gerritIdent;
            this.canonicalWebUrl = canonicalWebUrl;
            this.pluginValidators = pluginValidators;
            this.allUsers = allUsers;
            this.installCommitMsgHookCommand = cfg != null ? cfg.getString("gerrit", null, "installCommitMsgHookCommand") : null;
        }

        public CommitValidators create(Policy policy, RefControl refControl, SshInfo sshInfo, Repository repo) throws IOException {
            switch (policy) {
                case RECEIVE_COMMITS: {
                    return this.forReceiveCommits(refControl, sshInfo, repo);
                }
                case GERRIT: {
                    return this.forGerritCommits(refControl, sshInfo, repo);
                }
                case MERGED: {
                    return this.forMergedCommits(refControl);
                }
                case NONE: {
                    return this.none();
                }
            }
            throw new IllegalArgumentException("unspported policy: " + (Object)((Object)policy));
        }

        private CommitValidators forReceiveCommits(RefControl refControl, SshInfo sshInfo, Repository repo) throws IOException {
            try (RevWalk rw = new RevWalk(repo);){
                NoteMap rejectCommits = BanCommit.loadRejectCommitsMap(repo, rw);
                CommitValidators commitValidators = new CommitValidators(ImmutableList.of(new UploadMergesPermissionValidator(refControl), new AmendedGerritMergeCommitValidationListener(refControl, this.gerritIdent), new AuthorUploaderValidator(refControl, this.canonicalWebUrl), new CommitterUploaderValidator(refControl, this.canonicalWebUrl), new SignedOffByValidator(refControl), new ChangeIdValidator(refControl, this.canonicalWebUrl, this.installCommitMsgHookCommand, sshInfo), new ConfigValidator(refControl, repo, this.allUsers), new BannedCommitsValidator(rejectCommits), new PluginCommitValidationListener(this.pluginValidators), new BlockExternalIdUpdateListener(this.allUsers)));
                return commitValidators;
            }
        }

        private CommitValidators forGerritCommits(RefControl refControl, SshInfo sshInfo, Repository repo) {
            return new CommitValidators(ImmutableList.of(new UploadMergesPermissionValidator(refControl), new AmendedGerritMergeCommitValidationListener(refControl, this.gerritIdent), new AuthorUploaderValidator(refControl, this.canonicalWebUrl), new SignedOffByValidator(refControl), new ChangeIdValidator(refControl, this.canonicalWebUrl, this.installCommitMsgHookCommand, sshInfo), new ConfigValidator(refControl, repo, this.allUsers), new PluginCommitValidationListener(this.pluginValidators), new BlockExternalIdUpdateListener(this.allUsers)));
        }

        private CommitValidators forMergedCommits(RefControl refControl) {
            return new CommitValidators(ImmutableList.of(new UploadMergesPermissionValidator(refControl), new AuthorUploaderValidator(refControl, this.canonicalWebUrl), new CommitterUploaderValidator(refControl, this.canonicalWebUrl)));
        }

        private CommitValidators none() {
            return new CommitValidators(ImmutableList.of());
        }
    }

    public static enum Policy {
        GERRIT,
        RECEIVE_COMMITS,
        MERGED,
        NONE;

    }
}

