/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.mail.send;

import com.google.common.io.BaseEncoding;
import com.google.common.primitives.Ints;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.common.TimeUtil;
import com.google.gerrit.common.Version;
import com.google.gerrit.common.errors.EmailException;
import com.google.gerrit.server.config.ConfigUtil;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.mail.Address;
import com.google.gerrit.server.mail.Encryption;
import com.google.gerrit.server.mail.send.EmailHeader;
import com.google.gerrit.server.mail.send.EmailSender;
import com.google.inject.AbstractModule;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.apache.commons.net.smtp.AuthSMTPClient;
import org.apache.commons.net.smtp.SMTPClient;
import org.apache.commons.net.smtp.SMTPReply;
import org.apache.james.mime4j.codec.QuotedPrintableOutputStream;
import org.eclipse.jgit.lib.Config;

@Singleton
public class SmtpEmailSender
implements EmailSender {
    private static final int DEFAULT_CONNECT_TIMEOUT = 0;
    private final boolean enabled;
    private final int connectTimeout;
    private String smtpHost;
    private int smtpPort;
    private String smtpUser;
    private String smtpPass;
    private Encryption smtpEncryption;
    private boolean sslVerify;
    private Set<String> allowrcpt;
    private String importance;
    private int expiryDays;

    @Inject
    SmtpEmailSender(@GerritServerConfig Config cfg) {
        int defaultPort;
        this.enabled = cfg.getBoolean("sendemail", null, "enable", true);
        this.connectTimeout = Ints.checkedCast(ConfigUtil.getTimeUnit(cfg, "sendemail", null, "connectTimeout", 0L, TimeUnit.MILLISECONDS));
        this.smtpHost = cfg.getString("sendemail", null, "smtpserver");
        if (this.smtpHost == null) {
            this.smtpHost = "127.0.0.1";
        }
        this.smtpEncryption = cfg.getEnum("sendemail", null, "smtpencryption", Encryption.NONE);
        this.sslVerify = cfg.getBoolean("sendemail", null, "sslverify", true);
        switch (this.smtpEncryption) {
            case SSL: {
                defaultPort = 465;
                break;
            }
            default: {
                defaultPort = 25;
            }
        }
        this.smtpPort = cfg.getInt("sendemail", null, "smtpserverport", defaultPort);
        this.smtpUser = cfg.getString("sendemail", null, "smtpuser");
        this.smtpPass = cfg.getString("sendemail", null, "smtppass");
        HashSet<String> rcpt = new HashSet<String>();
        for (String addr : cfg.getStringList("sendemail", null, "allowrcpt")) {
            rcpt.add(addr);
        }
        this.allowrcpt = Collections.unmodifiableSet(rcpt);
        this.importance = cfg.getString("sendemail", null, "importance");
        this.expiryDays = cfg.getInt("sendemail", null, "expiryDays", 0);
    }

    @Override
    public boolean isEnabled() {
        return this.enabled;
    }

    @Override
    public boolean canEmail(String address) {
        if (!this.isEnabled()) {
            return false;
        }
        if (this.allowrcpt.isEmpty()) {
            return true;
        }
        if (this.allowrcpt.contains(address)) {
            return true;
        }
        String domain = address.substring(address.lastIndexOf(64) + 1);
        return this.allowrcpt.contains(domain) || this.allowrcpt.contains("@" + domain);
    }

    @Override
    public void send(Address from, Collection<Address> rcpt, Map<String, EmailHeader> callerHeaders, String body) throws EmailException {
        this.send(from, rcpt, callerHeaders, body, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void send(Address from, Collection<Address> rcpt, Map<String, EmailHeader> callerHeaders, String textBody, @Nullable String htmlBody) throws EmailException {
        if (!this.isEnabled()) {
            throw new EmailException("Sending email is disabled");
        }
        StringBuffer rejected = new StringBuffer();
        try {
            SMTPClient client = this.open();
            try {
                if (!client.setSender(from.getEmail())) {
                    throw new EmailException("Server " + this.smtpHost + " rejected from address " + from.getEmail());
                }
                for (Address addr : rcpt) {
                    if (client.addRecipient(addr.getEmail())) continue;
                    String error = client.getReplyString();
                    rejected.append("Server ").append(this.smtpHost).append(" rejected recipient ").append(addr).append(": ").append(error);
                }
                try (Writer messageDataWriter = client.sendMessageData();){
                    if (messageDataWriter == null) {
                        throw new EmailException(rejected + "Server " + this.smtpHost + " rejected DATA command: " + client.getReplyString());
                    }
                    this.render(messageDataWriter, callerHeaders, textBody, htmlBody);
                    if (!client.completePendingCommand()) {
                        throw new EmailException("Server " + this.smtpHost + " rejected message body: " + client.getReplyString());
                    }
                    client.logout();
                    if (rejected.length() > 0) {
                        throw new EmailException(rejected.toString());
                    }
                }
            }
            finally {
                client.disconnect();
            }
        }
        catch (IOException e) {
            throw new EmailException("Cannot send outgoing email", e);
        }
    }

    private void render(Writer out, Map<String, EmailHeader> callerHeaders, String textBody, @Nullable String htmlBody) throws IOException, EmailException {
        String encodedBody;
        LinkedHashMap<String, EmailHeader> hdrs = new LinkedHashMap<String, EmailHeader>(callerHeaders);
        SmtpEmailSender.setMissingHeader(hdrs, "MIME-Version", "1.0");
        SmtpEmailSender.setMissingHeader(hdrs, "Content-Transfer-Encoding", "8bit");
        SmtpEmailSender.setMissingHeader(hdrs, "Content-Disposition", "inline");
        SmtpEmailSender.setMissingHeader(hdrs, "User-Agent", "Gerrit/" + Version.getVersion());
        if (this.importance != null) {
            SmtpEmailSender.setMissingHeader(hdrs, "Importance", this.importance);
        }
        if (this.expiryDays > 0) {
            Date expiry = new Date(TimeUtil.nowMs() + (long)(this.expiryDays * 24 * 60 * 60) * 1000L);
            SmtpEmailSender.setMissingHeader(hdrs, "Expiry-Date", new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z").format(expiry));
        }
        if (htmlBody == null) {
            SmtpEmailSender.setMissingHeader(hdrs, "Content-Type", "text/plain; charset=UTF-8");
            encodedBody = textBody;
        } else {
            String boundary = SmtpEmailSender.generateMultipartBoundary(textBody, htmlBody);
            SmtpEmailSender.setMissingHeader(hdrs, "Content-Type", "multipart/alternative; boundary=\"" + boundary + "\"; charset=UTF-8");
            encodedBody = this.buildMultipartBody(boundary, textBody, htmlBody);
        }
        try (BufferedWriter w = new BufferedWriter(out);){
            for (Map.Entry h : hdrs.entrySet()) {
                if (((EmailHeader)h.getValue()).isEmpty()) continue;
                w.write((String)h.getKey());
                w.write(": ");
                ((EmailHeader)h.getValue()).write(w);
                w.write("\r\n");
            }
            w.write("\r\n");
            w.write(encodedBody);
            ((Writer)w).flush();
        }
    }

    public static String generateMultipartBoundary(String textBody, String htmlBody) throws EmailException {
        byte[] bytes = new byte[8];
        ThreadLocalRandom rng = ThreadLocalRandom.current();
        for (int i = 0; i < 2; ++i) {
            rng.nextBytes(bytes);
            String boundary = BaseEncoding.base64().encode(bytes);
            String encBoundary = "--" + boundary;
            if (textBody.contains(encBoundary) || htmlBody.contains(encBoundary)) continue;
            return boundary;
        }
        throw new EmailException("Gave up generating unique MIME boundary");
    }

    protected String buildMultipartBody(String boundary, String textPart, String htmlPart) throws IOException {
        String encodedTextPart = this.quotedPrintableEncode(textPart);
        String encodedHtmlPart = this.quotedPrintableEncode(htmlPart);
        String textTransferEncoding = textPart.equals(encodedTextPart) ? "7bit" : "quoted-printable";
        String htmlTransferEncoding = htmlPart.equals(encodedHtmlPart) ? "7bit" : "quoted-printable";
        return "--" + boundary + "\r\nContent-Type: text/plain; charset=UTF-8\r\nContent-Transfer-Encoding: " + textTransferEncoding + "\r\n\r\n" + encodedTextPart + "\r\n--" + boundary + "\r\nContent-Type: text/html; charset=UTF-8\r\nContent-Transfer-Encoding: " + htmlTransferEncoding + "\r\n\r\n" + encodedHtmlPart + "\r\n--" + boundary + "--\r\n";
    }

    protected String quotedPrintableEncode(String input) throws IOException {
        ByteArrayOutputStream s = new ByteArrayOutputStream();
        try (QuotedPrintableOutputStream qp = new QuotedPrintableOutputStream(s, false);){
            qp.write(input.getBytes(StandardCharsets.UTF_8));
        }
        return s.toString();
    }

    private static void setMissingHeader(Map<String, EmailHeader> hdrs, String name, String value) {
        if (!hdrs.containsKey(name) || hdrs.get(name).isEmpty()) {
            hdrs.put(name, new EmailHeader.String(value));
        }
    }

    private SMTPClient open() throws EmailException {
        AuthSMTPClient client = new AuthSMTPClient(StandardCharsets.UTF_8.name());
        if (this.smtpEncryption == Encryption.SSL) {
            client.enableSSL(this.sslVerify);
        }
        client.setConnectTimeout(this.connectTimeout);
        try {
            client.connect(this.smtpHost, this.smtpPort);
            int replyCode = client.getReplyCode();
            String replyString = client.getReplyString();
            if (!SMTPReply.isPositiveCompletion(replyCode)) {
                throw new EmailException(String.format("SMTP server rejected connection: %d: %s", replyCode, replyString));
            }
            if (!client.login()) {
                throw new EmailException("SMTP server rejected HELO/EHLO greeting: " + replyString);
            }
            if (this.smtpEncryption == Encryption.TLS) {
                if (!client.startTLS(this.smtpHost, this.smtpPort, this.sslVerify)) {
                    throw new EmailException("SMTP server does not support TLS");
                }
                if (!client.login()) {
                    throw new EmailException("SMTP server rejected login: " + replyString);
                }
            }
            if (this.smtpUser != null && !client.auth(this.smtpUser, this.smtpPass)) {
                throw new EmailException("SMTP server rejected auth: " + replyString);
            }
            return client;
        }
        catch (EmailException | IOException e) {
            if (client.isConnected()) {
                try {
                    client.disconnect();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            if (e instanceof EmailException) {
                throw (EmailException)e;
            }
            throw new EmailException(e.getMessage(), e);
        }
    }

    public static class Module
    extends AbstractModule {
        @Override
        protected void configure() {
            this.bind(EmailSender.class).to(SmtpEmailSender.class);
        }
    }
}

