/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.plugins;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import com.google.gerrit.extensions.events.LifecycleListener;
import com.google.gerrit.extensions.systemstatus.ServerInformation;
import com.google.gerrit.server.PluginUser;
import com.google.gerrit.server.cache.PersistentCacheFactory;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.config.ConfigUtil;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.config.SitePaths;
import com.google.gerrit.server.plugins.CleanupHandle;
import com.google.gerrit.server.plugins.InvalidPluginException;
import com.google.gerrit.server.plugins.JsPlugin;
import com.google.gerrit.server.plugins.Plugin;
import com.google.gerrit.server.plugins.PluginCleanerTask;
import com.google.gerrit.server.plugins.PluginGuiceEnvironment;
import com.google.gerrit.server.plugins.PluginInstallException;
import com.google.gerrit.server.plugins.PluginScannerThread;
import com.google.gerrit.server.plugins.PluginUtil;
import com.google.gerrit.server.plugins.ServerInformationImpl;
import com.google.gerrit.server.plugins.ServerPlugin;
import com.google.gerrit.server.plugins.ServerPluginProvider;
import com.google.gerrit.server.plugins.UniversalServerPluginProvider;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.AbstractMap;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.eclipse.jgit.internal.storage.file.FileSnapshot;
import org.eclipse.jgit.lib.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class PluginLoader
implements LifecycleListener {
    private static final Logger log = LoggerFactory.getLogger(PluginLoader.class);
    private final Path pluginsDir;
    private final Path dataDir;
    private final Path tempDir;
    private final PluginGuiceEnvironment env;
    private final ServerInformationImpl srvInfoImpl;
    private final PluginUser.Factory pluginUserFactory;
    private final ConcurrentMap<String, Plugin> running;
    private final ConcurrentMap<String, Plugin> disabled;
    private final Map<String, FileSnapshot> broken;
    private final Map<Plugin, CleanupHandle> cleanupHandles;
    private final Queue<Plugin> toCleanup;
    private final Provider<PluginCleanerTask> cleaner;
    private final PluginScannerThread scanner;
    private final Provider<String> urlProvider;
    private final PersistentCacheFactory persistentCacheFactory;
    private final boolean remoteAdmin;
    private final UniversalServerPluginProvider serverPluginFactory;

    public String getPluginName(Path srcPath) {
        return MoreObjects.firstNonNull(this.getGerritPluginName(srcPath), PluginUtil.nameOf(srcPath));
    }

    @Inject
    public PluginLoader(SitePaths sitePaths, PluginGuiceEnvironment pe, ServerInformationImpl sii, PluginUser.Factory puf, Provider<PluginCleanerTask> pct, @GerritServerConfig Config cfg, @CanonicalWebUrl Provider<String> provider, PersistentCacheFactory cacheFactory, UniversalServerPluginProvider pluginFactory) {
        this.pluginsDir = sitePaths.plugins_dir;
        this.dataDir = sitePaths.data_dir;
        this.tempDir = sitePaths.tmp_dir;
        this.env = pe;
        this.srvInfoImpl = sii;
        this.pluginUserFactory = puf;
        this.running = Maps.newConcurrentMap();
        this.disabled = Maps.newConcurrentMap();
        this.broken = new HashMap<String, FileSnapshot>();
        this.toCleanup = new ArrayDeque<Plugin>();
        this.cleanupHandles = Maps.newConcurrentMap();
        this.cleaner = pct;
        this.urlProvider = provider;
        this.persistentCacheFactory = cacheFactory;
        this.serverPluginFactory = pluginFactory;
        this.remoteAdmin = cfg.getBoolean("plugins", null, "allowRemoteAdmin", false);
        long checkFrequency = ConfigUtil.getTimeUnit(cfg, "plugins", null, "checkFrequency", TimeUnit.MINUTES.toMillis(1L), TimeUnit.MILLISECONDS);
        this.scanner = checkFrequency > 0L ? new PluginScannerThread(this, checkFrequency) : null;
    }

    public boolean isRemoteAdminEnabled() {
        return this.remoteAdmin;
    }

    public Plugin get(String name) {
        Plugin p = (Plugin)this.running.get(name);
        if (p != null) {
            return p;
        }
        return (Plugin)this.disabled.get(name);
    }

    public Iterable<Plugin> getPlugins(boolean all) {
        if (!all) {
            return this.running.values();
        }
        ArrayList<Plugin> plugins = new ArrayList<Plugin>(this.running.values());
        plugins.addAll(this.disabled.values());
        return plugins;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String installPluginFromStream(String originalName, InputStream in) throws IOException, PluginInstallException {
        this.checkRemoteInstall();
        String fileName = originalName;
        Path tmp = PluginUtil.asTemp(in, ".next_" + fileName + "_", ".tmp", this.pluginsDir);
        String name = MoreObjects.firstNonNull(this.getGerritPluginName(tmp), PluginUtil.nameOf(fileName));
        if (!originalName.equals(name)) {
            log.warn("Plugin provides its own name: <{}>, use it instead of the input name: <{}>", (Object)name, (Object)originalName);
        }
        String fileExtension = this.getExtension(fileName);
        Path dst = this.pluginsDir.resolve(name + fileExtension);
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            Plugin active = (Plugin)this.running.get(name);
            if (active != null) {
                fileName = active.getSrcFile().getFileName().toString();
                log.info("Replacing plugin {}", (Object)active.getName());
                Path old = this.pluginsDir.resolve(".last_" + fileName);
                Files.deleteIfExists(old);
                Files.move(active.getSrcFile(), old, new CopyOption[0]);
            }
            Files.deleteIfExists(this.pluginsDir.resolve(fileName + ".disabled"));
            Files.move(tmp, dst, new CopyOption[0]);
            try {
                Plugin plugin = this.runPlugin(name, dst, active);
                if (active == null) {
                    log.info("Installed plugin {}", (Object)plugin.getName());
                }
            }
            catch (PluginInstallException e) {
                Files.deleteIfExists(dst);
                throw e;
            }
            this.cleanInBackground();
        }
        return name;
    }

    private synchronized void unloadPlugin(Plugin plugin) {
        this.persistentCacheFactory.onStop(plugin);
        String name = plugin.getName();
        log.info("Unloading plugin {}, version {}", (Object)name, (Object)plugin.getVersion());
        plugin.stop(this.env);
        this.env.onStopPlugin(plugin);
        this.running.remove(name);
        this.disabled.remove(name);
        this.toCleanup.add(plugin);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void disablePlugins(Set<String> names) {
        if (!this.isRemoteAdminEnabled()) {
            log.warn("Remote plugin administration is disabled, ignoring disablePlugins({})", (Object)names);
            return;
        }
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            for (String name : names) {
                Plugin active = (Plugin)this.running.get(name);
                if (active == null) continue;
                log.info("Disabling plugin {}", (Object)active.getName());
                Path off = active.getSrcFile().resolveSibling(active.getSrcFile().getFileName() + ".disabled");
                try {
                    Files.move(active.getSrcFile(), off, new CopyOption[0]);
                }
                catch (IOException e) {
                    log.error("Failed to disable plugin", e);
                    continue;
                }
                this.unloadPlugin(active);
                try {
                    FileSnapshot snapshot = FileSnapshot.save(off.toFile());
                    Plugin offPlugin = this.loadPlugin(name, off, snapshot);
                    this.disabled.put(name, offPlugin);
                }
                catch (Throwable e) {
                    log.warn("Cannot load disabled plugin {}", (Object)active.getName(), (Object)e.getCause());
                }
            }
            this.cleanInBackground();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void enablePlugins(Set<String> names) throws PluginInstallException {
        if (!this.isRemoteAdminEnabled()) {
            log.warn("Remote plugin administration is disabled, ignoring enablePlugins({})", (Object)names);
            return;
        }
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            for (String name : names) {
                Plugin off = (Plugin)this.disabled.get(name);
                if (off == null) continue;
                log.info("Enabling plugin {}", (Object)name);
                String n = off.getSrcFile().toFile().getName();
                if (n.endsWith(".disabled")) {
                    n = n.substring(0, n.lastIndexOf(46));
                }
                Path on = this.pluginsDir.resolve(n);
                try {
                    Files.move(off.getSrcFile(), on, new CopyOption[0]);
                }
                catch (IOException e) {
                    log.error("Failed to move plugin {} into place", (Object)name, (Object)e);
                    continue;
                }
                this.disabled.remove(name);
                this.runPlugin(name, on, null);
            }
            this.cleanInBackground();
        }
    }

    private void removeStalePluginFiles() {
        DirectoryStream.Filter<Path> filter = new DirectoryStream.Filter<Path>(){

            @Override
            public boolean accept(Path entry) throws IOException {
                return entry.getFileName().toString().startsWith("plugin_");
            }
        };
        try (DirectoryStream<Path> files = Files.newDirectoryStream(this.tempDir, (DirectoryStream.Filter<? super Path>)filter);){
            for (Path file : files) {
                log.info("Removing stale plugin file: {}", (Object)file.toFile().getName());
                try {
                    Files.delete(file);
                }
                catch (IOException e) {
                    log.error("Failed to remove stale plugin file {}: {}", (Object)file.toFile().getName(), (Object)e.getMessage());
                }
            }
        }
        catch (IOException e) {
            log.warn("Unable to discover stale plugin files: {}", (Object)e.getMessage());
        }
    }

    @Override
    public synchronized void start() {
        this.removeStalePluginFiles();
        Path absolutePath = this.pluginsDir.toAbsolutePath();
        if (!Files.exists(absolutePath, new LinkOption[0])) {
            log.info("{} does not exist; creating", (Object)absolutePath);
            try {
                Files.createDirectories(absolutePath, new FileAttribute[0]);
            }
            catch (IOException e) {
                log.error("Failed to create {}: {}", (Object)absolutePath, (Object)e.getMessage());
            }
        }
        log.info("Loading plugins from {}", (Object)absolutePath);
        this.srvInfoImpl.state = ServerInformation.State.STARTUP;
        this.rescan();
        this.srvInfoImpl.state = ServerInformation.State.RUNNING;
        if (this.scanner != null) {
            this.scanner.start();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void stop() {
        if (this.scanner != null) {
            this.scanner.end();
        }
        this.srvInfoImpl.state = ServerInformation.State.SHUTDOWN;
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            for (Plugin p : this.running.values()) {
                this.unloadPlugin(p);
            }
            this.running.clear();
            this.disabled.clear();
            this.broken.clear();
            if (!this.toCleanup.isEmpty()) {
                System.gc();
                this.processPendingCleanups();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reload(List<String> names) throws InvalidPluginException, PluginInstallException {
        PluginLoader pluginLoader = this;
        synchronized (pluginLoader) {
            ArrayList<Plugin> reload = Lists.newArrayListWithCapacity(names.size());
            ArrayList<String> bad = Lists.newArrayListWithExpectedSize(4);
            for (String name : names) {
                Plugin active = (Plugin)this.running.get(name);
                if (active != null) {
                    reload.add(active);
                    continue;
                }
                bad.add(name);
            }
            if (!bad.isEmpty()) {
                throw new InvalidPluginException(String.format("Plugin(s) \"%s\" not running", Joiner.on("\", \"").join(bad)));
            }
            for (Plugin active : reload) {
                String name = active.getName();
                try {
                    log.info("Reloading plugin {}", (Object)name);
                    Plugin newPlugin = this.runPlugin(name, active.getSrcFile(), active);
                    log.info("Reloaded plugin {}, version {}", (Object)newPlugin.getName(), (Object)newPlugin.getVersion());
                }
                catch (PluginInstallException e) {
                    log.warn("Cannot reload plugin {}", (Object)name, (Object)e.getCause());
                    throw e;
                }
            }
            this.cleanInBackground();
        }
    }

    public synchronized void rescan() {
        SetMultimap<String, Path> pluginsFiles = this.prunePlugins(this.pluginsDir);
        if (pluginsFiles.isEmpty()) {
            return;
        }
        this.syncDisabledPlugins(pluginsFiles);
        Map<String, Path> activePlugins = this.filterDisabled(pluginsFiles);
        for (Map.Entry<String, Path> entry : this.jarsFirstSortedPluginsSet(activePlugins)) {
            Plugin active;
            String name = entry.getKey();
            Path path = entry.getValue();
            String fileName = path.getFileName().toString();
            if (!this.isUiPlugin(fileName) && !this.serverPluginFactory.handles(path)) {
                log.warn("No Plugin provider was found that handles this file format: {}", (Object)fileName);
                continue;
            }
            FileSnapshot brokenTime = this.broken.get(name);
            if (brokenTime != null && !brokenTime.isModified(path.toFile()) || (active = (Plugin)this.running.get(name)) != null && !active.isModified(path)) continue;
            if (active != null) {
                log.info("Reloading plugin {}", (Object)active.getName());
            }
            try {
                Plugin loadedPlugin = this.runPlugin(name, path, active);
                if (loadedPlugin.isDisabled()) continue;
                log.info("{} plugin {}, version {}", active == null ? "Loaded" : "Reloaded", loadedPlugin.getName(), loadedPlugin.getVersion());
            }
            catch (PluginInstallException e) {
                log.warn("Cannot load plugin {}", (Object)name, (Object)e.getCause());
            }
        }
        this.cleanInBackground();
    }

    private void addAllEntries(Map<String, Path> from, TreeSet<Map.Entry<String, Path>> to) {
        for (Map.Entry<String, Path> entry : from.entrySet()) {
            to.add(new AbstractMap.SimpleImmutableEntry<String, Path>(entry.getKey(), entry.getValue()));
        }
    }

    private TreeSet<Map.Entry<String, Path>> jarsFirstSortedPluginsSet(Map<String, Path> activePlugins) {
        TreeSet<Map.Entry<String, Path>> sortedPlugins = Sets.newTreeSet(new Comparator<Map.Entry<String, Path>>(){

            @Override
            public int compare(Map.Entry<String, Path> e1, Map.Entry<String, Path> e2) {
                Path n1 = e1.getValue().getFileName();
                Path n2 = e2.getValue().getFileName();
                return ComparisonChain.start().compareTrueFirst(this.isJar(n1), this.isJar(n2)).compare(n1, n2).result();
            }

            private boolean isJar(Path n1) {
                return n1.toString().endsWith(".jar");
            }
        });
        this.addAllEntries(activePlugins, sortedPlugins);
        return sortedPlugins;
    }

    private void syncDisabledPlugins(SetMultimap<String, Path> jars) {
        this.stopRemovedPlugins(jars);
        this.dropRemovedDisabledPlugins(jars);
    }

    private Plugin runPlugin(String name, Path plugin, Plugin oldPlugin) throws PluginInstallException {
        FileSnapshot snapshot = FileSnapshot.save(plugin.toFile());
        try {
            boolean reload;
            Plugin newPlugin = this.loadPlugin(name, plugin, snapshot);
            if (newPlugin.getCleanupHandle() != null) {
                this.cleanupHandles.put(newPlugin, newPlugin.getCleanupHandle());
            }
            name = newPlugin.getName();
            boolean bl = reload = oldPlugin != null && oldPlugin.canReload() && newPlugin.canReload();
            if (!reload && oldPlugin != null) {
                this.unloadPlugin(oldPlugin);
            }
            if (!newPlugin.isDisabled()) {
                newPlugin.start(this.env);
            }
            if (reload) {
                this.env.onReloadPlugin(oldPlugin, newPlugin);
                this.unloadPlugin(oldPlugin);
            } else if (!newPlugin.isDisabled()) {
                this.env.onStartPlugin(newPlugin);
            }
            if (!newPlugin.isDisabled()) {
                this.running.put(name, newPlugin);
            } else {
                this.disabled.put(name, newPlugin);
            }
            this.broken.remove(name);
            return newPlugin;
        }
        catch (Throwable err) {
            this.broken.put(name, snapshot);
            throw new PluginInstallException(err);
        }
    }

    private void stopRemovedPlugins(SetMultimap<String, Path> jars) {
        HashSet<String> unload = Sets.newHashSet(this.running.keySet());
        for (Map.Entry<String, Collection<Path>> entry : jars.asMap().entrySet()) {
            for (Path path : entry.getValue()) {
                if (path.getFileName().toString().endsWith(".disabled")) continue;
                unload.remove(entry.getKey());
            }
        }
        for (String name : unload) {
            this.unloadPlugin((Plugin)this.running.get(name));
        }
    }

    private void dropRemovedDisabledPlugins(SetMultimap<String, Path> jars) {
        HashSet<String> unload = Sets.newHashSet(this.disabled.keySet());
        for (Map.Entry<String, Collection<Path>> entry : jars.asMap().entrySet()) {
            for (Path path : entry.getValue()) {
                if (!path.getFileName().toString().endsWith(".disabled")) continue;
                unload.remove(entry.getKey());
            }
        }
        for (String name : unload) {
            this.disabled.remove(name);
        }
    }

    synchronized int processPendingCleanups() {
        Iterator iterator = this.toCleanup.iterator();
        while (iterator.hasNext()) {
            Plugin plugin = (Plugin)iterator.next();
            iterator.remove();
            CleanupHandle cleanupHandle = this.cleanupHandles.remove(plugin);
            if (cleanupHandle == null) continue;
            cleanupHandle.cleanup();
        }
        return this.toCleanup.size();
    }

    private void cleanInBackground() {
        int cnt = this.toCleanup.size();
        if (0 < cnt) {
            this.cleaner.get().clean(cnt);
        }
    }

    private String getExtension(String name) {
        int ext = name.lastIndexOf(46);
        return 0 < ext ? name.substring(ext) : "";
    }

    private Plugin loadPlugin(String name, Path srcPlugin, FileSnapshot snapshot) throws InvalidPluginException {
        String pluginName = srcPlugin.getFileName().toString();
        if (this.isUiPlugin(pluginName)) {
            return this.loadJsPlugin(name, srcPlugin, snapshot);
        }
        if (this.serverPluginFactory.handles(srcPlugin)) {
            return this.loadServerPlugin(srcPlugin, snapshot);
        }
        throw new InvalidPluginException(String.format("Unsupported plugin type: %s", srcPlugin.getFileName()));
    }

    private Path getPluginDataDir(String name) {
        return this.dataDir.resolve(name);
    }

    private String getPluginCanonicalWebUrl(String name) {
        String canonicalWebUrl = this.urlProvider.get();
        if (Strings.isNullOrEmpty(canonicalWebUrl)) {
            return "/plugins/" + name;
        }
        String url = String.format("%s/plugins/%s/", CharMatcher.is('/').trimTrailingFrom(canonicalWebUrl), name);
        return url;
    }

    private Plugin loadJsPlugin(String name, Path srcJar, FileSnapshot snapshot) {
        return new JsPlugin(name, srcJar, this.pluginUserFactory.create(name), snapshot);
    }

    private ServerPlugin loadServerPlugin(Path scriptFile, FileSnapshot snapshot) throws InvalidPluginException {
        String name = this.serverPluginFactory.getPluginName(scriptFile);
        return this.serverPluginFactory.get(scriptFile, snapshot, new ServerPluginProvider.PluginDescription(this.pluginUserFactory.create(name), this.getPluginCanonicalWebUrl(name), this.getPluginDataDir(name)));
    }

    private Map<String, Path> filterDisabled(SetMultimap<String, Path> pluginPaths) {
        HashMap<String, Path> activePlugins = Maps.newHashMapWithExpectedSize(pluginPaths.keys().size());
        for (String name : pluginPaths.keys()) {
            for (Path pluginPath : pluginPaths.asMap().get(name)) {
                if (pluginPath.getFileName().toString().endsWith(".disabled")) continue;
                assert (!activePlugins.containsKey(name));
                activePlugins.put(name, pluginPath);
            }
        }
        return activePlugins;
    }

    public SetMultimap<String, Path> prunePlugins(Path pluginsDir) {
        List<Path> pluginPaths = this.scanPathsInPluginsDirectory(pluginsDir);
        SetMultimap<String, Path> map = this.asMultimap(pluginPaths);
        for (String plugin : map.keySet()) {
            Iterable<Path> enabled;
            Collection<Path> files = map.asMap().get(plugin);
            if (files.size() == 1 || !Iterables.skip(enabled = this.filterDisabledPlugins(files), 1).iterator().hasNext()) continue;
            Path winner = Iterables.getFirst(enabled, null);
            assert (winner != null);
            ArrayList<Path> elementsToRemove = new ArrayList<Path>();
            ArrayList<Path> elementsToAdd = new ArrayList<Path>();
            for (Path loser : Iterables.skip(enabled, 1)) {
                log.warn("Plugin <{}> was disabled, because another plugin <{}> with the same name <{}> already exists", loser, winner, plugin);
                Path disabledPlugin = Paths.get(loser + ".disabled", new String[0]);
                elementsToAdd.add(disabledPlugin);
                elementsToRemove.add(loser);
                try {
                    Files.move(loser, disabledPlugin, new CopyOption[0]);
                }
                catch (IOException e) {
                    log.warn("Failed to fully disable plugin {}", (Object)loser, (Object)e);
                }
            }
            Iterables.removeAll(files, elementsToRemove);
            Iterables.addAll(files, elementsToAdd);
        }
        return map;
    }

    private List<Path> scanPathsInPluginsDirectory(Path pluginsDir) {
        try {
            return PluginUtil.listPlugins(pluginsDir);
        }
        catch (IOException e) {
            log.error("Cannot list {}", (Object)pluginsDir.toAbsolutePath(), (Object)e);
            return ImmutableList.of();
        }
    }

    private Iterable<Path> filterDisabledPlugins(Collection<Path> paths) {
        return Iterables.filter(paths, p -> !p.getFileName().toString().endsWith(".disabled"));
    }

    public String getGerritPluginName(Path srcPath) {
        String fileName = srcPath.getFileName().toString();
        if (this.isUiPlugin(fileName)) {
            return fileName.substring(0, fileName.lastIndexOf(46));
        }
        if (this.serverPluginFactory.handles(srcPath)) {
            return this.serverPluginFactory.getPluginName(srcPath);
        }
        return null;
    }

    private SetMultimap<String, Path> asMultimap(List<Path> plugins) {
        LinkedHashMultimap<String, Path> map = LinkedHashMultimap.create();
        for (Path srcPath : plugins) {
            map.put(this.getPluginName(srcPath), srcPath);
        }
        return map;
    }

    private boolean isUiPlugin(String name) {
        return this.isPlugin(name, "js") || this.isPlugin(name, "html");
    }

    private boolean isPlugin(String fileName, String ext) {
        String fullExt = "." + ext;
        return fileName.endsWith(fullExt) || fileName.endsWith(fullExt + ".disabled");
    }

    private void checkRemoteInstall() throws PluginInstallException {
        if (!this.isRemoteAdminEnabled()) {
            throw new PluginInstallException("remote installation is disabled");
        }
    }
}

