/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.change;

import com.google.auto.value.AutoValue;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.SetMultimap;
import com.google.gerrit.common.FooterConstants;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.common.TimeUtil;
import com.google.gerrit.extensions.api.changes.FixInput;
import com.google.gerrit.extensions.api.changes.NotifyHandling;
import com.google.gerrit.extensions.common.ProblemInfo;
import com.google.gerrit.extensions.registration.DynamicItem;
import com.google.gerrit.extensions.restapi.RestApiException;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.reviewdb.server.ReviewDbUtil;
import com.google.gerrit.server.ChangeUtil;
import com.google.gerrit.server.CurrentUser;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.PatchSetUtil;
import com.google.gerrit.server.account.Accounts;
import com.google.gerrit.server.change.AccountPatchReviewStore;
import com.google.gerrit.server.change.AutoValue_ConsistencyChecker_Result;
import com.google.gerrit.server.change.DeleteChangeOp;
import com.google.gerrit.server.change.PatchSetInserter;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.gerrit.server.notedb.ChangeNotes;
import com.google.gerrit.server.notedb.PatchSetState;
import com.google.gerrit.server.patch.PatchSetInfoFactory;
import com.google.gerrit.server.patch.PatchSetInfoNotAvailableException;
import com.google.gerrit.server.update.BatchUpdate;
import com.google.gerrit.server.update.BatchUpdateOp;
import com.google.gerrit.server.update.ChangeContext;
import com.google.gerrit.server.update.RepoContext;
import com.google.gerrit.server.update.RetryHelper;
import com.google.gerrit.server.update.UpdateException;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.errors.IncorrectObjectTypeException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.errors.RepositoryNotFoundException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectInserter;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.RefUpdate;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistencyChecker {
    private static final Logger log = LoggerFactory.getLogger(ConsistencyChecker.class);
    private final ChangeNotes.Factory notesFactory;
    private final Accounts accounts;
    private final DynamicItem<AccountPatchReviewStore> accountPatchReviewStore;
    private final GitRepositoryManager repoManager;
    private final PatchSetInfoFactory patchSetInfoFactory;
    private final PatchSetInserter.Factory patchSetInserterFactory;
    private final PatchSetUtil psUtil;
    private final Provider<CurrentUser> user;
    private final Provider<PersonIdent> serverIdent;
    private final Provider<ReviewDb> db;
    private final RetryHelper retryHelper;
    private BatchUpdate.Factory updateFactory;
    private FixInput fix;
    private ChangeNotes notes;
    private Repository repo;
    private RevWalk rw;
    private ObjectInserter oi;
    private RevCommit tip;
    private SetMultimap<ObjectId, PatchSet> patchSetsBySha;
    private PatchSet currPs;
    private RevCommit currPsCommit;
    private List<ProblemInfo> problems;

    @Inject
    ConsistencyChecker(@GerritPersonIdent Provider<PersonIdent> serverIdent, ChangeNotes.Factory notesFactory, Accounts accounts, DynamicItem<AccountPatchReviewStore> accountPatchReviewStore, GitRepositoryManager repoManager, PatchSetInfoFactory patchSetInfoFactory, PatchSetInserter.Factory patchSetInserterFactory, PatchSetUtil psUtil, Provider<CurrentUser> user, Provider<ReviewDb> db, RetryHelper retryHelper) {
        this.accounts = accounts;
        this.accountPatchReviewStore = accountPatchReviewStore;
        this.db = db;
        this.notesFactory = notesFactory;
        this.patchSetInfoFactory = patchSetInfoFactory;
        this.patchSetInserterFactory = patchSetInserterFactory;
        this.psUtil = psUtil;
        this.repoManager = repoManager;
        this.retryHelper = retryHelper;
        this.serverIdent = serverIdent;
        this.user = user;
        this.reset();
    }

    private void reset() {
        this.updateFactory = null;
        this.notes = null;
        this.repo = null;
        this.rw = null;
        this.problems = new ArrayList<ProblemInfo>();
    }

    private Change change() {
        return this.notes.getChange();
    }

    public Result check(ChangeNotes notes, @Nullable FixInput f) {
        Preconditions.checkNotNull(notes);
        try {
            return this.retryHelper.execute(buf -> {
                try {
                    this.reset();
                    this.updateFactory = buf;
                    this.notes = notes;
                    this.fix = f;
                    this.checkImpl();
                    Result result = this.result();
                    return result;
                }
                finally {
                    if (this.rw != null) {
                        this.rw.getObjectReader().close();
                        this.rw.close();
                        this.oi.close();
                    }
                    if (this.repo != null) {
                        this.repo.close();
                    }
                }
            });
        }
        catch (RestApiException e) {
            return this.logAndReturnOneProblem(e, notes, "Error checking change: " + e.getMessage());
        }
        catch (UpdateException e) {
            return this.logAndReturnOneProblem(e, notes, "Error checking change");
        }
    }

    private Result logAndReturnOneProblem(Exception e, ChangeNotes notes, String problem) {
        log.warn("Error checking change " + notes.getChangeId(), e);
        return Result.create(notes, ImmutableList.of(this.problem(problem)));
    }

    private void checkImpl() {
        this.checkOwner();
        this.checkCurrentPatchSetEntity();
        if (!this.openRepo()) {
            return;
        }
        if (!this.checkPatchSets()) {
            return;
        }
        this.checkMerged();
    }

    private void checkOwner() {
        try {
            if (this.accounts.get(this.change().getOwner()) == null) {
                this.problem("Missing change owner: " + this.change().getOwner());
            }
        }
        catch (IOException | ConfigInvalidException e) {
            this.error("Failed to look up owner", e);
        }
    }

    private void checkCurrentPatchSetEntity() {
        try {
            this.currPs = this.psUtil.current(this.db.get(), this.notes);
            if (this.currPs == null) {
                this.problem(String.format("Current patch set %d not found", this.change().currentPatchSetId().get()));
            }
        }
        catch (OrmException e) {
            this.error("Failed to look up current patch set", e);
        }
    }

    private boolean openRepo() {
        Project.NameKey project = this.change().getDest().getParentKey();
        try {
            this.repo = this.repoManager.openRepository(project);
            this.oi = this.repo.newObjectInserter();
            this.rw = new RevWalk(this.oi.newReader());
            return true;
        }
        catch (RepositoryNotFoundException e) {
            return this.error("Destination repository not found: " + project, e);
        }
        catch (IOException e) {
            return this.error("Failed to open repository: " + project, e);
        }
    }

    private boolean checkPatchSets() {
        Map<Object, Object> refs;
        List<PatchSet> all;
        try {
            all = ChangeUtil.PS_ID_ORDER.sortedCopy(this.psUtil.byChange(this.db.get(), this.notes));
        }
        catch (OrmException e) {
            return this.error("Failed to look up patch sets", e);
        }
        this.patchSetsBySha = MultimapBuilder.hashKeys(all.size()).treeSetValues(ChangeUtil.PS_ID_ORDER).build();
        try {
            refs = this.repo.getRefDatabase().exactRef((String[])all.stream().map(ps -> ps.getId().toRefName()).toArray(String[]::new));
        }
        catch (IOException e) {
            this.error("error reading refs", e);
            refs = Collections.emptyMap();
        }
        ArrayList<DeletePatchSetFromDbOp> deletePatchSetOps = new ArrayList<DeletePatchSetFromDbOp>();
        for (PatchSet patchSet : all) {
            int psNum = patchSet.getId().get();
            String refName = patchSet.getId().toRefName();
            ObjectId objId = this.parseObjectId(patchSet.getRevision().get(), "patch set " + psNum);
            if (objId == null) continue;
            this.patchSetsBySha.put(objId, patchSet);
            ProblemInfo refProblem = null;
            Ref ref = (Ref)refs.get(refName);
            if (ref == null) {
                refProblem = this.problem("Ref missing: " + refName);
            } else if (!objId.equals(ref.getObjectId())) {
                String actual = ref.getObjectId() != null ? ref.getObjectId().name() : "null";
                refProblem = this.problem(String.format("Expected %s to point to %s, found %s", ref.getName(), objId.name(), actual));
            }
            RevCommit psCommit = this.parseCommit(objId, String.format("patch set %d", psNum));
            if (psCommit == null) {
                if (this.fix == null || !this.fix.deletePatchSetIfCommitMissing) continue;
                deletePatchSetOps.add(new DeletePatchSetFromDbOp(this.lastProblem(), patchSet.getId()));
                continue;
            }
            if (refProblem != null && this.fix != null) {
                this.fixPatchSetRef(refProblem, patchSet);
            }
            if (!patchSet.getId().equals(this.change().currentPatchSetId())) continue;
            this.currPsCommit = psCommit;
        }
        this.deletePatchSets(deletePatchSetOps);
        for (Map.Entry entry : this.patchSetsBySha.asMap().entrySet()) {
            if (((Collection)entry.getValue()).size() <= 1) continue;
            this.problem(String.format("Multiple patch sets pointing to %s: %s", ((ObjectId)entry.getKey()).name(), Collections2.transform((Collection)entry.getValue(), PatchSet::getPatchSetId)));
        }
        return this.currPs != null && this.currPsCommit != null;
    }

    private void checkMerged() {
        Ref dest;
        String refName = this.change().getDest().get();
        try {
            dest = this.repo.getRefDatabase().exactRef(refName);
        }
        catch (IOException e) {
            this.problem("Failed to look up destination ref: " + refName);
            return;
        }
        if (dest == null) {
            this.problem("Destination ref not found (may be new branch): " + refName);
            return;
        }
        this.tip = this.parseCommit(dest.getObjectId(), "destination ref " + refName);
        if (this.tip == null) {
            return;
        }
        if (this.fix != null && this.fix.expectMergedAs != null) {
            this.checkExpectMergedAs();
        } else {
            boolean merged;
            try {
                merged = this.rw.isMergedInto(this.currPsCommit, this.tip);
            }
            catch (IOException e) {
                this.problem("Error checking whether patch set " + this.currPs.getId().get() + " is merged");
                return;
            }
            this.checkMergedBitMatchesStatus(this.currPs.getId(), this.currPsCommit, merged);
        }
    }

    private ProblemInfo wrongChangeStatus(PatchSet.Id psId, RevCommit commit) {
        String refName = this.change().getDest().get();
        return this.problem(String.format("Patch set %d (%s) is merged into destination ref %s (%s), but change status is %s", new Object[]{psId.get(), commit.name(), refName, this.tip.name(), this.change().getStatus()}));
    }

    private void checkMergedBitMatchesStatus(PatchSet.Id psId, RevCommit commit, boolean merged) {
        String refName = this.change().getDest().get();
        if (merged && this.change().getStatus() != Change.Status.MERGED) {
            ProblemInfo p = this.wrongChangeStatus(psId, commit);
            if (this.fix != null) {
                this.fixMerged(p);
            }
        } else if (!merged && this.change().getStatus() == Change.Status.MERGED) {
            this.problem(String.format("Patch set %d (%s) is not merged into destination ref %s (%s), but change status is %s", new Object[]{this.currPs.getId().get(), commit.name(), refName, this.tip.name(), this.change().getStatus()}));
        }
    }

    private void checkExpectMergedAs() {
        ObjectId objId = this.parseObjectId(this.fix.expectMergedAs, "expected merged commit");
        RevCommit commit = this.parseCommit(objId, "expected merged commit");
        if (commit == null) {
            return;
        }
        try {
            if (!this.rw.isMergedInto(commit, this.tip)) {
                this.problem(String.format("Expected merged commit %s is not merged into destination ref %s (%s)", commit.name(), this.change().getDest().get(), this.tip.name()));
                return;
            }
            ArrayList<PatchSet.Id> thisCommitPsIds = new ArrayList<PatchSet.Id>();
            for (Ref ref : this.repo.getRefDatabase().getRefs("refs/changes/").values()) {
                PatchSet.Id psId;
                if (!ref.getObjectId().equals(commit) || (psId = PatchSet.Id.fromRef(ref.getName())) == null) continue;
                try {
                    Change c = this.notesFactory.createChecked(this.db.get(), this.change().getProject(), psId.getParentKey()).getChange();
                    if (!c.getDest().equals(this.change().getDest())) {
                        continue;
                    }
                }
                catch (OrmException e) {
                    this.warn(e);
                }
                thisCommitPsIds.add(psId);
            }
            switch (thisCommitPsIds.size()) {
                case 0: {
                    this.rw.parseBody(commit);
                    String changeId = Iterables.getFirst(commit.getFooterLines(FooterConstants.CHANGE_ID), null);
                    if (changeId != null && !changeId.equals(this.change().getKey().get())) {
                        this.problem(String.format("Expected merged commit %s has Change-Id: %s, but expected %s", commit.name(), changeId, this.change().getKey().get()));
                        return;
                    }
                    this.insertMergedPatchSet(commit, null, false);
                    break;
                }
                case 1: {
                    PatchSet.Id id = (PatchSet.Id)thisCommitPsIds.get(0);
                    if (id.equals(this.change().currentPatchSetId())) {
                        this.fixMerged(this.wrongChangeStatus(id, commit));
                        break;
                    }
                    if (id.get() > this.change().currentPatchSetId().get()) {
                        this.insertMergedPatchSet(commit, id, true);
                        break;
                    }
                    this.insertMergedPatchSet(commit, id, false);
                    break;
                }
                default: {
                    this.problem(String.format("Multiple patch sets for expected merged commit %s: %s", commit.name(), ReviewDbUtil.intKeyOrdering().sortedCopy(thisCommitPsIds)));
                    break;
                }
            }
        }
        catch (IOException e) {
            this.error("Error looking up expected merged commit " + this.fix.expectMergedAs, e);
        }
    }

    private void insertMergedPatchSet(final RevCommit commit, final @Nullable PatchSet.Id psIdToDelete, boolean reuseOldPsId) {
        ProblemInfo deleteOldPatchSetProblem;
        ProblemInfo insertPatchSetProblem;
        ProblemInfo notFound = this.problem("No patch set found for merged commit " + commit.name());
        if (!this.user.get().isIdentifiedUser()) {
            notFound.status = ProblemInfo.Status.FIX_FAILED;
            notFound.outcome = "Must be called by an identified user to insert new patch set";
            return;
        }
        if (psIdToDelete == null) {
            insertPatchSetProblem = this.problem(String.format("Expected merged commit %s has no associated patch set", commit.name()));
            deleteOldPatchSetProblem = null;
        } else {
            String msg = String.format("Expected merge commit %s corresponds to patch set %s, not the current patch set %s", commit.name(), psIdToDelete.get(), this.change().currentPatchSetId().get());
            deleteOldPatchSetProblem = reuseOldPsId ? null : this.problem(msg);
            insertPatchSetProblem = this.problem(msg);
        }
        ArrayList<ProblemInfo> currProblems = new ArrayList<ProblemInfo>(3);
        currProblems.add(notFound);
        if (deleteOldPatchSetProblem != null) {
            currProblems.add(insertPatchSetProblem);
        }
        currProblems.add(insertPatchSetProblem);
        try {
            PatchSet.Id psId = psIdToDelete != null && reuseOldPsId ? psIdToDelete : ChangeUtil.nextPatchSetId(this.repo, this.change().currentPatchSetId());
            PatchSetInserter inserter = this.patchSetInserterFactory.create(this.notes, psId, commit);
            try (BatchUpdate bu = this.newBatchUpdate();){
                bu.setRepository(this.repo, this.rw, this.oi);
                if (psIdToDelete != null) {
                    bu.addOp(this.notes.getChangeId(), new BatchUpdateOp(){

                        @Override
                        public void updateRepo(RepoContext ctx) throws IOException {
                            ctx.addRefUpdate(commit, ObjectId.zeroId(), psIdToDelete.toRefName());
                        }
                    });
                    if (!reuseOldPsId) {
                        bu.addOp(this.notes.getChangeId(), new DeletePatchSetFromDbOp(Preconditions.checkNotNull(deleteOldPatchSetProblem), psIdToDelete));
                    }
                }
                bu.addOp(this.notes.getChangeId(), inserter.setValidate(false).setFireRevisionCreated(false).setNotify(NotifyHandling.NONE).setAllowClosed(true).setMessage("Patch set for merged commit inserted by consistency checker"));
                bu.addOp(this.notes.getChangeId(), new FixMergedOp(notFound));
                bu.execute();
            }
            this.notes = this.notesFactory.createChecked(this.db.get(), inserter.getChange());
            insertPatchSetProblem.status = ProblemInfo.Status.FIXED;
            insertPatchSetProblem.outcome = "Inserted as patch set " + psId.get();
        }
        catch (RestApiException | UpdateException | OrmException | IOException e) {
            this.warn(e);
            for (ProblemInfo pi : currProblems) {
                pi.status = ProblemInfo.Status.FIX_FAILED;
                pi.outcome = "Error inserting merged patch set";
            }
            return;
        }
    }

    private void fixMerged(ProblemInfo p) {
        try (BatchUpdate bu = this.newBatchUpdate();){
            bu.setRepository(this.repo, this.rw, this.oi);
            bu.addOp(this.notes.getChangeId(), new FixMergedOp(p));
            bu.execute();
        }
        catch (RestApiException | UpdateException e) {
            log.warn("Error marking " + this.notes.getChangeId() + "as merged", e);
            p.status = ProblemInfo.Status.FIX_FAILED;
            p.outcome = "Error updating status to merged";
        }
    }

    private BatchUpdate newBatchUpdate() {
        return this.updateFactory.create(this.db.get(), this.change().getProject(), this.user.get(), TimeUtil.nowTs());
    }

    private void fixPatchSetRef(ProblemInfo p, PatchSet ps) {
        try {
            RefUpdate ru = this.repo.updateRef(ps.getId().toRefName());
            ru.setForceUpdate(true);
            ru.setNewObjectId(ObjectId.fromString(ps.getRevision().get()));
            ru.setRefLogIdent(this.newRefLogIdent());
            ru.setRefLogMessage("Repair patch set ref", true);
            RefUpdate.Result result = ru.update();
            switch (result) {
                case NEW: 
                case FORCED: 
                case FAST_FORWARD: 
                case NO_CHANGE: {
                    p.status = ProblemInfo.Status.FIXED;
                    p.outcome = "Repaired patch set ref";
                    return;
                }
            }
            p.status = ProblemInfo.Status.FIX_FAILED;
            p.outcome = "Failed to update patch set ref: " + (Object)((Object)result);
            return;
        }
        catch (IOException e) {
            String msg = "Error fixing patch set ref";
            log.warn(msg + ' ' + ps.getId().toRefName(), e);
            p.status = ProblemInfo.Status.FIX_FAILED;
            p.outcome = msg;
            return;
        }
    }

    private void deletePatchSets(List<DeletePatchSetFromDbOp> ops) {
        try (BatchUpdate bu = this.newBatchUpdate();){
            bu.setRepository(this.repo, this.rw, this.oi);
            for (DeletePatchSetFromDbOp op : ops) {
                Preconditions.checkArgument(op.psId.getParentKey().equals(this.notes.getChangeId()));
                bu.addOp(this.notes.getChangeId(), op);
            }
            bu.addOp(this.notes.getChangeId(), new UpdateCurrentPatchSetOp(ops));
            bu.execute();
        }
        catch (NoPatchSetsWouldRemainException e) {
            for (DeletePatchSetFromDbOp op : ops) {
                ((DeletePatchSetFromDbOp)op).p.status = ProblemInfo.Status.FIX_FAILED;
                ((DeletePatchSetFromDbOp)op).p.outcome = e.getMessage();
            }
        }
        catch (RestApiException | UpdateException e) {
            String msg = "Error deleting patch set";
            log.warn(msg + " of change " + ops.get(0).psId.getParentKey(), e);
            for (DeletePatchSetFromDbOp op : ops) {
                ((DeletePatchSetFromDbOp)op).p.status = ProblemInfo.Status.FIX_FAILED;
                ((DeletePatchSetFromDbOp)op).p.outcome = msg;
            }
        }
    }

    private PersonIdent newRefLogIdent() {
        CurrentUser u = this.user.get();
        if (u.isIdentifiedUser()) {
            return u.asIdentifiedUser().newRefLogIdent();
        }
        return this.serverIdent.get();
    }

    private ObjectId parseObjectId(String objIdStr, String desc) {
        try {
            return ObjectId.fromString(objIdStr);
        }
        catch (IllegalArgumentException e) {
            this.problem(String.format("Invalid revision on %s: %s", desc, objIdStr));
            return null;
        }
    }

    private RevCommit parseCommit(ObjectId objId, String desc) {
        try {
            return this.rw.parseCommit(objId);
        }
        catch (MissingObjectException e) {
            this.problem(String.format("Object missing: %s: %s", desc, objId.name()));
        }
        catch (IncorrectObjectTypeException e) {
            this.problem(String.format("Not a commit: %s: %s", desc, objId.name()));
        }
        catch (IOException e) {
            this.problem(String.format("Failed to look up: %s: %s", desc, objId.name()));
        }
        return null;
    }

    private ProblemInfo problem(String msg) {
        ProblemInfo p = new ProblemInfo();
        p.message = Preconditions.checkNotNull(msg);
        this.problems.add(p);
        return p;
    }

    private ProblemInfo lastProblem() {
        return this.problems.get(this.problems.size() - 1);
    }

    private boolean error(String msg, Throwable t) {
        this.problem(msg);
        this.warn(t);
        return false;
    }

    private void warn(Throwable t) {
        log.warn("Error in consistency check of change " + this.notes.getChangeId(), t);
    }

    private Result result() {
        return Result.create(this.notes, this.problems);
    }

    private class UpdateCurrentPatchSetOp
    implements BatchUpdateOp {
        private final Set<PatchSet.Id> toDelete = new HashSet<PatchSet.Id>();

        private UpdateCurrentPatchSetOp(List<DeletePatchSetFromDbOp> deleteOps) {
            for (DeletePatchSetFromDbOp op : deleteOps) {
                this.toDelete.add(op.psId);
            }
        }

        @Override
        public boolean updateChange(ChangeContext ctx) throws OrmException, PatchSetInfoNotAvailableException, NoPatchSetsWouldRemainException {
            if (!this.toDelete.contains(ctx.getChange().currentPatchSetId())) {
                return false;
            }
            HashSet<PatchSet.Id> all = new HashSet<PatchSet.Id>();
            for (PatchSet ps : ConsistencyChecker.this.psUtil.byChange(ctx.getDb(), ctx.getNotes())) {
                if (this.toDelete.contains(ps.getId())) continue;
                all.add(ps.getId());
            }
            if (all.isEmpty()) {
                throw new NoPatchSetsWouldRemainException();
            }
            PatchSet.Id latest = (PatchSet.Id)ReviewDbUtil.intKeyOrdering().max(all);
            ctx.getChange().setCurrentPatchSet(ConsistencyChecker.this.patchSetInfoFactory.get(ctx.getDb(), ctx.getNotes(), latest));
            return true;
        }
    }

    private static class NoPatchSetsWouldRemainException
    extends RestApiException {
        private static final long serialVersionUID = 1L;

        private NoPatchSetsWouldRemainException() {
            super("Cannot delete patch set; no patch sets would remain");
        }
    }

    private class DeletePatchSetFromDbOp
    implements BatchUpdateOp {
        private final ProblemInfo p;
        private final PatchSet.Id psId;

        private DeletePatchSetFromDbOp(ProblemInfo p, PatchSet.Id psId) {
            this.p = p;
            this.psId = psId;
        }

        @Override
        public boolean updateChange(ChangeContext ctx) throws OrmException, PatchSetInfoNotAvailableException {
            ReviewDb db = DeleteChangeOp.unwrap(ctx.getDb());
            ((AccountPatchReviewStore)ConsistencyChecker.this.accountPatchReviewStore.get()).clearReviewed(this.psId);
            db.changeMessages().delete(db.changeMessages().byChange(this.psId.getParentKey()));
            db.patchSetApprovals().delete(db.patchSetApprovals().byPatchSet(this.psId));
            db.patchComments().delete(db.patchComments().byPatchSet(this.psId));
            db.patchSets().deleteKeys(Collections.singleton(this.psId));
            ctx.getUpdate(this.psId).setPatchSetState(PatchSetState.DELETED);
            this.p.status = ProblemInfo.Status.FIXED;
            this.p.outcome = "Deleted patch set";
            return true;
        }
    }

    private static class FixMergedOp
    implements BatchUpdateOp {
        private final ProblemInfo p;

        private FixMergedOp(ProblemInfo p) {
            this.p = p;
        }

        @Override
        public boolean updateChange(ChangeContext ctx) throws OrmException {
            ctx.getChange().setStatus(Change.Status.MERGED);
            ctx.getUpdate(ctx.getChange().currentPatchSetId()).fixStatus(Change.Status.MERGED);
            this.p.status = ProblemInfo.Status.FIXED;
            this.p.outcome = "Marked change as merged";
            return true;
        }
    }

    @AutoValue
    public static abstract class Result {
        private static Result create(ChangeNotes notes, List<ProblemInfo> problems) {
            return new AutoValue_ConsistencyChecker_Result(notes.getChangeId(), notes.getChange(), problems);
        }

        public abstract Change.Id id();

        @Nullable
        public abstract Change change();

        public abstract List<ProblemInfo> problems();
    }
}

