/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.index.change;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ComparisonChain;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.gerrit.index.SiteIndexer;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.gerrit.server.git.MultiProgressMonitor;
import com.google.gerrit.server.git.QueueProvider;
import com.google.gerrit.server.index.IndexExecutor;
import com.google.gerrit.server.index.change.ChangeIndex;
import com.google.gerrit.server.index.change.ChangeIndexer;
import com.google.gerrit.server.notedb.ChangeNotes;
import com.google.gerrit.server.project.ProjectCache;
import com.google.gerrit.server.query.change.ChangeData;
import com.google.gwtorm.server.SchemaFactory;
import com.google.inject.Inject;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Objects;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.eclipse.jgit.errors.RepositoryNotFoundException;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.TextProgressMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AllChangesIndexer
extends SiteIndexer<Change.Id, ChangeData, ChangeIndex> {
    private static final Logger log = LoggerFactory.getLogger(AllChangesIndexer.class);
    private final SchemaFactory<ReviewDb> schemaFactory;
    private final ChangeData.Factory changeDataFactory;
    private final GitRepositoryManager repoManager;
    private final ListeningExecutorService executor;
    private final ChangeIndexer.Factory indexerFactory;
    private final ChangeNotes.Factory notesFactory;
    private final ProjectCache projectCache;

    @Inject
    AllChangesIndexer(SchemaFactory<ReviewDb> schemaFactory, ChangeData.Factory changeDataFactory, GitRepositoryManager repoManager, @IndexExecutor(value=QueueProvider.QueueType.BATCH) ListeningExecutorService executor, ChangeIndexer.Factory indexerFactory, ChangeNotes.Factory notesFactory, ProjectCache projectCache) {
        this.schemaFactory = schemaFactory;
        this.changeDataFactory = changeDataFactory;
        this.repoManager = repoManager;
        this.executor = executor;
        this.indexerFactory = indexerFactory;
        this.notesFactory = notesFactory;
        this.projectCache = projectCache;
    }

    @Override
    public SiteIndexer.Result indexAll(ChangeIndex index) {
        TextProgressMonitor pm = new TextProgressMonitor();
        pm.beginTask("Collecting projects", 0);
        TreeSet<ProjectHolder> projects = new TreeSet<ProjectHolder>();
        int changeCount = 0;
        Stopwatch sw = Stopwatch.createStarted();
        int projectsFailed = 0;
        for (Project.NameKey name : this.projectCache.all()) {
            block9: {
                try (Repository repo = this.repoManager.openRepository(name);){
                    long size = this.estimateSize(repo);
                    changeCount = (int)((long)changeCount + size);
                    projects.add(new ProjectHolder(name, size));
                }
                catch (IOException e) {
                    log.error("Error collecting project {}", (Object)name, (Object)e);
                    if (++projectsFailed <= projects.size() / 2) break block9;
                    log.error("Over 50% of the projects could not be collected: aborted");
                    return new SiteIndexer.Result(sw, false, 0, 0);
                }
            }
            pm.update(1);
        }
        pm.endTask();
        this.setTotalWork(changeCount);
        return this.indexAll(index, projects);
    }

    private long estimateSize(Repository repo) throws IOException {
        return repo.getRefDatabase().getRefs("refs/changes/").values().stream().map(r -> Change.Id.fromRef(r.getName())).filter(Objects::nonNull).distinct().count();
    }

    private SiteIndexer.Result indexAll(ChangeIndex index, SortedSet<ProjectHolder> projects) {
        Stopwatch sw = Stopwatch.createStarted();
        MultiProgressMonitor mpm = new MultiProgressMonitor(this.progressOut, "Reindexing changes");
        MultiProgressMonitor.Task projTask = mpm.beginSubTask("projects", projects.size());
        Preconditions.checkState(this.totalWork >= 0);
        MultiProgressMonitor.Task doneTask = mpm.beginSubTask(null, this.totalWork);
        MultiProgressMonitor.Task failedTask = mpm.beginSubTask("failed", 0);
        ArrayList<Future> futures = new ArrayList<Future>();
        AtomicBoolean ok = new AtomicBoolean(true);
        for (ProjectHolder project : projects) {
            Future future = this.executor.submit(this.reindexProject(this.indexerFactory.create(this.executor, index), project.name, doneTask, failedTask));
            this.addErrorListener((ListenableFuture<?>)future, "project " + project.name, projTask, ok);
            futures.add(future);
        }
        try {
            mpm.waitFor(Futures.transform(Futures.successfulAsList(futures), x -> {
                mpm.end();
                return null;
            }, MoreExecutors.directExecutor()));
        }
        catch (ExecutionException e) {
            log.error("Error in batch indexer", e);
            ok.set(false);
        }
        int nFailed = failedTask.getCount();
        int nDone = doneTask.getCount();
        int nTotal = nFailed + nDone;
        double pctFailed = (double)nFailed / (double)nTotal * 100.0;
        if (pctFailed > 10.0) {
            log.error("Failed {}/{} changes ({}%); not marking new index as ready", nFailed, nTotal, Math.round(pctFailed));
            ok.set(false);
        }
        return new SiteIndexer.Result(sw, ok.get(), nDone, nFailed);
    }

    public Callable<Void> reindexProject(ChangeIndexer indexer, Project.NameKey project, MultiProgressMonitor.Task done, MultiProgressMonitor.Task failed) {
        return new ProjectIndexer(indexer, project, done, failed);
    }

    private class ProjectIndexer
    implements Callable<Void> {
        private final ChangeIndexer indexer;
        private final Project.NameKey project;
        private final ProgressMonitor done;
        private final ProgressMonitor failed;

        private ProjectIndexer(ChangeIndexer indexer, Project.NameKey project, ProgressMonitor done, ProgressMonitor failed) {
            this.indexer = indexer;
            this.project = project;
            this.done = done;
            this.failed = failed;
        }

        @Override
        public Void call() throws Exception {
            try (Repository repo = AllChangesIndexer.this.repoManager.openRepository(this.project);
                 ReviewDb db = (ReviewDb)AllChangesIndexer.this.schemaFactory.open();){
                AllChangesIndexer.this.notesFactory.scan(repo, db, this.project).forEach(r -> this.index(db, (ChangeNotes.Factory.ChangeNotesResult)r));
            }
            catch (RepositoryNotFoundException rnfe) {
                log.error(rnfe.getMessage());
            }
            return null;
        }

        private void index(ReviewDb db, ChangeNotes.Factory.ChangeNotesResult r) {
            if (r.error().isPresent()) {
                this.fail("Failed to read change " + r.id() + " for indexing", true, r.error().get());
                return;
            }
            try {
                this.indexer.index(AllChangesIndexer.this.changeDataFactory.create(db, r.notes()));
                this.done.update(1);
                AllChangesIndexer.this.verboseWriter.println("Reindexed change " + r.id());
            }
            catch (RejectedExecutionException e) {
                this.failSilently();
            }
            catch (Exception e) {
                this.fail("Failed to index change " + r.id(), true, e);
            }
        }

        private void fail(String error, boolean failed, Exception e) {
            if (failed) {
                this.failed.update(1);
            }
            if (e != null) {
                log.warn(error, e);
            } else {
                log.warn(error);
            }
            AllChangesIndexer.this.verboseWriter.println(error);
        }

        private void failSilently() {
            this.failed.update(1);
        }

        public String toString() {
            return "Index all changes of project " + this.project.get();
        }
    }

    private static class ProjectHolder
    implements Comparable<ProjectHolder> {
        final Project.NameKey name;
        private final long size;

        ProjectHolder(Project.NameKey name, long size) {
            this.name = name;
            this.size = size;
        }

        @Override
        public int compareTo(ProjectHolder other) {
            return ComparisonChain.start().compare(other.size, this.size).compare((Comparable<?>)((Object)other.name.get()), (Comparable<?>)((Object)this.name.get())).result();
        }
    }
}

