/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git.validators;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.gerrit.extensions.api.projects.ProjectConfigEntryType;
import com.google.gerrit.extensions.registration.DynamicMap;
import com.google.gerrit.extensions.registration.DynamicSet;
import com.google.gerrit.extensions.restapi.AuthException;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Branch;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.config.AllProjectsName;
import com.google.gerrit.server.config.AllUsersName;
import com.google.gerrit.server.config.PluginConfig;
import com.google.gerrit.server.config.ProjectConfigEntry;
import com.google.gerrit.server.git.CodeReviewCommit;
import com.google.gerrit.server.git.ProjectConfig;
import com.google.gerrit.server.git.validators.AccountValidator;
import com.google.gerrit.server.git.validators.MergeValidationException;
import com.google.gerrit.server.git.validators.MergeValidationListener;
import com.google.gerrit.server.permissions.GlobalPermission;
import com.google.gerrit.server.permissions.PermissionBackend;
import com.google.gerrit.server.permissions.PermissionBackendException;
import com.google.gerrit.server.project.ProjectCache;
import com.google.gerrit.server.project.ProjectState;
import com.google.gerrit.server.query.change.ChangeData;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.io.IOException;
import java.util.List;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevWalk;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MergeValidators {
    private static final Logger log = LoggerFactory.getLogger(MergeValidators.class);
    private final DynamicSet<MergeValidationListener> mergeValidationListeners;
    private final ProjectConfigValidator.Factory projectConfigValidatorFactory;
    private final AccountMergeValidator.Factory accountValidatorFactory;

    @Inject
    MergeValidators(DynamicSet<MergeValidationListener> mergeValidationListeners, ProjectConfigValidator.Factory projectConfigValidatorFactory, AccountMergeValidator.Factory accountValidatorFactory) {
        this.mergeValidationListeners = mergeValidationListeners;
        this.projectConfigValidatorFactory = projectConfigValidatorFactory;
        this.accountValidatorFactory = accountValidatorFactory;
    }

    public void validatePreMerge(Repository repo, CodeReviewCommit commit, ProjectState destProject, Branch.NameKey destBranch, PatchSet.Id patchSetId, IdentifiedUser caller) throws MergeValidationException {
        ImmutableList<AccountMergeValidator> validators = ImmutableList.of(new PluginMergeValidationListener(this.mergeValidationListeners), this.projectConfigValidatorFactory.create(), this.accountValidatorFactory.create());
        for (MergeValidationListener mergeValidationListener : validators) {
            mergeValidationListener.onPreMerge(repo, commit, destProject, destBranch, patchSetId, caller);
        }
    }

    public static class AccountMergeValidator
    implements MergeValidationListener {
        private final Provider<ReviewDb> dbProvider;
        private final AllUsersName allUsersName;
        private final ChangeData.Factory changeDataFactory;
        private final AccountValidator accountValidator;

        @Inject
        public AccountMergeValidator(Provider<ReviewDb> dbProvider, AllUsersName allUsersName, ChangeData.Factory changeDataFactory, AccountValidator accountValidator) {
            this.dbProvider = dbProvider;
            this.allUsersName = allUsersName;
            this.changeDataFactory = changeDataFactory;
            this.accountValidator = accountValidator;
        }

        @Override
        public void onPreMerge(Repository repo, CodeReviewCommit commit, ProjectState destProject, Branch.NameKey destBranch, PatchSet.Id patchSetId, IdentifiedUser caller) throws MergeValidationException {
            Account.Id accountId = Account.Id.fromRef(destBranch.get());
            if (!this.allUsersName.equals(destProject.getNameKey()) || accountId == null) {
                return;
            }
            ChangeData cd = this.changeDataFactory.create(this.dbProvider.get(), destProject.getProject().getNameKey(), patchSetId.getParentKey());
            try {
                if (!cd.currentFilePaths().contains("account.config")) {
                    return;
                }
            }
            catch (OrmException | IOException e) {
                log.error("Cannot validate account update", e);
                throw new MergeValidationException("account validation unavailable");
            }
            try (RevWalk rw = new RevWalk(repo);){
                List<String> errorMessages = this.accountValidator.validate(accountId, rw, null, commit);
                if (!errorMessages.isEmpty()) {
                    throw new MergeValidationException("invalid account configuration: " + Joiner.on("; ").join(errorMessages));
                }
            }
            catch (IOException e) {
                log.error("Cannot validate account update", e);
                throw new MergeValidationException("account validation unavailable");
            }
        }

        public static interface Factory {
            public AccountMergeValidator create();
        }
    }

    public static class PluginMergeValidationListener
    implements MergeValidationListener {
        private final DynamicSet<MergeValidationListener> mergeValidationListeners;

        public PluginMergeValidationListener(DynamicSet<MergeValidationListener> mergeValidationListeners) {
            this.mergeValidationListeners = mergeValidationListeners;
        }

        @Override
        public void onPreMerge(Repository repo, CodeReviewCommit commit, ProjectState destProject, Branch.NameKey destBranch, PatchSet.Id patchSetId, IdentifiedUser caller) throws MergeValidationException {
            for (MergeValidationListener validator : this.mergeValidationListeners) {
                validator.onPreMerge(repo, commit, destProject, destBranch, patchSetId, caller);
            }
        }
    }

    public static class ProjectConfigValidator
    implements MergeValidationListener {
        private static final String INVALID_CONFIG = "Change contains an invalid project configuration.";
        private static final String PARENT_NOT_FOUND = "Change contains an invalid project configuration:\nParent project does not exist.";
        private static final String PLUGIN_VALUE_NOT_EDITABLE = "Change contains an invalid project configuration:\nOne of the plugin configuration parameters is not editable.";
        private static final String PLUGIN_VALUE_NOT_PERMITTED = "Change contains an invalid project configuration:\nOne of the plugin configuration parameters has a value that is not permitted.";
        private static final String ROOT_NO_PARENT = "Change contains an invalid project configuration:\nThe root project cannot have a parent.";
        private static final String SET_BY_ADMIN = "Change contains a project configuration that changes the parent project.\nThe change must be submitted by a Gerrit administrator.";
        private final AllProjectsName allProjectsName;
        private final ProjectCache projectCache;
        private final PermissionBackend permissionBackend;
        private final DynamicMap<ProjectConfigEntry> pluginConfigEntries;

        @Inject
        public ProjectConfigValidator(AllProjectsName allProjectsName, ProjectCache projectCache, PermissionBackend permissionBackend, DynamicMap<ProjectConfigEntry> pluginConfigEntries) {
            this.allProjectsName = allProjectsName;
            this.projectCache = projectCache;
            this.permissionBackend = permissionBackend;
            this.pluginConfigEntries = pluginConfigEntries;
        }

        @Override
        public void onPreMerge(Repository repo, CodeReviewCommit commit, ProjectState destProject, Branch.NameKey destBranch, PatchSet.Id patchSetId, IdentifiedUser caller) throws MergeValidationException {
            if ("refs/meta/config".equals(destBranch.get())) {
                try {
                    ProjectConfig cfg = new ProjectConfig(destProject.getNameKey());
                    cfg.load(repo, (ObjectId)commit);
                    Project.NameKey newParent = cfg.getProject().getParent(this.allProjectsName);
                    Project.NameKey oldParent = destProject.getProject().getParent(this.allProjectsName);
                    if (oldParent == null) {
                        if (newParent != null) {
                            throw new MergeValidationException(ROOT_NO_PARENT);
                        }
                    } else if (!oldParent.equals(newParent)) {
                        try {
                            this.permissionBackend.user(caller).check(GlobalPermission.ADMINISTRATE_SERVER);
                        }
                        catch (AuthException e) {
                            throw new MergeValidationException(SET_BY_ADMIN);
                        }
                        catch (PermissionBackendException e) {
                            log.warn("Cannot check ADMINISTRATE_SERVER", e);
                            throw new MergeValidationException("validation unavailable");
                        }
                        if (this.projectCache.get(newParent) == null) {
                            throw new MergeValidationException(PARENT_NOT_FOUND);
                        }
                    }
                    for (DynamicMap.Entry<ProjectConfigEntry> entry : this.pluginConfigEntries) {
                        PluginConfig pluginCfg = cfg.getPluginConfig(entry.getPluginName());
                        ProjectConfigEntry configEntry = entry.getProvider().get();
                        String value = pluginCfg.getString(entry.getExportName());
                        String oldValue = destProject.getConfig().getPluginConfig(entry.getPluginName()).getString(entry.getExportName());
                        if ((value == null ? oldValue != null : !value.equals(oldValue)) && !configEntry.isEditable(destProject)) {
                            throw new MergeValidationException(PLUGIN_VALUE_NOT_EDITABLE);
                        }
                        if (!ProjectConfigEntryType.LIST.equals((Object)configEntry.getType()) || value == null || configEntry.getPermittedValues().contains(value)) continue;
                        throw new MergeValidationException(PLUGIN_VALUE_NOT_PERMITTED);
                    }
                }
                catch (IOException | ConfigInvalidException e) {
                    throw new MergeValidationException(INVALID_CONFIG);
                }
            }
        }

        public static interface Factory {
            public ProjectConfigValidator create();
        }
    }

    public static interface Factory {
        public MergeValidators create();
    }
}

