/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git.validators;

import com.google.common.base.CharMatcher;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.gerrit.common.FooterConstants;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.extensions.api.config.ConsistencyCheckInfo;
import com.google.gerrit.extensions.registration.DynamicSet;
import com.google.gerrit.extensions.restapi.AuthException;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Branch;
import com.google.gerrit.reviewdb.client.RefNames;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.account.WatchConfig;
import com.google.gerrit.server.account.externalids.ExternalIdsConsistencyChecker;
import com.google.gerrit.server.config.AllUsersName;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.events.CommitReceivedEvent;
import com.google.gerrit.server.git.BanCommit;
import com.google.gerrit.server.git.ProjectConfig;
import com.google.gerrit.server.git.ValidationError;
import com.google.gerrit.server.git.validators.AccountValidator;
import com.google.gerrit.server.git.validators.CommitValidationException;
import com.google.gerrit.server.git.validators.CommitValidationListener;
import com.google.gerrit.server.git.validators.CommitValidationMessage;
import com.google.gerrit.server.permissions.PermissionBackend;
import com.google.gerrit.server.permissions.PermissionBackendException;
import com.google.gerrit.server.permissions.RefPermission;
import com.google.gerrit.server.project.ProjectCache;
import com.google.gerrit.server.project.ProjectState;
import com.google.gerrit.server.ssh.SshInfo;
import com.google.gerrit.server.util.MagicBranch;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.jcraft.jsch.HostKey;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.notes.NoteMap;
import org.eclipse.jgit.revwalk.FooterKey;
import org.eclipse.jgit.revwalk.FooterLine;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.util.SystemReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitValidators {
    private static final Logger log = LoggerFactory.getLogger(CommitValidators.class);
    public static final Pattern NEW_PATCHSET_PATTERN = Pattern.compile("^refs/changes/(?:[0-9][0-9]/)?([1-9][0-9]*)(?:/[1-9][0-9]*)?$");
    private final List<CommitValidationListener> validators;

    CommitValidators(List<CommitValidationListener> validators) {
        this.validators = validators;
    }

    public List<CommitValidationMessage> validate(CommitReceivedEvent receiveEvent) throws CommitValidationException {
        ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
        try {
            for (CommitValidationListener commitValidator : this.validators) {
                messages.addAll(commitValidator.onCommitReceived(receiveEvent));
            }
        }
        catch (CommitValidationException e) {
            log.debug("CommitValidationException occurred: {}", (Object)e.getFullMessage(), (Object)e);
            messages.addAll(e.getMessages());
            throw new CommitValidationException(e.getMessage(), messages);
        }
        return messages;
    }

    public boolean hasAllCommitsValidators() {
        return this.validators.stream().anyMatch(v -> v.shouldValidateAllCommits());
    }

    private static CommitValidationMessage invalidEmail(String type, PersonIdent who, IdentifiedUser currentUser, String canonicalWebUrl) {
        StringBuilder sb = new StringBuilder();
        sb.append("email address ").append(who.getEmailAddress()).append(" is not registered in your account, and you lack 'forge ").append(type).append("' permission.\n");
        if (currentUser.getEmailAddresses().isEmpty()) {
            sb.append("You have not registered any email addresses.\n");
        } else {
            sb.append("The following addresses are currently registered:\n");
            for (String address : currentUser.getEmailAddresses()) {
                sb.append("   ").append(address).append("\n");
            }
        }
        if (canonicalWebUrl != null) {
            sb.append("To register an email address, visit:\n");
            sb.append(canonicalWebUrl).append("#").append("/settings/contact").append("\n");
        }
        sb.append("\n");
        return new CommitValidationMessage(sb.toString(), true);
    }

    private static String getGerritUrl(String canonicalWebUrl) {
        if (canonicalWebUrl != null) {
            return CharMatcher.is('/').trimTrailingFrom(canonicalWebUrl);
        }
        return "http://" + CommitValidators.getGerritHost(canonicalWebUrl);
    }

    private static String getGerritHost(String canonicalWebUrl) {
        String host;
        if (canonicalWebUrl != null) {
            try {
                host = new URL(canonicalWebUrl).getHost();
            }
            catch (MalformedURLException e) {
                host = SystemReader.getInstance().getHostname();
            }
        } else {
            host = SystemReader.getInstance().getHostname();
        }
        return host;
    }

    private static void addError(String error, List<CommitValidationMessage> messages) {
        messages.add(new CommitValidationMessage(error, true));
    }

    public static class AccountCommitValidator
    implements CommitValidationListener {
        private final AllUsersName allUsers;
        private final AccountValidator accountValidator;

        public AccountCommitValidator(AllUsersName allUsers, AccountValidator accountValidator) {
            this.allUsers = allUsers;
            this.accountValidator = accountValidator;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (!this.allUsers.equals(receiveEvent.project.getNameKey())) {
                return Collections.emptyList();
            }
            if (receiveEvent.command.getRefName().startsWith("refs/for/")) {
                return Collections.emptyList();
            }
            Account.Id accountId = Account.Id.fromRef(receiveEvent.refName);
            if (accountId == null) {
                return Collections.emptyList();
            }
            try {
                List<String> errorMessages = this.accountValidator.validate(accountId, receiveEvent.revWalk, receiveEvent.command.getOldId(), receiveEvent.commit);
                if (!errorMessages.isEmpty()) {
                    throw new CommitValidationException("invalid account configuration", errorMessages.stream().map(m -> new CommitValidationMessage((String)m, true)).collect(Collectors.toList()));
                }
            }
            catch (IOException e) {
                String m2 = String.format("Validating update for account %s failed", accountId.get());
                log.error(m2, e);
                throw new CommitValidationException(m2, e);
            }
            return Collections.emptyList();
        }
    }

    public static class ExternalIdUpdateListener
    implements CommitValidationListener {
        private final AllUsersName allUsers;
        private final ExternalIdsConsistencyChecker externalIdsConsistencyChecker;

        public ExternalIdUpdateListener(AllUsersName allUsers, ExternalIdsConsistencyChecker externalIdsConsistencyChecker) {
            this.externalIdsConsistencyChecker = externalIdsConsistencyChecker;
            this.allUsers = allUsers;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (this.allUsers.equals(receiveEvent.project.getNameKey()) && "refs/meta/external-ids".equals(receiveEvent.refName)) {
                try {
                    List<ConsistencyCheckInfo.ConsistencyProblemInfo> problems = this.externalIdsConsistencyChecker.check(receiveEvent.commit);
                    List<CommitValidationMessage> msgs = problems.stream().map(p -> new CommitValidationMessage(p.message, p.status == ConsistencyCheckInfo.ConsistencyProblemInfo.Status.ERROR)).collect(Collectors.toList());
                    if (msgs.stream().anyMatch(m -> m.isError())) {
                        throw new CommitValidationException("invalid external IDs", msgs);
                    }
                    return msgs;
                }
                catch (IOException e) {
                    String m2 = "error validating external IDs";
                    log.warn(m2, e);
                    throw new CommitValidationException(m2, e);
                }
            }
            return Collections.emptyList();
        }
    }

    public static class BannedCommitsValidator
    implements CommitValidationListener {
        private final NoteMap rejectCommits;

        public BannedCommitsValidator(NoteMap rejectCommits) {
            this.rejectCommits = rejectCommits;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            try {
                if (this.rejectCommits.contains(receiveEvent.commit)) {
                    throw new CommitValidationException("contains banned commit " + receiveEvent.commit.getName());
                }
                return Collections.emptyList();
            }
            catch (IOException e) {
                String m = "error checking banned commits";
                log.warn(m, e);
                throw new CommitValidationException(m, e);
            }
        }
    }

    public static class AmendedGerritMergeCommitValidationListener
    implements CommitValidationListener {
        private final PermissionBackend.ForRef perm;
        private final PersonIdent gerritIdent;

        public AmendedGerritMergeCommitValidationListener(PermissionBackend.ForRef perm, PersonIdent gerritIdent) {
            this.perm = perm;
            this.gerritIdent = gerritIdent;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            if (receiveEvent.commit.getParentCount() > 1 && author.getName().equals(this.gerritIdent.getName()) && author.getEmailAddress().equals(this.gerritIdent.getEmailAddress())) {
                try {
                    this.perm.check(RefPermission.FORGE_SERVER);
                }
                catch (AuthException denied) {
                    throw new CommitValidationException(String.format("pushing merge commit %s by %s requires '%s' permission", receiveEvent.commit.getId(), this.gerritIdent.getEmailAddress(), RefPermission.FORGE_SERVER.name()));
                }
                catch (PermissionBackendException e) {
                    log.error("cannot check FORGE_SERVER", e);
                    throw new CommitValidationException("internal auth error");
                }
            }
            return Collections.emptyList();
        }
    }

    public static class CommitterUploaderValidator
    implements CommitValidationListener {
        private final IdentifiedUser user;
        private final PermissionBackend.ForRef perm;
        private final String canonicalWebUrl;

        public CommitterUploaderValidator(IdentifiedUser user, PermissionBackend.ForRef perm, String canonicalWebUrl) {
            this.user = user;
            this.perm = perm;
            this.canonicalWebUrl = canonicalWebUrl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent committer = receiveEvent.commit.getCommitterIdent();
            if (this.user.hasEmailAddress(committer.getEmailAddress())) {
                return Collections.emptyList();
            }
            try {
                this.perm.check(RefPermission.FORGE_COMMITTER);
                return Collections.emptyList();
            }
            catch (AuthException e) {
                throw new CommitValidationException("invalid committer", CommitValidators.invalidEmail("committer", committer, this.user, this.canonicalWebUrl));
            }
            catch (PermissionBackendException e) {
                log.error("cannot check FORGE_COMMITTER", e);
                throw new CommitValidationException("internal auth error");
            }
        }
    }

    public static class AuthorUploaderValidator
    implements CommitValidationListener {
        private final IdentifiedUser user;
        private final PermissionBackend.ForRef perm;
        private final String canonicalWebUrl;

        public AuthorUploaderValidator(IdentifiedUser user, PermissionBackend.ForRef perm, String canonicalWebUrl) {
            this.user = user;
            this.perm = perm;
            this.canonicalWebUrl = canonicalWebUrl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            if (this.user.hasEmailAddress(author.getEmailAddress())) {
                return Collections.emptyList();
            }
            try {
                this.perm.check(RefPermission.FORGE_AUTHOR);
                return Collections.emptyList();
            }
            catch (AuthException e) {
                throw new CommitValidationException("invalid author", CommitValidators.invalidEmail("author", author, this.user, this.canonicalWebUrl));
            }
            catch (PermissionBackendException e) {
                log.error("cannot check FORGE_AUTHOR", e);
                throw new CommitValidationException("internal auth error");
            }
        }
    }

    public static class SignedOffByValidator
    implements CommitValidationListener {
        private final IdentifiedUser user;
        private final PermissionBackend.ForRef perm;
        private final ProjectState state;

        public SignedOffByValidator(IdentifiedUser user, PermissionBackend.ForRef perm, ProjectState state) {
            this.user = user;
            this.perm = perm;
            this.state = state;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (!this.state.isUseSignedOffBy()) {
                return Collections.emptyList();
            }
            RevCommit commit = receiveEvent.commit;
            PersonIdent committer = commit.getCommitterIdent();
            PersonIdent author = commit.getAuthorIdent();
            boolean sboAuthor = false;
            boolean sboCommitter = false;
            boolean sboMe = false;
            for (FooterLine footer : commit.getFooterLines()) {
                String e;
                if (!footer.matches(FooterKey.SIGNED_OFF_BY) || (e = footer.getEmailAddress()) == null) continue;
                sboAuthor |= author.getEmailAddress().equals(e);
                sboCommitter |= committer.getEmailAddress().equals(e);
                sboMe |= this.user.hasEmailAddress(e);
            }
            if (!(sboAuthor || sboCommitter || sboMe)) {
                try {
                    this.perm.check(RefPermission.FORGE_COMMITTER);
                }
                catch (AuthException denied) {
                    throw new CommitValidationException("not Signed-off-by author/committer/uploader in message footer");
                }
                catch (PermissionBackendException e) {
                    log.error("cannot check FORGE_COMMITTER", e);
                    throw new CommitValidationException("internal auth error");
                }
            }
            return Collections.emptyList();
        }
    }

    public static class PluginCommitValidationListener
    implements CommitValidationListener {
        private boolean skipValidation;
        private final DynamicSet<CommitValidationListener> commitValidationListeners;

        public PluginCommitValidationListener(DynamicSet<CommitValidationListener> commitValidationListeners) {
            this(commitValidationListeners, false);
        }

        public PluginCommitValidationListener(DynamicSet<CommitValidationListener> commitValidationListeners, boolean skipValidation) {
            this.skipValidation = skipValidation;
            this.commitValidationListeners = commitValidationListeners;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
            for (CommitValidationListener validator : this.commitValidationListeners) {
                if (this.skipValidation && !validator.shouldValidateAllCommits()) continue;
                try {
                    messages.addAll(validator.onCommitReceived(receiveEvent));
                }
                catch (CommitValidationException e) {
                    messages.addAll(e.getMessages());
                    throw new CommitValidationException(e.getMessage(), messages);
                }
            }
            return messages;
        }

        @Override
        public boolean shouldValidateAllCommits() {
            return this.commitValidationListeners.stream().anyMatch(v -> v.shouldValidateAllCommits());
        }
    }

    public static class UploadMergesPermissionValidator
    implements CommitValidationListener {
        private final PermissionBackend.ForRef perm;

        public UploadMergesPermissionValidator(PermissionBackend.ForRef perm) {
            this.perm = perm;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (receiveEvent.commit.getParentCount() <= 1) {
                return Collections.emptyList();
            }
            try {
                this.perm.check(RefPermission.MERGE);
                return Collections.emptyList();
            }
            catch (AuthException e) {
                throw new CommitValidationException("you are not allowed to upload merges");
            }
            catch (PermissionBackendException e) {
                log.error("cannot check MERGE", e);
                throw new CommitValidationException("internal auth error");
            }
        }
    }

    public static class ConfigValidator
    implements CommitValidationListener {
        private final Branch.NameKey branch;
        private final IdentifiedUser user;
        private final RevWalk rw;
        private final AllUsersName allUsers;

        public ConfigValidator(Branch.NameKey branch, IdentifiedUser user, RevWalk rw, AllUsersName allUsers) {
            this.branch = branch;
            this.user = user;
            this.rw = rw;
            this.allUsers = allUsers;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            ArrayList<CommitValidationMessage> messages;
            if ("refs/meta/config".equals(this.branch.get())) {
                messages = new ArrayList<CommitValidationMessage>();
                try {
                    ProjectConfig cfg = new ProjectConfig(receiveEvent.project.getNameKey());
                    cfg.load(this.rw, receiveEvent.command.getNewId());
                    if (!cfg.getValidationErrors().isEmpty()) {
                        CommitValidators.addError("Invalid project configuration:", messages);
                        for (ValidationError err : cfg.getValidationErrors()) {
                            CommitValidators.addError("  " + err.getMessage(), messages);
                        }
                        throw new ConfigInvalidException("invalid project configuration");
                    }
                }
                catch (IOException | ConfigInvalidException e) {
                    log.error("User " + this.user.getUserName() + " tried to push an invalid project configuration " + receiveEvent.command.getNewId().name() + " for project " + receiveEvent.project, e);
                    throw new CommitValidationException("invalid project configuration", messages);
                }
            }
            if (this.allUsers.equals(this.branch.getParentKey()) && RefNames.isRefsUsers(this.branch.get())) {
                messages = new ArrayList();
                Account.Id accountId = Account.Id.fromRef(this.branch.get());
                if (accountId != null) {
                    try {
                        WatchConfig wc = new WatchConfig(accountId);
                        wc.load(this.rw, receiveEvent.command.getNewId());
                        if (!wc.getValidationErrors().isEmpty()) {
                            CommitValidators.addError("Invalid project configuration:", messages);
                            for (ValidationError err : wc.getValidationErrors()) {
                                CommitValidators.addError("  " + err.getMessage(), messages);
                            }
                            throw new ConfigInvalidException("invalid watch configuration");
                        }
                    }
                    catch (IOException | ConfigInvalidException e) {
                        log.error("User " + this.user.getUserName() + " tried to push an invalid watch configuration " + receiveEvent.command.getNewId().name() + " for account " + accountId.get(), e);
                        throw new CommitValidationException("invalid watch configuration", messages);
                    }
                }
            }
            return Collections.emptyList();
        }
    }

    public static class ChangeIdValidator
    implements CommitValidationListener {
        private static final String CHANGE_ID_PREFIX = FooterConstants.CHANGE_ID.getName() + ":";
        private static final String MISSING_CHANGE_ID_MSG = "missing Change-Id in message footer";
        private static final String MISSING_SUBJECT_MSG = "missing subject; Change-Id must be in message footer";
        private static final String MULTIPLE_CHANGE_ID_MSG = "multiple Change-Id lines in message footer";
        private static final String INVALID_CHANGE_ID_MSG = "invalid Change-Id line format in message footer";
        private static final Pattern CHANGE_ID = Pattern.compile("^[iI][0-9a-f]{4,}.*$");
        private final ProjectState projectState;
        private final String canonicalWebUrl;
        private final String installCommitMsgHookCommand;
        private final SshInfo sshInfo;
        private final IdentifiedUser user;

        public ChangeIdValidator(ProjectState projectState, IdentifiedUser user, String canonicalWebUrl, String installCommitMsgHookCommand, SshInfo sshInfo) {
            this.projectState = projectState;
            this.canonicalWebUrl = canonicalWebUrl;
            this.installCommitMsgHookCommand = installCommitMsgHookCommand;
            this.sshInfo = sshInfo;
            this.user = user;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (!ChangeIdValidator.shouldValidateChangeId(receiveEvent)) {
                return Collections.emptyList();
            }
            RevCommit commit = receiveEvent.commit;
            ArrayList<CommitValidationMessage> messages = new ArrayList<CommitValidationMessage>();
            List<String> idList = commit.getFooterLines(FooterConstants.CHANGE_ID);
            if (idList.isEmpty()) {
                String shortMsg = commit.getShortMessage();
                if (shortMsg.startsWith(CHANGE_ID_PREFIX) && CHANGE_ID.matcher(shortMsg.substring(CHANGE_ID_PREFIX.length()).trim()).matches()) {
                    throw new CommitValidationException(MISSING_SUBJECT_MSG);
                }
                if (this.projectState.isRequireChangeID()) {
                    messages.add(this.getMissingChangeIdErrorMsg(MISSING_CHANGE_ID_MSG, commit));
                    throw new CommitValidationException(MISSING_CHANGE_ID_MSG, messages);
                }
            } else {
                if (idList.size() > 1) {
                    throw new CommitValidationException(MULTIPLE_CHANGE_ID_MSG, messages);
                }
                String v = idList.get(idList.size() - 1).trim();
                if (!CHANGE_ID.matcher(v).matches() || v.matches("^I00*$")) {
                    messages.add(this.getMissingChangeIdErrorMsg(INVALID_CHANGE_ID_MSG, receiveEvent.commit));
                    throw new CommitValidationException(INVALID_CHANGE_ID_MSG, messages);
                }
            }
            return Collections.emptyList();
        }

        private static boolean shouldValidateChangeId(CommitReceivedEvent event) {
            return MagicBranch.isMagicBranch(event.command.getRefName()) || NEW_PATCHSET_PATTERN.matcher(event.command.getRefName()).matches();
        }

        private CommitValidationMessage getMissingChangeIdErrorMsg(String errMsg, RevCommit c) {
            String lastLine;
            StringBuilder sb = new StringBuilder();
            sb.append("ERROR: ").append(errMsg).append("\n");
            boolean hinted = false;
            if (c.getFullMessage().contains(CHANGE_ID_PREFIX) && !(lastLine = Iterables.getLast(Splitter.on('\n').split(c.getFullMessage()), "")).contains(CHANGE_ID_PREFIX)) {
                hinted = true;
                sb.append("\n").append("Hint: run\n").append("  git commit --amend\n").append("and move 'Change-Id: Ixxx..' to the bottom on a separate line\n");
            }
            if (!hinted) {
                sb.append("\nHint: to automatically insert a Change-Id, install the hook:\n").append(this.getCommitMessageHookInstallationHint()).append("\n").append("and then amend the commit:\n").append("  git commit --amend\n");
            }
            return new CommitValidationMessage(sb.toString(), false);
        }

        private String getCommitMessageHookInstallationHint() {
            int sshPort;
            String sshHost;
            if (this.installCommitMsgHookCommand != null) {
                return this.installCommitMsgHookCommand;
            }
            List<HostKey> hostKeys = this.sshInfo.getHostKeys();
            if (hostKeys.isEmpty()) {
                String p = "${gitdir}/hooks/commit-msg";
                return String.format("  gitdir=$(git rev-parse --git-dir); curl -o %s %s/tools/hooks/commit-msg ; chmod +x %s", p, CommitValidators.getGerritUrl(this.canonicalWebUrl), p);
            }
            String host = hostKeys.get(0).getHost();
            int c = host.lastIndexOf(58);
            if (0 <= c) {
                sshHost = host.startsWith("*:") ? CommitValidators.getGerritHost(this.canonicalWebUrl) : host.substring(0, c);
                sshPort = Integer.parseInt(host.substring(c + 1));
            } else {
                sshHost = host;
                sshPort = 22;
            }
            return String.format("  gitdir=$(git rev-parse --git-dir); scp -p -P %d %s@%s:hooks/commit-msg ${gitdir}/hooks/", sshPort, this.user.getUserName(), sshHost);
        }
    }

    @Singleton
    public static class Factory {
        private final PersonIdent gerritIdent;
        private final String canonicalWebUrl;
        private final DynamicSet<CommitValidationListener> pluginValidators;
        private final AllUsersName allUsers;
        private final ExternalIdsConsistencyChecker externalIdsConsistencyChecker;
        private final AccountValidator accountValidator;
        private final String installCommitMsgHookCommand;
        private final ProjectCache projectCache;

        @Inject
        Factory(@GerritPersonIdent PersonIdent gerritIdent, @CanonicalWebUrl @Nullable String canonicalWebUrl, @GerritServerConfig Config cfg, DynamicSet<CommitValidationListener> pluginValidators, AllUsersName allUsers, ExternalIdsConsistencyChecker externalIdsConsistencyChecker, AccountValidator accountValidator, ProjectCache projectCache) {
            this.gerritIdent = gerritIdent;
            this.canonicalWebUrl = canonicalWebUrl;
            this.pluginValidators = pluginValidators;
            this.allUsers = allUsers;
            this.externalIdsConsistencyChecker = externalIdsConsistencyChecker;
            this.accountValidator = accountValidator;
            this.installCommitMsgHookCommand = cfg != null ? cfg.getString("gerrit", null, "installCommitMsgHookCommand") : null;
            this.projectCache = projectCache;
        }

        public CommitValidators forReceiveCommits(PermissionBackend.ForRef perm, Branch.NameKey branch, IdentifiedUser user, SshInfo sshInfo, Repository repo, RevWalk rw) throws IOException {
            return this.forReceiveCommits(perm, branch, user, sshInfo, repo, rw, false);
        }

        public CommitValidators forReceiveCommits(PermissionBackend.ForRef perm, Branch.NameKey branch, IdentifiedUser user, SshInfo sshInfo, Repository repo, RevWalk rw, boolean skipValidation) throws IOException {
            NoteMap rejectCommits = BanCommit.loadRejectCommitsMap(repo, rw);
            ProjectState projectState = this.projectCache.checkedGet(branch.getParentKey());
            return new CommitValidators(ImmutableList.of(new UploadMergesPermissionValidator(perm), new AmendedGerritMergeCommitValidationListener(perm, this.gerritIdent), new AuthorUploaderValidator(user, perm, this.canonicalWebUrl), new CommitterUploaderValidator(user, perm, this.canonicalWebUrl), new SignedOffByValidator(user, perm, projectState), new ChangeIdValidator(projectState, user, this.canonicalWebUrl, this.installCommitMsgHookCommand, sshInfo), new ConfigValidator(branch, user, rw, this.allUsers), new BannedCommitsValidator(rejectCommits), new PluginCommitValidationListener(this.pluginValidators, skipValidation), new ExternalIdUpdateListener(this.allUsers, this.externalIdsConsistencyChecker), new AccountCommitValidator(this.allUsers, this.accountValidator)));
        }

        public CommitValidators forGerritCommits(PermissionBackend.ForRef perm, Branch.NameKey branch, IdentifiedUser user, SshInfo sshInfo, RevWalk rw) throws IOException {
            return new CommitValidators(ImmutableList.of(new UploadMergesPermissionValidator(perm), new AmendedGerritMergeCommitValidationListener(perm, this.gerritIdent), new AuthorUploaderValidator(user, perm, this.canonicalWebUrl), new SignedOffByValidator(user, perm, this.projectCache.checkedGet(branch.getParentKey())), new ChangeIdValidator(this.projectCache.checkedGet(branch.getParentKey()), user, this.canonicalWebUrl, this.installCommitMsgHookCommand, sshInfo), new ConfigValidator(branch, user, rw, this.allUsers), new PluginCommitValidationListener(this.pluginValidators), new ExternalIdUpdateListener(this.allUsers, this.externalIdsConsistencyChecker), new AccountCommitValidator(this.allUsers, this.accountValidator)));
        }

        public CommitValidators forMergedCommits(PermissionBackend.ForRef perm, IdentifiedUser user) {
            return new CommitValidators(ImmutableList.of(new UploadMergesPermissionValidator(perm), new AuthorUploaderValidator(user, perm, this.canonicalWebUrl), new CommitterUploaderValidator(user, perm, this.canonicalWebUrl)));
        }
    }
}

