/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git;

import com.google.gerrit.extensions.events.LifecycleListener;
import com.google.gerrit.lifecycle.LifecycleModule;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.config.SitePaths;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.gerrit.server.git.RepositoryCaseMismatchException;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collections;
import java.util.EnumSet;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.jgit.errors.RepositoryNotFoundException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryCache;
import org.eclipse.jgit.lib.RepositoryCacheConfig;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.storage.file.WindowCacheConfig;
import org.eclipse.jgit.util.FS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class LocalDiskRepositoryManager
implements GitRepositoryManager {
    private static final Logger log = LoggerFactory.getLogger(LocalDiskRepositoryManager.class);
    private final Path basePath;
    private final Lock namesUpdateLock;
    private volatile SortedSet<Project.NameKey> names = new TreeSet<Project.NameKey>();

    @Inject
    LocalDiskRepositoryManager(SitePaths site, @GerritServerConfig Config cfg) {
        this.basePath = site.resolve(cfg.getString("gerrit", null, "basePath"));
        if (this.basePath == null) {
            throw new IllegalStateException("gerrit.basePath must be configured");
        }
        this.namesUpdateLock = new ReentrantLock(true);
    }

    public Path getBasePath(Project.NameKey name) {
        return this.basePath;
    }

    @Override
    public Repository openRepository(Project.NameKey name) throws RepositoryNotFoundException {
        return this.openRepository(this.getBasePath(name), name);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private Repository openRepository(Path path, Project.NameKey name) throws RepositoryNotFoundException {
        if (this.isUnreasonableName(name)) {
            throw new RepositoryNotFoundException("Invalid name: " + name);
        }
        File gitDir = path.resolve(name.get()).toFile();
        if (!this.names.contains(name)) {
            if (!name.get().endsWith(".git")) {
                if (RepositoryCache.FileKey.resolve(gitDir, FS.DETECTED) == null) throw new RepositoryNotFoundException(gitDir);
                this.onCreateProject(name);
            } else {
                File directory = gitDir;
                if (RepositoryCache.FileKey.isGitRepository(new File(directory, ".git"), FS.DETECTED)) {
                    this.onCreateProject(name);
                } else {
                    if (!RepositoryCache.FileKey.isGitRepository(new File(directory.getParentFile(), directory.getName() + ".git"), FS.DETECTED)) throw new RepositoryNotFoundException(gitDir);
                    this.onCreateProject(name);
                }
            }
        }
        RepositoryCache.FileKey loc = RepositoryCache.FileKey.lenient(gitDir, FS.DETECTED);
        try {
            return RepositoryCache.open(loc);
        }
        catch (IOException e1) {
            RepositoryNotFoundException e2 = new RepositoryNotFoundException("Cannot open repository " + name);
            e2.initCause(e1);
            throw e2;
        }
    }

    @Override
    public Repository createRepository(Project.NameKey name) throws RepositoryNotFoundException, RepositoryCaseMismatchException, IOException {
        RepositoryCache.FileKey loc;
        Path path = this.getBasePath(name);
        if (this.isUnreasonableName(name)) {
            throw new RepositoryNotFoundException("Invalid name: " + name);
        }
        File dir = RepositoryCache.FileKey.resolve(path.resolve(name.get()).toFile(), FS.DETECTED);
        if (dir != null) {
            Project.NameKey onDiskName = LocalDiskRepositoryManager.getProjectName(path, dir.getCanonicalFile().toPath());
            this.onCreateProject(onDiskName);
            loc = RepositoryCache.FileKey.exact(dir, FS.DETECTED);
            if (!this.names.contains(name)) {
                throw new RepositoryCaseMismatchException(name);
            }
        } else {
            String n = name.get() + ".git";
            loc = RepositoryCache.FileKey.exact(path.resolve(n).toFile(), FS.DETECTED);
        }
        try {
            Repository db = RepositoryCache.open(loc, false);
            db.create(true);
            StoredConfig config = db.getConfig();
            config.setBoolean("core", null, "logallrefupdates", true);
            config.save();
            File metaConfigLog = new File(db.getDirectory(), "logs/refs/meta/config");
            if (!metaConfigLog.getParentFile().mkdirs() || !metaConfigLog.createNewFile()) {
                log.error("Failed to create ref log for {} in repository {}", (Object)"refs/meta/config", (Object)name);
            }
            this.onCreateProject(name);
            return db;
        }
        catch (IOException e1) {
            RepositoryNotFoundException e2 = new RepositoryNotFoundException("Cannot create repository " + name);
            e2.initCause(e1);
            throw e2;
        }
    }

    private void onCreateProject(Project.NameKey newProjectName) {
        this.namesUpdateLock.lock();
        try {
            TreeSet<Project.NameKey> n = new TreeSet<Project.NameKey>(this.names);
            n.add(newProjectName);
            this.names = Collections.unmodifiableSortedSet(n);
        }
        finally {
            this.namesUpdateLock.unlock();
        }
    }

    private boolean isUnreasonableName(Project.NameKey nameKey) {
        String name = nameKey.get();
        return name.length() == 0 || name.charAt(name.length() - 1) == '/' || name.indexOf(92) >= 0 || name.charAt(0) == '/' || new File(name).isAbsolute() || name.startsWith("../") || name.contains("/../") || name.contains("/./") || name.contains("//") || name.contains(".git/") || name.contains("?") || name.contains("%") || name.contains("*") || name.contains(":") || name.contains("<") || name.contains(">") || name.contains("|") || name.contains("$") || name.contains("\r") || name.contains("/+") || name.contains("~");
    }

    @Override
    public SortedSet<Project.NameKey> list() {
        ProjectVisitor visitor = new ProjectVisitor(this.basePath);
        this.scanProjects(visitor);
        this.namesUpdateLock.lock();
        try {
            this.names = Collections.unmodifiableSortedSet(visitor.found);
        }
        finally {
            this.namesUpdateLock.unlock();
        }
        return this.names;
    }

    protected void scanProjects(ProjectVisitor visitor) {
        try {
            Files.walkFileTree(visitor.startFolder, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, visitor);
        }
        catch (IOException e) {
            log.error("Error walking repository tree {}", (Object)visitor.startFolder.toAbsolutePath(), (Object)e);
        }
    }

    private static Project.NameKey getProjectName(Path startFolder, Path p) {
        String projectName = startFolder.relativize(p).toString();
        if (File.separatorChar != '/') {
            projectName = projectName.replace(File.separatorChar, '/');
        }
        if (projectName.endsWith(".git")) {
            int newLen = projectName.length() - ".git".length();
            projectName = projectName.substring(0, newLen);
        }
        return new Project.NameKey(projectName);
    }

    protected class ProjectVisitor
    extends SimpleFileVisitor<Path> {
        private final SortedSet<Project.NameKey> found = new TreeSet<Project.NameKey>();
        private Path startFolder;

        public ProjectVisitor(Path startFolder) {
            this.setStartFolder(startFolder);
        }

        public void setStartFolder(Path startFolder) {
            this.startFolder = startFolder;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (!dir.equals(this.startFolder) && this.isRepo(dir)) {
                this.addProject(dir);
                return FileVisitResult.SKIP_SUBTREE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException e) {
            log.warn(e.getMessage());
            return FileVisitResult.CONTINUE;
        }

        private boolean isRepo(Path p) {
            String name = p.getFileName().toString();
            return !name.equals(".git") && (name.endsWith(".git") || RepositoryCache.FileKey.isGitRepository(p.toFile(), FS.DETECTED));
        }

        private void addProject(Path p) {
            Project.NameKey nameKey = LocalDiskRepositoryManager.getProjectName(this.startFolder, p);
            if (LocalDiskRepositoryManager.this.getBasePath(nameKey).equals(this.startFolder)) {
                if (LocalDiskRepositoryManager.this.isUnreasonableName(nameKey)) {
                    log.warn("Ignoring unreasonably named repository {}", (Object)p.toAbsolutePath());
                } else {
                    this.found.add(nameKey);
                }
            }
        }
    }

    public static class Lifecycle
    implements LifecycleListener {
        private final Config serverConfig;

        @Inject
        Lifecycle(@GerritServerConfig Config cfg) {
            this.serverConfig = cfg;
        }

        @Override
        public void start() {
            RepositoryCacheConfig repoCacheCfg = new RepositoryCacheConfig();
            repoCacheCfg.fromConfig(this.serverConfig);
            repoCacheCfg.install();
            WindowCacheConfig cfg = new WindowCacheConfig();
            cfg.fromConfig(this.serverConfig);
            if (this.serverConfig.getString("core", null, "streamFileThreshold") == null) {
                long mx = Runtime.getRuntime().maxMemory();
                int limit = (int)Math.min(mx / 4L, 0x7FF00000L);
                if (0x500000 < limit && limit % 0x100000 != 0) {
                    limit = limit / 0x100000 + 1 << 20;
                }
                String desc = limit % 0x100000 == 0 ? String.format("%dm", limit / 0x100000) : (limit % 1024 == 0 ? String.format("%dk", limit / 1024) : String.format("%d", limit));
                log.info("Defaulting core.streamFileThreshold to {}", (Object)desc);
                cfg.setStreamFileThreshold(limit);
            }
            cfg.install();
        }

        @Override
        public void stop() {
        }
    }

    public static class Module
    extends LifecycleModule {
        @Override
        protected void configure() {
            this.listener().to(Lifecycle.class);
        }
    }
}

