/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.mail.receive;

import com.google.common.base.Strings;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.gerrit.common.TimeUtil;
import com.google.gerrit.extensions.api.changes.NotifyHandling;
import com.google.gerrit.extensions.client.Side;
import com.google.gerrit.extensions.registration.DynamicMap;
import com.google.gerrit.extensions.restapi.RestApiException;
import com.google.gerrit.extensions.restapi.UnprocessableEntityException;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.Change;
import com.google.gerrit.reviewdb.client.ChangeMessage;
import com.google.gerrit.reviewdb.client.Comment;
import com.google.gerrit.reviewdb.client.PatchLineComment;
import com.google.gerrit.reviewdb.client.PatchSet;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.server.ApprovalsUtil;
import com.google.gerrit.server.ChangeMessagesUtil;
import com.google.gerrit.server.CommentsUtil;
import com.google.gerrit.server.PatchSetUtil;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.account.Emails;
import com.google.gerrit.server.change.EmailReviewComments;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.extensions.events.CommentAdded;
import com.google.gerrit.server.mail.MailFilter;
import com.google.gerrit.server.mail.receive.HtmlParser;
import com.google.gerrit.server.mail.receive.MailComment;
import com.google.gerrit.server.mail.receive.MailMessage;
import com.google.gerrit.server.mail.receive.MailMetadata;
import com.google.gerrit.server.mail.receive.MetadataParser;
import com.google.gerrit.server.mail.receive.TextParser;
import com.google.gerrit.server.notedb.ChangeNotes;
import com.google.gerrit.server.patch.PatchListCache;
import com.google.gerrit.server.patch.PatchListNotAvailableException;
import com.google.gerrit.server.query.change.ChangeData;
import com.google.gerrit.server.query.change.InternalChangeQuery;
import com.google.gerrit.server.update.BatchUpdate;
import com.google.gerrit.server.update.BatchUpdateOp;
import com.google.gerrit.server.update.ChangeContext;
import com.google.gerrit.server.update.Context;
import com.google.gerrit.server.update.RetryHelper;
import com.google.gerrit.server.update.UpdateException;
import com.google.gerrit.server.util.ManualRequestContext;
import com.google.gerrit.server.util.OneOffRequestContext;
import com.google.gwtorm.server.OrmException;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class MailProcessor {
    private static final Logger log = LoggerFactory.getLogger(MailProcessor.class);
    private final Emails emails;
    private final RetryHelper retryHelper;
    private final ChangeMessagesUtil changeMessagesUtil;
    private final CommentsUtil commentsUtil;
    private final OneOffRequestContext oneOffRequestContext;
    private final PatchListCache patchListCache;
    private final PatchSetUtil psUtil;
    private final Provider<InternalChangeQuery> queryProvider;
    private final DynamicMap<MailFilter> mailFilters;
    private final EmailReviewComments.Factory outgoingMailFactory;
    private final CommentAdded commentAdded;
    private final ApprovalsUtil approvalsUtil;
    private final AccountCache accountCache;
    private final Provider<String> canonicalUrl;

    @Inject
    public MailProcessor(Emails emails, RetryHelper retryHelper, ChangeMessagesUtil changeMessagesUtil, CommentsUtil commentsUtil, OneOffRequestContext oneOffRequestContext, PatchListCache patchListCache, PatchSetUtil psUtil, Provider<InternalChangeQuery> queryProvider, DynamicMap<MailFilter> mailFilters, EmailReviewComments.Factory outgoingMailFactory, ApprovalsUtil approvalsUtil, CommentAdded commentAdded, AccountCache accountCache, @CanonicalWebUrl Provider<String> canonicalUrl) {
        this.emails = emails;
        this.retryHelper = retryHelper;
        this.changeMessagesUtil = changeMessagesUtil;
        this.commentsUtil = commentsUtil;
        this.oneOffRequestContext = oneOffRequestContext;
        this.patchListCache = patchListCache;
        this.psUtil = psUtil;
        this.queryProvider = queryProvider;
        this.mailFilters = mailFilters;
        this.outgoingMailFactory = outgoingMailFactory;
        this.commentAdded = commentAdded;
        this.approvalsUtil = approvalsUtil;
        this.accountCache = accountCache;
        this.canonicalUrl = canonicalUrl;
    }

    public void process(MailMessage message) throws RestApiException, UpdateException {
        this.retryHelper.execute(buf -> {
            this.processImpl(buf, message);
            return null;
        });
    }

    private void processImpl(BatchUpdate.Factory buf, MailMessage message) throws OrmException, UpdateException, RestApiException, IOException {
        for (DynamicMap.Entry<MailFilter> entry : this.mailFilters) {
            if (entry.getProvider().get().shouldProcessMessage(message)) continue;
            log.warn("Message {} filtered by plugin {} {}. Will delete message.", message.id(), entry.getPluginName(), entry.getExportName());
            return;
        }
        MailMetadata metadata = MetadataParser.parse(message);
        if (!metadata.hasRequiredFields()) {
            log.error("Message {} is missing required metadata, have {}. Will delete message.", (Object)message.id(), (Object)metadata);
            return;
        }
        ImmutableSet<Account.Id> immutableSet = this.emails.getAccountFor(metadata.author);
        if (immutableSet.size() != 1) {
            log.error("Address {} could not be matched to a unique account. It was matched to {}. Will delete message.", (Object)metadata.author, (Object)immutableSet);
            return;
        }
        Account.Id account = (Account.Id)immutableSet.iterator().next();
        if (!this.accountCache.get(account).getAccount().isActive()) {
            log.warn("Mail: Account {} is inactive. Will delete message.", (Object)account);
            return;
        }
        this.persistComments(buf, message, metadata, account);
    }

    private void persistComments(BatchUpdate.Factory buf, MailMessage message, MailMetadata metadata, Account.Id sender) throws OrmException, UpdateException, RestApiException {
        try (ManualRequestContext ctx = this.oneOffRequestContext.openAs(sender);){
            List<ChangeData> changeDataList = this.queryProvider.get().byLegacyChangeId(new Change.Id(metadata.changeNumber));
            if (changeDataList.size() != 1) {
                log.error("Message {} references unique change {}, but there are {} matching changes in the index. Will delete message.", message.id(), metadata.changeNumber, changeDataList.size());
                return;
            }
            ChangeData cd = changeDataList.get(0);
            if (this.existingMessageIds(cd).contains(message.id())) {
                log.info("Message {} was already processed. Will delete message.", (Object)message.id());
                return;
            }
            Collection comments = cd.publishedComments().stream().filter(c -> c.writtenOn.getTime() / 1000L == metadata.timestamp.getTime() / 1000L).sorted(CommentsUtil.COMMENT_ORDER).collect(Collectors.toList());
            Project.NameKey project = cd.project();
            String changeUrl = this.canonicalUrl.get() + "#/c/" + cd.getId().get();
            List<MailComment> parsedComments = MailProcessor.useHtmlParser(message) ? HtmlParser.parse(message, comments, changeUrl) : TextParser.parse(message, comments, changeUrl);
            if (parsedComments.isEmpty()) {
                log.warn("Could not parse any comments from {}. Will delete message.", (Object)message.id());
                return;
            }
            Op o = new Op(new PatchSet.Id(cd.getId(), metadata.patchSet), parsedComments, message.id());
            BatchUpdate batchUpdate = buf.create(cd.db(), project, ctx.getUser(), TimeUtil.nowTs());
            batchUpdate.addOp(cd.getId(), o);
            batchUpdate.execute();
        }
    }

    private static boolean useHtmlParser(MailMessage m) {
        return !Strings.isNullOrEmpty(m.htmlContent());
    }

    private static String numComments(int numComments) {
        return "(" + numComments + (numComments > 1 ? " comments)" : " comment)");
    }

    private Set<String> existingMessageIds(ChangeData cd) throws OrmException {
        HashSet<String> existingMessageIds = new HashSet<String>();
        cd.messages().stream().forEach(m -> {
            String messageId = CommentsUtil.extractMessageId(m.getTag());
            if (messageId != null) {
                existingMessageIds.add(messageId);
            }
        });
        cd.publishedComments().stream().forEach(c -> {
            String messageId = CommentsUtil.extractMessageId(c.tag);
            if (messageId != null) {
                existingMessageIds.add(messageId);
            }
        });
        return existingMessageIds;
    }

    private class Op
    implements BatchUpdateOp {
        private final PatchSet.Id psId;
        private final List<MailComment> parsedComments;
        private final String tag;
        private ChangeMessage changeMessage;
        private List<Comment> comments;
        private PatchSet patchSet;
        private ChangeNotes notes;

        private Op(PatchSet.Id psId, List<MailComment> parsedComments, String messageId) {
            this.psId = psId;
            this.parsedComments = parsedComments;
            this.tag = "mailMessageId=" + messageId;
        }

        @Override
        public boolean updateChange(ChangeContext ctx) throws OrmException, UnprocessableEntityException, PatchListNotAvailableException {
            this.patchSet = MailProcessor.this.psUtil.get(ctx.getDb(), ctx.getNotes(), this.psId);
            this.notes = ctx.getNotes();
            if (this.patchSet == null) {
                throw new OrmException("patch set not found: " + this.psId);
            }
            this.changeMessage = this.generateChangeMessage(ctx);
            MailProcessor.this.changeMessagesUtil.addChangeMessage(ctx.getDb(), ctx.getUpdate(this.psId), this.changeMessage);
            this.comments = new ArrayList<Comment>();
            for (MailComment c : this.parsedComments) {
                if (c.type == MailComment.CommentType.CHANGE_MESSAGE) continue;
                this.comments.add(this.persistentCommentFromMailComment(ctx, c, this.targetPatchSetForComment(ctx, c, this.patchSet)));
            }
            MailProcessor.this.commentsUtil.putComments(ctx.getDb(), ctx.getUpdate(ctx.getChange().currentPatchSetId()), PatchLineComment.Status.PUBLISHED, this.comments);
            return true;
        }

        @Override
        public void postUpdate(Context ctx) throws Exception {
            String patchSetComment = null;
            if (this.parsedComments.get((int)0).type == MailComment.CommentType.CHANGE_MESSAGE) {
                patchSetComment = this.parsedComments.get((int)0).message;
            }
            MailProcessor.this.outgoingMailFactory.create(NotifyHandling.ALL, ArrayListMultimap.create(), this.notes, this.patchSet, ctx.getUser().asIdentifiedUser(), this.changeMessage, this.comments, patchSetComment, ImmutableList.of()).sendAsync();
            HashMap<String, Short> approvals = new HashMap<String, Short>();
            MailProcessor.this.approvalsUtil.byPatchSetUser(ctx.getDb(), this.notes, ctx.getUser(), this.psId, ctx.getAccountId(), ctx.getRevWalk(), ctx.getRepoView().getConfig()).forEach(a -> approvals.put(a.getLabel(), a.getValue()));
            MailProcessor.this.commentAdded.fire(this.notes.getChange(), this.patchSet, ctx.getAccount(), this.changeMessage.getMessage(), approvals, approvals, ctx.getWhen());
        }

        private ChangeMessage generateChangeMessage(ChangeContext ctx) {
            String changeMsg = "Patch Set " + this.psId.get() + ":";
            if (this.parsedComments.get((int)0).type == MailComment.CommentType.CHANGE_MESSAGE) {
                if (this.parsedComments.size() > 1) {
                    changeMsg = changeMsg + "\n\n" + MailProcessor.numComments(this.parsedComments.size() - 1);
                }
                changeMsg = changeMsg + "\n\n" + this.parsedComments.get((int)0).message;
            } else {
                changeMsg = changeMsg + "\n\n" + MailProcessor.numComments(this.parsedComments.size());
            }
            return ChangeMessagesUtil.newMessage(ctx, changeMsg, this.tag);
        }

        private PatchSet targetPatchSetForComment(ChangeContext ctx, MailComment mailComment, PatchSet current) throws OrmException {
            if (mailComment.inReplyTo != null) {
                return MailProcessor.this.psUtil.get(ctx.getDb(), ctx.getNotes(), new PatchSet.Id(ctx.getChange().getId(), mailComment.inReplyTo.key.patchSetId));
            }
            return current;
        }

        private Comment persistentCommentFromMailComment(ChangeContext ctx, MailComment mailComment, PatchSet patchSetForComment) throws OrmException, UnprocessableEntityException, PatchListNotAvailableException {
            Side side;
            String fileName;
            if (mailComment.inReplyTo != null) {
                fileName = mailComment.inReplyTo.key.filename;
                side = Side.fromShort(mailComment.inReplyTo.side);
            } else {
                fileName = mailComment.fileName;
                side = Side.REVISION;
            }
            Comment comment = MailProcessor.this.commentsUtil.newComment(ctx, fileName, patchSetForComment.getId(), (short)side.ordinal(), mailComment.message, false, null);
            comment.tag = this.tag;
            if (mailComment.inReplyTo != null) {
                comment.parentUuid = mailComment.inReplyTo.key.uuid;
                comment.lineNbr = mailComment.inReplyTo.lineNbr;
                comment.range = mailComment.inReplyTo.range;
                comment.unresolved = mailComment.inReplyTo.unresolved;
            }
            CommentsUtil.setCommentRevId(comment, MailProcessor.this.patchListCache, ctx.getChange(), patchSetForComment);
            return comment;
        }
    }
}

