/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.account.externalids;

import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.google.gerrit.extensions.api.config.ConsistencyCheckInfo;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.account.HashedPassword;
import com.google.gerrit.server.account.externalids.ExternalId;
import com.google.gerrit.server.account.externalids.ExternalIdReader;
import com.google.gerrit.server.config.AllUsersName;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.gerrit.server.mail.send.OutgoingEmailValidator;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.codec.DecoderException;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.notes.Note;
import org.eclipse.jgit.notes.NoteMap;
import org.eclipse.jgit.revwalk.RevWalk;

@Singleton
public class ExternalIdsConsistencyChecker {
    private final GitRepositoryManager repoManager;
    private final AllUsersName allUsers;
    private final AccountCache accountCache;
    private final OutgoingEmailValidator validator;

    @Inject
    ExternalIdsConsistencyChecker(GitRepositoryManager repoManager, AllUsersName allUsers, AccountCache accountCache, OutgoingEmailValidator validator) {
        this.repoManager = repoManager;
        this.allUsers = allUsers;
        this.accountCache = accountCache;
        this.validator = validator;
    }

    public List<ConsistencyCheckInfo.ConsistencyProblemInfo> check() throws IOException {
        try (Repository repo = this.repoManager.openRepository(this.allUsers);){
            List<ConsistencyCheckInfo.ConsistencyProblemInfo> list = this.check(repo, ExternalIdReader.readRevision(repo));
            return list;
        }
    }

    public List<ConsistencyCheckInfo.ConsistencyProblemInfo> check(ObjectId rev) throws IOException {
        try (Repository repo = this.repoManager.openRepository(this.allUsers);){
            List<ConsistencyCheckInfo.ConsistencyProblemInfo> list = this.check(repo, rev);
            return list;
        }
    }

    private List<ConsistencyCheckInfo.ConsistencyProblemInfo> check(Repository repo, ObjectId commit) throws IOException {
        ArrayList<ConsistencyCheckInfo.ConsistencyProblemInfo> problems = new ArrayList<ConsistencyCheckInfo.ConsistencyProblemInfo>();
        Multimap emails = MultimapBuilder.hashKeys().arrayListValues().build();
        try (RevWalk rw = new RevWalk(repo);){
            NoteMap noteMap = ExternalIdReader.readNoteMap(rw, commit);
            for (Note note : noteMap) {
                byte[] raw = rw.getObjectReader().open(note.getData(), 3).getCachedBytes(524288);
                try {
                    ExternalId extId = ExternalId.parse(note.getName(), raw, note.getData());
                    problems.addAll(this.validateExternalId(extId));
                    if (extId.email() == null) continue;
                    emails.put(extId.email(), extId.key());
                }
                catch (ConfigInvalidException e2) {
                    ExternalIdsConsistencyChecker.addError(String.format(e2.getMessage(), new Object[0]), problems);
                }
            }
        }
        emails.asMap().entrySet().stream().filter(e -> ((Collection)e.getValue()).size() > 1).forEach(e -> ExternalIdsConsistencyChecker.addError(String.format("Email '%s' is not unique, it's used by the following external IDs: %s", e.getKey(), ((Collection)e.getValue()).stream().map(k -> "'" + k.get() + "'").sorted().collect(Collectors.joining(", "))), problems));
        return problems;
    }

    private List<ConsistencyCheckInfo.ConsistencyProblemInfo> validateExternalId(ExternalId extId) {
        ArrayList<ConsistencyCheckInfo.ConsistencyProblemInfo> problems = new ArrayList<ConsistencyCheckInfo.ConsistencyProblemInfo>();
        if (this.accountCache.getOrNull(extId.accountId()) == null) {
            ExternalIdsConsistencyChecker.addError(String.format("External ID '%s' belongs to account that doesn't exist: %s", extId.key().get(), extId.accountId().get()), problems);
        }
        if (extId.email() != null && !this.validator.isValid(extId.email())) {
            ExternalIdsConsistencyChecker.addError(String.format("External ID '%s' has an invalid email: %s", extId.key().get(), extId.email()), problems);
        }
        if (extId.password() != null && extId.isScheme("username")) {
            try {
                HashedPassword.decode(extId.password());
            }
            catch (DecoderException e) {
                ExternalIdsConsistencyChecker.addError(String.format("External ID '%s' has an invalid password: %s", extId.key().get(), e.getMessage()), problems);
            }
        }
        return problems;
    }

    private static void addError(String error, List<ConsistencyCheckInfo.ConsistencyProblemInfo> problems) {
        problems.add(new ConsistencyCheckInfo.ConsistencyProblemInfo(ConsistencyCheckInfo.ConsistencyProblemInfo.Status.ERROR, error));
    }
}

