/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.account;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.gerrit.common.data.Permission;
import com.google.gerrit.common.errors.NameAlreadyUsedException;
import com.google.gerrit.common.errors.NoSuchGroupException;
import com.google.gerrit.extensions.client.AccountFieldName;
import com.google.gerrit.reviewdb.client.Account;
import com.google.gerrit.reviewdb.client.AccountGroup;
import com.google.gerrit.reviewdb.server.ReviewDb;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.Sequences;
import com.google.gerrit.server.account.AccountCache;
import com.google.gerrit.server.account.AccountException;
import com.google.gerrit.server.account.AccountUserNameException;
import com.google.gerrit.server.account.Accounts;
import com.google.gerrit.server.account.AccountsUpdate;
import com.google.gerrit.server.account.AuthRequest;
import com.google.gerrit.server.account.AuthResult;
import com.google.gerrit.server.account.ChangeUserName;
import com.google.gerrit.server.account.InvalidUserNameException;
import com.google.gerrit.server.account.Realm;
import com.google.gerrit.server.account.SetInactiveFlag;
import com.google.gerrit.server.account.externalids.ExternalId;
import com.google.gerrit.server.account.externalids.ExternalIds;
import com.google.gerrit.server.account.externalids.ExternalIdsUpdate;
import com.google.gerrit.server.auth.NoSuchUserException;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.group.GroupsUpdate;
import com.google.gerrit.server.project.ProjectCache;
import com.google.gwtorm.server.OrmException;
import com.google.gwtorm.server.SchemaFactory;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class AccountManager {
    private static final Logger log = LoggerFactory.getLogger(AccountManager.class);
    private final SchemaFactory<ReviewDb> schema;
    private final Sequences sequences;
    private final Accounts accounts;
    private final AccountsUpdate.Server accountsUpdateFactory;
    private final AccountCache byIdCache;
    private final Realm realm;
    private final IdentifiedUser.GenericFactory userFactory;
    private final ChangeUserName.Factory changeUserNameFactory;
    private final ProjectCache projectCache;
    private final AtomicBoolean awaitsFirstAccountCheck;
    private final ExternalIds externalIds;
    private final ExternalIdsUpdate.Server externalIdsUpdateFactory;
    private final GroupsUpdate.Factory groupsUpdateFactory;
    private final boolean autoUpdateAccountActiveStatus;
    private final SetInactiveFlag setInactiveFlag;

    @Inject
    AccountManager(SchemaFactory<ReviewDb> schema, Sequences sequences, @GerritServerConfig Config cfg, Accounts accounts, AccountsUpdate.Server accountsUpdateFactory, AccountCache byIdCache, Realm accountMapper, IdentifiedUser.GenericFactory userFactory, ChangeUserName.Factory changeUserNameFactory, ProjectCache projectCache, ExternalIds externalIds, ExternalIdsUpdate.Server externalIdsUpdateFactory, GroupsUpdate.Factory groupsUpdateFactory, SetInactiveFlag setInactiveFlag) {
        this.schema = schema;
        this.sequences = sequences;
        this.accounts = accounts;
        this.accountsUpdateFactory = accountsUpdateFactory;
        this.byIdCache = byIdCache;
        this.realm = accountMapper;
        this.userFactory = userFactory;
        this.changeUserNameFactory = changeUserNameFactory;
        this.projectCache = projectCache;
        this.awaitsFirstAccountCheck = new AtomicBoolean(cfg.getBoolean("capability", "makeFirstUserAdmin", true));
        this.externalIds = externalIds;
        this.externalIdsUpdateFactory = externalIdsUpdateFactory;
        this.groupsUpdateFactory = groupsUpdateFactory;
        this.autoUpdateAccountActiveStatus = cfg.getBoolean("auth", "autoUpdateAccountActiveStatus", false);
        this.setInactiveFlag = setInactiveFlag;
    }

    public Optional<Account.Id> lookup(String externalId) throws AccountException {
        try {
            ExternalId extId = this.externalIds.get(ExternalId.Key.parse(externalId));
            return extId != null ? Optional.of(extId.accountId()) : Optional.empty();
        }
        catch (IOException | ConfigInvalidException e) {
            throw new AccountException("Cannot lookup account " + externalId, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public AuthResult authenticate(AuthRequest who) throws AccountException, IOException {
        try {
            who = this.realm.authenticate(who);
        }
        catch (NoSuchUserException e) {
            this.deactivateAccountIfItExists(who);
            throw e;
        }
        try (ReviewDb db = this.schema.open();){
            ExternalId id = this.externalIds.get(who.getExternalIdKey());
            if (id == null) {
                AuthResult authResult2 = this.create(db, who);
                return authResult2;
            }
            Account act = this.updateAccountActiveStatus(who, this.byIdCache.get(id.accountId()).getAccount());
            if (!act.isActive()) {
                throw new AccountException("Authentication error, account inactive");
            }
            this.update(who, id);
            AuthResult authResult = new AuthResult(id.accountId(), who.getExternalIdKey(), false);
            return authResult;
        }
        catch (OrmException | ConfigInvalidException e) {
            throw new AccountException("Authentication error", e);
        }
    }

    private void deactivateAccountIfItExists(AuthRequest authRequest) {
        if (!this.shouldUpdateActiveStatus(authRequest)) {
            return;
        }
        try {
            ExternalId id = this.externalIds.get(authRequest.getExternalIdKey());
            if (id == null) {
                return;
            }
            this.setInactiveFlag.deactivate(id.accountId());
        }
        catch (Exception e) {
            log.error("Unable to deactivate account " + authRequest.getUserName(), e);
        }
    }

    private Account updateAccountActiveStatus(AuthRequest authRequest, Account account) throws AccountException {
        if (!this.shouldUpdateActiveStatus(authRequest) || authRequest.isActive() == account.isActive()) {
            return account;
        }
        if (authRequest.isActive()) {
            try {
                this.setInactiveFlag.activate(account.getId());
            }
            catch (Exception e) {
                throw new AccountException("Unable to activate account " + account.getId(), e);
            }
        }
        try {
            this.setInactiveFlag.deactivate(account.getId());
        }
        catch (Exception e) {
            throw new AccountException("Unable to deactivate account " + account.getId(), e);
        }
        return this.byIdCache.get(account.getId()).getAccount();
    }

    private boolean shouldUpdateActiveStatus(AuthRequest authRequest) {
        return this.autoUpdateAccountActiveStatus && authRequest.authProvidesAccountActiveStatus();
    }

    private void update(AuthRequest who, ExternalId extId) throws OrmException, IOException, ConfigInvalidException {
        Account account;
        IdentifiedUser user = this.userFactory.create(extId.accountId());
        ArrayList<Consumer<Account>> accountUpdates = new ArrayList<Consumer<Account>>();
        String newEmail = who.getEmailAddress();
        String oldEmail = extId.email();
        if (newEmail != null && !newEmail.equals(oldEmail)) {
            if (oldEmail != null && oldEmail.equals(user.getAccount().getPreferredEmail())) {
                accountUpdates.add(a -> a.setPreferredEmail(newEmail));
            }
            this.externalIdsUpdateFactory.create().replace(extId, ExternalId.create(extId.key(), extId.accountId(), newEmail, extId.password()));
        }
        if (!Strings.isNullOrEmpty(who.getDisplayName()) && !AccountManager.eq(user.getAccount().getFullName(), who.getDisplayName())) {
            if (this.realm.allowsEdit(AccountFieldName.FULL_NAME)) {
                accountUpdates.add(a -> a.setFullName(who.getDisplayName()));
            } else {
                log.warn("Not changing already set display name '{}' to '{}'", (Object)user.getAccount().getFullName(), (Object)who.getDisplayName());
            }
        }
        if (!this.realm.allowsEdit(AccountFieldName.USER_NAME) && who.getUserName() != null && !AccountManager.eq(user.getUserName(), who.getUserName())) {
            log.warn("Not changing already set username {} to {}", (Object)user.getUserName(), (Object)who.getUserName());
        }
        if (!accountUpdates.isEmpty() && (account = this.accountsUpdateFactory.create().update(user.getAccountId(), accountUpdates)) == null) {
            throw new OrmException("Account " + user.getAccountId() + " has been deleted");
        }
    }

    private static boolean eq(String a, String b) {
        return a == null && b == null || a != null && a.equals(b);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private AuthResult create(ReviewDb db, AuthRequest who) throws OrmException, AccountException, IOException, ConfigInvalidException {
        Account account;
        Account.Id newId = new Account.Id(this.sequences.nextAccountId());
        log.debug("Assigning new Id {} to account", (Object)newId);
        ExternalId extId = ExternalId.createWithEmail(who.getExternalIdKey(), newId, who.getEmailAddress());
        log.debug("Created external Id: {}", (Object)extId);
        boolean isFirstAccount = this.awaitsFirstAccountCheck.getAndSet(false) && !this.accounts.hasAnyAccount();
        try {
            AccountsUpdate accountsUpdate = this.accountsUpdateFactory.create();
            account = accountsUpdate.insert(newId, a -> {
                a.setFullName(who.getDisplayName());
                a.setPreferredEmail(extId.email());
            });
            ExternalId existingExtId = this.externalIds.get(extId.key());
            if (existingExtId != null && !existingExtId.accountId().equals(extId.accountId())) {
                accountsUpdate.delete(account);
                throw new AccountException("Cannot assign external ID \"" + extId.key().get() + "\" to account " + newId + "; external ID already in use.");
            }
            this.externalIdsUpdateFactory.create().upsert(extId);
        }
        finally {
            this.awaitsFirstAccountCheck.set(isFirstAccount);
        }
        IdentifiedUser user = this.userFactory.create(newId);
        if (isFirstAccount) {
            Permission admin = this.projectCache.getAllProjects().getConfig().getAccessSection("GLOBAL_CAPABILITIES").getPermission("administrateServer");
            AccountGroup.UUID uuid = admin.getRules().get(0).getGroup().getUUID();
            GroupsUpdate groupsUpdate = this.groupsUpdateFactory.create(user);
            try {
                groupsUpdate.addGroupMember(db, uuid, newId);
            }
            catch (NoSuchGroupException e) {
                throw new AccountException(String.format("Group %s not found", uuid));
            }
        }
        log.debug("Username from AuthRequest: {}", (Object)who.getUserName());
        if (who.getUserName() != null) {
            String message;
            log.debug("Setting username for: {}", (Object)who.getUserName());
            try {
                this.changeUserNameFactory.create(user, who.getUserName()).call();
                log.debug("Identified user {} was created from {}", (Object)user, (Object)who.getUserName());
            }
            catch (NameAlreadyUsedException e) {
                message = "Cannot assign user name \"" + who.getUserName() + "\" to account " + newId + "; name already in use.";
                this.handleSettingUserNameFailure(account, extId, message, e, false);
            }
            catch (InvalidUserNameException e) {
                message = "Cannot assign user name \"" + who.getUserName() + "\" to account " + newId + "; name does not conform.";
                this.handleSettingUserNameFailure(account, extId, message, e, false);
            }
            catch (OrmException e) {
                message = "Cannot assign user name";
                this.handleSettingUserNameFailure(account, extId, message, e, true);
            }
        }
        this.realm.onCreateAccount(who, account);
        return new AuthResult(newId, extId.key(), true);
    }

    private void handleSettingUserNameFailure(Account account, ExternalId extId, String errorMessage, Exception e, boolean logException) throws AccountUserNameException, OrmException, IOException, ConfigInvalidException {
        if (logException) {
            log.error(errorMessage, e);
        } else {
            log.error(errorMessage);
        }
        if (!this.realm.allowsEdit(AccountFieldName.USER_NAME)) {
            this.accountsUpdateFactory.create().delete(account);
            this.externalIdsUpdateFactory.create().delete(extId);
            throw new AccountUserNameException(errorMessage, e);
        }
    }

    public AuthResult link(Account.Id to, AuthRequest who) throws AccountException, OrmException, IOException, ConfigInvalidException {
        ExternalId extId = this.externalIds.get(who.getExternalIdKey());
        if (extId != null) {
            if (!extId.accountId().equals(to)) {
                throw new AccountException("Identity '" + extId.key().get() + "' in use by another account");
            }
            this.update(who, extId);
        } else {
            this.externalIdsUpdateFactory.create().insert(ExternalId.createWithEmail(who.getExternalIdKey(), to, who.getEmailAddress()));
            if (who.getEmailAddress() != null) {
                this.accountsUpdateFactory.create().update(to, (Account a) -> {
                    if (a.getPreferredEmail() == null) {
                        a.setPreferredEmail(who.getEmailAddress());
                    }
                });
            }
        }
        return new AuthResult(to, who.getExternalIdKey(), false);
    }

    public AuthResult updateLink(Account.Id to, AuthRequest who) throws OrmException, AccountException, IOException, ConfigInvalidException {
        Set<ExternalId> filteredExtIdsByScheme = this.externalIds.byAccount(to, who.getExternalIdKey().scheme());
        if (!(filteredExtIdsByScheme.isEmpty() || filteredExtIdsByScheme.size() <= 1 && filteredExtIdsByScheme.stream().filter(e -> e.key().equals(who.getExternalIdKey())).findAny().isPresent())) {
            this.externalIdsUpdateFactory.create().delete(filteredExtIdsByScheme);
        }
        return this.link(to, who);
    }

    public void unlink(Account.Id from, ExternalId.Key extIdKey) throws AccountException, OrmException, IOException, ConfigInvalidException {
        this.unlink(from, ImmutableList.of(extIdKey));
    }

    public void unlink(Account.Id from, Collection<ExternalId.Key> extIdKeys) throws AccountException, OrmException, IOException, ConfigInvalidException {
        if (extIdKeys.isEmpty()) {
            return;
        }
        ArrayList<ExternalId> extIds = new ArrayList<ExternalId>(extIdKeys.size());
        for (ExternalId.Key extIdKey : extIdKeys) {
            ExternalId extId = this.externalIds.get(extIdKey);
            if (extId != null) {
                if (!extId.accountId().equals(from)) {
                    throw new AccountException("Identity '" + extIdKey.get() + "' in use by another account");
                }
                extIds.add(extId);
                continue;
            }
            throw new AccountException("Identity '" + extIdKey.get() + "' not found");
        }
        this.externalIdsUpdateFactory.create().delete(extIds);
        if (extIds.stream().anyMatch(e -> e.email() != null)) {
            this.accountsUpdateFactory.create().update(from, (Account a) -> {
                if (a.getPreferredEmail() != null) {
                    for (ExternalId extId : extIds) {
                        if (!a.getPreferredEmail().equals(extId.email())) continue;
                        a.setPreferredEmail(null);
                        break;
                    }
                }
            });
        }
    }
}

