/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.lucene;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.gerrit.index.FieldDef;
import com.google.gerrit.index.FieldType;
import com.google.gerrit.index.Index;
import com.google.gerrit.index.Schema;
import com.google.gerrit.lucene.AutoCommitWriter;
import com.google.gerrit.lucene.GerritIndexWriterConfig;
import com.google.gerrit.lucene.WrappableSearcherManager;
import com.google.gerrit.server.config.SitePaths;
import com.google.gerrit.server.index.IndexUtils;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.IntField;
import org.apache.lucene.document.LongField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.TrackingIndexWriter;
import org.apache.lucene.search.ControlledRealTimeReopenThread;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.ReferenceManager;
import org.apache.lucene.search.SearcherFactory;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.Directory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractLuceneIndex<K, V>
implements Index<K, V> {
    private static final Logger log = LoggerFactory.getLogger(AbstractLuceneIndex.class);
    private final Schema<V> schema;
    private final SitePaths sitePaths;
    private final Directory dir;
    private final String name;
    private final ListeningExecutorService writerThread;
    private final TrackingIndexWriter writer;
    private final ReferenceManager<IndexSearcher> searcherManager;
    private final ControlledRealTimeReopenThread<IndexSearcher> reopenThread;
    private final Set<NrtFuture> notDoneNrtFutures;
    private ScheduledThreadPoolExecutor autoCommitExecutor;

    static String sortFieldName(FieldDef<?, ?> f) {
        return f.getName() + "_SORT";
    }

    AbstractLuceneIndex(Schema<V> schema, SitePaths sitePaths, Directory dir, String name, String subIndex, GerritIndexWriterConfig writerConfig, SearcherFactory searcherFactory) throws IOException {
        AutoCommitWriter delegateWriter;
        this.schema = schema;
        this.sitePaths = sitePaths;
        this.dir = dir;
        this.name = name;
        String index = Joiner.on('_').skipNulls().join(name, subIndex, new Object[0]);
        long commitPeriod = writerConfig.getCommitWithinMs();
        if (commitPeriod < 0L) {
            delegateWriter = new AutoCommitWriter(dir, writerConfig.getLuceneConfig());
        } else if (commitPeriod == 0L) {
            delegateWriter = new AutoCommitWriter(dir, writerConfig.getLuceneConfig(), true);
        } else {
            AutoCommitWriter autoCommitWriter;
            delegateWriter = autoCommitWriter = new AutoCommitWriter(dir, writerConfig.getLuceneConfig());
            this.autoCommitExecutor = new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setNameFormat(index + " Commit-%d").setDaemon(true).build());
            ScheduledFuture<?> scheduledFuture = this.autoCommitExecutor.scheduleAtFixedRate(() -> {
                try {
                    if (autoCommitWriter.hasUncommittedChanges()) {
                        autoCommitWriter.manualFlush();
                        autoCommitWriter.commit();
                    }
                }
                catch (IOException e) {
                    log.error("Error committing " + index + " Lucene index", e);
                }
                catch (OutOfMemoryError e) {
                    log.error("Error committing " + index + " Lucene index", e);
                    try {
                        autoCommitWriter.close();
                    }
                    catch (IOException e2) {
                        log.error("SEVERE: Error closing " + index + " Lucene index after OOM; index may be corrupted.", e);
                    }
                }
            }, commitPeriod, commitPeriod, TimeUnit.MILLISECONDS);
        }
        this.writer = new TrackingIndexWriter(delegateWriter);
        this.searcherManager = new WrappableSearcherManager(this.writer.getIndexWriter(), true, searcherFactory);
        this.notDoneNrtFutures = Sets.newConcurrentHashSet();
        this.writerThread = MoreExecutors.listeningDecorator(Executors.newFixedThreadPool(1, new ThreadFactoryBuilder().setNameFormat(index + " Write-%d").setDaemon(true).build()));
        this.reopenThread = new ControlledRealTimeReopenThread<IndexSearcher>(this.writer, this.searcherManager, 0.5, 0.01);
        this.reopenThread.setName(index + " NRT");
        this.reopenThread.setPriority(Math.min(Thread.currentThread().getPriority() + 2, 10));
        this.reopenThread.setDaemon(true);
        this.searcherManager.addListener(new ReferenceManager.RefreshListener(){

            @Override
            public void beforeRefresh() throws IOException {
            }

            @Override
            public void afterRefresh(boolean didRefresh) throws IOException {
                for (NrtFuture f : AbstractLuceneIndex.this.notDoneNrtFutures) {
                    f.removeIfDone();
                }
            }
        });
        this.reopenThread.start();
    }

    @Override
    public void markReady(boolean ready) throws IOException {
        IndexUtils.setReady(this.sitePaths, this.name, this.schema.getVersion(), ready);
    }

    @Override
    public void close() {
        if (this.autoCommitExecutor != null) {
            this.autoCommitExecutor.shutdown();
        }
        this.writerThread.shutdown();
        try {
            if (!this.writerThread.awaitTermination(5L, TimeUnit.SECONDS)) {
                log.warn("shutting down " + this.name + " index with pending Lucene writes");
            }
        }
        catch (InterruptedException e) {
            log.warn("interrupted waiting for pending Lucene writes of " + this.name + " index", e);
        }
        this.reopenThread.close();
        try {
            this.searcherManager.maybeRefreshBlocking();
        }
        catch (IOException e) {
            log.warn("error finishing pending Lucene writes", e);
        }
        try {
            this.writer.getIndexWriter().close();
        }
        catch (AlreadyClosedException e) {
        }
        catch (IOException e) {
            log.warn("error closing Lucene writer", e);
        }
        try {
            this.dir.close();
        }
        catch (IOException e) {
            log.warn("error closing Lucene directory", e);
        }
    }

    ListenableFuture<?> insert(Document doc) {
        return this.submit(() -> this.writer.addDocument(doc));
    }

    ListenableFuture<?> replace(Term term, Document doc) {
        return this.submit(() -> this.writer.updateDocument(term, doc));
    }

    ListenableFuture<?> delete(Term term) {
        return this.submit(() -> this.writer.deleteDocuments(term));
    }

    private ListenableFuture<?> submit(Callable<Long> task) {
        ListenableFuture future = Futures.nonCancellationPropagating(this.writerThread.submit(task));
        return Futures.transformAsync(future, gen -> {
            this.reopenThread.waitForGeneration((long)gen, 0);
            return new NrtFuture((long)gen);
        }, MoreExecutors.directExecutor());
    }

    @Override
    public void deleteAll() throws IOException {
        this.writer.deleteAll();
    }

    public TrackingIndexWriter getWriter() {
        return this.writer;
    }

    IndexSearcher acquire() throws IOException {
        return this.searcherManager.acquire();
    }

    void release(IndexSearcher searcher) throws IOException {
        this.searcherManager.release(searcher);
    }

    Document toDocument(V obj) {
        Document result = new Document();
        for (Schema.Values<V> vs : this.schema.buildFields(obj)) {
            if (vs.getValues() == null) continue;
            this.add(result, vs);
        }
        return result;
    }

    void add(Document doc, Schema.Values<V> values) {
        String name = values.getField().getName();
        FieldType<?> type = values.getField().getType();
        Field.Store store = AbstractLuceneIndex.store(values.getField());
        if (type == FieldType.INTEGER || type == FieldType.INTEGER_RANGE) {
            for (Object value : values.getValues()) {
                doc.add(new IntField(name, (int)((Integer)value), store));
            }
        } else if (type == FieldType.LONG) {
            for (Object value : values.getValues()) {
                doc.add(new LongField(name, (long)((Long)value), store));
            }
        } else if (type == FieldType.TIMESTAMP) {
            for (Object value : values.getValues()) {
                doc.add(new LongField(name, ((Timestamp)value).getTime(), store));
            }
        } else if (type == FieldType.EXACT || type == FieldType.PREFIX) {
            for (Object value : values.getValues()) {
                doc.add(new StringField(name, (String)value, store));
            }
        } else if (type == FieldType.FULL_TEXT) {
            for (Object value : values.getValues()) {
                doc.add(new TextField(name, (String)value, store));
            }
        } else if (type == FieldType.STORED_ONLY) {
            for (Object value : values.getValues()) {
                doc.add(new StoredField(name, (byte[])value));
            }
        } else {
            throw FieldType.badFieldType(type);
        }
    }

    private static Field.Store store(FieldDef<?, ?> f) {
        return f.isStored() ? Field.Store.YES : Field.Store.NO;
    }

    @Override
    public Schema<V> getSchema() {
        return this.schema;
    }

    private final class NrtFuture
    extends AbstractFuture<Void> {
        private final long gen;

        NrtFuture(long gen) {
            this.gen = gen;
        }

        @Override
        public Void get() throws InterruptedException, ExecutionException {
            if (!this.isDone()) {
                AbstractLuceneIndex.this.reopenThread.waitForGeneration(this.gen);
                this.set(null);
            }
            return (Void)super.get();
        }

        @Override
        public Void get(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException, ExecutionException {
            if (!this.isDone()) {
                if (!AbstractLuceneIndex.this.reopenThread.waitForGeneration(this.gen, (int)unit.toMillis(timeout))) {
                    throw new TimeoutException();
                }
                this.set(null);
            }
            return (Void)super.get(timeout, unit);
        }

        @Override
        public boolean isDone() {
            if (super.isDone()) {
                return true;
            }
            if (this.isGenAvailableNowForCurrentSearcher()) {
                this.set(null);
                return true;
            }
            if (!AbstractLuceneIndex.this.reopenThread.isAlive()) {
                this.setException(new IllegalStateException("NRT thread is dead"));
                return true;
            }
            return false;
        }

        @Override
        public void addListener(Runnable listener, Executor executor) {
            if (this.isGenAvailableNowForCurrentSearcher() && !this.isCancelled()) {
                this.set(null);
            } else if (!this.isDone()) {
                AbstractLuceneIndex.this.notDoneNrtFutures.add(this);
            }
            super.addListener(listener, executor);
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            boolean result = super.cancel(mayInterruptIfRunning);
            if (result) {
                AbstractLuceneIndex.this.notDoneNrtFutures.remove(this);
            }
            return result;
        }

        void removeIfDone() {
            if (this.isGenAvailableNowForCurrentSearcher()) {
                AbstractLuceneIndex.this.notDoneNrtFutures.remove(this);
                if (!this.isCancelled()) {
                    this.set(null);
                }
            }
        }

        private boolean isGenAvailableNowForCurrentSearcher() {
            try {
                return AbstractLuceneIndex.this.reopenThread.waitForGeneration(this.gen, 0);
            }
            catch (InterruptedException e) {
                log.warn("Interrupted waiting for searcher generation", e);
                return false;
            }
        }
    }
}

