/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git.validators;

import com.google.common.base.CharMatcher;
import com.google.gerrit.common.Nullable;
import com.google.gerrit.extensions.registration.DynamicSet;
import com.google.gerrit.server.GerritPersonIdent;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.config.CanonicalWebUrl;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.events.CommitReceivedEvent;
import com.google.gerrit.server.git.ProjectConfig;
import com.google.gerrit.server.git.ReceiveCommits;
import com.google.gerrit.server.git.ValidationError;
import com.google.gerrit.server.git.validators.CommitValidationException;
import com.google.gerrit.server.git.validators.CommitValidationListener;
import com.google.gerrit.server.git.validators.CommitValidationMessage;
import com.google.gerrit.server.project.ProjectControl;
import com.google.gerrit.server.project.RefControl;
import com.google.gerrit.server.ssh.SshInfo;
import com.google.gerrit.server.util.MagicBranch;
import com.google.inject.Inject;
import com.google.inject.assistedinject.Assisted;
import com.jcraft.jsch.HostKey;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.FooterKey;
import org.eclipse.jgit.revwalk.FooterLine;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.util.SystemReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitValidators {
    private static final Logger log = LoggerFactory.getLogger(CommitValidators.class);
    private static final FooterKey CHANGE_ID = new FooterKey("Change-Id");
    private final PersonIdent gerritIdent;
    private final RefControl refControl;
    private final String canonicalWebUrl;
    private final String installCommitMsgHookCommand;
    private final SshInfo sshInfo;
    private final Repository repo;
    private final DynamicSet<CommitValidationListener> commitValidationListeners;

    @Inject
    CommitValidators(@GerritPersonIdent PersonIdent gerritIdent, @CanonicalWebUrl @Nullable String canonicalWebUrl, @GerritServerConfig Config config, DynamicSet<CommitValidationListener> commitValidationListeners, @Assisted SshInfo sshInfo, @Assisted Repository repo, @Assisted RefControl refControl) {
        this.gerritIdent = gerritIdent;
        this.refControl = refControl;
        this.canonicalWebUrl = canonicalWebUrl;
        this.installCommitMsgHookCommand = config.getString("gerrit", null, "installCommitMsgHookCommand");
        this.sshInfo = sshInfo;
        this.repo = repo;
        this.commitValidationListeners = commitValidationListeners;
    }

    public List<CommitValidationMessage> validateForReceiveCommits(CommitReceivedEvent receiveEvent) throws CommitValidationException {
        LinkedList<CommitValidationListener> validators = new LinkedList<CommitValidationListener>();
        validators.add(new UploadMergesPermissionValidator(this.refControl));
        validators.add(new AmendedGerritMergeCommitValidationListener(this.refControl, this.gerritIdent));
        validators.add(new AuthorUploaderValidator(this.refControl, this.canonicalWebUrl));
        validators.add(new CommitterUploaderValidator(this.refControl, this.canonicalWebUrl));
        validators.add(new SignedOffByValidator(this.refControl, this.canonicalWebUrl));
        if (MagicBranch.isMagicBranch(receiveEvent.command.getRefName()) || ReceiveCommits.NEW_PATCHSET.matcher(receiveEvent.command.getRefName()).matches()) {
            validators.add(new ChangeIdValidator(this.refControl, this.canonicalWebUrl, this.installCommitMsgHookCommand, this.sshInfo));
        }
        validators.add(new ConfigValidator(this.refControl, this.repo));
        validators.add(new PluginCommitValidationListener(this.commitValidationListeners));
        LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
        try {
            for (CommitValidationListener commitValidator : validators) {
                messages.addAll(commitValidator.onCommitReceived(receiveEvent));
            }
        }
        catch (CommitValidationException e) {
            messages.addAll(e.getMessages());
            throw new CommitValidationException(e.getMessage(), messages);
        }
        return messages;
    }

    public List<CommitValidationMessage> validateForGerritCommits(CommitReceivedEvent receiveEvent) throws CommitValidationException {
        LinkedList<CommitValidationListener> validators = new LinkedList<CommitValidationListener>();
        validators.add(new UploadMergesPermissionValidator(this.refControl));
        validators.add(new AmendedGerritMergeCommitValidationListener(this.refControl, this.gerritIdent));
        validators.add(new AuthorUploaderValidator(this.refControl, this.canonicalWebUrl));
        validators.add(new SignedOffByValidator(this.refControl, this.canonicalWebUrl));
        if (MagicBranch.isMagicBranch(receiveEvent.command.getRefName()) || ReceiveCommits.NEW_PATCHSET.matcher(receiveEvent.command.getRefName()).matches()) {
            validators.add(new ChangeIdValidator(this.refControl, this.canonicalWebUrl, this.installCommitMsgHookCommand, this.sshInfo));
        }
        validators.add(new ConfigValidator(this.refControl, this.repo));
        validators.add(new PluginCommitValidationListener(this.commitValidationListeners));
        LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
        try {
            for (CommitValidationListener commitValidator : validators) {
                messages.addAll(commitValidator.onCommitReceived(receiveEvent));
            }
        }
        catch (CommitValidationException e) {
            messages.addAll(e.getMessages());
            throw new CommitValidationException(e.getMessage(), messages);
        }
        return messages;
    }

    private static CommitValidationMessage getInvalidEmailError(RevCommit c, String type, PersonIdent who, IdentifiedUser currentUser, String canonicalWebUrl) {
        StringBuilder sb = new StringBuilder();
        sb.append("\n");
        sb.append("ERROR:  In commit ").append(c.name()).append("\n");
        sb.append("ERROR:  ").append(type).append(" email address ").append(who.getEmailAddress()).append("\n");
        sb.append("ERROR:  does not match your user account.\n");
        sb.append("ERROR:\n");
        if (currentUser.getEmailAddresses().isEmpty()) {
            sb.append("ERROR:  You have not registered any email addresses.\n");
        } else {
            sb.append("ERROR:  The following addresses are currently registered:\n");
            for (String address : currentUser.getEmailAddresses()) {
                sb.append("ERROR:    ").append(address).append("\n");
            }
        }
        sb.append("ERROR:\n");
        if (canonicalWebUrl != null) {
            sb.append("ERROR:  To register an email address, please visit:\n");
            sb.append("ERROR:  ").append(canonicalWebUrl).append("#").append("/settings/contact").append("\n");
        }
        sb.append("\n");
        return new CommitValidationMessage(sb.toString(), false);
    }

    private static String getGerritUrl(String canonicalWebUrl) {
        if (canonicalWebUrl != null) {
            return CharMatcher.is('/').trimTrailingFrom(canonicalWebUrl);
        }
        return "http://" + CommitValidators.getGerritHost(canonicalWebUrl);
    }

    private static String getGerritHost(String canonicalWebUrl) {
        String host;
        if (canonicalWebUrl != null) {
            try {
                host = new URL(canonicalWebUrl).getHost();
            }
            catch (MalformedURLException e) {
                host = SystemReader.getInstance().getHostname();
            }
        } else {
            host = SystemReader.getInstance().getHostname();
        }
        return host;
    }

    private static void addError(String error, List<CommitValidationMessage> messages) {
        messages.add(new CommitValidationMessage(error, true));
    }

    public static class AmendedGerritMergeCommitValidationListener
    implements CommitValidationListener {
        private final PersonIdent gerritIdent;
        private final RefControl refControl;

        public AmendedGerritMergeCommitValidationListener(RefControl refControl, PersonIdent gerritIdent) {
            this.refControl = refControl;
            this.gerritIdent = gerritIdent;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            if (receiveEvent.commit.getParentCount() > 1 && author.getName().equals(this.gerritIdent.getName()) && author.getEmailAddress().equals(this.gerritIdent.getEmailAddress()) && !this.refControl.canForgeGerritServerIdentity()) {
                throw new CommitValidationException("do not amend merges not made by you");
            }
            return Collections.emptyList();
        }
    }

    public static class CommitterUploaderValidator
    implements CommitValidationListener {
        private final RefControl refControl;
        private final String canonicalWebUrl;

        public CommitterUploaderValidator(RefControl refControl, String canonicalWebUrl) {
            this.refControl = refControl;
            this.canonicalWebUrl = canonicalWebUrl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            IdentifiedUser currentUser = (IdentifiedUser)this.refControl.getCurrentUser();
            PersonIdent committer = receiveEvent.commit.getCommitterIdent();
            if (!currentUser.getEmailAddresses().contains(committer.getEmailAddress()) && !this.refControl.canForgeCommitter()) {
                LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
                messages.add(CommitValidators.getInvalidEmailError(receiveEvent.commit, "committer", committer, currentUser, this.canonicalWebUrl));
                throw new CommitValidationException("invalid committer", messages);
            }
            return Collections.emptyList();
        }
    }

    public static class AuthorUploaderValidator
    implements CommitValidationListener {
        private final RefControl refControl;
        private final String canonicalWebUrl;

        public AuthorUploaderValidator(RefControl refControl, String canonicalWebUrl) {
            this.refControl = refControl;
            this.canonicalWebUrl = canonicalWebUrl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            IdentifiedUser currentUser = (IdentifiedUser)this.refControl.getCurrentUser();
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            if (!currentUser.getEmailAddresses().contains(author.getEmailAddress()) && !this.refControl.canForgeAuthor()) {
                LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
                messages.add(CommitValidators.getInvalidEmailError(receiveEvent.commit, "author", author, currentUser, this.canonicalWebUrl));
                throw new CommitValidationException("invalid author", messages);
            }
            return Collections.emptyList();
        }
    }

    public static class SignedOffByValidator
    implements CommitValidationListener {
        private final RefControl refControl;

        public SignedOffByValidator(RefControl refControl, String canonicalWebUrl) {
            this.refControl = refControl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            IdentifiedUser currentUser = (IdentifiedUser)this.refControl.getCurrentUser();
            PersonIdent committer = receiveEvent.commit.getCommitterIdent();
            PersonIdent author = receiveEvent.commit.getAuthorIdent();
            ProjectControl projectControl = this.refControl.getProjectControl();
            if (projectControl.getProjectState().isUseSignedOffBy()) {
                boolean sboAuthor = false;
                boolean sboCommitter = false;
                boolean sboMe = false;
                for (FooterLine footer : receiveEvent.commit.getFooterLines()) {
                    String e;
                    if (!footer.matches(FooterKey.SIGNED_OFF_BY) || (e = footer.getEmailAddress()) == null) continue;
                    sboAuthor |= author.getEmailAddress().equals(e);
                    sboCommitter |= committer.getEmailAddress().equals(e);
                    sboMe |= currentUser.getEmailAddresses().contains(e);
                }
                if (!(sboAuthor || sboCommitter || sboMe || this.refControl.canForgeCommitter())) {
                    throw new CommitValidationException("not Signed-off-by author/committer/uploader in commit message footer");
                }
            }
            return Collections.emptyList();
        }
    }

    public static class PluginCommitValidationListener
    implements CommitValidationListener {
        private final DynamicSet<CommitValidationListener> commitValidationListeners;

        public PluginCommitValidationListener(DynamicSet<CommitValidationListener> commitValidationListeners) {
            this.commitValidationListeners = commitValidationListeners;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
            for (CommitValidationListener validator : this.commitValidationListeners) {
                try {
                    messages.addAll(validator.onCommitReceived(receiveEvent));
                }
                catch (CommitValidationException e) {
                    messages.addAll(e.getMessages());
                    throw new CommitValidationException(e.getMessage(), messages);
                }
            }
            return messages;
        }
    }

    public static class UploadMergesPermissionValidator
    implements CommitValidationListener {
        private final RefControl refControl;

        public UploadMergesPermissionValidator(RefControl refControl) {
            this.refControl = refControl;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            if (receiveEvent.commit.getParentCount() > 1 && !this.refControl.canUploadMerges()) {
                throw new CommitValidationException("you are not allowed to upload merges");
            }
            return Collections.emptyList();
        }
    }

    public static class ConfigValidator
    implements CommitValidationListener {
        private final RefControl refControl;
        private final Repository repo;

        public ConfigValidator(RefControl refControl, Repository repo) {
            this.refControl = refControl;
            this.repo = repo;
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            IdentifiedUser currentUser = (IdentifiedUser)this.refControl.getCurrentUser();
            if ("refs/meta/config".equals(this.refControl.getRefName())) {
                LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
                try {
                    ProjectConfig cfg = new ProjectConfig(receiveEvent.project.getNameKey());
                    cfg.load(this.repo, receiveEvent.command.getNewId());
                    if (!cfg.getValidationErrors().isEmpty()) {
                        CommitValidators.addError("Invalid project configuration:", messages);
                        for (ValidationError err : cfg.getValidationErrors()) {
                            CommitValidators.addError("  " + err.getMessage(), messages);
                        }
                        throw new ConfigInvalidException("invalid project configuration");
                    }
                }
                catch (Exception e) {
                    log.error("User " + currentUser.getUserName() + " tried to push invalid project configuration " + receiveEvent.command.getNewId().name() + " for " + receiveEvent.project.getName(), e);
                    throw new CommitValidationException("invalid project configuration", messages);
                }
            }
            return Collections.emptyList();
        }
    }

    public static class ChangeIdValidator
    implements CommitValidationListener {
        private final ProjectControl projectControl;
        private final String canonicalWebUrl;
        private final String installCommitMsgHookCommand;
        private final SshInfo sshInfo;
        private final IdentifiedUser user;

        public ChangeIdValidator(RefControl refControl, String canonicalWebUrl, String installCommitMsgHookCommand, SshInfo sshInfo) {
            this.projectControl = refControl.getProjectControl();
            this.canonicalWebUrl = canonicalWebUrl;
            this.installCommitMsgHookCommand = installCommitMsgHookCommand;
            this.sshInfo = sshInfo;
            this.user = (IdentifiedUser)this.projectControl.getCurrentUser();
        }

        @Override
        public List<CommitValidationMessage> onCommitReceived(CommitReceivedEvent receiveEvent) throws CommitValidationException {
            List<String> idList = receiveEvent.commit.getFooterLines(CHANGE_ID);
            LinkedList<CommitValidationMessage> messages = new LinkedList<CommitValidationMessage>();
            if (idList.isEmpty()) {
                if (this.projectControl.getProjectState().isRequireChangeID()) {
                    String changeIdPrefix;
                    String shortMsg = receiveEvent.commit.getShortMessage();
                    if (shortMsg.startsWith(changeIdPrefix = CHANGE_ID.getName() + ":") && shortMsg.substring(changeIdPrefix.length()).trim().matches("^I[0-9a-f]{8,}.*$")) {
                        throw new CommitValidationException("missing subject; Change-Id must be in commit message footer");
                    }
                    String errMsg = "missing Change-Id in commit message footer";
                    messages.add(this.getFixedCommitMsgWithChangeId(errMsg, receiveEvent.commit));
                    throw new CommitValidationException(errMsg, messages);
                }
            } else {
                if (idList.size() > 1) {
                    throw new CommitValidationException("multiple Change-Id lines in commit message footer", messages);
                }
                String v = idList.get(idList.size() - 1).trim();
                if (!v.matches("^I[0-9a-f]{8,}.*$")) {
                    String errMsg = "missing or invalid Change-Id line format in commit message footer";
                    messages.add(this.getFixedCommitMsgWithChangeId("missing or invalid Change-Id line format in commit message footer", receiveEvent.commit));
                    throw new CommitValidationException("missing or invalid Change-Id line format in commit message footer", messages);
                }
            }
            return Collections.emptyList();
        }

        private CommitValidationMessage getFixedCommitMsgWithChangeId(String errMsg, RevCommit c) {
            String changeId = "Change-Id:";
            StringBuilder sb = new StringBuilder();
            sb.append("ERROR: ").append(errMsg);
            sb.append('\n');
            sb.append("Suggestion for commit message:\n");
            if (c.getFullMessage().indexOf("Change-Id:") == -1) {
                sb.append(c.getFullMessage());
                sb.append('\n');
                sb.append("Change-Id:").append(" I").append(c.name());
            } else {
                String lastLine;
                String[] lines = c.getFullMessage().trim().split("\n");
                String string = lastLine = lines.length > 0 ? lines[lines.length - 1] : "";
                if (lastLine.indexOf("Change-Id:") == 0) {
                    for (int i = 0; i < lines.length - 1; ++i) {
                        sb.append(lines[i]);
                        sb.append('\n');
                    }
                    sb.append('\n');
                    sb.append(lastLine);
                } else {
                    sb.append(c.getFullMessage());
                    sb.append('\n');
                    sb.append("Change-Id:").append(" I").append(c.name());
                    sb.append('\n');
                    sb.append("Hint: A potential Change-Id was found, but it was not in the ");
                    sb.append("footer (last paragraph) of the commit message.");
                }
            }
            sb.append('\n');
            sb.append('\n');
            sb.append("Hint: To automatically insert Change-Id, install the hook:\n");
            sb.append(this.getCommitMessageHookInstallationHint()).append('\n');
            sb.append('\n');
            return new CommitValidationMessage(sb.toString(), false);
        }

        private String getCommitMessageHookInstallationHint() {
            int sshPort;
            String sshHost;
            if (this.installCommitMsgHookCommand != null) {
                return this.installCommitMsgHookCommand;
            }
            List<HostKey> hostKeys = this.sshInfo.getHostKeys();
            if (hostKeys.isEmpty()) {
                String p = "${gitdir}/hooks/commit-msg";
                return String.format("  gitdir=$(git rev-parse --git-dir); curl -o %s %s/tools/hooks/commit-msg ; chmod +x %s", p, CommitValidators.getGerritUrl(this.canonicalWebUrl), p);
            }
            String host = hostKeys.get(0).getHost();
            int c = host.lastIndexOf(58);
            if (0 <= c) {
                sshHost = host.startsWith("*:") ? CommitValidators.getGerritHost(this.canonicalWebUrl) : host.substring(0, c);
                sshPort = Integer.parseInt(host.substring(c + 1));
            } else {
                sshHost = host;
                sshPort = 22;
            }
            return String.format("  gitdir=$(git rev-parse --git-dir); scp -p -P %d %s@%s:hooks/commit-msg ${gitdir}/hooks/", sshPort, this.user.getUserName(), sshHost);
        }
    }

    public static interface Factory {
        public CommitValidators create(RefControl var1, SshInfo var2, Repository var3);
    }
}

