/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.sshd;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.gerrit.audit.AuditService;
import com.google.gerrit.audit.SshAuditEvent;
import com.google.gerrit.extensions.events.LifecycleListener;
import com.google.gerrit.server.CurrentUser;
import com.google.gerrit.server.IdentifiedUser;
import com.google.gerrit.server.PeerDaemonUser;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.config.SitePaths;
import com.google.gerrit.server.util.IdGenerator;
import com.google.gerrit.server.util.LogUtil;
import com.google.gerrit.server.util.TimeUtil;
import com.google.gerrit.sshd.DispatchCommand;
import com.google.gerrit.sshd.SshLogLayout;
import com.google.gerrit.sshd.SshScope;
import com.google.gerrit.sshd.SshSession;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.io.File;
import java.io.IOException;
import org.apache.log4j.Appender;
import org.apache.log4j.AsyncAppender;
import org.apache.log4j.DailyRollingFileAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.spi.ErrorHandler;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.jgit.lib.Config;

@Singleton
class SshLog
implements LifecycleListener {
    private static final Logger log = Logger.getLogger(SshLog.class);
    private static final String LOG_NAME = "sshd_log";
    private static final String P_SESSION = "session";
    private static final String P_USER_NAME = "userName";
    private static final String P_ACCOUNT_ID = "accountId";
    private static final String P_WAIT = "queueWaitTime";
    private static final String P_EXEC = "executionTime";
    private static final String P_STATUS = "status";
    private final Provider<SshSession> session;
    private final Provider<SshScope.Context> context;
    private final AsyncAppender async;
    private final AuditService auditService;

    @Inject
    SshLog(Provider<SshSession> session, Provider<SshScope.Context> context, SitePaths site, @GerritServerConfig Config config, AuditService auditService) {
        this.session = session;
        this.context = context;
        this.auditService = auditService;
        if (!config.getBoolean("sshd", "requestLog", true)) {
            this.async = null;
            return;
        }
        this.async = new AsyncAppender();
        this.async.setBlocking(true);
        this.async.setBufferSize(config.getInt("core", "asyncLoggingBufferSize", 64));
        this.async.setLocationInfo(false);
        if (LogUtil.shouldConfigureLogSystem()) {
            DailyRollingFileAppender dst = new DailyRollingFileAppender();
            dst.setName(LOG_NAME);
            dst.setLayout(new SshLogLayout());
            dst.setEncoding("UTF-8");
            dst.setFile(new File(SshLog.resolve(site.logs_dir), LOG_NAME).getPath());
            dst.setImmediateFlush(true);
            dst.setAppend(true);
            dst.setThreshold(Level.INFO);
            dst.setErrorHandler(new DieErrorHandler());
            dst.activateOptions();
            dst.setErrorHandler(new LogLogHandler());
            this.async.addAppender(dst);
        } else {
            Appender appender = log.getAppender(LOG_NAME);
            if (appender != null) {
                this.async.addAppender(appender);
            } else {
                log.warn("No appender with the name: sshd_log was found. SSHD logging is disabled");
            }
        }
        this.async.activateOptions();
    }

    @Override
    public void start() {
    }

    @Override
    public void stop() {
        if (this.async != null) {
            this.async.close();
        }
    }

    void onLogin() {
        LoggingEvent entry = this.log("LOGIN FROM " + this.session.get().getRemoteAddressAsString());
        if (this.async != null) {
            this.async.append(entry);
        }
        this.audit(this.context.get(), (Object)"0", "LOGIN");
    }

    void onAuthFail(SshSession sd) {
        LoggingEvent event = new LoggingEvent(Logger.class.getName(), log, TimeUtil.nowMs(), Level.INFO, "AUTH FAILURE FROM " + sd.getRemoteAddressAsString(), "SSHD", null, null, null, null);
        event.setProperty(P_SESSION, SshLog.id(sd.getSessionId()));
        event.setProperty(P_USER_NAME, sd.getUsername());
        String error = sd.getAuthenticationError();
        if (error != null) {
            event.setProperty(P_STATUS, error);
        }
        if (this.async != null) {
            this.async.append(event);
        }
        this.audit(null, (Object)"FAIL", "AUTH");
    }

    void onExecute(DispatchCommand dcmd, int exitValue) {
        String status;
        SshScope.Context ctx = this.context.get();
        ctx.finished = TimeUtil.nowMs();
        String cmd = this.extractWhat(dcmd);
        LoggingEvent event = this.log(cmd);
        event.setProperty(P_WAIT, ctx.started - ctx.created + "ms");
        event.setProperty(P_EXEC, ctx.finished - ctx.started + "ms");
        switch (exitValue) {
            case 0x40000001: {
                status = "killed";
                break;
            }
            case 0x40000002: {
                status = "not-found";
                break;
            }
            case 0x40000003: {
                status = "not-admin";
                break;
            }
            default: {
                status = String.valueOf(exitValue);
            }
        }
        event.setProperty(P_STATUS, status);
        if (this.async != null) {
            this.async.append(event);
        }
        this.audit(this.context.get(), (Object)status, dcmd);
    }

    private Multimap<String, ?> extractParameters(DispatchCommand dcmd) {
        String[] cmdArgs = dcmd.getArguments();
        String paramName = null;
        int argPos = 0;
        ArrayListMultimap<String, String> parms = ArrayListMultimap.create();
        for (int i = 2; i < cmdArgs.length; ++i) {
            String arg = cmdArgs[i];
            if (arg.equals("--")) {
                ++i;
                while (i < cmdArgs.length) {
                    parms.put("$" + argPos++, cmdArgs[i]);
                    ++i;
                }
                break;
            }
            int eqPos = arg.indexOf(61);
            if (arg.startsWith("--") && eqPos > 0) {
                parms.put(arg.substring(0, eqPos), arg.substring(eqPos + 1));
                continue;
            }
            if (arg.startsWith("-")) {
                if (paramName != null) {
                    parms.put(paramName, null);
                }
                paramName = arg;
                continue;
            }
            if (paramName == null) {
                parms.put("$" + argPos++, arg);
                continue;
            }
            parms.put(paramName, arg);
            paramName = null;
        }
        if (paramName != null) {
            parms.put(paramName, null);
        }
        return parms;
    }

    void onLogout() {
        LoggingEvent entry = this.log("LOGOUT");
        if (this.async != null) {
            this.async.append(entry);
        }
        this.audit(this.context.get(), (Object)"0", "LOGOUT");
    }

    private LoggingEvent log(String msg) {
        SshSession sd = this.session.get();
        CurrentUser user = sd.getCurrentUser();
        LoggingEvent event = new LoggingEvent(Logger.class.getName(), log, TimeUtil.nowMs(), Level.INFO, msg, "SSHD", null, null, null, null);
        event.setProperty(P_SESSION, SshLog.id(sd.getSessionId()));
        String userName = "-";
        String accountId = "-";
        if (user != null && user.isIdentifiedUser()) {
            IdentifiedUser u = (IdentifiedUser)user;
            userName = u.getAccount().getUserName();
            accountId = "a/" + u.getAccountId().toString();
        } else if (user instanceof PeerDaemonUser) {
            userName = "Gerrit Code Review";
        }
        event.setProperty(P_USER_NAME, userName);
        event.setProperty(P_ACCOUNT_ID, accountId);
        return event;
    }

    private static String id(int id) {
        return IdGenerator.format(id);
    }

    private static File resolve(File logs_dir) {
        try {
            return logs_dir.getCanonicalFile();
        }
        catch (IOException e) {
            return logs_dir.getAbsoluteFile();
        }
    }

    void audit(SshScope.Context ctx, Object result, String cmd) {
        String sid = this.extractSessionId(ctx);
        long created = this.extractCreated(ctx);
        this.auditService.dispatch(new SshAuditEvent(sid, this.extractCurrentUser(ctx), cmd, created, null, result));
    }

    void audit(SshScope.Context ctx, Object result, DispatchCommand cmd) {
        String sid = this.extractSessionId(ctx);
        long created = this.extractCreated(ctx);
        this.auditService.dispatch(new SshAuditEvent(sid, this.extractCurrentUser(ctx), this.extractWhat(cmd), created, this.extractParameters(cmd), result));
    }

    private String extractWhat(DispatchCommand dcmd) {
        String commandName = dcmd.getCommandName();
        String[] args = dcmd.getArguments();
        if (args.length > 1) {
            return commandName + "." + args[1];
        }
        return commandName;
    }

    private long extractCreated(SshScope.Context ctx) {
        return ctx != null ? ctx.created : TimeUtil.nowMs();
    }

    private CurrentUser extractCurrentUser(SshScope.Context ctx) {
        if (ctx != null) {
            SshSession session = ctx.getSession();
            return session == null ? null : session.getCurrentUser();
        }
        return null;
    }

    private String extractSessionId(SshScope.Context ctx) {
        if (ctx != null) {
            SshSession session = ctx.getSession();
            return session == null ? null : IdGenerator.format(session.getSessionId());
        }
        return null;
    }

    private static final class LogLogHandler
    implements ErrorHandler {
        private LogLogHandler() {
        }

        @Override
        public void error(String message, Exception e, int errorCode, LoggingEvent event) {
            log.error(message, e);
        }

        @Override
        public void error(String message, Exception e, int errorCode) {
            log.error(message, e);
        }

        @Override
        public void error(String message) {
            log.error(message);
        }

        @Override
        public void activateOptions() {
        }

        @Override
        public void setAppender(Appender appender) {
        }

        @Override
        public void setBackupAppender(Appender appender) {
        }

        @Override
        public void setLogger(Logger logger) {
        }
    }

    private static final class DieErrorHandler
    implements ErrorHandler {
        private DieErrorHandler() {
        }

        @Override
        public void error(String message, Exception e, int errorCode, LoggingEvent event) {
            this.error(e != null ? e.getMessage() : message);
        }

        @Override
        public void error(String message, Exception e, int errorCode) {
            this.error(e != null ? e.getMessage() : message);
        }

        @Override
        public void error(String message) {
            throw new RuntimeException("Cannot open log file: " + message);
        }

        @Override
        public void activateOptions() {
        }

        @Override
        public void setAppender(Appender appender) {
        }

        @Override
        public void setBackupAppender(Appender appender) {
        }

        @Override
        public void setLogger(Logger logger) {
        }
    }
}

