/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.base.Charsets;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.hash.AbstractNonStreamingHashFunction;
import com.google.common.hash.AbstractStreamingHashFunction;
import com.google.common.hash.AndroidIncompatible;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashTestUtils;
import com.google.common.hash.Hasher;
import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Random;
import junit.framework.Assert;
import junit.framework.TestCase;

public class AbstractStreamingHasherTest
extends TestCase {
    public void testBytes() {
        Sink sink = new Sink(4);
        byte[] expected = new byte[]{1, 2, 3, 4, 5, 6, 7, 8};
        sink.putByte((byte)1);
        sink.putBytes(new byte[]{2, 3, 4, 5, 6});
        sink.putByte((byte)7);
        sink.putBytes(new byte[0]);
        sink.putBytes(new byte[]{8});
        HashCode unused = sink.hash();
        sink.assertInvariants(8);
        sink.assertBytes(expected);
    }

    public void testShort() {
        Sink sink = new Sink(4);
        sink.putShort((short)513);
        HashCode unused = sink.hash();
        sink.assertInvariants(2);
        sink.assertBytes(new byte[]{1, 2, 0, 0});
    }

    public void testInt() {
        Sink sink = new Sink(4);
        sink.putInt(67305985);
        HashCode unused = sink.hash();
        sink.assertInvariants(4);
        sink.assertBytes(new byte[]{1, 2, 3, 4});
    }

    public void testLong() {
        Sink sink = new Sink(8);
        sink.putLong(578437695752307201L);
        HashCode unused = sink.hash();
        sink.assertInvariants(8);
        sink.assertBytes(new byte[]{1, 2, 3, 4, 5, 6, 7, 8});
    }

    public void testChar() {
        Sink sink = new Sink(4);
        sink.putChar('\u0201');
        HashCode unused = sink.hash();
        sink.assertInvariants(2);
        sink.assertBytes(new byte[]{1, 2, 0, 0});
    }

    public void testString() {
        Random random = new Random();
        for (int i = 0; i < 100; ++i) {
            byte[] bytes = new byte[64];
            random.nextBytes(bytes);
            String s = new String(bytes, Charsets.UTF_16LE);
            AbstractStreamingHasherTest.assertEquals((Object)new Sink(4).putUnencodedChars(s).hash(), (Object)new Sink(4).putBytes(s.getBytes(Charsets.UTF_16LE)).hash());
            AbstractStreamingHasherTest.assertEquals((Object)new Sink(4).putUnencodedChars(s).hash(), (Object)new Sink(4).putString(s, Charsets.UTF_16LE).hash());
        }
    }

    public void testFloat() {
        Sink sink = new Sink(4);
        sink.putFloat(Float.intBitsToFloat(67305985));
        HashCode unused = sink.hash();
        sink.assertInvariants(4);
        sink.assertBytes(new byte[]{1, 2, 3, 4});
    }

    public void testDouble() {
        Sink sink = new Sink(8);
        sink.putDouble(Double.longBitsToDouble(578437695752307201L));
        HashCode unused = sink.hash();
        sink.assertInvariants(8);
        sink.assertBytes(new byte[]{1, 2, 3, 4, 5, 6, 7, 8});
    }

    public void testCorrectExceptions() {
        Sink sink = new Sink(4);
        try {
            sink.putBytes(new byte[8], -1, 4);
            AbstractStreamingHasherTest.fail();
        }
        catch (IndexOutOfBoundsException ok) {
            // empty catch block
        }
        try {
            sink.putBytes(new byte[8], 0, 16);
            AbstractStreamingHasherTest.fail();
        }
        catch (IndexOutOfBoundsException ok) {
            // empty catch block
        }
        try {
            sink.putBytes(new byte[8], 0, -1);
            AbstractStreamingHasherTest.fail();
        }
        catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            // empty catch block
        }
    }

    @AndroidIncompatible
    public void testExhaustive() throws Exception {
        Random random = new Random(0L);
        for (int totalInsertions = 0; totalInsertions < 200; ++totalInsertions) {
            ArrayList sinks = Lists.newArrayList();
            for (int chunkSize = 4; chunkSize <= 32; ++chunkSize) {
                for (int bufferSize = chunkSize; bufferSize <= chunkSize * 4; bufferSize += chunkSize) {
                    sinks.add(new Sink(chunkSize, bufferSize));
                }
            }
            Control control = new Control();
            Hasher controlSink = control.newHasher(1024);
            Iterable sinksAndControl = Iterables.concat((Iterable)sinks, Collections.singleton(controlSink));
            for (int insertion = 0; insertion < totalInsertions; ++insertion) {
                HashTestUtils.RandomHasherAction.pickAtRandom(random).performAction(random, sinksAndControl);
            }
            int intToPut = random.nextInt();
            for (Hasher hasher : sinksAndControl) {
                hasher.putInt(intToPut);
            }
            for (Sink sink : sinks) {
                HashCode unused = sink.hash();
            }
            byte[] expected = controlSink.hash().asBytes();
            for (Sink sink : sinks) {
                sink.assertInvariants(expected.length);
                sink.assertBytes(expected);
            }
        }
    }

    private static class Control
    extends AbstractNonStreamingHashFunction {
        private Control() {
        }

        public HashCode hashBytes(byte[] input) {
            return HashCode.fromBytes((byte[])input);
        }

        public HashCode hashBytes(byte[] input, int off, int len) {
            return this.hashBytes(Arrays.copyOfRange(input, off, off + len));
        }

        public int bits() {
            throw new UnsupportedOperationException();
        }

        public HashCode hashString(CharSequence input, Charset charset) {
            throw new UnsupportedOperationException();
        }

        public HashCode hashLong(long input) {
            throw new UnsupportedOperationException();
        }

        public HashCode hashInt(int input) {
            throw new UnsupportedOperationException();
        }
    }

    private static class Sink
    extends AbstractStreamingHashFunction.AbstractStreamingHasher {
        final int chunkSize;
        final int bufferSize;
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        int processCalled = 0;
        boolean remainingCalled = false;

        Sink(int chunkSize, int bufferSize) {
            super(chunkSize, bufferSize);
            this.chunkSize = chunkSize;
            this.bufferSize = bufferSize;
        }

        Sink(int chunkSize) {
            super(chunkSize);
            this.chunkSize = chunkSize;
            this.bufferSize = chunkSize;
        }

        HashCode makeHash() {
            return HashCode.fromBytes((byte[])this.out.toByteArray());
        }

        protected void process(ByteBuffer bb) {
            ++this.processCalled;
            Assert.assertEquals((Object)ByteOrder.LITTLE_ENDIAN, (Object)bb.order());
            Assert.assertTrue((bb.remaining() >= this.chunkSize ? 1 : 0) != 0);
            for (int i = 0; i < this.chunkSize; ++i) {
                this.out.write(bb.get());
            }
        }

        protected void processRemaining(ByteBuffer bb) {
            Assert.assertFalse((boolean)this.remainingCalled);
            this.remainingCalled = true;
            Assert.assertEquals((Object)ByteOrder.LITTLE_ENDIAN, (Object)bb.order());
            Assert.assertTrue((bb.remaining() > 0 ? 1 : 0) != 0);
            Assert.assertTrue((bb.remaining() < this.bufferSize ? 1 : 0) != 0);
            int before = this.processCalled;
            super.processRemaining(bb);
            int after = this.processCalled--;
            Assert.assertEquals((int)(before + 1), (int)after);
        }

        void assertInvariants(int expectedBytes) {
            Assert.assertEquals((int)this.out.toByteArray().length, (int)Sink.ceilToMultiple(expectedBytes, this.chunkSize));
            Assert.assertEquals((int)(expectedBytes / this.chunkSize), (int)this.processCalled);
            Assert.assertEquals((expectedBytes % this.chunkSize != 0 ? 1 : 0) != 0, (boolean)this.remainingCalled);
        }

        private static int ceilToMultiple(int a, int b) {
            int remainder = a % b;
            return remainder == 0 ? a : a + b - remainder;
        }

        void assertBytes(byte[] expected) {
            byte[] got = this.out.toByteArray();
            for (int i = 0; i < expected.length; ++i) {
                Assert.assertEquals((byte)expected[i], (byte)got[i]);
            }
        }
    }
}

