/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableTable;
import com.google.common.collect.Table;
import com.google.common.hash.AndroidIncompatible;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.HashTestUtils;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.common.hash.MacHashFunction;
import com.google.common.io.BaseEncoding;
import com.google.common.testing.NullPointerTester;
import java.security.Key;
import java.security.Provider;
import java.util.Arrays;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import junit.framework.TestCase;
import sun.security.jca.ProviderList;
import sun.security.jca.Providers;

public class MacHashFunctionTest
extends TestCase {
    private static final ImmutableSet<String> INPUTS = ImmutableSet.of((Object)"", (Object)"Z", (Object)"foobar");
    private static final SecretKey MD5_KEY = new SecretKeySpec("secret key".getBytes(Charsets.UTF_8), "HmacMD5");
    private static final SecretKey SHA1_KEY = new SecretKeySpec("secret key".getBytes(Charsets.UTF_8), "HmacSHA1");
    private static final SecretKey SHA256_KEY = new SecretKeySpec("secret key".getBytes(Charsets.UTF_8), "HmacSHA256");
    private static final SecretKey SHA512_KEY = new SecretKeySpec("secret key".getBytes(Charsets.UTF_8), "HmacSHA512");
    private static final ImmutableTable<String, SecretKey, HashFunction> ALGORITHMS = new ImmutableTable.Builder().put((Object)"HmacMD5", (Object)MD5_KEY, (Object)Hashing.hmacMd5((Key)MD5_KEY)).put((Object)"HmacSHA1", (Object)SHA1_KEY, (Object)Hashing.hmacSha1((Key)SHA1_KEY)).put((Object)"HmacSHA256", (Object)SHA256_KEY, (Object)Hashing.hmacSha256((Key)SHA256_KEY)).put((Object)"HmacSHA512", (Object)SHA512_KEY, (Object)Hashing.hmacSha512((Key)SHA512_KEY)).build();

    public void testNulls() {
        NullPointerTester tester = new NullPointerTester().setDefault(String.class, (Object)"HmacMD5").setDefault(Key.class, (Object)MD5_KEY);
        tester.testAllPublicConstructors(MacHashFunction.class);
        tester.testAllPublicInstanceMethods((Object)new MacHashFunction("HmacMD5", (Key)MD5_KEY, "toString"));
    }

    public void testHashing() throws Exception {
        for (String stringToTest : INPUTS) {
            for (Table.Cell cell : ALGORITHMS.cellSet()) {
                String algorithm = (String)cell.getRowKey();
                SecretKey key = (SecretKey)cell.getColumnKey();
                HashFunction hashFunc = (HashFunction)cell.getValue();
                MacHashFunctionTest.assertMacHashing(HashTestUtils.ascii(stringToTest), algorithm, key, hashFunc);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @AndroidIncompatible
    public void testNoProviders() {
        ProviderList providers = Providers.getProviderList();
        Providers.setProviderList(ProviderList.newList(new Provider[0]));
        try {
            Hashing.hmacMd5((Key)MD5_KEY);
            MacHashFunctionTest.fail((String)"expected ISE");
        }
        catch (IllegalStateException illegalStateException) {
        }
        finally {
            Providers.setProviderList(providers);
        }
    }

    public void testMultipleUpdates() throws Exception {
        Mac mac = Mac.getInstance("HmacSHA1");
        mac.init(SHA1_KEY);
        mac.update("hello".getBytes(Charsets.UTF_8));
        mac.update("world".getBytes(Charsets.UTF_8));
        MacHashFunctionTest.assertEquals((Object)HashCode.fromBytes((byte[])mac.doFinal()), (Object)Hashing.hmacSha1((Key)SHA1_KEY).newHasher().putString((CharSequence)"hello", Charsets.UTF_8).putString((CharSequence)"world", Charsets.UTF_8).hash());
    }

    public void testMultipleUpdatesDoFinal() throws Exception {
        Mac mac = Mac.getInstance("HmacSHA1");
        mac.init(SHA1_KEY);
        mac.update("hello".getBytes(Charsets.UTF_8));
        mac.update("world".getBytes(Charsets.UTF_8));
        MacHashFunctionTest.assertEquals((Object)HashCode.fromBytes((byte[])mac.doFinal("!!!".getBytes(Charsets.UTF_8))), (Object)Hashing.hmacSha1((Key)SHA1_KEY).newHasher().putString((CharSequence)"hello", Charsets.UTF_8).putString((CharSequence)"world", Charsets.UTF_8).putString((CharSequence)"!!!", Charsets.UTF_8).hash());
    }

    public void testCustomKey() throws Exception {
        SecretKey customKey = new SecretKey(){

            @Override
            public String getAlgorithm() {
                return "HmacMD5";
            }

            @Override
            public byte[] getEncoded() {
                return new byte[8];
            }

            @Override
            public String getFormat() {
                return "RAW";
            }
        };
        MacHashFunctionTest.assertEquals((String)"ad262969c53bc16032f160081c4a07a0", (String)Hashing.hmacMd5((Key)customKey).hashString((CharSequence)"The quick brown fox jumps over the lazy dog", Charsets.UTF_8).toString());
    }

    public void testBadKey_emptyKey() throws Exception {
        SecretKey badKey = new SecretKey(){

            @Override
            public String getAlgorithm() {
                return "HmacMD5";
            }

            @Override
            public byte[] getEncoded() {
                return null;
            }

            @Override
            public String getFormat() {
                return "RAW";
            }
        };
        try {
            Hashing.hmacMd5((Key)badKey);
            MacHashFunctionTest.fail();
        }
        catch (IllegalArgumentException expected) {
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    public void testEmptyInputs() throws Exception {
        String knownOutput = "8cbf764cbe2e4623d99a41354adfd390";
        Mac mac = Mac.getInstance("HmacMD5");
        mac.init(MD5_KEY);
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)HashCode.fromBytes((byte[])mac.doFinal()).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)Hashing.hmacMd5((Key)MD5_KEY).newHasher().hash().toString());
    }

    public void testEmptyInputs_mixedAlgorithms() throws Exception {
        String knownOutput = "8cbf764cbe2e4623d99a41354adfd390";
        Mac mac = Mac.getInstance("HmacMD5");
        mac.init(SHA1_KEY);
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)HashCode.fromBytes((byte[])mac.doFinal()).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)Hashing.hmacMd5((Key)SHA1_KEY).newHasher().hash().toString());
    }

    public void testKnownInputs() throws Exception {
        String knownOutput = "9753980fe94daa8ecaa82216519393a9";
        String input = "The quick brown fox jumps over the lazy dog";
        Mac mac = Mac.getInstance("HmacMD5");
        mac.init(MD5_KEY);
        mac.update(input.getBytes(Charsets.UTF_8));
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)HashCode.fromBytes((byte[])mac.doFinal()).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)HashCode.fromBytes((byte[])mac.doFinal(input.getBytes(Charsets.UTF_8))).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)Hashing.hmacMd5((Key)MD5_KEY).hashString((CharSequence)input, Charsets.UTF_8).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)Hashing.hmacMd5((Key)MD5_KEY).hashBytes(input.getBytes(Charsets.UTF_8)).toString());
    }

    public void testKnownInputs_mixedAlgorithms() throws Exception {
        String knownOutput = "9753980fe94daa8ecaa82216519393a9";
        String input = "The quick brown fox jumps over the lazy dog";
        Mac mac = Mac.getInstance("HmacMD5");
        mac.init(SHA1_KEY);
        mac.update(input.getBytes(Charsets.UTF_8));
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)HashCode.fromBytes((byte[])mac.doFinal()).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)HashCode.fromBytes((byte[])mac.doFinal(input.getBytes(Charsets.UTF_8))).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)Hashing.hmacMd5((Key)SHA1_KEY).hashString((CharSequence)input, Charsets.UTF_8).toString());
        MacHashFunctionTest.assertEquals((String)knownOutput, (String)Hashing.hmacMd5((Key)SHA1_KEY).hashBytes(input.getBytes(Charsets.UTF_8)).toString());
    }

    public void testPutAfterHash() {
        Hasher hasher = Hashing.hmacMd5((Key)MD5_KEY).newHasher();
        MacHashFunctionTest.assertEquals((String)"9753980fe94daa8ecaa82216519393a9", (String)hasher.putString((CharSequence)"The quick brown fox jumps over the lazy dog", Charsets.UTF_8).hash().toString());
        try {
            hasher.putInt(42);
            MacHashFunctionTest.fail();
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    public void testHashTwice() {
        Hasher hasher = Hashing.hmacMd5((Key)MD5_KEY).newHasher();
        MacHashFunctionTest.assertEquals((String)"9753980fe94daa8ecaa82216519393a9", (String)hasher.putString((CharSequence)"The quick brown fox jumps over the lazy dog", Charsets.UTF_8).hash().toString());
        try {
            hasher.hash();
            MacHashFunctionTest.fail();
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    public void testToString() {
        byte[] keyData = "secret key".getBytes(Charsets.UTF_8);
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacMd5(Key[algorithm=HmacMD5, format=RAW])", (String)Hashing.hmacMd5((Key)MD5_KEY).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacMd5(Key[algorithm=HmacMD5, format=RAW])", (String)Hashing.hmacMd5((byte[])keyData).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacSha1(Key[algorithm=HmacSHA1, format=RAW])", (String)Hashing.hmacSha1((Key)SHA1_KEY).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacSha1(Key[algorithm=HmacSHA1, format=RAW])", (String)Hashing.hmacSha1((byte[])keyData).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacSha256(Key[algorithm=HmacSHA256, format=RAW])", (String)Hashing.hmacSha256((Key)SHA256_KEY).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacSha256(Key[algorithm=HmacSHA256, format=RAW])", (String)Hashing.hmacSha256((byte[])keyData).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacSha512(Key[algorithm=HmacSHA512, format=RAW])", (String)Hashing.hmacSha512((Key)SHA512_KEY).toString());
        MacHashFunctionTest.assertEquals((String)"Hashing.hmacSha512(Key[algorithm=HmacSHA512, format=RAW])", (String)Hashing.hmacSha512((byte[])keyData).toString());
    }

    private static void assertMacHashing(byte[] input, String algorithm, SecretKey key, HashFunction hashFunc) throws Exception {
        Mac mac = Mac.getInstance(algorithm);
        mac.init(key);
        mac.update(input);
        MacHashFunctionTest.assertEquals((Object)HashCode.fromBytes((byte[])mac.doFinal()), (Object)hashFunc.hashBytes(input));
        MacHashFunctionTest.assertEquals((Object)HashCode.fromBytes((byte[])mac.doFinal(input)), (Object)hashFunc.hashBytes(input));
    }

    public void testRfc2202_hmacSha1_case1() {
        byte[] key = MacHashFunctionTest.fillByteArray(20, 11);
        String data = "Hi There";
        MacHashFunctionTest.checkSha1("b617318655057264e28bc0b6fb378c8ef146be00", key, data);
    }

    public void testRfc2202_hmacSha1_case2() {
        byte[] key = "Jefe".getBytes(Charsets.UTF_8);
        String data = "what do ya want for nothing?";
        MacHashFunctionTest.checkSha1("effcdf6ae5eb2fa2d27416d5f184df9c259a7c79", key, data);
    }

    public void testRfc2202_hmacSha1_case3() {
        byte[] key = MacHashFunctionTest.fillByteArray(20, 170);
        byte[] data = MacHashFunctionTest.fillByteArray(50, 221);
        MacHashFunctionTest.checkSha1("125d7342b9ac11cd91a39af48aa17b4f63f175d3", key, data);
    }

    public void testRfc2202_hmacSha1_case4() {
        byte[] key = BaseEncoding.base16().lowerCase().decode((CharSequence)"0102030405060708090a0b0c0d0e0f10111213141516171819");
        byte[] data = MacHashFunctionTest.fillByteArray(50, 205);
        MacHashFunctionTest.checkSha1("4c9007f4026250c6bc8414f9bf50c86c2d7235da", key, data);
    }

    public void testRfc2202_hmacSha1_case5() {
        byte[] key = MacHashFunctionTest.fillByteArray(20, 12);
        String data = "Test With Truncation";
        MacHashFunctionTest.checkSha1("4c1a03424b55e07fe7f27be1d58bb9324a9a5a04", key, data);
    }

    public void testRfc2202_hmacSha1_case6() {
        byte[] key = MacHashFunctionTest.fillByteArray(80, 170);
        String data = "Test Using Larger Than Block-Size Key - Hash Key First";
        MacHashFunctionTest.checkSha1("aa4ae5e15272d00e95705637ce8a3b55ed402112", key, data);
    }

    public void testRfc2202_hmacSha1_case7() {
        byte[] key = MacHashFunctionTest.fillByteArray(80, 170);
        String data = "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data";
        MacHashFunctionTest.checkSha1("e8e99d0f45237d786d6bbaa7965c7808bbff1a91", key, data);
    }

    public void testRfc2202_hmacMd5_case1() {
        byte[] key = MacHashFunctionTest.fillByteArray(16, 11);
        String data = "Hi There";
        MacHashFunctionTest.checkMd5("9294727a3638bb1c13f48ef8158bfc9d", key, data);
    }

    public void testRfc2202_hmacMd5_case2() {
        byte[] key = "Jefe".getBytes(Charsets.UTF_8);
        String data = "what do ya want for nothing?";
        MacHashFunctionTest.checkMd5("750c783e6ab0b503eaa86e310a5db738", key, data);
    }

    public void testRfc2202_hmacMd5_case3() {
        byte[] key = MacHashFunctionTest.fillByteArray(16, 170);
        byte[] data = MacHashFunctionTest.fillByteArray(50, 221);
        MacHashFunctionTest.checkMd5("56be34521d144c88dbb8c733f0e8b3f6", key, data);
    }

    public void testRfc2202_hmacMd5_case4() {
        byte[] key = BaseEncoding.base16().lowerCase().decode((CharSequence)"0102030405060708090a0b0c0d0e0f10111213141516171819");
        byte[] data = MacHashFunctionTest.fillByteArray(50, 205);
        MacHashFunctionTest.checkMd5("697eaf0aca3a3aea3a75164746ffaa79", key, data);
    }

    public void testRfc2202_hmacMd5_case5() {
        byte[] key = MacHashFunctionTest.fillByteArray(16, 12);
        String data = "Test With Truncation";
        MacHashFunctionTest.checkMd5("56461ef2342edc00f9bab995690efd4c", key, data);
    }

    public void testRfc2202_hmacMd5_case6() {
        byte[] key = MacHashFunctionTest.fillByteArray(80, 170);
        String data = "Test Using Larger Than Block-Size Key - Hash Key First";
        MacHashFunctionTest.checkMd5("6b1ab7fe4bd7bf8f0b62e6ce61b9d0cd", key, data);
    }

    public void testRfc2202_hmacMd5_case7() {
        byte[] key = MacHashFunctionTest.fillByteArray(80, 170);
        String data = "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data";
        MacHashFunctionTest.checkMd5("6f630fad67cda0ee1fb1f562db3aa53e", key, data);
    }

    private static void checkSha1(String expected, byte[] key, String data) {
        MacHashFunctionTest.checkSha1(expected, key, data.getBytes(Charsets.UTF_8));
    }

    private static void checkSha1(String expected, byte[] key, byte[] data) {
        MacHashFunctionTest.checkHmac(expected, Hashing.hmacSha1((byte[])key), data);
    }

    private static void checkMd5(String expected, byte[] key, String data) {
        MacHashFunctionTest.checkMd5(expected, key, data.getBytes(Charsets.UTF_8));
    }

    private static void checkMd5(String expected, byte[] key, byte[] data) {
        MacHashFunctionTest.checkHmac(expected, Hashing.hmacMd5((byte[])key), data);
    }

    private static void checkHmac(String expected, HashFunction hashFunc, byte[] data) {
        MacHashFunctionTest.assertEquals((Object)HashCode.fromString((String)expected), (Object)hashFunc.hashBytes(data));
    }

    private static byte[] fillByteArray(int size, int toFillWith) {
        byte[] array = new byte[size];
        Arrays.fill(array, (byte)toFillWith);
        return array;
    }
}

