/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.client.connection.nio;

import com.hazelcast.client.config.ClientIcmpPingConfig;
import com.hazelcast.client.connection.nio.ClientConnection;
import com.hazelcast.client.connection.nio.ClientConnectionManagerImpl;
import com.hazelcast.client.connection.nio.HeartbeatManager;
import com.hazelcast.client.spi.impl.ClientExecutionServiceImpl;
import com.hazelcast.internal.cluster.fd.PingFailureDetector;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.LoggingService;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.Connection;
import com.hazelcast.nio.ConnectionListener;
import com.hazelcast.util.EmptyStatement;
import com.hazelcast.util.ICMPHelper;
import java.io.IOException;
import java.net.ConnectException;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

public class ClientICMPManager
implements ConnectionListener {
    private static final long MIN_ICMP_INTERVAL_MILLIS = TimeUnit.SECONDS.toMillis(1L);
    private final ClientExecutionServiceImpl clientExecutionService;
    private final ClientConnectionManagerImpl clientConnectionManager;
    private final HeartbeatManager heartbeatManager;
    private final ILogger logger;
    private final PingFailureDetector<Connection> icmpFailureDetector;
    private final boolean icmpEnabled;
    private final int icmpTtl;
    private final int icmpTimeoutMillis;
    private final int icmpIntervalMillis;

    public ClientICMPManager(ClientIcmpPingConfig clientIcmpPingConfig, ClientExecutionServiceImpl clientExecutionService, LoggingService loggingService, ClientConnectionManagerImpl clientConnectionManager, HeartbeatManager heartbeatManager) {
        this.clientExecutionService = clientExecutionService;
        this.clientConnectionManager = clientConnectionManager;
        this.heartbeatManager = heartbeatManager;
        this.logger = loggingService.getLogger(ClientICMPManager.class);
        this.icmpTtl = clientIcmpPingConfig.getTtl();
        this.icmpTimeoutMillis = clientIcmpPingConfig.getTimeoutMilliseconds();
        this.icmpIntervalMillis = clientIcmpPingConfig.getIntervalMilliseconds();
        this.icmpEnabled = clientIcmpPingConfig.isEnabled();
        int icmpMaxAttempts = clientIcmpPingConfig.getMaxAttempts();
        if (this.icmpTimeoutMillis > this.icmpIntervalMillis) {
            throw new IllegalStateException("ICMP timeout is set to a value greater than the ICMP interval, this is not allowed.");
        }
        if ((long)this.icmpIntervalMillis < MIN_ICMP_INTERVAL_MILLIS) {
            throw new IllegalStateException("ICMP interval is set to a value less than the min allowed, " + MIN_ICMP_INTERVAL_MILLIS + "ms");
        }
        if (this.icmpEnabled) {
            if (clientIcmpPingConfig.isEchoFailFastOnStartup()) {
                this.echoFailFast();
            }
            this.icmpFailureDetector = new PingFailureDetector(icmpMaxAttempts);
        } else {
            this.icmpFailureDetector = null;
        }
    }

    private void echoFailFast() {
        this.logger.info("Checking that ICMP failure-detector is permitted. Attempting to create a raw-socket using JNI.");
        if (!ICMPHelper.isRawSocketPermitted()) {
            throw new IllegalStateException("ICMP failure-detector can't be used in this environment. Check Hazelcast Documentation Chapter on the Ping Failure Detector for supported platforms and how to enable this capability for your operating system");
        }
        this.logger.info("ICMP failure-detector is supported, enabling.");
    }

    public void start() {
        if (!this.icmpEnabled) {
            return;
        }
        this.clientConnectionManager.addConnectionListener(this);
        this.clientExecutionService.scheduleWithRepetition(new Runnable(){

            @Override
            public void run() {
                for (ClientConnection connection : ClientICMPManager.this.clientConnectionManager.getActiveConnections()) {
                    try {
                        ClientICMPManager.this.clientExecutionService.getUserExecutor().execute(new PeriodicPingTask(connection));
                    }
                    catch (Throwable e) {
                        ClientICMPManager.this.logger.severe(e);
                    }
                }
            }
        }, this.icmpIntervalMillis, this.icmpIntervalMillis, TimeUnit.MILLISECONDS);
    }

    @Override
    public void connectionAdded(Connection connection) {
    }

    @Override
    public void connectionRemoved(Connection connection) {
        if (this.icmpEnabled) {
            this.icmpFailureDetector.remove(connection);
        }
    }

    public void shutdown() {
        if (this.icmpEnabled) {
            this.icmpFailureDetector.reset();
        }
    }

    private class PeriodicPingTask
    implements Runnable {
        final ClientConnection connection;

        PeriodicPingTask(ClientConnection connection) {
            this.connection = connection;
        }

        boolean doPing(Address address, Level level) throws IOException {
            try {
                if (address.getInetAddress().isReachable(null, ClientICMPManager.this.icmpTtl, ClientICMPManager.this.icmpTimeoutMillis)) {
                    String msg = String.format("%s is pinged successfully", address);
                    ClientICMPManager.this.logger.log(level, msg);
                    return true;
                }
            }
            catch (ConnectException ignored) {
                EmptyStatement.ignore(ignored);
            }
            return false;
        }

        @Override
        public void run() {
            try {
                Address address = this.connection.getEndPoint();
                ClientICMPManager.this.logger.fine(String.format("will ping %s", address));
                if (this.doPing(address, Level.FINE)) {
                    ClientICMPManager.this.icmpFailureDetector.heartbeat(this.connection);
                    return;
                }
                ClientICMPManager.this.icmpFailureDetector.logAttempt(this.connection);
                String reason = String.format("Could not ping %s", address);
                ClientICMPManager.this.logger.warning(reason);
                if (!ClientICMPManager.this.icmpFailureDetector.isAlive(this.connection)) {
                    this.connection.onHeartbeatFailed();
                    ClientICMPManager.this.heartbeatManager.fireHeartbeatStopped(this.connection);
                }
            }
            catch (Throwable ignored) {
                EmptyStatement.ignore(ignored);
            }
            finally {
                if (!this.connection.isAlive()) {
                    ClientICMPManager.this.icmpFailureDetector.remove(this.connection);
                }
            }
        }
    }
}

