/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.map;

import com.hazelcast.config.MapConfig;
import com.hazelcast.config.MaxSizeConfig;
import com.hazelcast.core.EntryEventType;
import com.hazelcast.map.MapContainer;
import com.hazelcast.map.MapService;
import com.hazelcast.map.PartitionContainer;
import com.hazelcast.map.RecordStore;
import com.hazelcast.map.operation.EvictKeysOperation;
import com.hazelcast.map.record.Record;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.partition.InternalPartitionService;
import com.hazelcast.spi.NodeEngine;
import com.hazelcast.spi.OperationAccessor;
import com.hazelcast.spi.impl.ResponseHandlerFactory;
import com.hazelcast.util.EmptyArrays;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public class MapEvictionManager {
    static long SCHEDULER_INITIAL_DELAY = 1L;
    static long SCHEDULER_PERIOD = 1L;
    static TimeUnit SCHEDULER_TIME_UNIT = TimeUnit.SECONDS;
    private final MapService mapService;

    public MapEvictionManager(MapService mapService) {
        this.mapService = mapService;
    }

    public void init() {
        this.mapService.getNodeEngine().getExecutionService().scheduleAtFixedRate(new MapEvictTask(), SCHEDULER_INITIAL_DELAY, SCHEDULER_PERIOD, SCHEDULER_TIME_UNIT);
    }

    private Object[][] getEvictableRecords(RecordStore recordStore, MapConfig mapConfig) {
        int partitionSize = recordStore.size();
        if (partitionSize < 1) {
            return EmptyArrays.EMPTY_2D_OBJECT_ARRAY;
        }
        int evictableSize = this.getEvictableSize(partitionSize, mapConfig);
        if (evictableSize < 1) {
            return EmptyArrays.EMPTY_2D_OBJECT_ARRAY;
        }
        MapConfig.EvictionPolicy evictionPolicy = mapConfig.getEvictionPolicy();
        Map<Data, Record> entries = recordStore.getReadonlyRecordMap();
        int size = entries.size();
        if (entries.isEmpty()) {
            return EmptyArrays.EMPTY_2D_OBJECT_ARRAY;
        }
        long[] criterias = new long[size];
        int index = 0;
        for (Record record : entries.values()) {
            criterias[index] = this.getEvictionCriteriaValue(record, evictionPolicy);
            if (++index != size) continue;
            break;
        }
        if (criterias.length == 0) {
            return EmptyArrays.EMPTY_2D_OBJECT_ARRAY;
        }
        if (index < criterias.length) {
            for (int i = index; i < criterias.length; ++i) {
                criterias[i] = Long.MAX_VALUE;
            }
        }
        Arrays.sort(criterias);
        Object[][] evictableKeyValuePairs = new Object[evictableSize][2];
        int indexKVP = 0;
        int evictableBaseIndex = Math.min(evictableSize, index - 1);
        long criteriaValue = criterias[evictableBaseIndex];
        for (Map.Entry<Data, Record> entry : entries.entrySet()) {
            Record record = entry.getValue();
            long value = this.getEvictionCriteriaValue(record, evictionPolicy);
            if (value <= criteriaValue) {
                evictableKeyValuePairs[indexKVP][0] = record.getKey();
                evictableKeyValuePairs[indexKVP][1] = record.getValue();
                ++indexKVP;
            }
            if (indexKVP < evictableSize) continue;
            break;
        }
        if (evictableKeyValuePairs.length == 0) {
            return EmptyArrays.EMPTY_2D_OBJECT_ARRAY;
        }
        return evictableKeyValuePairs;
    }

    private int getEvictableSize(int currentPartitionSize, MapConfig mapConfig) {
        int evictableSize;
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = mapConfig.getMaxSizeConfig().getMaxSizePolicy();
        int evictionPercentage = mapConfig.getEvictionPercentage();
        switch (maxSizePolicy) {
            case PER_PARTITION: {
                int maxSize = mapConfig.getMaxSizeConfig().getSize();
                int targetSizePerPartition = Double.valueOf((double)maxSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case PER_NODE: {
                int maxSize = mapConfig.getMaxSizeConfig().getSize();
                int memberCount = this.mapService.getNodeEngine().getClusterService().getMembers().size();
                int maxPartitionSize = maxSize * memberCount / this.mapService.getNodeEngine().getPartitionService().getPartitionCount();
                int targetSizePerPartition = Double.valueOf((double)maxPartitionSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case USED_HEAP_PERCENTAGE: 
            case USED_HEAP_SIZE: {
                evictableSize = Math.max(currentPartitionSize * evictionPercentage / 100, 1);
                break;
            }
            default: {
                throw new IllegalArgumentException("Max size policy is not defined [" + (Object)((Object)maxSizePolicy) + "]");
            }
        }
        return evictableSize;
    }

    private long getEvictionCriteriaValue(Record record, MapConfig.EvictionPolicy evictionPolicy) {
        long value;
        switch (evictionPolicy) {
            case LRU: 
            case LFU: {
                value = record.getEvictionCriteriaNumber();
                break;
            }
            default: {
                throw new IllegalArgumentException("Not an appropriate eviction policy [" + (Object)((Object)evictionPolicy) + ']');
            }
        }
        return value;
    }

    private class EvictRunner
    implements Runnable {
        private final int mod;
        private final MapConfig mapConfig;
        private final String mapName;

        private EvictRunner(MapContainer mapContainer, int mode) {
            this.mod = mode;
            this.mapName = mapContainer.getName();
            this.mapConfig = mapContainer.getMapConfig();
        }

        @Override
        public void run() {
            MapConfig mapConfig = this.mapConfig;
            String mapName = this.mapName;
            MapService mapService = MapEvictionManager.this.mapService;
            NodeEngine nodeEngine = mapService.getNodeEngine();
            Set<Data> keysGatheredForNearCacheEviction = Collections.emptySet();
            for (int i = 0; i < nodeEngine.getPartitionService().getPartitionCount(); ++i) {
                Object[][] evictableKeyValuePairs;
                PartitionContainer pc;
                RecordStore recordStore;
                Collection<Record> values;
                if (i % 8 != this.mod) continue;
                Address owner = nodeEngine.getPartitionService().getPartitionOwner(i);
                if (!nodeEngine.getThisAddress().equals(owner) || (values = (recordStore = (pc = mapService.getPartitionContainer(i)).getRecordStore(mapName)).getReadonlyRecordMap().values()).isEmpty() || (evictableKeyValuePairs = MapEvictionManager.this.getEvictableRecords(recordStore, mapConfig)).length == 0) continue;
                HashSet<Data> keySet = new HashSet<Data>(evictableKeyValuePairs.length);
                for (Object[] kvp : evictableKeyValuePairs) {
                    if (kvp[0] == null) continue;
                    keySet.add((Data)kvp[0]);
                }
                if (keySet.isEmpty()) continue;
                keysGatheredForNearCacheEviction = new HashSet<Data>(keySet.size());
                keysGatheredForNearCacheEviction.addAll(keySet);
                EvictKeysOperation evictKeysOperation = new EvictKeysOperation(mapName, keySet);
                evictKeysOperation.setNodeEngine(nodeEngine);
                evictKeysOperation.setServiceName("hz:impl:mapService");
                evictKeysOperation.setResponseHandler(ResponseHandlerFactory.createEmptyResponseHandler());
                evictKeysOperation.setPartitionId(i);
                OperationAccessor.setCallerAddress(evictKeysOperation, nodeEngine.getThisAddress());
                nodeEngine.getOperationService().executeOperation(evictKeysOperation);
                for (Object[] kvp : evictableKeyValuePairs) {
                    if (kvp[0] == null) continue;
                    mapService.publishEvent(nodeEngine.getThisAddress(), mapName, EntryEventType.EVICTED, (Data)kvp[0], mapService.toData(kvp[1]), null);
                }
            }
            if (mapService.isNearCacheAndInvalidationEnabled(mapName)) {
                mapService.invalidateAllNearCaches(mapName, keysGatheredForNearCacheEviction);
            }
        }
    }

    private class MapEvictTask
    implements Runnable {
        private static final String EXECUTOR_NAME = "hz:map-evict";

        @Override
        public void run() {
            MapService mapService = MapEvictionManager.this.mapService;
            Map<String, MapContainer> mapContainers = mapService.getMapContainers();
            for (MapContainer mapContainer : mapContainers.values()) {
                if (!this.evictionPolicyConfigured(mapContainer) || !this.evictable(mapContainer)) continue;
                this.evictMap(mapContainer);
            }
        }

        private void evictMap(MapContainer mapContainer) {
            NodeEngine nodeEngine = MapEvictionManager.this.mapService.getNodeEngine();
            for (int i = 0; i < 8; ++i) {
                EvictRunner runner = new EvictRunner(mapContainer, i);
                nodeEngine.getExecutionService().execute(EXECUTOR_NAME, runner);
            }
        }

        private boolean evictionPolicyConfigured(MapContainer mapContainer) {
            MapConfig.EvictionPolicy evictionPolicy = mapContainer.getMapConfig().getEvictionPolicy();
            MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
            return !MapConfig.EvictionPolicy.NONE.equals((Object)evictionPolicy) && maxSizeConfig.getSize() > 0;
        }

        private boolean evictable(MapContainer mapContainer) {
            boolean result;
            MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
            MaxSizeConfig.MaxSizePolicy maxSizePolicy = maxSizeConfig.getMaxSizePolicy();
            switch (maxSizePolicy) {
                case PER_NODE: {
                    result = this.isEvictablePerNode(mapContainer);
                    break;
                }
                case PER_PARTITION: {
                    result = this.isEvictablePerPartition(mapContainer);
                    break;
                }
                case USED_HEAP_PERCENTAGE: {
                    result = this.isEvictableHeapPercentage(mapContainer);
                    break;
                }
                case USED_HEAP_SIZE: {
                    result = this.isEvictableHeapSize(mapContainer);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Not an appropriate max size policy [" + (Object)((Object)maxSizePolicy) + ']');
                }
            }
            return result;
        }

        private boolean isEvictablePerNode(MapContainer mapContainer) {
            int nodeTotalSize = 0;
            MapService mapService = MapEvictionManager.this.mapService;
            MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
            int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
            String mapName = mapContainer.getName();
            NodeEngine nodeEngine = mapService.getNodeEngine();
            InternalPartitionService partitionService = nodeEngine.getPartitionService();
            int partitionCount = partitionService.getPartitionCount();
            for (int i = 0; i < partitionCount; ++i) {
                PartitionContainer container;
                Address owner = partitionService.getPartitionOwner(i);
                if (!nodeEngine.getThisAddress().equals(owner) || (container = mapService.getPartitionContainer(i)) == null || (nodeTotalSize += this.getRecordStoreSize(mapName, container)) < maxSize) continue;
                return true;
            }
            return false;
        }

        private int getApproximateMaxSize(int maxSizeFromConfig) {
            return maxSizeFromConfig * 95 / 100;
        }

        private boolean isEvictablePerPartition(MapContainer mapContainer) {
            MapService mapService = MapEvictionManager.this.mapService;
            MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
            int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
            String mapName = mapContainer.getName();
            NodeEngine nodeEngine = mapService.getNodeEngine();
            InternalPartitionService partitionService = nodeEngine.getPartitionService();
            for (int i = 0; i < partitionService.getPartitionCount(); ++i) {
                int size;
                PartitionContainer container;
                Address owner = partitionService.getPartitionOwner(i);
                if (!nodeEngine.getThisAddress().equals(owner) || (container = mapService.getPartitionContainer(i)) == null || (size = this.getRecordStoreSize(mapName, container)) < maxSize) continue;
                return true;
            }
            return false;
        }

        private int getRecordStoreSize(String mapName, PartitionContainer partitionContainer) {
            RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
            if (existingRecordStore == null) {
                return 0;
            }
            return existingRecordStore.size();
        }

        private boolean isEvictableHeapSize(MapContainer mapContainer) {
            long usedHeapSize = this.getUsedHeapSize(mapContainer);
            if (usedHeapSize == -1L) {
                return false;
            }
            MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
            int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
            return (long)maxSize < usedHeapSize / 1024L / 1024L;
        }

        private boolean isEvictableHeapPercentage(MapContainer mapContainer) {
            long total;
            long usedHeapSize = this.getUsedHeapSize(mapContainer);
            if (usedHeapSize == -1L) {
                return false;
            }
            MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
            int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
            boolean result = (double)maxSize < 100.0 * (double)usedHeapSize / (double)(total = Runtime.getRuntime().totalMemory());
            return result;
        }

        private long getUsedHeapSize(MapContainer mapContainer) {
            long heapCost = 0L;
            MapService mapService = MapEvictionManager.this.mapService;
            String mapName = mapContainer.getName();
            NodeEngine nodeEngine = mapService.getNodeEngine();
            Address thisAddress = nodeEngine.getThisAddress();
            for (int i = 0; i < nodeEngine.getPartitionService().getPartitionCount(); ++i) {
                if (!nodeEngine.getPartitionService().getPartition(i).isOwnerOrBackup(thisAddress)) continue;
                PartitionContainer container = mapService.getPartitionContainer(i);
                if (container == null) {
                    return -1L;
                }
                heapCost += this.getRecordStoreHeapCost(mapName, container);
            }
            return heapCost += mapContainer.getNearCacheSizeEstimator().getSize();
        }

        private long getRecordStoreHeapCost(String mapName, PartitionContainer partitionContainer) {
            RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
            if (existingRecordStore == null) {
                return 0L;
            }
            return existingRecordStore.getHeapCost();
        }
    }
}

