/*
    This file is part of the iText (R) project.
    Copyright (c) 1998-2021 iText Group NV
    Authors: Bruno Lowagie, Paulo Soares, et al.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License version 3
    as published by the Free Software Foundation with the addition of the
    following permission added to Section 15 as permitted in Section 7(a):
    FOR ANY PART OF THE COVERED WORK IN WHICH THE COPYRIGHT IS OWNED BY
    ITEXT GROUP. ITEXT GROUP DISCLAIMS THE WARRANTY OF NON INFRINGEMENT
    OF THIRD PARTY RIGHTS
    
    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Affero General Public License for more details.
    You should have received a copy of the GNU Affero General Public License
    along with this program; if not, see http://www.gnu.org/licenses or write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA, 02110-1301 USA, or download the license from the following URL:
    http://itextpdf.com/terms-of-use/
    
    The interactive user interfaces in modified source and object code versions
    of this program must display Appropriate Legal Notices, as required under
    Section 5 of the GNU Affero General Public License.
    
    In accordance with Section 7(b) of the GNU Affero General Public License,
    a covered work must retain the producer line in every PDF that is created
    or manipulated using iText.
    
    You can be released from the requirements of the license by purchasing
    a commercial license. Buying such a license is mandatory as soon as you
    develop commercial activities involving the iText software without
    disclosing the source code of your own applications.
    These activities include: offering paid services to customers as an ASP,
    serving PDFs on the fly in a web application, shipping iText with a closed
    source product.
    
    For more information, please contact iText Software Corp. at this
    address: sales@itextpdf.com
 */
package com.itextpdf.html2pdf.css.apply.impl;

import com.itextpdf.html2pdf.css.CssConstants;
import com.itextpdf.html2pdf.css.apply.ICssApplier;
import com.itextpdf.html2pdf.css.apply.ICssApplierFactory;
import com.itextpdf.html2pdf.exception.CssApplierInitializationException;
import com.itextpdf.html2pdf.util.TagProcessorMapping;
import com.itextpdf.styledxmlparser.node.IElementNode;

/**
 * A factory for creating {@link ICssApplier} objects.
 */
public class DefaultCssApplierFactory implements ICssApplierFactory {

    private static final ICssApplierFactory INSTANCE = new DefaultCssApplierFactory();

    /** The default mapping of CSS keywords and CSS appliers. */
    private TagProcessorMapping defaultMapping;

    /**
     * Creates a new {@link DefaultCssApplierFactory} instance.
     */
    public DefaultCssApplierFactory() {
        defaultMapping = DefaultTagCssApplierMapping.getDefaultCssApplierMapping();
    }

    /**
     * Gets {@link DefaultCssApplierFactory} instance.
     * @return default instance that is used if custom css appliers are not configured
     */
    public static ICssApplierFactory getInstance() {
        return INSTANCE;
    }

    /* (non-Javadoc)
     * @see com.itextpdf.html2pdf.css.apply.ICssApplierFactory#getCssApplier(com.itextpdf.html2pdf.html.node.IElementNode)
     */
    @Override
    public final ICssApplier getCssApplier(IElementNode tag) {
        ICssApplier cssApplier = getCustomCssApplier(tag);

        if (cssApplier == null) {
            Class<?> cssApplierClass = getCssApplierClass(defaultMapping, tag);
            if ( cssApplierClass != null ) {
                try {
                    return (ICssApplier) cssApplierClass.newInstance();
                } catch (Exception e) {
                    throw new CssApplierInitializationException(CssApplierInitializationException.ReflectionFailed, cssApplierClass.getName(), tag.name());
                }
            }
        }

        return cssApplier;
    }

    /**
     * Gets a custom CSS applier.
     * This method needs to be overridden because the default CSS applier
     * factory will always return {@code null}.
     *
     * @param tag the key
     * @return the custom CSS applier
     */
    public ICssApplier getCustomCssApplier(IElementNode tag) {
        return null;
    }

    /**
     * Gets the css applier class.
     *
     * @param mapping the mapping
     * @param tag the tag
     * @return the css applier class
     */
    private static Class<?> getCssApplierClass(TagProcessorMapping mapping, IElementNode tag) {
        Class<?> cssApplierClass = null;
        String display = tag.getStyles() != null ? tag.getStyles().get(CssConstants.DISPLAY) : null;
        if (display != null) {
            cssApplierClass = mapping.getMapping(tag.name(), display);
        }
        if (cssApplierClass == null) {
            cssApplierClass = mapping.getMapping(tag.name());
        }
        return cssApplierClass;
    }

}
