package com.jpattern.orm.test;

import static org.junit.Assert.*;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;

import org.joda.time.DateMidnight;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.junit.Test;

import com.jpattern.orm.BaseTestShared;
import com.jpattern.orm.JPO;
import com.jpattern.orm.session.ResultSetReader;
import com.jpattern.orm.session.Session;
import com.jpattern.orm.test.domain.WrapperTypeTable;
import com.jpattern.orm.transaction.Transaction;

/**
 * 
 * @author Francesco Cina'
 *
 * Apr 17, 2012
 */
public class WrapperTypeTableTest extends BaseTestShared {

	@Override
	protected void setUp() throws Exception {
	}

	@Override
	protected void tearDown() throws Exception {
	}

	@Test
	public void testCrudWithWrapperType() {
		final JPO jpOrm = getJPOrm();

		DateMidnight endDate = new DateMidnight(DateTimeZone.UTC);
		DateTime startDate = new DateTime(DateTimeZone.UTC);
		final Date now = new Date();

		WrapperTypeTable wrapper1 = new WrapperTypeTable();
		wrapper1.setNow(now);
		wrapper1.setEndDate(endDate);
		wrapper1.setStartDate(startDate);

		assertEquals( Long.valueOf(-1l), wrapper1.getId() );

		// CREATE
		final Session conn = jpOrm.session();
		Transaction tx = conn.transaction();
		wrapper1 = conn.save(wrapper1);
		tx.commit();


		System.out.println("wrapper1 id: " + wrapper1.getId());
		assertTrue( wrapper1.getId() >= Long.valueOf(0) );

		seeDBValues(conn, wrapper1.getId());

		// LOAD
		WrapperTypeTable wrapperLoad1 = conn.find(WrapperTypeTable.class, wrapper1.getId() );
		assertNotNull(wrapperLoad1);
		assertEquals( wrapper1.getId(), wrapperLoad1.getId() );
		assertNull( wrapperLoad1.getValid() );
		assertEquals( now, wrapperLoad1.getNow() );
		assertEquals( DateTimeZone.UTC, wrapperLoad1.getStartDate().getZone() );
		assertEquals( startDate, wrapperLoad1.getStartDate() );
		assertEquals( endDate, wrapperLoad1.getEndDate() );

		//UPDATE
		tx = conn.transaction();
		endDate = new DateMidnight(DateTimeZone.UTC);
		startDate = new DateTime(DateTimeZone.UTC);
		final boolean valid = true;

		wrapperLoad1.setEndDate(endDate);
		wrapperLoad1.setStartDate(startDate);
		wrapperLoad1.setValid(valid);
		wrapperLoad1 = conn.update(wrapperLoad1);
		assertEquals( DateTimeZone.UTC, wrapperLoad1.getStartDate().getZone() );

		tx.commit();

		// LOAD
		final WrapperTypeTable wrapperLoad2 = conn.find(WrapperTypeTable.class, wrapper1.getId() );
		assertNotNull(wrapperLoad2);
		assertEquals( wrapperLoad1.getId(), wrapperLoad2.getId() );
		assertEquals( valid, wrapperLoad2.getValid() );
		assertEquals( DateTimeZone.UTC, wrapperLoad2.getStartDate().getZone() );
		assertEquals( startDate, wrapperLoad2.getStartDate() );
		assertEquals( endDate, wrapperLoad2.getEndDate() );
		assertEquals( now, wrapperLoad1.getNow() );

		//DELETE
		tx = conn.transaction();
		conn.delete(wrapperLoad2);
		tx.commit();
		final WrapperTypeTable wrapperLoad3 = conn.find(WrapperTypeTable.class, wrapper1.getId() );
		assertNull(wrapperLoad3);
	}

	@Test
	public void testQueryWithWrapperType() {
		final JPO jpOrm = getJPOrm();

		DateMidnight endDate = new DateMidnight(DateTimeZone.UTC);
		DateTime startDate = new DateTime(DateTimeZone.UTC);
		final Date now = new Date();

		WrapperTypeTable wrapper1 = new WrapperTypeTable();
		wrapper1.setNow(now);
		wrapper1.setEndDate(endDate);
		wrapper1.setStartDate(startDate);

		assertEquals( Long.valueOf(-1l), wrapper1.getId() );

		// CREATE
		final Session conn = jpOrm.session();
		Transaction tx = conn.transaction();
		wrapper1 = conn.save(wrapper1);
		tx.commit();


		System.out.println("wrapper1 id: " + wrapper1.getId());
		assertTrue( wrapper1.getId() >= Long.valueOf(0) );

		seeDBValues(conn, wrapper1.getId());

		// LOAD
		final WrapperTypeTable wrapperLoad1 = conn.findQuery(WrapperTypeTable.class).where().eq("startDate", startDate).eq("now", now).eq("endDate", endDate).findUnique();
		assertNotNull(wrapperLoad1);
		assertEquals( wrapper1.getId(), wrapperLoad1.getId() );
		assertNull( wrapperLoad1.getValid() );
		assertEquals( now, wrapperLoad1.getNow() );
		assertEquals( startDate, wrapperLoad1.getStartDate() );
		assertEquals( endDate, wrapperLoad1.getEndDate() );

		//UPDATE
		tx = conn.transaction();
		endDate = new DateMidnight(DateTimeZone.UTC);
		startDate = new DateTime(DateTimeZone.UTC);
		final boolean valid = true;

		//		conn.updateQuery(clazz)
		final int updated = conn.updateQuery(WrapperTypeTable.class)
				.set().eq("startDate", startDate).eq("valid", valid).eq("endDate", endDate)
				.where().eq("id", wrapper1.getId())
				.perform();

		assertEquals(1, updated);

		tx.commit();

		// LOAD
		final WrapperTypeTable wrapperLoad2 = conn.findQuery(WrapperTypeTable.class).where().eq("id", wrapper1.getId()).findUnique();

		assertNotNull(wrapperLoad2);
		assertEquals( wrapperLoad1.getId(), wrapperLoad2.getId() );
		assertEquals( valid, wrapperLoad2.getValid() );
		assertEquals( startDate, wrapperLoad2.getStartDate() );
		assertEquals( endDate, wrapperLoad2.getEndDate() );
		assertEquals( now, wrapperLoad1.getNow() );

		//DELETE
		tx = conn.transaction();
		final int deleted = conn.deleteQuery(WrapperTypeTable.class).where().eq("id", wrapper1.getId()).perform();
		assertEquals(1, deleted);
		tx.commit();

		assertTrue( conn.findQuery(WrapperTypeTable.class).where().eq("id", wrapper1.getId()).findList().isEmpty() );
	}

	private void seeDBValues(final Session conn, final Long id) {
		final ResultSetReader<Object> rse = new ResultSetReader<Object>() {

			@Override
			public Object read(final ResultSet resultSet) throws SQLException {

				while(resultSet.next()) {
					System.out.println("++++++++++++++++++++++++++++++++++++++++++++++++++++");
					System.out.println("Object found:");
					System.out.println("id: " + resultSet.getLong("id"));
					System.out.println("now: " + resultSet.getTimestamp("now"));
					System.out.println("start_date: " + resultSet.getTimestamp("start_date"));
					System.out.println("end_date: " + resultSet.getTimestamp("end_date"));
					System.out.println("valid: " + resultSet.getBigDecimal("valid"));
					System.out.println("++++++++++++++++++++++++++++++++++++++++++++++++++++");
				}

				return null;
			}
		};
		conn.sqlExecutor().findQuery("select * from WRAPPER_TYPE_TABLE").find(rse);

	}



}
