package com.jpattern.orm.test.query;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;

import org.junit.Test;

import com.jpattern.orm.BaseTestShared;
import com.jpattern.orm.JPO;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.query.sql.SqlExecutor;
import com.jpattern.orm.query.sql.SqlFindQuery;
import com.jpattern.orm.session.ResultSetReader;
import com.jpattern.orm.session.ResultSetRowReader;
import com.jpattern.orm.session.Session;
import com.jpattern.orm.test.domain.Employee;
import com.jpattern.orm.transaction.Transaction;

/**
 * 
 * @author Francesco Cina
 *
 * 02/lug/2011
 */
public class PlainSqlExecutorsRowReaderTest extends BaseTestShared {

	private Employee employee1;
	private Employee employee2;
	private Employee employee3;
	private Session session;

	@Override
	protected void setUp() throws Exception {
		final JPO jpOrm = getJPOrm();
		jpOrm.register(Employee.class);

		this.session = jpOrm.session();
		Transaction tx = this.session.transaction();

		this.session.deleteQuery(Employee.class).perform();

		final Random random = new Random();
		this.employee1 = new Employee();
		this.employee1.setId( random.nextInt() );
		this.employee1.setAge( 44 );
		this.session.save(this.employee1);

		this.employee2 = new Employee();
		this.employee2.setId( random.nextInt() );
		this.employee2.setAge( 44 );
		this.session.save(this.employee2);

		this.employee3 = new Employee();
		this.employee3.setId( random.nextInt() );
		this.employee3.setAge( 45 );
		this.session.save(this.employee3);

		tx.commit();
	}

	@Override
	protected void tearDown() throws Exception {
		Transaction tx = this.session.transaction();
		this.session.delete(this.employee1);
		this.session.delete(this.employee2);
		this.session.delete(this.employee3);
		tx.commit();
	}

	@Test
	public void testResultSetReaderWithTwoResults() {
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 44);
		List<Integer> results = findQuery.find(new ResultSetReader<List<Integer>>() {
			@Override
			public List<Integer> read(final ResultSet resultSet) throws SQLException {
				List<Integer> results = new ArrayList<Integer>();
				while (resultSet.next()) {
					results.add( resultSet.getInt("id") );
				}
				return results;
			}
		});
		System.out.println("Result is " + results);
		assertEquals( 2, results.size() );
		assertTrue( results.contains( this.employee1.getId() ) );
		assertTrue( results.contains( this.employee2.getId() ) );
	}

	@Test
	public void testResultSetRowReaderWithTwoResults() {
		final AtomicInteger atomicRownNum = new AtomicInteger(-1);
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 44);
		List<Integer> results = findQuery.find(new ResultSetRowReader<Integer>() {
			@Override
			public Integer readRow(final ResultSet rs, final int rowNum) throws SQLException {
				atomicRownNum.set(rowNum);
				return rs.getInt("id");
			}
		});
		System.out.println("Result is " + results);
		System.out.println("atomicRownNum is " + atomicRownNum);
		assertEquals( 2, results.size() );
		assertEquals( 1, atomicRownNum.get() );
		assertTrue( results.contains( this.employee1.getId() ) );
		assertTrue( results.contains( this.employee2.getId() ) );
	}

	@Test
	public void testResultSetRowReaderWithOneResult() {
		final AtomicInteger atomicRownNum = new AtomicInteger(-1);
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 45);
		List<Integer> results = findQuery.find(new ResultSetRowReader<Integer>() {
			@Override
			public Integer readRow(final ResultSet rs, final int rowNum) throws SQLException {
				atomicRownNum.set(rowNum);
				return rs.getInt("id");
			}
		});
		System.out.println("Result is " + results);
		System.out.println("atomicRownNum is " + atomicRownNum);
		assertEquals( 1, results.size() );
		assertEquals( 0, atomicRownNum.get() );
		assertTrue( results.contains( this.employee3.getId() ) );
	}

	@Test
	public void testResultSetRowReaderWithNoResult() {
		final AtomicInteger atomicRownNum = new AtomicInteger(-1);
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 46);
		List<Integer> results = findQuery.find(new ResultSetRowReader<Integer>() {
			@Override
			public Integer readRow(final ResultSet rs, final int rowNum) throws SQLException {
				atomicRownNum.set(rowNum);
				return rs.getInt("id");
			}
		});
		System.out.println("Result is " + results);
		System.out.println("atomicRownNum is " + atomicRownNum);
		assertEquals( 0, results.size() );
		assertEquals( -1, atomicRownNum.get() );
	}

	@Test
	public void testResultSetRowReaderUniqueWithTwoResults() {
		final AtomicInteger atomicRownNum = new AtomicInteger(-1);
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 44);
		try {
			findQuery.findUnique(new ResultSetRowReader<Integer>() {
				@Override
				public Integer readRow(final ResultSet rs, final int rowNum) throws SQLException {
					atomicRownNum.set(rowNum);
					return rs.getInt("id");
				}
			});
			fail("an exception should be thrown before");
		} catch (OrmNotUniqueResultException e) {
			assertTrue( e.getMessage().contains("higher") );
		}
	}

	@Test
	public void testResultSetRowReaderUniqueWithNoResults() {
		final AtomicInteger atomicRownNum = new AtomicInteger(-1);
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 46);
		try {
			findQuery.findUnique(new ResultSetRowReader<Integer>() {
				@Override
				public Integer readRow(final ResultSet rs, final int rowNum) throws SQLException {
					atomicRownNum.set(rowNum);
					return rs.getInt("id");
				}
			});
			fail("an exception should be thrown before");
		} catch (OrmNotUniqueResultException e) {
			assertTrue( e.getMessage().contains("zero") );
		}
	}

	@Test
	public void testResultSetRowReaderUniqueWithOneResult() {
		final AtomicInteger atomicRownNum = new AtomicInteger(-1);
		SqlExecutor sqlExecutor = this.session.sqlExecutor();
		SqlFindQuery findQuery = sqlExecutor.findQuery("select * from Employee where age = ?", 45);
		Integer result = findQuery.findUnique(new ResultSetRowReader<Integer>() {
			@Override
			public Integer readRow(final ResultSet rs, final int rowNum) throws SQLException {
				atomicRownNum.set(rowNum);
				return rs.getInt("id");
			}
		});
		assertEquals( this.employee3.getId(), result );
		assertEquals( 0, atomicRownNum.get() );
	}

}
