package com.jpattern.orm.test.query;

import static org.junit.Assert.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import org.junit.Test;

import com.jpattern.orm.JPOrm;
import com.jpattern.orm.BaseTestShared;
import com.jpattern.orm.JPO;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.query.clause.where.LeExpressionElement;
import com.jpattern.orm.query.find.FindQuery;
import com.jpattern.orm.session.Session;
import com.jpattern.orm.test.domain.Employee;
import com.jpattern.orm.test.domain.Zoo_People;
import com.jpattern.orm.transaction.Transaction;

/**
 * 
 * @author Francesco Cina
 *
 * 23/giu/2011
 */
public class QueryExecutionTest extends BaseTestShared {

	@Override
	protected void setUp() throws Exception {
	}

	@Override
	protected void tearDown() throws Exception {
	}

	@Test
	public void testQuery1() throws Exception {
		final JPOrm jpOrm = getJPOrm();
		final List<Class<?>> classes = new ArrayList<Class<?>>();
		classes.add(Employee.class);
		classes.add(Zoo_People.class);

		//		jpOrm.register(classes, true);

		final Session session =  jpOrm.session();
		final Employee employee = createEmployee(jpOrm);

		final FindQuery<Employee> query = session.findQuery(Employee.class);
		System.out.println(query.renderSql());
		System.out.println(query.getGeneratedRowCountSql());
		final String expectedSql = "SELECT Employee.ID, Employee.NAME, Employee.AGE, Employee.SURNAME, Employee.EMPLOYEE_NUMBER FROM EMPLOYEE Employee ";
		assertEquals(expectedSql , query.renderSql());

		final List<Employee> employeeList = query.findList();
		assertNotNull( employeeList );

		final long countRowQueryResult = query.findRowCount();

		System.out.println("found employees: " + employeeList.size());
		System.out.println("count row query result: " + countRowQueryResult);
		assertTrue( employeeList.size()>0 );
		assertEquals( employeeList.size(), countRowQueryResult );

		deleteEmployee(jpOrm, employee);
	}

	@Test
	public void testQuery2() throws Exception {
		final JPOrm jpOrm = getJPOrm();
		jpOrm.register(Employee.class);
		jpOrm.register(Zoo_People.class);

		final Session session =  jpOrm.session();
		final Employee employee = createEmployee(jpOrm);

		final int maxRows = 4;
		final FindQuery<Employee> query = session.findQuery(Employee.class);
		query.join(Zoo_People.class, "zp");
		query.setMaxRows(maxRows);
		query.where().not( new LeExpressionElement("Employee.id", "0") );
		query.where().ilike("zp.firstname", "%");
		System.out.println(query.renderSql());
		final String expectedSql = "SELECT Employee.ID, Employee.NAME, Employee.AGE, Employee.SURNAME, Employee.EMPLOYEE_NUMBER FROM EMPLOYEE Employee , ZOO.PEOPLE zp WHERE NOT ( Employee.ID <= ? ) AND LOWER(zp.FIRSTNAME) like LOWER(?) ";
		assertEquals(expectedSql , query.renderSql());

		final List<Employee> employeeList = query.findList();
		assertNotNull( employeeList );

		System.out.println("found employees: " + employeeList.size());
		assertTrue( employeeList.size()<=maxRows );

		deleteEmployee(jpOrm, employee);
	}

	@Test
	public void testQuery3() throws Exception {
		final JPOrm jpOrm = getJPOrm();
		jpOrm.register(Employee.class);
		jpOrm.register(Zoo_People.class);

		final Session session =  jpOrm.session();
		final Employee employee = createEmployee(jpOrm);

		final int maxRows = 4;
		final FindQuery<Employee> query = session.findQuery(Employee.class, "e");
		query.setMaxRows(maxRows);
		query.where().ge("e.id", "0");
		System.out.println(query.renderSql());
		final String expectedSql = "SELECT e.ID, e.NAME, e.AGE, e.SURNAME, e.EMPLOYEE_NUMBER FROM EMPLOYEE e WHERE e.ID >= ? ";
		assertEquals(expectedSql , query.renderSql());

		final List<Employee> employeeList = query.findList();
		assertNotNull( employeeList );

		System.out.println("found employees: " + employeeList.size());
		assertTrue( employeeList.size()>0 );
		assertTrue( employeeList.size()<=maxRows );

		deleteEmployee(jpOrm, employee);
	}

	@Test
	public void testQuery4() throws Exception {
		final JPOrm jpOrm = getJPOrm();
		jpOrm.register(Employee.class);
		jpOrm.register(Zoo_People.class);

		final Session session =  jpOrm.session();
		final Employee employee = createEmployee(jpOrm);

		final Transaction tx = session.transaction();

		//find list with one result
		final FindQuery<Employee> query1 = session.findQuery(Employee.class);
		query1.where().eq("Employee.id", employee.getId());
		assertEquals( 1 , query1.findList().size() );

		//find list with zero result
		final FindQuery<Employee> query2 = session.findQuery(Employee.class);
		query2.where().eq("Employee.id", (-employee.getId()) );
		assertEquals( 0 , query2.findList().size() );

		//find unique query
		final FindQuery<Employee> query3 = session.findQuery(Employee.class);
		query3.where().eq("Employee.id", employee.getId());
		assertNotNull( query3.findUnique() );

		//find unique query exception
		final FindQuery<Employee> query4 = session.findQuery(Employee.class);
		query4.where().eq("Employee.id", -employee.getId());
		boolean notUniqueResultException = false;
		try{
			assertNull( query4.findUnique() );
		} catch (final OrmNotUniqueResultException e) {
			notUniqueResultException = true;
		}
		assertTrue(notUniqueResultException);

		//find unique
		assertNotNull( session.findUnique(Employee.class, employee.getId()) );

		//find unique exception
		notUniqueResultException = false;
		try{
			assertNull( session.findUnique(Employee.class, -employee.getId()) );
		} catch (final OrmNotUniqueResultException e) {
			notUniqueResultException = true;
		}
		assertTrue(notUniqueResultException);

		tx.commit();

		deleteEmployee(jpOrm, employee);
	}

	private Employee createEmployee(final JPO jpOrm) {
		final Session ormSession = jpOrm.session();
		final Transaction tx = ormSession.transaction();
		final int id = new Random().nextInt(1000000);
		final Employee employee = new Employee();
		employee.setId( id );
		employee.setAge( 44 );
		employee.setEmployeeNumber( "empNumber" + id );
		employee.setName("Wizard");
		employee.setSurname("Cina");
		ormSession.save(employee);
		tx.commit();
		return employee;
	}

	private void deleteEmployee(final JPO jpOrm, final Employee employee) {
		final Session ormSession = jpOrm.session();
		final Transaction tx = ormSession.transaction();
		ormSession.delete(employee);
		tx.commit();
	}

}
