package com.jpattern.orm.test.query.forupdate;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;

import java.util.Random;

import org.junit.Test;

import com.jpattern.orm.BaseTestShared;
import com.jpattern.orm.JPO;
import com.jpattern.orm.query.OrmRowMapper;
import com.jpattern.orm.query.LockMode;
import com.jpattern.orm.query.find.FindQuery;
import com.jpattern.orm.session.Session;
import com.jpattern.orm.test.domain.Employee;
import com.jpattern.orm.test.domain.Zoo_People;
import com.jpattern.orm.transaction.Transaction;

/**
 * 
 * @author Francesco Cina'
 *
 * 30/ago/2011
 */
public class QuerySelectForUpdateExecutionTest extends BaseTestShared {

	private final long THREAD_SLEEP = 250l;

	@Override
	protected void setUp() throws Exception {
	}

	@Override
	protected void tearDown() throws Exception {
	}

	@Test
	public void testQuery1() throws Exception {
		final JPO jpOrm =getJPOrm();
		jpOrm.register(Employee.class);
		jpOrm.register(Zoo_People.class);

		final Session session =  jpOrm.session();
		final Employee employeeLocked = createEmployee(jpOrm);
		final Employee employeeUnlocked = createEmployee(jpOrm);

		final ActorLockForUpdate actor1 = new ActorLockForUpdate(jpOrm, employeeLocked.getId(), LockMode.FOR_UPDATE, "locked");
		final Thread thread1 = new Thread( actor1 );
		thread1.start();

		Thread.sleep(this.THREAD_SLEEP / 5);

		final ActorLockForUpdate actor2 = new ActorLockForUpdate(jpOrm, employeeLocked.getId(), LockMode.FOR_UPDATE, "locked2");
		final Thread thread2 = new Thread( actor2 );
		thread2.start();

		thread1.join();
		thread2.join();
		assertFalse(actor1.exception);
		assertFalse(actor2.exception);

		assertEquals( "name_locked_locked2" ,  session.find(Employee.class, employeeLocked.getId()).getName() );

		deleteEmployee(jpOrm, employeeLocked);
		deleteEmployee(jpOrm, employeeUnlocked);
	}




	public class ActorLockForUpdate implements Runnable {

		private final JPO jpOrm;
		private final LockMode lockMode;
		private final String name;
		private final long employeeId;
		boolean exception = false;

		public ActorLockForUpdate(final JPO jpOrm, final long employeeId, final LockMode lockMode, final String name) {
			this.jpOrm = jpOrm;
			this.employeeId = employeeId;
			this.lockMode = lockMode;
			this.name = name;
		}

		@Override
		public void run() {
			System.out.println("Run: " + this.name);
			final Session session;
			try {
				session = this.jpOrm.session();

				final Transaction tx = session.transaction();

				final FindQuery<Employee> query = session.findQuery(Employee.class);
				query.where().eq("Employee.id", this.employeeId);
				query.setLockMode(this.lockMode);
				System.out.println("Thread " + this.name + " executing query [" + query.renderSql() + "]");

				final OrmRowMapper<Employee> srr = new OrmRowMapper<Employee>() {
					@Override
					public void read(final Employee employee, final int rowCount) {
						System.out.println("Thread " + ActorLockForUpdate.this.name + " - employee.getName() = [" + employee.getName() + "]");
						assertNotNull(employee);

						try {
							Thread.sleep(QuerySelectForUpdateExecutionTest.this.THREAD_SLEEP);
						} catch (final InterruptedException e) {
						}

						employee.setName( employee.getName() + "_" + ActorLockForUpdate.this.name);
						System.out.println("Thread " + ActorLockForUpdate.this.name + " updating employee");
						session.update(employee);
					}
				};
				query.find(srr);

				tx.commit();


			} catch (final Exception e) {
				e.printStackTrace();
				this.exception = true;
			}
		}

	}



	private Employee createEmployee(final JPO jpOrm) {
		final Session ormSession = jpOrm.session();
		final Transaction tx = ormSession.transaction();
		final int id = new Random().nextInt();
		final Employee employee = new Employee();
		employee.setId( id );
		employee.setAge( 44 );
		employee.setEmployeeNumber( ("empNumber" + id) );
		employee.setName("name");
		employee.setSurname("Cina");
		ormSession.save(employee);
		tx.commit();
		return employee;
	}

	private void deleteEmployee(final JPO jpOrm, final Employee employee) {
		final Session ormSession = jpOrm.session();
		final Transaction tx = ormSession.transaction();
		ormSession.delete(employee);
		tx.commit();
	}



}
