/**
 * Copyright (c) 2000-2012 Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.portal.kernel.servlet.liferaypackage;

import com.liferay.portal.kernel.cluster.ClusterExecutorUtil;
import com.liferay.portal.kernel.cluster.ClusterNode;
import com.liferay.portal.kernel.cluster.ClusterNodeResponse;
import com.liferay.portal.kernel.cluster.ClusterNodeResponses;
import com.liferay.portal.kernel.cluster.ClusterRequest;
import com.liferay.portal.kernel.cluster.FutureClusterResponses;
import com.liferay.portal.kernel.dao.jdbc.DataAccess;
import com.liferay.portal.kernel.deploy.hot.LiferayPackageHotDeployException;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.MethodHandler;
import com.liferay.portal.kernel.util.MethodKey;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.workflow.WorkflowConstants;
import com.liferay.portal.license.util.LicenseManagerUtil;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author Amos Fong
 */
public class LiferayPackageUtil {

	public static void checkPackage() throws Exception {
		if (_getProductType() == _productTypeEE) {
			int licenseState = LicenseManagerUtil.getLicenseState(
				_productIdPortal);

			if (licenseState != _stateGood) {
				LicenseManagerUtil.checkLicense(_productIdPortal);

				licenseState = LicenseManagerUtil.getLicenseState(
					_productIdPortal);
			}

			if (licenseState != _stateGood) {
				throw new LiferayPackageHotDeployException(
					"This application requires a valid Liferay Portal EE " +
						"license.");
			}
		}

		if (Validator.isNull(_getProductId())) {
			return;
		}

		int licenseState = _getLicenseState(
			_getProductId(), _getProductVersion());

		if (licenseState != _stateGood) {
			LicenseManagerUtil.checkLicense(_getProductId());

			licenseState = _getLicenseState(
				_getProductId(), _getProductVersion());
		}

		if (licenseState != _stateGood) {
			throw new LiferayPackageHotDeployException(
				"This application does not have a valid license");
		}

		Map<String, String> licenseProperties =
			LicenseManagerUtil.getLicenseProperties(_getProductId());

		if (licenseProperties != null) {
			int requiredProductVersion = GetterUtil.getInteger(
				licenseProperties.get("productVersion"));

			if ((_getProductVersion() > 0) &&
				(requiredProductVersion > _getProductVersion())) {

				throw new LiferayPackageHotDeployException(
					"The version of your application is not compatible with " +
						"the registered license");
			}
		}

		List<ClusterNode> clusterNodes = ClusterExecutorUtil.getClusterNodes();

		if (clusterNodes.size() <= 1) {
			return;
		}

		clusterNodes.remove(ClusterExecutorUtil.getLocalClusterNode());

		for (ClusterNode clusterNode : clusterNodes) {
			MethodHandler methodHandler = new MethodHandler(
				_getLicenseStateMethodKey, _getProductId());

			ClusterRequest clusterRequest = ClusterRequest.createUnicastRequest(
				methodHandler, clusterNode.getClusterNodeId());

			FutureClusterResponses futureClusterResponses =
				ClusterExecutorUtil.execute(clusterRequest);

			ClusterNodeResponses clusterNodeResponses =
				futureClusterResponses.get(10000, TimeUnit.MILLISECONDS);

			ClusterNodeResponse clusterNodeResponse =
				clusterNodeResponses.getClusterResponse(clusterNode);

			Object result = clusterNodeResponse.getResult();

			if (result == null) {
				return;
			}

			Integer remoteLicenseState = (Integer)result;

			if (remoteLicenseState != _stateGood) {
				throw new LiferayPackageHotDeployException(
					"A clustered server has an invalid license.");
			}
		}
	}

	private static int _getLicenseState(String productId, int productVersion)
		throws Exception {

		Map<String, String> licenseProperties = new HashMap<String, String>();

		licenseProperties.put("productId", productId);
		licenseProperties.put("productVersion", String.valueOf(productVersion));
		licenseProperties.put("userCount", String.valueOf(_getUserCount()));

		return LicenseManagerUtil.getLicenseState(licenseProperties);
	}

	private static String _getProductId() {
		return _productId;
	}

	private static int _getProductType() {
		return GetterUtil.getInteger(_productType);
	}

	private static int _getProductVersion() {
		return GetterUtil.getInteger(_productVersion);
	}

	private static long _getUserCount() throws Exception {
		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;

		try {
			con = DataAccess.getConnection();

			ps = con.prepareStatement(
				"select count(*) from User_ where (defaultUser = ?) and " +
					"(status = ?)");

			ps.setBoolean(1, false);
			ps.setLong(2, WorkflowConstants.STATUS_APPROVED);

			rs = ps.executeQuery();

			while (rs.next()) {
				long count = rs.getLong(1);

				if (count > 0) {
					return count;
				}
			}
		}
		finally {
			DataAccess.cleanUp(con, ps, rs);
		}

		throw new Exception("Unable to count number of users on server");
	}

	private static final String _productId = "_PRODUCT_ID_";
	private static final String _productIdPortal = "Portal";
	private static final String _productType = "_PRODUCT_TYPE_";
	private static final int _productTypeEE = 2;
	private static final String _productVersion = "_PRODUCT_VERSION_";
	private static final int _stateGood = 3;

	private static MethodKey _getLicenseStateMethodKey = new MethodKey(
		LicenseManagerUtil.class.getName(), "getLicenseState", String.class);

}