/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.user.associated.data.display;

import com.liferay.petra.string.CharPool;
import com.liferay.portal.kernel.dao.orm.Disjunction;
import com.liferay.portal.kernel.dao.orm.DynamicQuery;
import com.liferay.portal.kernel.dao.orm.Order;
import com.liferay.portal.kernel.dao.orm.OrderFactoryUtil;
import com.liferay.portal.kernel.dao.orm.RestrictionsFactoryUtil;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.BaseModel;
import com.liferay.portal.kernel.model.GroupedModel;
import com.liferay.portal.kernel.model.TrashedModel;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.OrderByComparator;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.TextFormatter;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.user.associated.data.util.UADDynamicQueryUtil;

import java.io.Serializable;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * Provides the base implementation of {@link UADDisplay} for entities generated
 * with Service Builder. The count and retrieval are based on a dynamic query,
 * which is available in the service generated by Service Builder.
 *
 * @author Pei-Jung Lan
 * @author Drew Brokke
 */
public abstract class BaseModelUADDisplay<T extends BaseModel>
	implements UADDisplay<T> {

	@Override
	public long count(long userId) {
		return doCount(getDynamicQuery(userId));
	}

	@Override
	public Map<String, Object> getFieldValues(
		T t, String[] fieldNames, Locale locale) {

		Map<String, Object> modelAttributes = t.getModelAttributes();

		Set<String> modelAttributesKeySet = modelAttributes.keySet();

		modelAttributesKeySet.retainAll(Arrays.asList(fieldNames));

		return modelAttributes;
	}

	@Override
	public Serializable getPrimaryKey(T baseModel) {
		return baseModel.getPrimaryKeyObj();
	}

	@Override
	public List<T> getRange(long userId, int start, int end) {
		return doGetRange(getDynamicQuery(userId), start, end);
	}

	@Override
	public String[] getSortingFieldNames() {
		return ArrayUtil.append(
			new String[] {"createDate", "modifiedDate"}, getColumnFieldNames());
	}

	@Override
	public String getTypeName(Locale locale) {
		return getTypeClass().getSimpleName();
	}

	@Override
	public boolean isInTrash(T t)
		throws IllegalAccessException, InvocationTargetException {

		if (!TrashedModel.class.isAssignableFrom(t.getClass())) {
			return false;
		}

		try {
			Class<?> clazz = t.getClass();

			Method method = clazz.getMethod("isInTrash");

			return (boolean)method.invoke(t);
		}
		catch (NoSuchMethodException noSuchMethodException) {
			if (_log.isDebugEnabled()) {
				_log.debug(noSuchMethodException, noSuchMethodException);
			}

			return false;
		}
	}

	@Override
	public boolean isSiteScoped() {
		if (GroupedModel.class.isAssignableFrom(getTypeClass())) {
			return true;
		}

		return false;
	}

	@Override
	public List<T> search(
		long userId, long[] groupIds, String keywords, String orderByField,
		String orderByType, int start, int end) {

		return doGetRange(
			getSearchDynamicQuery(
				userId, groupIds, keywords, orderByField, orderByType),
			start, end);
	}

	@Override
	public long searchCount(long userId, long[] groupIds, String keywords) {
		return doCount(
			getSearchDynamicQuery(userId, groupIds, keywords, null, null));
	}

	/**
	 * Returns the number of type {@code T} entities associated with the user
	 * using the dynamic query.
	 *
	 * @param  dynamicQuery the dynamic query to pass to the service layer
	 * @return the number of type {@code T} entities associated with the user
	 *         using the dynamic query
	 */
	protected abstract long doCount(DynamicQuery dynamicQuery);

	/**
	 * Returns a new {@link DynamicQuery} from the relevant service for type
	 * {@code T}.
	 *
	 * @return a new {@link DynamicQuery} to be used by the service layer
	 */
	protected abstract DynamicQuery doGetDynamicQuery();

	/**
	 * Returns type {@code T} entities in the given range associated with a user
	 * using a dynamic query.
	 *
	 * @param  dynamicQuery the dynamic query to pass to the service layer
	 * @param  start the starting index of the result set, for pagination
	 * @param  end the ending index of the result set, for pagination
	 * @return a paginated list of type {@code T} entities
	 */
	protected abstract List<T> doGetRange(
		DynamicQuery dynamicQuery, int start, int end);

	/**
	 * Returns names identifying fields on the type {@code T} entity that
	 * contain the primary key of a user.
	 *
	 * @return the fields that contain the primary key of a user
	 */
	protected abstract String[] doGetUserIdFieldNames();

	/**
	 * Returns a dynamic query for type {@code T}. It should be populated with
	 * criteria and ready for use by the service.
	 *
	 * @param  userId the primary key of the user used to pre-filter the dynamic
	 *         query
	 * @return a pre-filtered dynamic query
	 */
	protected DynamicQuery getDynamicQuery(long userId) {
		return UADDynamicQueryUtil.addDynamicQueryCriteria(
			doGetDynamicQuery(), doGetUserIdFieldNames(), userId);
	}

	/**
	 * Returns an {@code OrderByComparator} for type {@code T} used to sort
	 * search results. If this returns {@code null}, the default dynamic query
	 * ordering is used.
	 *
	 * @param  orderByField the name of the field to use for ordering
	 * @param  orderByType the type of ordering
	 * @return an {@code OrderByComparator} for type {@code T}
	 */
	protected OrderByComparator<T> getOrderByComparator(
		String orderByField, String orderByType) {

		return null;
	}

	/**
	 * Returns the field names that are queried when using the {@link #search}
	 * or {@link #searchCount} methods.
	 *
	 * @return the field names to be queried
	 */
	protected String[] getSearchableFields() {
		return getDisplayFieldNames();
	}

	/**
	 * Returns a dynamic query that can be used to perform a database search for
	 * type {@code T} entities that are associated with the user.
	 *
	 * @param  userId the primary key of the user whose data to search
	 * @param  groupIds the optional group primary keys to filter by
	 * @param  keywords the optional search terms to filter by
	 * @param  orderByField the name of the field to use for ordering
	 * @param  orderByType the type of ordering
	 * @return a dynamic query to be used by the {@link #doGetRange} and {@link
	 *         #doCount} methods
	 */
	protected DynamicQuery getSearchDynamicQuery(
		long userId, long[] groupIds, String keywords, String orderByField,
		String orderByType) {

		DynamicQuery dynamicQuery = getDynamicQuery(userId);

		if (isSiteScoped() && ArrayUtil.isNotEmpty(groupIds)) {
			dynamicQuery.add(
				RestrictionsFactoryUtil.in(
					"groupId", ArrayUtil.toLongArray(groupIds)));
		}

		String[] searchableFields = getSearchableFields();

		if (Validator.isNotNull(keywords) && (searchableFields.length > 0)) {
			Disjunction disjunction = RestrictionsFactoryUtil.disjunction();

			String quotedKeywords = StringUtil.quote(
				keywords, CharPool.PERCENT);

			Class<?> clazz = getTypeClass();

			for (String searchableField : searchableFields) {
				try {
					String formattedSearchableField = TextFormatter.format(
						searchableField, TextFormatter.G);

					Method method = clazz.getMethod(
						"get" + formattedSearchableField);

					if (method.getReturnType() == String.class) {
						disjunction.add(
							RestrictionsFactoryUtil.ilike(
								searchableField, quotedKeywords));
					}
				}
				catch (NoSuchMethodException | SecurityException exception) {
					if (_log.isDebugEnabled()) {
						_log.debug(exception, exception);
					}
				}
			}

			dynamicQuery.add(disjunction);
		}

		if (orderByField != null) {
			OrderByComparator<T> orderByComparator = getOrderByComparator(
				orderByField, orderByType);

			if (orderByComparator != null) {
				OrderFactoryUtil.addOrderByComparator(
					dynamicQuery, orderByComparator);
			}
			else {
				Order order = null;

				if (Objects.equals(orderByType, "desc")) {
					order = OrderFactoryUtil.desc(orderByField);
				}
				else {
					order = OrderFactoryUtil.asc(orderByField);
				}

				dynamicQuery.addOrder(order);
			}
		}

		return dynamicQuery;
	}

	private static final Log _log = LogFactoryUtil.getLog(
		BaseModelUADDisplay.class);

}