/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.impl;

import io.vertx.core.AsyncResult;
import io.vertx.core.Context;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Verticle;
import io.vertx.core.impl.ConcurrentHashSet;
import io.vertx.core.impl.ContextImpl;
import io.vertx.core.impl.Deployment;
import io.vertx.core.impl.EventLoopContext;
import io.vertx.core.impl.IsolatingClassLoader;
import io.vertx.core.impl.JavaVerticleFactory;
import io.vertx.core.impl.Redeployer;
import io.vertx.core.impl.VertxInternal;
import io.vertx.core.json.JsonObject;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.impl.LoggerFactory;
import io.vertx.core.spi.VerticleFactory;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.UUID;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class DeploymentManager {
    private static final Logger log = LoggerFactory.getLogger(DeploymentManager.class);
    private final VertxInternal vertx;
    private final Map<String, Deployment> deployments = new ConcurrentHashMap<String, Deployment>();
    private final Map<String, ClassLoader> classloaders = new WeakHashMap<String, ClassLoader>();
    private final Map<String, List<VerticleFactory>> verticleFactories = new ConcurrentHashMap<String, List<VerticleFactory>>();
    private final List<VerticleFactory> defaultFactories = new ArrayList<VerticleFactory>();

    public DeploymentManager(VertxInternal vertx) {
        this.vertx = vertx;
        this.loadVerticleFactories();
    }

    private void loadVerticleFactories() {
        ServiceLoader<VerticleFactory> factories = ServiceLoader.load(VerticleFactory.class);
        for (VerticleFactory factory : factories) {
            this.registerVerticleFactory(factory);
        }
        JavaVerticleFactory defaultFactory = new JavaVerticleFactory();
        defaultFactory.init(this.vertx);
        this.defaultFactories.add(defaultFactory);
    }

    private String generateDeploymentID() {
        return UUID.randomUUID().toString();
    }

    public void deployVerticle(Verticle verticle, DeploymentOptions options, Handler<AsyncResult<String>> completionHandler) {
        ContextImpl currentContext = this.vertx.getOrCreateContext();
        this.doDeploy("java:" + verticle.getClass().getName(), this.generateDeploymentID(), options, currentContext, currentContext, completionHandler, this.getCurrentClassLoader(), null, verticle);
    }

    public void deployVerticle(String identifier, DeploymentOptions options, Handler<AsyncResult<String>> completionHandler) {
        ContextImpl callingContext = this.vertx.getOrCreateContext();
        ClassLoader cl = this.getClassLoader(options, callingContext);
        Redeployer redeployer = this.getRedeployer(options, cl, callingContext);
        this.doDeployVerticle(identifier, this.generateDeploymentID(), options, callingContext, callingContext, cl, redeployer, completionHandler);
    }

    private void doRedeployVerticle(String identifier, String deploymentID, DeploymentOptions options, ContextImpl parentContext, ContextImpl callingContext, Redeployer redeployer, Handler<AsyncResult<String>> completionHandler) {
        ClassLoader cl = this.getClassLoader(options, parentContext);
        this.doDeployVerticle(identifier, deploymentID, options, parentContext, callingContext, cl, redeployer, completionHandler);
    }

    private void doDeployVerticle(String identifier, String deploymentID, DeploymentOptions options, ContextImpl parentContext, ContextImpl callingContext, ClassLoader cl, Redeployer redeployer, Handler<AsyncResult<String>> completionHandler) {
        List<VerticleFactory> verticleFactories = this.resolveFactories(identifier);
        Iterator<VerticleFactory> iter = verticleFactories.iterator();
        while (iter.hasNext()) {
            try {
                String resolvedName;
                VerticleFactory verticleFactory = iter.next();
                if (verticleFactory.requiresResolve() && !(resolvedName = verticleFactory.resolve(identifier, options, cl)).equals(identifier)) {
                    this.deployVerticle(resolvedName, options, completionHandler);
                    return;
                }
                Verticle[] verticles = new Verticle[options.getInstances()];
                for (int i = 0; i < options.getInstances(); ++i) {
                    verticles[i] = verticleFactory.createVerticle(identifier, cl);
                    if (verticles[i] != null) continue;
                    throw new NullPointerException("VerticleFactory::createVerticle returned null");
                }
                this.doDeploy(identifier, deploymentID, options, parentContext, callingContext, completionHandler, cl, redeployer, verticles);
                return;
            }
            catch (Exception e) {
                if (iter.hasNext()) continue;
                this.reportFailure(e, callingContext, completionHandler);
            }
        }
    }

    private String getSuffix(int pos, String str) {
        if (pos + 1 >= str.length()) {
            throw new IllegalArgumentException("Invalid name: " + str);
        }
        return str.substring(pos + 1);
    }

    public void undeployVerticle(String deploymentID, Handler<AsyncResult<Void>> completionHandler) {
        Deployment deployment = this.deployments.get(deploymentID);
        ContextImpl currentContext = this.vertx.getOrCreateContext();
        if (deployment == null) {
            this.reportFailure(new IllegalStateException("Unknown deployment"), currentContext, completionHandler);
        } else {
            deployment.undeploy(completionHandler);
        }
    }

    public Set<String> deployments() {
        return Collections.unmodifiableSet(this.deployments.keySet());
    }

    public Deployment getDeployment(String deploymentID) {
        return this.deployments.get(deploymentID);
    }

    public void undeployAll(Handler<AsyncResult<Void>> completionHandler) {
        HashSet<String> deploymentIDs = new HashSet<String>();
        for (Map.Entry<String, Deployment> entry : this.deployments.entrySet()) {
            if (entry.getValue().isChild()) continue;
            deploymentIDs.add(entry.getKey());
        }
        if (!deploymentIDs.isEmpty()) {
            AtomicInteger count = new AtomicInteger(0);
            for (String deploymentID : deploymentIDs) {
                this.undeployVerticle(deploymentID, ar -> {
                    if (ar.failed()) {
                        log.error("Undeploy failed", ar.cause());
                    }
                    if (count.incrementAndGet() == deploymentIDs.size()) {
                        completionHandler.handle(Future.succeededFuture());
                    }
                });
            }
        } else {
            ContextImpl context = this.vertx.getOrCreateContext();
            context.runOnContext(v -> completionHandler.handle(Future.succeededFuture()));
        }
    }

    public void registerVerticleFactory(VerticleFactory factory) {
        String prefix = factory.prefix();
        if (prefix == null) {
            throw new IllegalArgumentException("factory.prefix() cannot be null");
        }
        List<VerticleFactory> facts = this.verticleFactories.get(prefix);
        if (facts == null) {
            facts = new ArrayList<VerticleFactory>();
            this.verticleFactories.put(prefix, facts);
        }
        if (facts.contains(factory)) {
            throw new IllegalArgumentException("Factory already registered");
        }
        facts.add(factory);
        facts.sort((fact1, fact2) -> fact1.order() - fact2.order());
        factory.init(this.vertx);
    }

    public void unregisterVerticleFactory(VerticleFactory factory) {
        String prefix = factory.prefix();
        if (prefix == null) {
            throw new IllegalArgumentException("factory.prefix() cannot be null");
        }
        List<VerticleFactory> facts = this.verticleFactories.get(prefix);
        boolean removed = false;
        if (facts != null) {
            if (facts.remove(factory)) {
                removed = true;
            }
            if (facts.isEmpty()) {
                this.verticleFactories.remove(prefix);
            }
        }
        if (!removed) {
            throw new IllegalArgumentException("factory isn't registered");
        }
    }

    public Set<VerticleFactory> verticleFactories() {
        HashSet<VerticleFactory> facts = new HashSet<VerticleFactory>();
        for (List<VerticleFactory> list : this.verticleFactories.values()) {
            facts.addAll(list);
        }
        return facts;
    }

    private List<VerticleFactory> resolveFactories(String identifier) {
        List<VerticleFactory> factoryList = null;
        int pos = identifier.indexOf(58);
        String lookup = null;
        if (pos != -1) {
            lookup = identifier.substring(0, pos);
        } else {
            pos = identifier.lastIndexOf(46);
            if (pos != -1) {
                lookup = this.getSuffix(pos, identifier);
            } else {
                factoryList = this.defaultFactories;
            }
        }
        if (factoryList == null && (factoryList = this.verticleFactories.get(lookup)) == null) {
            factoryList = this.defaultFactories;
        }
        return factoryList;
    }

    private boolean isTopMostDeployment(ContextImpl context) {
        return context.getDeployment() == null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ClassLoader getClassLoader(DeploymentOptions options, ContextImpl parentContext) {
        ClassLoader cl;
        String isolationGroup;
        if (this.shouldEnableRedployment(options, parentContext)) {
            this.setRedeployIsolationGroup(options);
        }
        if ((isolationGroup = options.getIsolationGroup()) == null) {
            cl = this.getCurrentClassLoader();
        } else {
            DeploymentManager deploymentManager = this;
            synchronized (deploymentManager) {
                cl = this.classloaders.get(isolationGroup);
                if (cl == null) {
                    ClassLoader current = this.getCurrentClassLoader();
                    if (!(current instanceof URLClassLoader)) {
                        throw new IllegalStateException("Current classloader must be URLClassLoader");
                    }
                    URLClassLoader urlc = (URLClassLoader)current;
                    ArrayList<URL> urls = new ArrayList<URL>();
                    List<String> extraClasspath = options.getExtraClasspath();
                    if (extraClasspath != null) {
                        for (String pathElement : extraClasspath) {
                            File file = new File(pathElement);
                            try {
                                URL url = file.toURI().toURL();
                                urls.add(url);
                            }
                            catch (MalformedURLException e) {
                                throw new IllegalStateException(e);
                            }
                        }
                    }
                    urls.addAll(Arrays.asList(urlc.getURLs()));
                    cl = new IsolatingClassLoader(urls.toArray(new URL[urls.size()]), this.getCurrentClassLoader());
                    this.classloaders.put(isolationGroup, cl);
                }
            }
        }
        return cl;
    }

    private void setRedeployIsolationGroup(DeploymentOptions options) {
        options.setIsolationGroup("redeploy-" + UUID.randomUUID().toString());
    }

    private boolean shouldEnableRedployment(DeploymentOptions options, ContextImpl parentContext) {
        return options.isRedeploy() && this.isTopMostDeployment(parentContext);
    }

    private Redeployer getRedeployer(DeploymentOptions options, ClassLoader cl, ContextImpl parentContext) {
        if (this.shouldEnableRedployment(options, parentContext)) {
            this.setRedeployIsolationGroup(options);
            URLClassLoader urlc = (URLClassLoader)cl;
            HashSet<File> filesToWatch = new HashSet<File>();
            for (URL url : urlc.getURLs()) {
                try {
                    filesToWatch.add(new File(url.toURI()));
                }
                catch (IllegalArgumentException | URISyntaxException ignore) {
                    // empty catch block
                }
            }
            return new Redeployer(filesToWatch, options.getRedeployGracePeriod());
        }
        return null;
    }

    private ClassLoader getCurrentClassLoader() {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        if (cl == null) {
            cl = this.getClass().getClassLoader();
        }
        return cl;
    }

    private <T> void reportFailure(Throwable t, Context context, Handler<AsyncResult<T>> completionHandler) {
        if (completionHandler != null) {
            this.reportResult(context, completionHandler, Future.failedFuture(t));
        } else {
            log.error(t.getMessage(), t);
        }
    }

    private <T> void reportSuccess(T result, Context context, Handler<AsyncResult<T>> completionHandler) {
        if (completionHandler != null) {
            this.reportResult(context, completionHandler, Future.succeededFuture(result));
        }
    }

    private <T> void reportResult(Context context, Handler<AsyncResult<T>> completionHandler, AsyncResult<T> result) {
        context.runOnContext(v -> {
            try {
                completionHandler.handle(result);
            }
            catch (Throwable t) {
                log.error("Failure in calling handler", t);
            }
        });
    }

    private void doDeploy(String identifier, String deploymentID, DeploymentOptions options, ContextImpl parentContext, ContextImpl callingContext, Handler<AsyncResult<String>> completionHandler, ClassLoader tccl, Redeployer redeployer, Verticle ... verticles) {
        if (options.isMultiThreaded() && !options.isWorker()) {
            throw new IllegalArgumentException("If multi-threaded then must be worker too");
        }
        JsonObject conf = options.getConfig() == null ? new JsonObject() : options.getConfig().copy();
        DeploymentImpl deployment = new DeploymentImpl(deploymentID, identifier, options, redeployer, parentContext);
        Deployment parent = parentContext.getDeployment();
        if (parent != null) {
            parent.addChild(deployment);
            deployment.child = true;
        }
        AtomicInteger deployCount = new AtomicInteger();
        AtomicBoolean failureReported = new AtomicBoolean();
        for (Verticle verticle : verticles) {
            EventLoopContext context = options.isWorker() ? this.vertx.createWorkerContext(options.isMultiThreaded(), deploymentID, conf, tccl) : this.vertx.createEventLoopContext(deploymentID, conf, tccl);
            context.setDeployment(deployment);
            deployment.addVerticle(new VerticleHolder(verticle, context));
            context.runOnContext(v -> {
                try {
                    verticle.init(this.vertx, context);
                    Future<Void> startFuture = Future.future();
                    verticle.start(startFuture);
                    startFuture.setHandler(ar -> {
                        if (ar.succeeded()) {
                            this.vertx.metricsSPI().verticleDeployed(verticle);
                            this.deployments.put(deploymentID, deployment);
                            if (deployCount.incrementAndGet() == verticles.length) {
                                this.reportSuccess(deploymentID, callingContext, completionHandler);
                                deployment.startRedeployTimer();
                            }
                        } else if (!failureReported.get()) {
                            this.reportFailure(ar.cause(), callingContext, completionHandler);
                        }
                    });
                }
                catch (Throwable t) {
                    this.reportFailure(t, callingContext, completionHandler);
                }
            });
        }
    }

    private class DeploymentImpl
    implements Deployment {
        private final String deploymentID;
        private final String verticleIdentifier;
        private List<VerticleHolder> verticles = new ArrayList<VerticleHolder>();
        private final Set<Deployment> children = new ConcurrentHashSet<Deployment>();
        private final DeploymentOptions options;
        private final Redeployer redeployer;
        private final ContextImpl parentContext;
        private boolean undeployed;
        private boolean broken;
        private volatile boolean child;
        private long redeployTimerID = -1L;

        private DeploymentImpl(String deploymentID, String verticleIdentifier, DeploymentOptions options, Redeployer redeployer, ContextImpl parentContext) {
            this.deploymentID = deploymentID;
            this.verticleIdentifier = verticleIdentifier;
            this.options = options;
            this.redeployer = redeployer;
            this.parentContext = parentContext;
        }

        public void addVerticle(VerticleHolder holder) {
            this.verticles.add(holder);
        }

        @Override
        public void undeploy(Handler<AsyncResult<Void>> completionHandler) {
            if (this.redeployTimerID != -1L) {
                DeploymentManager.this.vertx.cancelTimer(this.redeployTimerID);
            }
            ContextImpl currentContext = DeploymentManager.this.vertx.getOrCreateContext();
            if (!this.undeployed) {
                this.doUndeploy(currentContext, completionHandler);
            } else {
                DeploymentManager.this.reportFailure(new IllegalStateException("Already undeployed"), currentContext, completionHandler);
            }
        }

        @Override
        public void doUndeploy(ContextImpl undeployingContext, Handler<AsyncResult<Void>> completionHandler) {
            if (!this.children.isEmpty()) {
                int size = this.children.size();
                AtomicInteger childCount = new AtomicInteger();
                for (Deployment childDeployment : new HashSet<Deployment>(this.children)) {
                    childDeployment.doUndeploy(undeployingContext, ar -> {
                        this.children.remove(childDeployment);
                        if (ar.failed()) {
                            DeploymentManager.this.reportFailure(ar.cause(), undeployingContext, completionHandler);
                        } else if (childCount.incrementAndGet() == size) {
                            this.doUndeploy(undeployingContext, completionHandler);
                        }
                    });
                }
            } else {
                this.undeployed = true;
                AtomicInteger undeployCount = new AtomicInteger();
                for (VerticleHolder verticleHolder : this.verticles) {
                    ContextImpl context = verticleHolder.context;
                    context.runOnContext(v -> {
                        Future<Void> stopFuture = Future.future();
                        AtomicBoolean failureReported = new AtomicBoolean();
                        stopFuture.setHandler(ar -> {
                            DeploymentManager.this.deployments.remove(this.deploymentID);
                            DeploymentManager.this.vertx.metricsSPI().verticleUndeployed(verticleHolder.verticle);
                            context.runCloseHooks(ar2 -> {
                                if (ar2.failed()) {
                                    log.error("Failed to run close hook", ar2.cause());
                                }
                                if (ar.succeeded() && undeployCount.incrementAndGet() == this.verticles.size()) {
                                    DeploymentManager.this.reportSuccess(null, undeployingContext, completionHandler);
                                } else if (ar.failed() && !failureReported.get()) {
                                    failureReported.set(true);
                                    DeploymentManager.this.reportFailure(ar.cause(), undeployingContext, completionHandler);
                                }
                            });
                        });
                        try {
                            verticleHolder.verticle.stop(stopFuture);
                        }
                        catch (Throwable t) {
                            stopFuture.fail(t);
                        }
                    });
                }
            }
        }

        @Override
        public String verticleIdentifier() {
            return this.verticleIdentifier;
        }

        @Override
        public DeploymentOptions deploymentOptions() {
            return this.options;
        }

        @Override
        public synchronized void addChild(Deployment deployment) {
            this.children.add(deployment);
        }

        @Override
        public Set<Verticle> getVerticles() {
            HashSet<Verticle> verts = new HashSet<Verticle>();
            for (VerticleHolder holder : this.verticles) {
                verts.add(holder.verticle);
            }
            return verts;
        }

        @Override
        public boolean isChild() {
            return this.child;
        }

        @Override
        public String deploymentID() {
            return this.deploymentID;
        }

        private void startRedeployTimer() {
            if (this.redeployer != null) {
                this.doStartRedeployTimer();
            }
        }

        private void doStartRedeployTimer() {
            this.redeployTimerID = DeploymentManager.this.vertx.setTimer(this.options.getRedeployScanPeriod(), tid -> DeploymentManager.this.vertx.executeBlocking(this.redeployer, res -> {
                if (res.succeeded()) {
                    if (((Boolean)res.result()).booleanValue()) {
                        this.doRedeploy();
                    } else if (!this.undeployed || this.broken) {
                        this.doStartRedeployTimer();
                    }
                } else {
                    log.error("Failure in redeployer", res.cause());
                }
            }));
        }

        private void doRedeploy() {
            log.trace("Redeploying!");
            log.trace("Undeploying " + this.deploymentID);
            if (!this.broken) {
                this.undeploy(res -> {
                    if (res.succeeded()) {
                        log.trace("Undeployed ok");
                        this.tryRedeploy();
                    } else {
                        log.error("Can't find verticle to undeploy", res.cause());
                    }
                });
            } else {
                this.tryRedeploy();
            }
        }

        private void tryRedeploy() {
            ContextImpl callingContext = DeploymentManager.this.vertx.getContext();
            DeploymentManager.this.doRedeployVerticle(this.verticleIdentifier, this.deploymentID, this.options, this.parentContext, callingContext, this.redeployer, res2 -> {
                if (res2.succeeded()) {
                    this.broken = false;
                    this.undeployed = false;
                    log.trace("Redeployed ok");
                } else {
                    log.trace("Failed to deploy!!");
                    this.broken = true;
                    this.doStartRedeployTimer();
                }
            });
        }
    }

    static class VerticleHolder {
        final Verticle verticle;
        final ContextImpl context;

        VerticleHolder(Verticle verticle, ContextImpl context) {
            this.verticle = verticle;
            this.context = context;
        }
    }
}

