/*
 * Decompiled with CFR 0.152.
 */
package org.apache.archiva.metadata.repository.file;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import org.apache.archiva.configuration.ArchivaConfiguration;
import org.apache.archiva.configuration.ManagedRepositoryConfiguration;
import org.apache.archiva.metadata.model.ArtifactMetadata;
import org.apache.archiva.metadata.model.CiManagement;
import org.apache.archiva.metadata.model.Dependency;
import org.apache.archiva.metadata.model.IssueManagement;
import org.apache.archiva.metadata.model.License;
import org.apache.archiva.metadata.model.MailingList;
import org.apache.archiva.metadata.model.MetadataFacet;
import org.apache.archiva.metadata.model.MetadataFacetFactory;
import org.apache.archiva.metadata.model.Organization;
import org.apache.archiva.metadata.model.ProjectMetadata;
import org.apache.archiva.metadata.model.ProjectVersionMetadata;
import org.apache.archiva.metadata.model.ProjectVersionReference;
import org.apache.archiva.metadata.model.Scm;
import org.apache.archiva.metadata.repository.MetadataRepository;
import org.apache.archiva.metadata.repository.MetadataRepositoryException;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileMetadataRepository
implements MetadataRepository {
    private final Map<String, MetadataFacetFactory> metadataFacetFactories;
    private final ArchivaConfiguration configuration;
    private Logger log = LoggerFactory.getLogger(FileMetadataRepository.class);
    private static final String PROJECT_METADATA_KEY = "project-metadata";
    private static final String PROJECT_VERSION_METADATA_KEY = "version-metadata";
    private static final String NAMESPACE_METADATA_KEY = "namespace-metadata";
    private static final String METADATA_KEY = "metadata";

    public FileMetadataRepository(Map<String, MetadataFacetFactory> metadataFacetFactories, ArchivaConfiguration configuration) {
        this.metadataFacetFactories = metadataFacetFactories;
        this.configuration = configuration;
    }

    private File getBaseDirectory(String repoId) {
        String basedir = ((ManagedRepositoryConfiguration)this.configuration.getConfiguration().getManagedRepositoriesAsMap().get(repoId)).getLocation();
        return new File(basedir, ".archiva");
    }

    private File getDirectory(String repoId) {
        return new File(this.getBaseDirectory(repoId), "content");
    }

    public void updateProject(String repoId, ProjectMetadata project) {
        this.updateProject(repoId, project.getNamespace(), project.getId());
    }

    private void updateProject(String repoId, String namespace, String id) {
        this.updateNamespace(repoId, namespace);
        try {
            File namespaceDirectory = new File(this.getDirectory(repoId), namespace);
            Properties properties = new Properties();
            properties.setProperty("namespace", namespace);
            properties.setProperty("id", id);
            this.writeProperties(properties, new File(namespaceDirectory, id), PROJECT_METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public void updateProjectVersion(String repoId, String namespace, String projectId, ProjectVersionMetadata versionMetadata) {
        this.updateProject(repoId, namespace, projectId);
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + versionMetadata.getId());
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        for (Object key : new ArrayList<Object>(properties.keySet())) {
            String name = (String)key;
            if (!name.contains(":") && !name.equals("facetIds")) {
                properties.remove(name);
            }
            FileMetadataRepository.clearMetadataFacetProperties(versionMetadata.getFacetList(), properties, "");
        }
        properties.setProperty("id", versionMetadata.getId());
        this.setProperty(properties, "name", versionMetadata.getName());
        this.setProperty(properties, "description", versionMetadata.getDescription());
        this.setProperty(properties, "url", versionMetadata.getUrl());
        this.setProperty(properties, "incomplete", String.valueOf(versionMetadata.isIncomplete()));
        if (versionMetadata.getScm() != null) {
            this.setProperty(properties, "scm.connection", versionMetadata.getScm().getConnection());
            this.setProperty(properties, "scm.developerConnection", versionMetadata.getScm().getDeveloperConnection());
            this.setProperty(properties, "scm.url", versionMetadata.getScm().getUrl());
        }
        if (versionMetadata.getCiManagement() != null) {
            this.setProperty(properties, "ci.system", versionMetadata.getCiManagement().getSystem());
            this.setProperty(properties, "ci.url", versionMetadata.getCiManagement().getUrl());
        }
        if (versionMetadata.getIssueManagement() != null) {
            this.setProperty(properties, "issue.system", versionMetadata.getIssueManagement().getSystem());
            this.setProperty(properties, "issue.url", versionMetadata.getIssueManagement().getUrl());
        }
        if (versionMetadata.getOrganization() != null) {
            this.setProperty(properties, "org.name", versionMetadata.getOrganization().getName());
            this.setProperty(properties, "org.url", versionMetadata.getOrganization().getUrl());
        }
        int i = 0;
        for (License license : versionMetadata.getLicenses()) {
            this.setProperty(properties, "license." + i + ".name", license.getName());
            this.setProperty(properties, "license." + i + ".url", license.getUrl());
            ++i;
        }
        i = 0;
        for (MailingList mailingList : versionMetadata.getMailingLists()) {
            this.setProperty(properties, "mailingList." + i + ".archive", mailingList.getMainArchiveUrl());
            this.setProperty(properties, "mailingList." + i + ".name", mailingList.getName());
            this.setProperty(properties, "mailingList." + i + ".post", mailingList.getPostAddress());
            this.setProperty(properties, "mailingList." + i + ".unsubscribe", mailingList.getUnsubscribeAddress());
            this.setProperty(properties, "mailingList." + i + ".subscribe", mailingList.getSubscribeAddress());
            this.setProperty(properties, "mailingList." + i + ".otherArchives", this.join(mailingList.getOtherArchives()));
            ++i;
        }
        i = 0;
        ProjectVersionReference reference = new ProjectVersionReference();
        reference.setNamespace(namespace);
        reference.setProjectId(projectId);
        reference.setProjectVersion(versionMetadata.getId());
        reference.setReferenceType(ProjectVersionReference.ReferenceType.DEPENDENCY);
        for (Dependency dependency : versionMetadata.getDependencies()) {
            this.setProperty(properties, "dependency." + i + ".classifier", dependency.getClassifier());
            this.setProperty(properties, "dependency." + i + ".scope", dependency.getScope());
            this.setProperty(properties, "dependency." + i + ".systemPath", dependency.getSystemPath());
            this.setProperty(properties, "dependency." + i + ".artifactId", dependency.getArtifactId());
            this.setProperty(properties, "dependency." + i + ".groupId", dependency.getGroupId());
            this.setProperty(properties, "dependency." + i + ".version", dependency.getVersion());
            this.setProperty(properties, "dependency." + i + ".type", dependency.getType());
            this.setProperty(properties, "dependency." + i + ".optional", String.valueOf(dependency.isOptional()));
            this.updateProjectReference(repoId, dependency.getGroupId(), dependency.getArtifactId(), dependency.getVersion(), reference);
            ++i;
        }
        LinkedHashSet<String> facetIds = new LinkedHashSet<String>(versionMetadata.getFacetIds());
        facetIds.addAll(Arrays.asList(properties.getProperty("facetIds", "").split(",")));
        properties.setProperty("facetIds", this.join(facetIds));
        this.updateProjectVersionFacets(versionMetadata, properties);
        try {
            this.writeProperties(properties, directory, PROJECT_VERSION_METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    private void updateProjectVersionFacets(ProjectVersionMetadata versionMetadata, Properties properties) {
        for (MetadataFacet facet : versionMetadata.getFacetList()) {
            for (Map.Entry entry : facet.toProperties().entrySet()) {
                properties.setProperty(facet.getFacetId() + ":" + (String)entry.getKey(), (String)entry.getValue());
            }
        }
    }

    private static void clearMetadataFacetProperties(Collection<MetadataFacet> facetList, Properties properties, String prefix) {
        ArrayList<Object> propsToRemove = new ArrayList<Object>();
        for (MetadataFacet facet : facetList) {
            for (Object key : new ArrayList<Object>(properties.keySet())) {
                String keyString = (String)key;
                if (!keyString.startsWith(prefix + facet.getFacetId() + ":")) continue;
                propsToRemove.add(key);
            }
        }
        for (Object key : propsToRemove) {
            properties.remove(key);
        }
    }

    private void updateProjectReference(String repoId, String namespace, String projectId, String projectVersion, ProjectVersionReference reference) {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        int i = Integer.parseInt(properties.getProperty("ref:lastReferenceNum", "-1")) + 1;
        this.setProperty(properties, "ref:lastReferenceNum", Integer.toString(i));
        this.setProperty(properties, "ref:reference." + i + ".namespace", reference.getNamespace());
        this.setProperty(properties, "ref:reference." + i + ".projectId", reference.getProjectId());
        this.setProperty(properties, "ref:reference." + i + ".projectVersion", reference.getProjectVersion());
        this.setProperty(properties, "ref:reference." + i + ".referenceType", reference.getReferenceType().toString());
        try {
            this.writeProperties(properties, directory, PROJECT_VERSION_METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public void updateNamespace(String repoId, String namespace) {
        try {
            File namespaceDirectory = new File(this.getDirectory(repoId), namespace);
            Properties properties = new Properties();
            properties.setProperty("namespace", namespace);
            this.writeProperties(properties, namespaceDirectory, NAMESPACE_METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public List<String> getMetadataFacets(String repoId, String facetId) {
        File directory = this.getMetadataDirectory(repoId, facetId);
        ArrayList<String> facets = new ArrayList<String>();
        this.recurse(facets, "", directory);
        return facets;
    }

    public boolean hasMetadataFacet(String repositoryId, String facetId) throws MetadataRepositoryException {
        return !this.getMetadataFacets(repositoryId, facetId).isEmpty();
    }

    private void recurse(List<String> facets, String prefix, File directory) {
        File[] list = directory.listFiles();
        if (list != null) {
            for (File dir : list) {
                if (dir.isDirectory()) {
                    this.recurse(facets, prefix + "/" + dir.getName(), dir);
                    continue;
                }
                if (!dir.getName().equals("metadata.properties")) continue;
                facets.add(prefix.substring(1));
            }
        }
    }

    public MetadataFacet getMetadataFacet(String repositoryId, String facetId, String name) {
        Properties properties;
        try {
            properties = this.readProperties(new File(this.getMetadataDirectory(repositoryId, facetId), name), METADATA_KEY);
        }
        catch (FileNotFoundException e) {
            return null;
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
            return null;
        }
        MetadataFacet metadataFacet = null;
        MetadataFacetFactory metadataFacetFactory = this.metadataFacetFactories.get(facetId);
        if (metadataFacetFactory != null) {
            metadataFacet = metadataFacetFactory.createMetadataFacet(repositoryId, name);
            HashMap<String, String> map = new HashMap<String, String>();
            for (Object key : new ArrayList<Object>(properties.keySet())) {
                String property = (String)key;
                map.put(property, properties.getProperty(property));
            }
            metadataFacet.fromProperties(map);
        }
        return metadataFacet;
    }

    public void addMetadataFacet(String repositoryId, MetadataFacet metadataFacet) {
        Properties properties = new Properties();
        properties.putAll((Map<?, ?>)metadataFacet.toProperties());
        try {
            File directory = new File(this.getMetadataDirectory(repositoryId, metadataFacet.getFacetId()), metadataFacet.getName());
            this.writeProperties(properties, directory, METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public void removeMetadataFacets(String repositoryId, String facetId) {
        File dir = this.getMetadataDirectory(repositoryId, facetId);
        if (!FileUtils.deleteQuietly((File)dir)) {
            this.log.error("Cannot delete the metadata repository {}", (Object)dir);
        }
    }

    public void removeMetadataFacet(String repoId, String facetId, String name) {
        File dir = new File(this.getMetadataDirectory(repoId, facetId), name);
        if (!FileUtils.deleteQuietly((File)dir)) {
            this.log.error("Cannot delete the metadata repository {}", (Object)dir);
        }
    }

    public List<ArtifactMetadata> getArtifactsByDateRange(String repoId, Date startTime, Date endTime) {
        ArrayList<ArtifactMetadata> artifacts = new ArrayList<ArtifactMetadata>();
        for (String ns : this.getRootNamespaces(repoId)) {
            this.getArtifactsByDateRange(artifacts, repoId, ns, startTime, endTime);
        }
        Collections.sort(artifacts, new ArtifactComparator());
        return artifacts;
    }

    private void getArtifactsByDateRange(List<ArtifactMetadata> artifacts, String repoId, String ns, Date startTime, Date endTime) {
        for (String namespace : this.getNamespaces(repoId, ns)) {
            this.getArtifactsByDateRange(artifacts, repoId, ns + "." + namespace, startTime, endTime);
        }
        for (String project : this.getProjects(repoId, ns)) {
            for (String version : this.getProjectVersions(repoId, ns, project)) {
                for (ArtifactMetadata artifact : this.getArtifacts(repoId, ns, project, version)) {
                    if (startTime != null && !startTime.before(artifact.getWhenGathered()) || endTime != null && !endTime.after(artifact.getWhenGathered())) continue;
                    artifacts.add(artifact);
                }
            }
        }
    }

    public Collection<ArtifactMetadata> getArtifacts(String repoId, String namespace, String projectId, String projectVersion) {
        HashMap<String, ArtifactMetadata> artifacts = new HashMap<String, ArtifactMetadata>();
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            String name = (String)entry.getKey();
            StringTokenizer tok = new StringTokenizer(name, ":");
            if (!tok.hasMoreTokens() || !"artifact".equals(tok.nextToken())) continue;
            String field = tok.nextToken();
            String id = tok.nextToken();
            ArtifactMetadata artifact = (ArtifactMetadata)artifacts.get(id);
            if (artifact == null) {
                artifact = new ArtifactMetadata();
                artifact.setRepositoryId(repoId);
                artifact.setNamespace(namespace);
                artifact.setProject(projectId);
                artifact.setProjectVersion(projectVersion);
                artifact.setVersion(projectVersion);
                artifact.setId(id);
                artifacts.put(id, artifact);
            }
            String value = (String)entry.getValue();
            if ("updated".equals(field)) {
                artifact.setFileLastModified(Long.parseLong(value));
                continue;
            }
            if ("size".equals(field)) {
                artifact.setSize(Long.valueOf(value).longValue());
                continue;
            }
            if ("whenGathered".equals(field)) {
                artifact.setWhenGathered(new Date(Long.parseLong(value)));
                continue;
            }
            if ("version".equals(field)) {
                artifact.setVersion(value);
                continue;
            }
            if ("md5".equals(field)) {
                artifact.setMd5(value);
                continue;
            }
            if ("sha1".equals(field)) {
                artifact.setSha1(value);
                continue;
            }
            if (!"facetIds".equals(field)) continue;
            if (value.length() > 0) {
                String propertyPrefix = "artifact:facet:" + id + ":";
                for (String facetId : value.split(",")) {
                    MetadataFacetFactory factory = this.metadataFacetFactories.get(facetId);
                    if (factory == null) {
                        this.log.error("Attempted to load unknown artifact metadata facet: " + facetId);
                        continue;
                    }
                    MetadataFacet facet = factory.createMetadataFacet();
                    String prefix = propertyPrefix + facet.getFacetId();
                    HashMap<String, String> map = new HashMap<String, String>();
                    for (Object key : new ArrayList<Object>(properties.keySet())) {
                        String property = (String)key;
                        if (!property.startsWith(prefix)) continue;
                        map.put(property.substring(prefix.length() + 1), properties.getProperty(property));
                    }
                    facet.fromProperties(map);
                    artifact.addFacet(facet);
                }
            }
            this.updateArtifactFacets(artifact, properties);
        }
        return artifacts.values();
    }

    public void save() {
    }

    public void close() {
    }

    public void revert() {
        this.log.warn("Attempted to revert a session, but the file-based repository storage doesn't support it");
    }

    public boolean canObtainAccess(Class<?> aClass) {
        return false;
    }

    public <T> T obtainAccess(Class<T> aClass) {
        throw new IllegalArgumentException("Access using " + aClass + " is not supported on the file metadata storage");
    }

    private void updateArtifactFacets(ArtifactMetadata artifact, Properties properties) {
        String propertyPrefix = "artifact:facet:" + artifact.getId() + ":";
        for (MetadataFacet facet : artifact.getFacetList()) {
            for (Map.Entry e : facet.toProperties().entrySet()) {
                String key = propertyPrefix + facet.getFacetId() + ":" + (String)e.getKey();
                properties.setProperty(key, (String)e.getValue());
            }
        }
    }

    public Collection<String> getRepositories() {
        ArrayList<String> repositories = new ArrayList<String>();
        for (ManagedRepositoryConfiguration managedRepositoryConfiguration : this.configuration.getConfiguration().getManagedRepositories()) {
            repositories.add(managedRepositoryConfiguration.getId());
        }
        return repositories;
    }

    public List<ArtifactMetadata> getArtifactsByChecksum(String repositoryId, String checksum) {
        ArrayList<ArtifactMetadata> artifacts = new ArrayList<ArtifactMetadata>();
        for (String ns : this.getRootNamespaces(repositoryId)) {
            this.getArtifactsByChecksum(artifacts, repositoryId, ns, checksum);
        }
        return artifacts;
    }

    public void removeNamespace(String repositoryId, String project) throws MetadataRepositoryException {
        try {
            File namespaceDirectory = new File(this.getDirectory(repositoryId), project);
            FileUtils.deleteDirectory((File)namespaceDirectory);
        }
        catch (IOException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeArtifact(ArtifactMetadata artifactMetadata, String baseVersion) throws MetadataRepositoryException {
        File directory = new File(this.getDirectory(artifactMetadata.getRepositoryId()), artifactMetadata.getNamespace() + "/" + artifactMetadata.getProject() + "/" + baseVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        String id = artifactMetadata.getId();
        properties.remove("artifact:updated:" + id);
        properties.remove("artifact:whenGathered:" + id);
        properties.remove("artifact:size:" + id);
        properties.remove("artifact:md5:" + id);
        properties.remove("artifact:sha1:" + id);
        properties.remove("artifact:version:" + id);
        properties.remove("artifact:facetIds:" + id);
        String prefix = "artifact:facet:" + id + ":";
        for (Object key : new ArrayList<Object>(properties.keySet())) {
            String property = (String)key;
            if (!property.startsWith(prefix)) continue;
            properties.remove(property);
        }
        try {
            this.writeProperties(properties, directory, PROJECT_VERSION_METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public void removeArtifact(String repoId, String namespace, String project, String version, String id) {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + project + "/" + version);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        properties.remove("artifact:updated:" + id);
        properties.remove("artifact:whenGathered:" + id);
        properties.remove("artifact:size:" + id);
        properties.remove("artifact:md5:" + id);
        properties.remove("artifact:sha1:" + id);
        properties.remove("artifact:version:" + id);
        properties.remove("artifact:facetIds:" + id);
        String prefix = "artifact:facet:" + id + ":";
        for (Object key : new ArrayList<Object>(properties.keySet())) {
            String property = (String)key;
            if (!property.startsWith(prefix)) continue;
            properties.remove(property);
        }
        try {
            FileUtils.deleteDirectory((File)directory);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public void removeArtifact(String repositoryId, String namespace, String project, String projectVersion, MetadataFacet metadataFacet) throws MetadataRepositoryException {
        throw new UnsupportedOperationException("not implemented");
    }

    public void removeRepository(String repoId) {
        File dir = this.getDirectory(repoId);
        if (!FileUtils.deleteQuietly((File)dir)) {
            this.log.error("Cannot delete repository {}", (Object)dir);
        }
    }

    private void getArtifactsByChecksum(List<ArtifactMetadata> artifacts, String repositoryId, String ns, String checksum) {
        for (String namespace : this.getNamespaces(repositoryId, ns)) {
            this.getArtifactsByChecksum(artifacts, repositoryId, ns + "." + namespace, checksum);
        }
        for (String project : this.getProjects(repositoryId, ns)) {
            for (String version : this.getProjectVersions(repositoryId, ns, project)) {
                for (ArtifactMetadata artifact : this.getArtifacts(repositoryId, ns, project, version)) {
                    if (!checksum.equals(artifact.getMd5()) && !checksum.equals(artifact.getSha1())) continue;
                    artifacts.add(artifact);
                }
            }
        }
    }

    private File getMetadataDirectory(String repoId, String facetId) {
        return new File(this.getBaseDirectory(repoId), "facets/" + facetId);
    }

    private String join(Collection<String> ids) {
        if (ids != null && !ids.isEmpty()) {
            StringBuilder s = new StringBuilder();
            for (String id : ids) {
                s.append(id);
                s.append(",");
            }
            return s.substring(0, s.length() - 1);
        }
        return "";
    }

    private void setProperty(Properties properties, String name, String value) {
        if (value != null) {
            properties.setProperty(name, value);
        }
    }

    public void updateArtifact(String repoId, String namespace, String projectId, String projectVersion, ArtifactMetadata artifact) {
        ProjectVersionMetadata metadata = new ProjectVersionMetadata();
        metadata.setId(projectVersion);
        this.updateProjectVersion(repoId, namespace, projectId, metadata);
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        FileMetadataRepository.clearMetadataFacetProperties(artifact.getFacetList(), properties, "artifact:facet:" + artifact.getId() + ":");
        String id = artifact.getId();
        properties.setProperty("artifact:updated:" + id, Long.toString(artifact.getFileLastModified().getTime()));
        properties.setProperty("artifact:whenGathered:" + id, Long.toString(artifact.getWhenGathered().getTime()));
        properties.setProperty("artifact:size:" + id, Long.toString(artifact.getSize()));
        if (artifact.getMd5() != null) {
            properties.setProperty("artifact:md5:" + id, artifact.getMd5());
        }
        if (artifact.getSha1() != null) {
            properties.setProperty("artifact:sha1:" + id, artifact.getSha1());
        }
        properties.setProperty("artifact:version:" + id, artifact.getVersion());
        LinkedHashSet<String> facetIds = new LinkedHashSet<String>(artifact.getFacetIds());
        String property = "artifact:facetIds:" + id;
        facetIds.addAll(Arrays.asList(properties.getProperty(property, "").split(",")));
        properties.setProperty(property, this.join(facetIds));
        this.updateArtifactFacets(artifact, properties);
        try {
            this.writeProperties(properties, directory, PROJECT_VERSION_METADATA_KEY);
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    private Properties readOrCreateProperties(File directory, String propertiesKey) {
        try {
            return this.readProperties(directory, propertiesKey);
        }
        catch (FileNotFoundException | NoSuchFileException e) {
        }
        catch (IOException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
        return new Properties();
    }

    private Properties readProperties(File directory, String propertiesKey) throws IOException {
        Properties properties = new Properties();
        try (InputStream in = Files.newInputStream(new File(directory, propertiesKey + ".properties").toPath(), new OpenOption[0]);){
            properties.load(in);
        }
        return properties;
    }

    public ProjectMetadata getProject(String repoId, String namespace, String projectId) {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_METADATA_KEY);
        ProjectMetadata project = null;
        String id = properties.getProperty("id");
        if (id != null) {
            project = new ProjectMetadata();
            project.setNamespace(properties.getProperty("namespace"));
            project.setId(id);
        }
        return project;
    }

    public ProjectVersionMetadata getProjectVersion(String repoId, String namespace, String projectId, String projectVersion) {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        String id = properties.getProperty("id");
        ProjectVersionMetadata versionMetadata = null;
        if (id != null) {
            versionMetadata = new ProjectVersionMetadata();
            versionMetadata.setId(id);
            versionMetadata.setName(properties.getProperty("name"));
            versionMetadata.setDescription(properties.getProperty("description"));
            versionMetadata.setUrl(properties.getProperty("url"));
            versionMetadata.setIncomplete(Boolean.valueOf(properties.getProperty("incomplete", "false")).booleanValue());
            String scmConnection = properties.getProperty("scm.connection");
            String scmDeveloperConnection = properties.getProperty("scm.developerConnection");
            String scmUrl = properties.getProperty("scm.url");
            if (scmConnection != null || scmDeveloperConnection != null || scmUrl != null) {
                Scm scm = new Scm();
                scm.setConnection(scmConnection);
                scm.setDeveloperConnection(scmDeveloperConnection);
                scm.setUrl(scmUrl);
                versionMetadata.setScm(scm);
            }
            String ciSystem = properties.getProperty("ci.system");
            String ciUrl = properties.getProperty("ci.url");
            if (ciSystem != null || ciUrl != null) {
                CiManagement ci = new CiManagement();
                ci.setSystem(ciSystem);
                ci.setUrl(ciUrl);
                versionMetadata.setCiManagement(ci);
            }
            String issueSystem = properties.getProperty("issue.system");
            String issueUrl = properties.getProperty("issue.url");
            if (issueSystem != null || issueUrl != null) {
                IssueManagement issueManagement = new IssueManagement();
                issueManagement.setSystem(issueSystem);
                issueManagement.setUrl(issueUrl);
                versionMetadata.setIssueManagement(issueManagement);
            }
            String orgName = properties.getProperty("org.name");
            String orgUrl = properties.getProperty("org.url");
            if (orgName != null || orgUrl != null) {
                Organization org = new Organization();
                org.setName(orgName);
                org.setUrl(orgUrl);
                versionMetadata.setOrganization(org);
            }
            boolean done = false;
            int i = 0;
            while (!done) {
                String licenseName = properties.getProperty("license." + i + ".name");
                String licenseUrl = properties.getProperty("license." + i + ".url");
                if (licenseName != null || licenseUrl != null) {
                    License license = new License();
                    license.setName(licenseName);
                    license.setUrl(licenseUrl);
                    versionMetadata.addLicense(license);
                } else {
                    done = true;
                }
                ++i;
            }
            done = false;
            i = 0;
            while (!done) {
                String mailingListName = properties.getProperty("mailingList." + i + ".name");
                if (mailingListName != null) {
                    MailingList mailingList = new MailingList();
                    mailingList.setName(mailingListName);
                    mailingList.setMainArchiveUrl(properties.getProperty("mailingList." + i + ".archive"));
                    String p = properties.getProperty("mailingList." + i + ".otherArchives");
                    if (p != null && p.length() > 0) {
                        mailingList.setOtherArchives(Arrays.asList(p.split(",")));
                    } else {
                        mailingList.setOtherArchives(Collections.emptyList());
                    }
                    mailingList.setPostAddress(properties.getProperty("mailingList." + i + ".post"));
                    mailingList.setSubscribeAddress(properties.getProperty("mailingList." + i + ".subscribe"));
                    mailingList.setUnsubscribeAddress(properties.getProperty("mailingList." + i + ".unsubscribe"));
                    versionMetadata.addMailingList(mailingList);
                } else {
                    done = true;
                }
                ++i;
            }
            done = false;
            i = 0;
            while (!done) {
                String dependencyArtifactId = properties.getProperty("dependency." + i + ".artifactId");
                if (dependencyArtifactId != null) {
                    Dependency dependency = new Dependency();
                    dependency.setArtifactId(dependencyArtifactId);
                    dependency.setGroupId(properties.getProperty("dependency." + i + ".groupId"));
                    dependency.setClassifier(properties.getProperty("dependency." + i + ".classifier"));
                    dependency.setOptional(Boolean.valueOf(properties.getProperty("dependency." + i + ".optional")).booleanValue());
                    dependency.setScope(properties.getProperty("dependency." + i + ".scope"));
                    dependency.setSystemPath(properties.getProperty("dependency." + i + ".systemPath"));
                    dependency.setType(properties.getProperty("dependency." + i + ".type"));
                    dependency.setVersion(properties.getProperty("dependency." + i + ".version"));
                    dependency.setOptional(Boolean.valueOf(properties.getProperty("dependency." + i + ".optional")).booleanValue());
                    versionMetadata.addDependency(dependency);
                } else {
                    done = true;
                }
                ++i;
            }
            String facetIds = properties.getProperty("facetIds", "");
            if (facetIds.length() > 0) {
                for (String facetId : facetIds.split(",")) {
                    MetadataFacetFactory factory = this.metadataFacetFactories.get(facetId);
                    if (factory == null) {
                        this.log.error("Attempted to load unknown project version metadata facet: {}", (Object)facetId);
                        continue;
                    }
                    MetadataFacet facet = factory.createMetadataFacet();
                    HashMap<String, String> map = new HashMap<String, String>();
                    for (Object key : new ArrayList<Object>(properties.keySet())) {
                        String property = (String)key;
                        if (!property.startsWith(facet.getFacetId())) continue;
                        map.put(property.substring(facet.getFacetId().length() + 1), properties.getProperty(property));
                    }
                    facet.fromProperties(map);
                    versionMetadata.addFacet(facet);
                }
            }
            this.updateProjectVersionFacets(versionMetadata, properties);
        }
        return versionMetadata;
    }

    public Collection<String> getArtifactVersions(String repoId, String namespace, String projectId, String projectVersion) {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        HashSet<String> versions = new HashSet<String>();
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            String name = (String)entry.getKey();
            if (!name.startsWith("artifact:version:")) continue;
            versions.add((String)entry.getValue());
        }
        return versions;
    }

    public Collection<ProjectVersionReference> getProjectReferences(String repoId, String namespace, String projectId, String projectVersion) {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        Properties properties = this.readOrCreateProperties(directory, PROJECT_VERSION_METADATA_KEY);
        int numberOfRefs = Integer.parseInt(properties.getProperty("ref:lastReferenceNum", "-1")) + 1;
        ArrayList<ProjectVersionReference> references = new ArrayList<ProjectVersionReference>();
        for (int i = 0; i < numberOfRefs; ++i) {
            ProjectVersionReference reference = new ProjectVersionReference();
            reference.setProjectId(properties.getProperty("ref:reference." + i + ".projectId"));
            reference.setNamespace(properties.getProperty("ref:reference." + i + ".namespace"));
            reference.setProjectVersion(properties.getProperty("ref:reference." + i + ".projectVersion"));
            reference.setReferenceType(ProjectVersionReference.ReferenceType.valueOf((String)properties.getProperty("ref:reference." + i + ".referenceType")));
            references.add(reference);
        }
        return references;
    }

    public Collection<String> getRootNamespaces(String repoId) {
        return this.getNamespaces(repoId, null);
    }

    public Collection<String> getNamespaces(String repoId, String baseNamespace) {
        ArrayList<String> allNamespaces = new ArrayList<String>();
        File directory = this.getDirectory(repoId);
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                if (!new File(file, "namespace-metadata.properties").exists()) continue;
                allNamespaces.add(file.getName());
            }
        }
        LinkedHashSet<String> namespaces = new LinkedHashSet<String>();
        int fromIndex = baseNamespace != null ? baseNamespace.length() + 1 : 0;
        for (String string : allNamespaces) {
            if (baseNamespace != null && !string.startsWith(baseNamespace + ".")) continue;
            int i = string.indexOf(46, fromIndex);
            if (i >= 0) {
                namespaces.add(string.substring(fromIndex, i));
                continue;
            }
            namespaces.add(string.substring(fromIndex));
        }
        return new ArrayList<String>(namespaces);
    }

    public Collection<String> getProjects(String repoId, String namespace) {
        ArrayList<String> projects = new ArrayList<String>();
        File directory = new File(this.getDirectory(repoId), namespace);
        File[] files = directory.listFiles();
        if (files != null) {
            for (File project : files) {
                if (!new File(project, "project-metadata.properties").exists()) continue;
                projects.add(project.getName());
            }
        }
        return projects;
    }

    public Collection<String> getProjectVersions(String repoId, String namespace, String projectId) {
        ArrayList<String> projectVersions = new ArrayList<String>();
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId);
        File[] files = directory.listFiles();
        if (files != null) {
            for (File projectVersion : files) {
                if (!new File(projectVersion, "version-metadata.properties").exists()) continue;
                projectVersions.add(projectVersion.getName());
            }
        }
        return projectVersions;
    }

    public void removeProject(String repositoryId, String namespace, String projectId) throws MetadataRepositoryException {
        File directory = new File(this.getDirectory(repositoryId), namespace + "/" + projectId);
        try {
            if (directory.exists()) {
                FileUtils.deleteDirectory((File)directory);
            }
        }
        catch (IOException e) {
            throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
        }
    }

    public void removeProjectVersion(String repoId, String namespace, String projectId, String projectVersion) throws MetadataRepositoryException {
        File directory = new File(this.getDirectory(repoId), namespace + "/" + projectId + "/" + projectVersion);
        if (directory.exists()) {
            try {
                FileUtils.deleteDirectory((File)directory);
            }
            catch (IOException e) {
                throw new MetadataRepositoryException(e.getMessage(), (Throwable)e);
            }
        }
    }

    private void writeProperties(Properties properties, File directory, String propertiesKey) throws IOException {
        directory.mkdirs();
        try (OutputStream os = Files.newOutputStream(new File(directory, propertiesKey + ".properties").toPath(), new OpenOption[0]);){
            properties.store(os, null);
        }
    }

    public List<ArtifactMetadata> getArtifacts(String repoId) {
        ArrayList<ArtifactMetadata> artifacts = new ArrayList<ArtifactMetadata>();
        for (String ns : this.getRootNamespaces(repoId)) {
            this.getArtifacts(artifacts, repoId, ns);
        }
        return artifacts;
    }

    private void getArtifacts(List<ArtifactMetadata> artifacts, String repoId, String ns) {
        for (String namespace : this.getNamespaces(repoId, ns)) {
            this.getArtifacts(artifacts, repoId, ns + "." + namespace);
        }
        for (String project : this.getProjects(repoId, ns)) {
            for (String version : this.getProjectVersions(repoId, ns, project)) {
                for (ArtifactMetadata artifact : this.getArtifacts(repoId, ns, project, version)) {
                    artifacts.add(artifact);
                }
            }
        }
    }

    private static class ArtifactComparator
    implements Comparator<ArtifactMetadata> {
        private ArtifactComparator() {
        }

        @Override
        public int compare(ArtifactMetadata artifact1, ArtifactMetadata artifact2) {
            if (artifact1.getWhenGathered() == artifact2.getWhenGathered()) {
                return 0;
            }
            if (artifact1.getWhenGathered() == null) {
                return 1;
            }
            if (artifact2.getWhenGathered() == null) {
                return -1;
            }
            return artifact1.getWhenGathered().compareTo(artifact2.getWhenGathered());
        }
    }
}

