/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.dht;

import com.google.common.base.Charsets;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.gms.IFailureDetector;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.locator.TokenMetadata;
import org.apache.cassandra.net.IAsyncCallback;
import org.apache.cassandra.net.IVerbHandler;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.streaming.OperationType;
import org.apache.cassandra.streaming.StreamIn;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.SimpleCondition;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BootStrapper {
    private static final Logger logger = LoggerFactory.getLogger(BootStrapper.class);
    protected final InetAddress address;
    protected final Token<?> token;
    protected final TokenMetadata tokenMetadata;
    private static final long BOOTSTRAP_TIMEOUT = 30000L;

    public BootStrapper(InetAddress address, Token token, TokenMetadata tmd) {
        assert (address != null);
        assert (token != null);
        this.address = address;
        this.token = token;
        this.tokenMetadata = tmd;
    }

    public void bootstrap() throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("Beginning bootstrap process");
        }
        HashMultimap rangesToFetch = HashMultimap.create();
        int requests = 0;
        for (String table : Schema.instance.getNonSystemTables()) {
            Map workMap = BootStrapper.getWorkMap(this.getRangesWithSources(table)).asMap();
            for (Map.Entry entry : workMap.entrySet()) {
                ++requests;
                rangesToFetch.put((Object)table, entry);
            }
        }
        final CountDownLatch latch = new CountDownLatch(requests);
        for (final String table : rangesToFetch.keySet()) {
            for (Map.Entry entry : rangesToFetch.get((Object)table)) {
                final InetAddress source = (InetAddress)entry.getKey();
                Runnable callback = new Runnable(){

                    @Override
                    public void run() {
                        latch.countDown();
                        if (logger.isDebugEnabled()) {
                            logger.debug(String.format("Removed %s/%s as a bootstrap source; remaining is %s", source, table, latch.getCount()));
                        }
                    }
                };
                if (logger.isDebugEnabled()) {
                    logger.debug("Bootstrapping from " + source + " ranges " + StringUtils.join((Collection)((Collection)entry.getValue()), (String)", "));
                }
                StreamIn.requestRanges(source, table, (Collection)entry.getValue(), callback, OperationType.BOOTSTRAP);
            }
        }
        try {
            latch.await();
            StorageService.instance.finishBootstrapping();
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
    }

    public static Token getBootstrapToken(TokenMetadata metadata, Map<InetAddress, Double> load) throws IOException, ConfigurationException {
        if (DatabaseDescriptor.getInitialToken() != null) {
            logger.debug("token manually specified as " + DatabaseDescriptor.getInitialToken());
            Token token = StorageService.getPartitioner().getTokenFactory().fromString(DatabaseDescriptor.getInitialToken());
            if (metadata.getEndpoint(token) != null) {
                throw new ConfigurationException("Bootstraping to existing token " + token + " is not allowed (decommission/removetoken the old node first).");
            }
            return token;
        }
        return BootStrapper.getBalancedToken(metadata, load);
    }

    public static Token getBalancedToken(TokenMetadata metadata, Map<InetAddress, Double> load) {
        InetAddress maxEndpoint = BootStrapper.getBootstrapSource(metadata, load);
        Token<?> t = BootStrapper.getBootstrapTokenFrom(maxEndpoint);
        logger.info("New token will be " + t + " to assume load from " + maxEndpoint);
        return t;
    }

    static InetAddress getBootstrapSource(final TokenMetadata metadata, final Map<InetAddress, Double> load) {
        ArrayList<InetAddress> endpoints = new ArrayList<InetAddress>(load.size());
        for (InetAddress endpoint : load.keySet()) {
            if (!metadata.isMember(endpoint)) continue;
            endpoints.add(endpoint);
        }
        if (endpoints.isEmpty()) {
            throw new RuntimeException("No other nodes seen!  Unable to bootstrap.If you intended to start a single-node cluster, you should make sure your broadcast_address (or listen_address) is listed as a seed.  Otherwise, you need to determine why the seed being contacted has no knowledge of the rest of the cluster.  Usually, this can be solved by giving all nodes the same seed list.");
        }
        Collections.sort(endpoints, new Comparator<InetAddress>(){

            @Override
            public int compare(InetAddress ia1, InetAddress ia2) {
                double load2;
                int n2;
                int n1 = metadata.pendingRangeChanges(ia1);
                if (n1 != (n2 = metadata.pendingRangeChanges(ia2))) {
                    return -(n1 - n2);
                }
                double load1 = (Double)load.get(ia1);
                if (load1 == (load2 = ((Double)load.get(ia2)).doubleValue())) {
                    return 0;
                }
                return load1 < load2 ? -1 : 1;
            }
        });
        InetAddress maxEndpoint = (InetAddress)endpoints.get(endpoints.size() - 1);
        assert (!maxEndpoint.equals(FBUtilities.getBroadcastAddress()));
        if (metadata.pendingRangeChanges(maxEndpoint) > 0) {
            throw new RuntimeException("Every node is a bootstrap source! Please specify an initial token manually or wait for an existing bootstrap operation to finish.");
        }
        return maxEndpoint;
    }

    Multimap<Range, InetAddress> getRangesWithSources(String table) {
        assert (this.tokenMetadata.sortedTokens().size() > 0);
        AbstractReplicationStrategy strat = Table.open(table).getReplicationStrategy();
        Collection<Range> myRanges = strat.getPendingAddressRanges(this.tokenMetadata, this.token, this.address);
        ArrayListMultimap myRangeAddresses = ArrayListMultimap.create();
        Multimap<Range, InetAddress> rangeAddresses = strat.getRangeAddresses(this.tokenMetadata);
        for (Range myRange : myRanges) {
            for (Range range : rangeAddresses.keySet()) {
                if (!range.contains(myRange)) continue;
                List<InetAddress> preferred = DatabaseDescriptor.getEndpointSnitch().getSortedListByProximity(this.address, rangeAddresses.get((Object)range));
                myRangeAddresses.putAll((Object)myRange, preferred);
                break;
            }
            assert (myRangeAddresses.keySet().contains(myRange));
        }
        return myRangeAddresses;
    }

    static Token<?> getBootstrapTokenFrom(InetAddress maxEndpoint) {
        Message message = new Message(FBUtilities.getBroadcastAddress(), StorageService.Verb.BOOTSTRAP_TOKEN, ArrayUtils.EMPTY_BYTE_ARRAY, Gossiper.instance.getVersion(maxEndpoint));
        long timeout = Math.max(MessagingService.getDefaultCallbackTimeout(), 30000L);
        for (int retries = 5; retries > 0; --retries) {
            BootstrapTokenCallback btc = new BootstrapTokenCallback();
            MessagingService.instance().sendRR(message, maxEndpoint, btc, timeout);
            Token<?> token = btc.getToken(timeout);
            if (token == null) continue;
            return token;
        }
        throw new RuntimeException("Bootstrap failed, could not obtain token from: " + maxEndpoint);
    }

    public static Multimap<InetAddress, Range> getWorkMap(Multimap<Range, InetAddress> rangesWithSourceTarget) {
        return BootStrapper.getWorkMap(rangesWithSourceTarget, FailureDetector.instance);
    }

    static Multimap<InetAddress, Range> getWorkMap(Multimap<Range, InetAddress> rangesWithSourceTarget, IFailureDetector failureDetector) {
        ArrayListMultimap sources = ArrayListMultimap.create();
        block0: for (Range range : rangesWithSourceTarget.keySet()) {
            for (InetAddress source : rangesWithSourceTarget.get((Object)range)) {
                if (!failureDetector.isAlive(source) || source.equals(FBUtilities.getBroadcastAddress())) continue;
                sources.put((Object)source, (Object)range);
                continue block0;
            }
        }
        return sources;
    }

    private static class BootstrapTokenCallback
    implements IAsyncCallback {
        private volatile Token<?> token;
        private final Condition condition = new SimpleCondition();

        private BootstrapTokenCallback() {
        }

        public Token<?> getToken(long timeout) {
            boolean success;
            try {
                success = this.condition.await(timeout, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            return success ? this.token : null;
        }

        @Override
        public void response(Message msg) {
            this.token = StorageService.getPartitioner().getTokenFactory().fromString(new String(msg.getMessageBody(), Charsets.UTF_8));
            this.condition.signalAll();
        }

        @Override
        public boolean isLatencyForSnitch() {
            return false;
        }
    }

    public static class BootstrapTokenVerbHandler
    implements IVerbHandler {
        @Override
        public void doVerb(Message message, String id) {
            StorageService ss = StorageService.instance;
            String tokenString = StorageService.getPartitioner().getTokenFactory().toString(ss.getBootstrapToken());
            Message response = message.getInternalReply(tokenString.getBytes(Charsets.UTF_8), message.getVersion());
            MessagingService.instance().sendReply(response, id, message.getFrom());
        }
    }
}

