/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.gms;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.GossipDigest;
import org.apache.cassandra.gms.GossipDigestSyn;
import org.apache.cassandra.gms.GossiperMBean;
import org.apache.cassandra.gms.HeartBeatState;
import org.apache.cassandra.gms.IEndpointStateChangeSubscriber;
import org.apache.cassandra.gms.IFailureDetectionEventListener;
import org.apache.cassandra.gms.IFailureDetector;
import org.apache.cassandra.gms.VersionedValue;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Gossiper
implements IFailureDetectionEventListener,
GossiperMBean {
    private static final String MBEAN_NAME = "org.apache.cassandra.net:type=Gossiper";
    private static final DebuggableScheduledThreadPoolExecutor executor = new DebuggableScheduledThreadPoolExecutor("GossipTasks");
    static final ApplicationState[] STATES = ApplicationState.values();
    static final List<String> DEAD_STATES = Arrays.asList("removing", "removed", "LEFT", "hibernate");
    private ScheduledFuture<?> scheduledGossipTask;
    public static final int intervalInMillis = 1000;
    public static final int QUARANTINE_DELAY = StorageService.RING_DELAY * 2;
    private static final Logger logger = LoggerFactory.getLogger(Gossiper.class);
    public static final Gossiper instance = new Gossiper();
    public static final long aVeryLongTime = 259200000L;
    private long FatClientTimeout;
    private final Random random = new Random();
    private final Comparator<InetAddress> inetcomparator = new Comparator<InetAddress>(){

        @Override
        public int compare(InetAddress addr1, InetAddress addr2) {
            return addr1.getHostAddress().compareTo(addr2.getHostAddress());
        }
    };
    private final List<IEndpointStateChangeSubscriber> subscribers = new CopyOnWriteArrayList<IEndpointStateChangeSubscriber>();
    private final Set<InetAddress> liveEndpoints = new ConcurrentSkipListSet<InetAddress>(this.inetcomparator);
    private final Map<InetAddress, Long> unreachableEndpoints = new ConcurrentHashMap<InetAddress, Long>();
    private final Set<InetAddress> seeds = new ConcurrentSkipListSet<InetAddress>(this.inetcomparator);
    final ConcurrentMap<InetAddress, EndpointState> endpointStateMap = new ConcurrentHashMap<InetAddress, EndpointState>();
    private final Map<InetAddress, Long> justRemovedEndpoints = new ConcurrentHashMap<InetAddress, Long>();
    private final Map<InetAddress, Long> expireTimeEndpointMap = new ConcurrentHashMap<InetAddress, Long>();
    private boolean seedContacted = false;

    private Gossiper() {
        this.FatClientTimeout = QUARANTINE_DELAY / 2;
        FailureDetector.instance.registerFailureDetectionEventListener(this);
        try {
            MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    protected void checkSeedContact(InetAddress ep) {
        if (!this.seedContacted && this.seeds.contains(ep)) {
            this.seedContacted = true;
        }
    }

    public boolean seenAnySeed() {
        return this.seedContacted;
    }

    public void register(IEndpointStateChangeSubscriber subscriber) {
        this.subscribers.add(subscriber);
    }

    public void unregister(IEndpointStateChangeSubscriber subscriber) {
        this.subscribers.remove(subscriber);
    }

    public Set<InetAddress> getLiveMembers() {
        HashSet<InetAddress> liveMbrs = new HashSet<InetAddress>(this.liveEndpoints);
        if (!liveMbrs.contains(FBUtilities.getBroadcastAddress())) {
            liveMbrs.add(FBUtilities.getBroadcastAddress());
        }
        return liveMbrs;
    }

    public Set<InetAddress> getUnreachableMembers() {
        return this.unreachableEndpoints.keySet();
    }

    public long getEndpointDowntime(InetAddress ep) {
        Long downtime = this.unreachableEndpoints.get(ep);
        if (downtime != null) {
            return System.currentTimeMillis() - downtime;
        }
        return 0L;
    }

    @Override
    public void convict(InetAddress endpoint, double phi) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState.isAlive() && !this.isDeadState(epState).booleanValue()) {
            this.markDead(endpoint, epState);
        } else {
            epState.markDead();
        }
    }

    int getMaxEndpointStateVersion(EndpointState epState) {
        int maxVersion = epState.getHeartBeatState().getHeartBeatVersion();
        for (VersionedValue value : epState.getApplicationStateMap().values()) {
            maxVersion = Math.max(maxVersion, value.version);
        }
        return maxVersion;
    }

    private void evictFromMembership(InetAddress endpoint) {
        this.unreachableEndpoints.remove(endpoint);
        this.endpointStateMap.remove(endpoint);
        this.expireTimeEndpointMap.remove(endpoint);
        this.quarantineEndpoint(endpoint);
        if (logger.isDebugEnabled()) {
            logger.debug("evicting " + endpoint + " from gossip");
        }
    }

    public void removeEndpoint(InetAddress endpoint) {
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onRemove(endpoint);
        }
        if (this.seeds.contains(endpoint)) {
            this.buildSeedsList();
            this.seeds.remove(endpoint);
            logger.info("removed {} from seeds, updated seeds list = {}", (Object)endpoint, this.seeds);
        }
        this.liveEndpoints.remove(endpoint);
        this.unreachableEndpoints.remove(endpoint);
        FailureDetector.instance.remove(endpoint);
        MessagingService.instance().resetVersion(endpoint);
        this.quarantineEndpoint(endpoint);
        MessagingService.instance().destroyConnectionPool(endpoint);
        if (logger.isDebugEnabled()) {
            logger.debug("removing endpoint " + endpoint);
        }
    }

    private void quarantineEndpoint(InetAddress endpoint) {
        this.justRemovedEndpoints.put(endpoint, System.currentTimeMillis());
    }

    public void replacedEndpoint(InetAddress endpoint) {
        this.removeEndpoint(endpoint);
        this.evictFromMembership(endpoint);
    }

    private void makeRandomGossipDigest(List<GossipDigest> gDigests) {
        int generation = 0;
        int maxVersion = 0;
        ArrayList endpoints = new ArrayList(this.endpointStateMap.keySet());
        Collections.shuffle(endpoints, this.random);
        for (InetAddress endpoint : endpoints) {
            EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
            if (epState != null) {
                generation = epState.getHeartBeatState().getGeneration();
                maxVersion = this.getMaxEndpointStateVersion(epState);
            }
            gDigests.add(new GossipDigest(endpoint, generation, maxVersion));
        }
        if (logger.isTraceEnabled()) {
            StringBuilder sb = new StringBuilder();
            for (GossipDigest gDigest : gDigests) {
                sb.append(gDigest);
                sb.append(" ");
            }
            logger.trace("Gossip Digests are : " + sb.toString());
        }
    }

    public void advertiseRemoving(InetAddress endpoint, UUID hostId, UUID localHostId) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        int generation = epState.getHeartBeatState().getGeneration();
        logger.info("Removing host: {}", (Object)hostId);
        logger.info("Sleeping for " + StorageService.RING_DELAY + "ms to ensure " + endpoint + " does not change");
        try {
            Thread.sleep(StorageService.RING_DELAY);
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
        epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState.getHeartBeatState().getGeneration() != generation) {
            throw new RuntimeException("Endpoint " + endpoint + " generation changed while trying to remove it");
        }
        logger.info("Advertising removal for " + endpoint);
        epState.updateTimestamp();
        epState.getHeartBeatState().forceNewerGenerationUnsafe();
        epState.addApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.removingNonlocal(hostId));
        epState.addApplicationState(ApplicationState.REMOVAL_COORDINATOR, StorageService.instance.valueFactory.removalCoordinator(localHostId));
        this.endpointStateMap.put(endpoint, epState);
    }

    public void advertiseTokenRemoved(InetAddress endpoint, UUID hostId) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        epState.updateTimestamp();
        epState.getHeartBeatState().forceNewerGenerationUnsafe();
        long expireTime = Gossiper.computeExpireTime();
        epState.addApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.removedNonlocal(hostId, expireTime));
        logger.info("Completing removal of " + endpoint);
        this.addExpireTimeForEndpoint(endpoint, expireTime);
        this.endpointStateMap.put(endpoint, epState);
        try {
            Thread.sleep(2000L);
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public void unsafeAssassinateEndpoint(String address) throws UnknownHostException {
        InetAddress endpoint = InetAddress.getByName(address);
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        Collection<Token> tokens = null;
        logger.warn("Assassinating {} via gossip", (Object)endpoint);
        if (epState == null) {
            epState = new EndpointState(new HeartBeatState((int)((System.currentTimeMillis() + 60000L) / 1000L), 9999));
        } else {
            try {
                tokens = StorageService.instance.getTokenMetadata().getTokens(endpoint);
            }
            catch (AssertionError e) {
                // empty catch block
            }
            int generation = epState.getHeartBeatState().getGeneration();
            logger.info("Sleeping for " + StorageService.RING_DELAY + "ms to ensure " + endpoint + " does not change");
            try {
                Thread.sleep(StorageService.RING_DELAY);
            }
            catch (InterruptedException e) {
                throw new AssertionError((Object)e);
            }
            epState = (EndpointState)this.endpointStateMap.get(endpoint);
            if (epState.getHeartBeatState().getGeneration() != generation) {
                throw new RuntimeException("Endpoint " + endpoint + " generation changed while trying to remove it");
            }
            epState.updateTimestamp();
            epState.getHeartBeatState().forceNewerGenerationUnsafe();
        }
        if (tokens == null) {
            tokens = Arrays.asList(StorageService.instance.getBootstrapToken());
        }
        epState.addApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.left(tokens, Gossiper.computeExpireTime()));
        this.handleMajorStateChange(endpoint, epState);
        try {
            Thread.sleep(4000L);
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
        logger.warn("Finished assassinating {}", (Object)endpoint);
    }

    public boolean isKnownEndpoint(InetAddress endpoint) {
        return this.endpointStateMap.containsKey(endpoint);
    }

    public int getCurrentGenerationNumber(InetAddress endpoint) {
        return ((EndpointState)this.endpointStateMap.get(endpoint)).getHeartBeatState().getGeneration();
    }

    private boolean sendGossip(MessageOut<GossipDigestSyn> message, Set<InetAddress> epSet) {
        ImmutableList liveEndpoints = ImmutableList.copyOf(epSet);
        int size = liveEndpoints.size();
        if (size < 1) {
            return false;
        }
        int index = size == 1 ? 0 : this.random.nextInt(size);
        InetAddress to = (InetAddress)liveEndpoints.get(index);
        if (logger.isTraceEnabled()) {
            logger.trace("Sending a GossipDigestSyn to {} ...", (Object)to);
        }
        MessagingService.instance().sendOneWay(message, to);
        return this.seeds.contains(to);
    }

    private boolean doGossipToLiveMember(MessageOut<GossipDigestSyn> message) {
        int size = this.liveEndpoints.size();
        if (size == 0) {
            return false;
        }
        return this.sendGossip(message, this.liveEndpoints);
    }

    private void doGossipToUnreachableMember(MessageOut<GossipDigestSyn> message) {
        double liveEndpointCount = this.liveEndpoints.size();
        double unreachableEndpointCount = this.unreachableEndpoints.size();
        if (unreachableEndpointCount > 0.0) {
            double prob = unreachableEndpointCount / (liveEndpointCount + 1.0);
            double randDbl = this.random.nextDouble();
            if (randDbl < prob) {
                this.sendGossip(message, this.unreachableEndpoints.keySet());
            }
        }
    }

    private void doGossipToSeed(MessageOut<GossipDigestSyn> prod) {
        int size = this.seeds.size();
        if (size > 0) {
            if (size == 1 && this.seeds.contains(FBUtilities.getBroadcastAddress())) {
                return;
            }
            if (this.liveEndpoints.size() == 0) {
                this.sendGossip(prod, this.seeds);
            } else {
                double probability = (double)this.seeds.size() / (double)(this.liveEndpoints.size() + this.unreachableEndpoints.size());
                double randDbl = this.random.nextDouble();
                if (randDbl <= probability) {
                    this.sendGossip(prod, this.seeds);
                }
            }
        }
    }

    public boolean isFatClient(InetAddress endpoint) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState == null) {
            return false;
        }
        return this.isDeadState(epState) == false && !epState.isAlive() && !StorageService.instance.getTokenMetadata().isMember(endpoint);
    }

    private void doStatusCheck() {
        long now = System.currentTimeMillis();
        Set eps = this.endpointStateMap.keySet();
        for (InetAddress inetAddress : eps) {
            if (inetAddress.equals(FBUtilities.getBroadcastAddress())) continue;
            FailureDetector.instance.interpret(inetAddress);
            EndpointState epState = (EndpointState)this.endpointStateMap.get(inetAddress);
            if (epState == null) continue;
            long duration = now - epState.getUpdateTimestamp();
            if (this.isFatClient(inetAddress) && !this.justRemovedEndpoints.containsKey(inetAddress) && duration > this.FatClientTimeout) {
                logger.info("FatClient " + inetAddress + " has been silent for " + this.FatClientTimeout + "ms, removing from gossip");
                this.removeEndpoint(inetAddress);
                this.evictFromMembership(inetAddress);
            }
            long expireTime = this.getExpireTimeForEndpoint(inetAddress);
            if (epState.isAlive() || now <= expireTime || StorageService.instance.getTokenMetadata().isMember(inetAddress)) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("time is expiring for endpoint : " + inetAddress + " (" + expireTime + ")");
            }
            this.evictFromMembership(inetAddress);
        }
        if (!this.justRemovedEndpoints.isEmpty()) {
            for (Map.Entry entry : this.justRemovedEndpoints.entrySet()) {
                if (now - (Long)entry.getValue() <= (long)QUARANTINE_DELAY) continue;
                if (logger.isDebugEnabled()) {
                    logger.debug(QUARANTINE_DELAY + " elapsed, " + entry.getKey() + " gossip quarantine over");
                }
                this.justRemovedEndpoints.remove(entry.getKey());
            }
        }
    }

    protected long getExpireTimeForEndpoint(InetAddress endpoint) {
        Long storedTime = this.expireTimeEndpointMap.get(endpoint);
        return storedTime == null ? Gossiper.computeExpireTime() : storedTime;
    }

    public EndpointState getEndpointStateForEndpoint(InetAddress ep) {
        return (EndpointState)this.endpointStateMap.get(ep);
    }

    public Set<Map.Entry<InetAddress, EndpointState>> getEndpointStates() {
        return this.endpointStateMap.entrySet();
    }

    public boolean usesHostId(InetAddress endpoint) {
        if (MessagingService.instance().knowsVersion(endpoint) && MessagingService.instance().getVersion(endpoint) >= 6) {
            return true;
        }
        return this.getEndpointStateForEndpoint(endpoint).getApplicationState(ApplicationState.NET_VERSION) != null && Integer.parseInt(this.getEndpointStateForEndpoint((InetAddress)endpoint).getApplicationState((ApplicationState)ApplicationState.NET_VERSION).value) >= 6;
    }

    public boolean usesVnodes(InetAddress endpoint) {
        return this.usesHostId(endpoint) && this.getEndpointStateForEndpoint(endpoint).getApplicationState(ApplicationState.TOKENS) != null;
    }

    public UUID getHostId(InetAddress endpoint) {
        if (!this.usesHostId(endpoint)) {
            throw new RuntimeException("Host " + endpoint + " does not use new-style tokens!");
        }
        return UUID.fromString(this.getEndpointStateForEndpoint((InetAddress)endpoint).getApplicationState((ApplicationState)ApplicationState.HOST_ID).value);
    }

    EndpointState getStateForVersionBiggerThan(InetAddress forEndpoint, int version) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(forEndpoint);
        EndpointState reqdEndpointState = null;
        if (epState != null) {
            int localHbVersion = epState.getHeartBeatState().getHeartBeatVersion();
            if (localHbVersion > version) {
                reqdEndpointState = new EndpointState(epState.getHeartBeatState());
                if (logger.isTraceEnabled()) {
                    logger.trace("local heartbeat version " + localHbVersion + " greater than " + version + " for " + forEndpoint);
                }
            }
            for (Map.Entry<ApplicationState, VersionedValue> entry : epState.getApplicationStateMap().entrySet()) {
                VersionedValue value = entry.getValue();
                if (value.version <= version) continue;
                if (reqdEndpointState == null) {
                    reqdEndpointState = new EndpointState(epState.getHeartBeatState());
                }
                ApplicationState key = entry.getKey();
                if (logger.isTraceEnabled()) {
                    logger.trace("Adding state " + (Object)((Object)key) + ": " + value.value);
                }
                reqdEndpointState.addApplicationState(key, value);
            }
        }
        return reqdEndpointState;
    }

    public int compareEndpointStartup(InetAddress addr1, InetAddress addr2) {
        EndpointState ep1 = this.getEndpointStateForEndpoint(addr1);
        EndpointState ep2 = this.getEndpointStateForEndpoint(addr2);
        assert (ep1 != null && ep2 != null);
        return ep1.getHeartBeatState().getGeneration() - ep2.getHeartBeatState().getGeneration();
    }

    void notifyFailureDetector(Map<InetAddress, EndpointState> remoteEpStateMap) {
        for (Map.Entry<InetAddress, EndpointState> entry : remoteEpStateMap.entrySet()) {
            this.notifyFailureDetector(entry.getKey(), entry.getValue());
        }
    }

    void notifyFailureDetector(InetAddress endpoint, EndpointState remoteEndpointState) {
        EndpointState localEndpointState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (localEndpointState != null) {
            IFailureDetector fd = FailureDetector.instance;
            int localGeneration = localEndpointState.getHeartBeatState().getGeneration();
            int remoteGeneration = remoteEndpointState.getHeartBeatState().getGeneration();
            if (remoteGeneration > localGeneration) {
                localEndpointState.updateTimestamp();
                if (!localEndpointState.isAlive()) {
                    logger.debug("Clearing interval times for {} due to generation change", (Object)endpoint);
                    fd.clear(endpoint);
                }
                fd.report(endpoint);
                return;
            }
            if (remoteGeneration == localGeneration) {
                int localVersion = this.getMaxEndpointStateVersion(localEndpointState);
                int remoteVersion = remoteEndpointState.getHeartBeatState().getHeartBeatVersion();
                if (remoteVersion > localVersion) {
                    localEndpointState.updateTimestamp();
                    fd.report(endpoint);
                }
            }
        }
    }

    private void markAlive(InetAddress addr, EndpointState localState) {
        if (logger.isTraceEnabled()) {
            logger.trace("marking as alive {}", (Object)addr);
        }
        localState.markAlive();
        localState.updateTimestamp();
        this.liveEndpoints.add(addr);
        this.unreachableEndpoints.remove(addr);
        this.expireTimeEndpointMap.remove(addr);
        logger.debug("removing expire time for endpoint : " + addr);
        logger.info("InetAddress {} is now UP", (Object)addr);
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onAlive(addr, localState);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Notified " + this.subscribers);
        }
    }

    private void markDead(InetAddress addr, EndpointState localState) {
        if (logger.isTraceEnabled()) {
            logger.trace("marking as down {}", (Object)addr);
        }
        localState.markDead();
        this.liveEndpoints.remove(addr);
        this.unreachableEndpoints.put(addr, System.currentTimeMillis());
        logger.info("InetAddress {} is now DOWN", (Object)addr);
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onDead(addr, localState);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Notified " + this.subscribers);
        }
    }

    private void handleMajorStateChange(InetAddress ep, EndpointState epState) {
        if (!this.isDeadState(epState).booleanValue()) {
            if (this.endpointStateMap.get(ep) != null) {
                logger.info("Node {} has restarted, now UP", (Object)ep);
            } else {
                logger.info("Node {} is now part of the cluster", (Object)ep);
            }
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Adding endpoint state for " + ep);
        }
        this.endpointStateMap.put(ep, epState);
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onRestart(ep, epState);
        }
        if (!this.isDeadState(epState).booleanValue()) {
            this.markAlive(ep, epState);
        } else {
            logger.debug("Not marking " + ep + " alive due to dead state");
            this.markDead(ep, epState);
        }
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onJoin(ep, epState);
        }
    }

    private Boolean isDeadState(EndpointState epState) {
        if (epState.getApplicationState(ApplicationState.STATUS) == null) {
            return false;
        }
        String value = epState.getApplicationState((ApplicationState)ApplicationState.STATUS).value;
        String[] pieces = value.split(VersionedValue.DELIMITER_STR, -1);
        assert (pieces.length > 0);
        String state = pieces[0];
        for (String deadstate : DEAD_STATES) {
            if (!state.equals(deadstate)) continue;
            return true;
        }
        return false;
    }

    void applyStateLocally(Map<InetAddress, EndpointState> epStateMap) {
        for (Map.Entry<InetAddress, EndpointState> entry : epStateMap.entrySet()) {
            InetAddress ep = entry.getKey();
            if (ep.equals(FBUtilities.getBroadcastAddress())) continue;
            if (this.justRemovedEndpoints.containsKey(ep)) {
                if (!logger.isTraceEnabled()) continue;
                logger.trace("Ignoring gossip for " + ep + " because it is quarantined");
                continue;
            }
            EndpointState localEpStatePtr = (EndpointState)this.endpointStateMap.get(ep);
            EndpointState remoteState = entry.getValue();
            if (localEpStatePtr != null) {
                int localGeneration = localEpStatePtr.getHeartBeatState().getGeneration();
                int remoteGeneration = remoteState.getHeartBeatState().getGeneration();
                if (logger.isTraceEnabled()) {
                    logger.trace(ep + "local generation " + localGeneration + ", remote generation " + remoteGeneration);
                }
                if (remoteGeneration > localGeneration) {
                    if (logger.isTraceEnabled()) {
                        logger.trace("Updating heartbeat state generation to " + remoteGeneration + " from " + localGeneration + " for " + ep);
                    }
                    this.handleMajorStateChange(ep, remoteState);
                    continue;
                }
                if (remoteGeneration == localGeneration) {
                    int localMaxVersion = this.getMaxEndpointStateVersion(localEpStatePtr);
                    int remoteMaxVersion = this.getMaxEndpointStateVersion(remoteState);
                    if (remoteMaxVersion > localMaxVersion) {
                        this.applyNewStates(ep, localEpStatePtr, remoteState);
                    } else if (logger.isTraceEnabled()) {
                        logger.trace("Ignoring remote version " + remoteMaxVersion + " <= " + localMaxVersion + " for " + ep);
                    }
                    if (localEpStatePtr.isAlive() || this.isDeadState(localEpStatePtr).booleanValue()) continue;
                    this.markAlive(ep, localEpStatePtr);
                    continue;
                }
                if (!logger.isTraceEnabled()) continue;
                logger.trace("Ignoring remote generation " + remoteGeneration + " < " + localGeneration);
                continue;
            }
            FailureDetector.instance.report(ep);
            this.handleMajorStateChange(ep, remoteState);
        }
    }

    private void applyNewStates(InetAddress addr, EndpointState localState, EndpointState remoteState) {
        int oldVersion = localState.getHeartBeatState().getHeartBeatVersion();
        localState.setHeartBeatState(remoteState.getHeartBeatState());
        if (logger.isTraceEnabled()) {
            logger.trace("Updating heartbeat state version to " + localState.getHeartBeatState().getHeartBeatVersion() + " from " + oldVersion + " for " + addr + " ...");
        }
        for (Map.Entry<ApplicationState, VersionedValue> remoteEntry : remoteState.getApplicationStateMap().entrySet()) {
            ApplicationState remoteKey = remoteEntry.getKey();
            VersionedValue remoteValue = remoteEntry.getValue();
            assert (remoteState.getHeartBeatState().getGeneration() == localState.getHeartBeatState().getGeneration());
            localState.addApplicationState(remoteKey, remoteValue);
        }
        for (Map.Entry<ApplicationState, VersionedValue> remoteEntry : remoteState.getApplicationStateMap().entrySet()) {
            this.doNotifications(addr, remoteEntry.getKey(), remoteEntry.getValue());
        }
    }

    private void doNotifications(InetAddress addr, ApplicationState state, VersionedValue value) {
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onChange(addr, state, value);
        }
    }

    private void requestAll(GossipDigest gDigest, List<GossipDigest> deltaGossipDigestList, int remoteGeneration) {
        deltaGossipDigestList.add(new GossipDigest(gDigest.getEndpoint(), remoteGeneration, 0));
        if (logger.isTraceEnabled()) {
            logger.trace("requestAll for " + gDigest.getEndpoint());
        }
    }

    private void sendAll(GossipDigest gDigest, Map<InetAddress, EndpointState> deltaEpStateMap, int maxRemoteVersion) {
        EndpointState localEpStatePtr = this.getStateForVersionBiggerThan(gDigest.getEndpoint(), maxRemoteVersion);
        if (localEpStatePtr != null) {
            deltaEpStateMap.put(gDigest.getEndpoint(), localEpStatePtr);
        }
    }

    void examineGossiper(List<GossipDigest> gDigestList, List<GossipDigest> deltaGossipDigestList, Map<InetAddress, EndpointState> deltaEpStateMap) {
        for (GossipDigest gDigest : gDigestList) {
            int remoteGeneration = gDigest.getGeneration();
            int maxRemoteVersion = gDigest.getMaxVersion();
            EndpointState epStatePtr = (EndpointState)this.endpointStateMap.get(gDigest.getEndpoint());
            if (epStatePtr != null) {
                int localGeneration = epStatePtr.getHeartBeatState().getGeneration();
                int maxLocalVersion = this.getMaxEndpointStateVersion(epStatePtr);
                if (remoteGeneration == localGeneration && maxRemoteVersion == maxLocalVersion) continue;
                if (remoteGeneration > localGeneration) {
                    this.requestAll(gDigest, deltaGossipDigestList, remoteGeneration);
                    continue;
                }
                if (remoteGeneration < localGeneration) {
                    this.sendAll(gDigest, deltaEpStateMap, 0);
                    continue;
                }
                if (remoteGeneration != localGeneration) continue;
                if (maxRemoteVersion > maxLocalVersion) {
                    deltaGossipDigestList.add(new GossipDigest(gDigest.getEndpoint(), remoteGeneration, maxLocalVersion));
                    continue;
                }
                if (maxRemoteVersion >= maxLocalVersion) continue;
                this.sendAll(gDigest, deltaEpStateMap, maxRemoteVersion);
                continue;
            }
            this.requestAll(gDigest, deltaGossipDigestList, remoteGeneration);
        }
    }

    public void start(int generationNumber) {
        this.start(generationNumber, new HashMap<ApplicationState, VersionedValue>());
    }

    public void start(int generationNbr, Map<ApplicationState, VersionedValue> preloadLocalStates) {
        this.buildSeedsList();
        this.maybeInitializeLocalState(generationNbr);
        EndpointState localState = (EndpointState)this.endpointStateMap.get(FBUtilities.getBroadcastAddress());
        for (Map.Entry<ApplicationState, VersionedValue> entry : preloadLocalStates.entrySet()) {
            localState.addApplicationState(entry.getKey(), entry.getValue());
        }
        DatabaseDescriptor.getEndpointSnitch().gossiperStarting();
        if (logger.isTraceEnabled()) {
            logger.trace("gossip started with generation " + localState.getHeartBeatState().getGeneration());
        }
        this.scheduledGossipTask = executor.scheduleWithFixedDelay(new GossipTask(), 1000L, 1000L, TimeUnit.MILLISECONDS);
    }

    private void buildSeedsList() {
        for (InetAddress seed : DatabaseDescriptor.getSeeds()) {
            if (seed.equals(FBUtilities.getBroadcastAddress())) continue;
            this.seeds.add(seed);
        }
    }

    public void maybeInitializeLocalState(int generationNbr) {
        HeartBeatState hbState = new HeartBeatState(generationNbr);
        EndpointState localState = new EndpointState(hbState);
        localState.markAlive();
        this.endpointStateMap.putIfAbsent(FBUtilities.getBroadcastAddress(), localState);
    }

    public void addSavedEndpoint(InetAddress ep) {
        if (ep.equals(FBUtilities.getBroadcastAddress())) {
            logger.debug("Attempt to add self as saved endpoint");
            return;
        }
        EndpointState epState = (EndpointState)this.endpointStateMap.get(ep);
        if (epState != null) {
            logger.debug("not replacing a previous epState for {}, but reusing it: {}", (Object)ep, (Object)epState);
            epState.setHeartBeatState(new HeartBeatState(0));
        } else {
            epState = new EndpointState(new HeartBeatState(0));
        }
        epState.markDead();
        this.endpointStateMap.put(ep, epState);
        this.unreachableEndpoints.put(ep, System.currentTimeMillis());
        if (logger.isTraceEnabled()) {
            logger.trace("Adding saved endpoint " + ep + " " + epState.getHeartBeatState().getGeneration());
        }
    }

    public void addLocalApplicationState(ApplicationState state, VersionedValue value) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(FBUtilities.getBroadcastAddress());
        assert (epState != null);
        epState.addApplicationState(state, value);
        this.doNotifications(FBUtilities.getBroadcastAddress(), state, value);
    }

    public void stop() {
        this.scheduledGossipTask.cancel(false);
        logger.info("Announcing shutdown");
        try {
            Thread.sleep(2000L);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        MessageOut message = new MessageOut(MessagingService.Verb.GOSSIP_SHUTDOWN);
        for (InetAddress ep : this.liveEndpoints) {
            MessagingService.instance().sendOneWay(message, ep);
        }
    }

    public boolean isEnabled() {
        return this.scheduledGossipTask != null && !this.scheduledGossipTask.isCancelled();
    }

    @VisibleForTesting
    public void initializeNodeUnsafe(InetAddress addr, UUID uuid, int generationNbr) {
        HeartBeatState hbState = new HeartBeatState(generationNbr);
        EndpointState newState = new EndpointState(hbState);
        newState.markAlive();
        EndpointState oldState = this.endpointStateMap.putIfAbsent(addr, newState);
        EndpointState localState = oldState == null ? newState : oldState;
        localState.addApplicationState(ApplicationState.NET_VERSION, StorageService.instance.valueFactory.networkVersion());
        localState.addApplicationState(ApplicationState.HOST_ID, StorageService.instance.valueFactory.hostId(uuid));
    }

    @VisibleForTesting
    public void injectApplicationState(InetAddress endpoint, ApplicationState state, VersionedValue value) {
        EndpointState localState = (EndpointState)this.endpointStateMap.get(endpoint);
        localState.addApplicationState(state, value);
    }

    @Override
    public long getEndpointDowntime(String address) throws UnknownHostException {
        return this.getEndpointDowntime(InetAddress.getByName(address));
    }

    @Override
    public int getCurrentGenerationNumber(String address) throws UnknownHostException {
        return this.getCurrentGenerationNumber(InetAddress.getByName(address));
    }

    public void addExpireTimeForEndpoint(InetAddress endpoint, long expireTime) {
        if (logger.isDebugEnabled()) {
            logger.debug("adding expire time for endpoint : " + endpoint + " (" + expireTime + ")");
        }
        this.expireTimeEndpointMap.put(endpoint, expireTime);
    }

    public static long computeExpireTime() {
        return System.currentTimeMillis() + 259200000L;
    }

    private class GossipTask
    implements Runnable {
        private GossipTask() {
        }

        @Override
        public void run() {
            try {
                MessagingService.instance().waitUntilListening();
                ((EndpointState)Gossiper.this.endpointStateMap.get(FBUtilities.getBroadcastAddress())).getHeartBeatState().updateHeartBeat();
                if (logger.isTraceEnabled()) {
                    logger.trace("My heartbeat is now " + ((EndpointState)Gossiper.this.endpointStateMap.get(FBUtilities.getBroadcastAddress())).getHeartBeatState().getHeartBeatVersion());
                }
                ArrayList<GossipDigest> gDigests = new ArrayList<GossipDigest>();
                instance.makeRandomGossipDigest(gDigests);
                if (gDigests.size() > 0) {
                    GossipDigestSyn digestSynMessage = new GossipDigestSyn(DatabaseDescriptor.getClusterName(), DatabaseDescriptor.getPartitionerName(), gDigests);
                    MessageOut<GossipDigestSyn> message = new MessageOut<GossipDigestSyn>(MessagingService.Verb.GOSSIP_DIGEST_SYN, digestSynMessage, GossipDigestSyn.serializer);
                    boolean gossipedToSeed = Gossiper.this.doGossipToLiveMember(message);
                    Gossiper.this.doGossipToUnreachableMember(message);
                    if (!gossipedToSeed || Gossiper.this.liveEndpoints.size() < Gossiper.this.seeds.size()) {
                        Gossiper.this.doGossipToSeed(message);
                    }
                    if (logger.isTraceEnabled()) {
                        logger.trace("Performing status check ...");
                    }
                    Gossiper.this.doStatusCheck();
                }
            }
            catch (Exception e) {
                logger.error("Gossip error", (Throwable)e);
            }
        }
    }
}

