/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.thrift;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.TreeSet;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.cql3.CFDefinition;
import org.apache.cassandra.cql3.ColumnIdentifier;
import org.apache.cassandra.db.ColumnFamilyType;
import org.apache.cassandra.db.ExpiringColumn;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.KeyspaceNotDefinedException;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.SuperColumns;
import org.apache.cassandra.db.filter.IDiskAtomFilter;
import org.apache.cassandra.db.filter.NamesQueryFilter;
import org.apache.cassandra.db.filter.SliceQueryFilter;
import org.apache.cassandra.db.index.SecondaryIndex;
import org.apache.cassandra.db.index.SecondaryIndexManager;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.ColumnToCollectionType;
import org.apache.cassandra.db.marshal.CompositeType;
import org.apache.cassandra.db.marshal.UTF8Type;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.ColumnOrSuperColumn;
import org.apache.cassandra.thrift.ColumnParent;
import org.apache.cassandra.thrift.ColumnPath;
import org.apache.cassandra.thrift.Deletion;
import org.apache.cassandra.thrift.IndexClause;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.thrift.IndexOperator;
import org.apache.cassandra.thrift.KeyRange;
import org.apache.cassandra.thrift.Mutation;
import org.apache.cassandra.thrift.SlicePredicate;
import org.apache.cassandra.thrift.SliceRange;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThriftValidation {
    private static final Logger logger = LoggerFactory.getLogger(ThriftValidation.class);

    public static void validateKey(CFMetaData metadata, ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
        try {
            metadata.getKeyValidator().validate(key);
        }
        catch (MarshalException e) {
            throw new InvalidRequestException(e.getMessage());
        }
    }

    public static void validateKeyspace(String keyspaceName) throws KeyspaceNotDefinedException {
        if (!Schema.instance.getKeyspaces().contains(keyspaceName)) {
            throw new KeyspaceNotDefinedException("Keyspace " + keyspaceName + " does not exist");
        }
    }

    public static CFMetaData validateColumnFamily(String keyspaceName, String cfName, boolean isCommutativeOp) throws InvalidRequestException {
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspaceName, cfName);
        if (isCommutativeOp) {
            if (!metadata.getDefaultValidator().isCommutative()) {
                throw new InvalidRequestException("invalid operation for non commutative columnfamily " + cfName);
            }
        } else if (metadata.getDefaultValidator().isCommutative()) {
            throw new InvalidRequestException("invalid operation for commutative columnfamily " + cfName);
        }
        return metadata;
    }

    public static CFMetaData validateColumnFamily(String keyspaceName, String cfName) throws InvalidRequestException {
        ThriftValidation.validateKeyspace(keyspaceName);
        if (cfName.isEmpty()) {
            throw new InvalidRequestException("non-empty columnfamily is required");
        }
        CFMetaData metadata = Schema.instance.getCFMetaData(keyspaceName, cfName);
        if (metadata == null) {
            throw new InvalidRequestException("unconfigured columnfamily " + cfName);
        }
        return metadata;
    }

    public static void validateColumnPath(CFMetaData metadata, ColumnPath column_path) throws InvalidRequestException {
        if (metadata.cfType == ColumnFamilyType.Standard) {
            if (column_path.super_column != null) {
                throw new InvalidRequestException("supercolumn parameter is invalid for standard CF " + metadata.cfName);
            }
            if (column_path.column == null) {
                throw new InvalidRequestException("column parameter is not optional for standard CF " + metadata.cfName);
            }
        } else if (column_path.super_column == null) {
            throw new InvalidRequestException("supercolumn parameter is not optional for super CF " + metadata.cfName);
        }
        if (column_path.column != null) {
            ThriftValidation.validateColumnNames(metadata, column_path.super_column, Arrays.asList(column_path.column));
        }
        if (column_path.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(column_path.super_column));
        }
    }

    public static void validateColumnParent(CFMetaData metadata, ColumnParent column_parent) throws InvalidRequestException {
        if (metadata.cfType == ColumnFamilyType.Standard && column_parent.super_column != null) {
            throw new InvalidRequestException("columnfamily alone is required for standard CF " + metadata.cfName);
        }
        if (column_parent.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(column_parent.super_column));
        }
    }

    static void validateColumnPathOrParent(CFMetaData metadata, ColumnPath column_path_or_parent) throws InvalidRequestException {
        if (metadata.cfType == ColumnFamilyType.Standard && column_path_or_parent.super_column != null) {
            throw new InvalidRequestException("supercolumn may not be specified for standard CF " + metadata.cfName);
        }
        if (metadata.cfType == ColumnFamilyType.Super && column_path_or_parent.super_column == null && column_path_or_parent.column != null) {
            throw new InvalidRequestException("A column cannot be specified without specifying a super column for removal on super CF " + metadata.cfName);
        }
        if (column_path_or_parent.column != null) {
            ThriftValidation.validateColumnNames(metadata, column_path_or_parent.super_column, Arrays.asList(column_path_or_parent.column));
        }
        if (column_path_or_parent.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(column_path_or_parent.super_column));
        }
    }

    private static void validateColumnNames(CFMetaData metadata, ByteBuffer superColumnName, Iterable<ByteBuffer> column_names) throws InvalidRequestException {
        int maxNameLength = 65535;
        if (superColumnName != null) {
            if (superColumnName.remaining() > maxNameLength) {
                throw new InvalidRequestException("supercolumn name length must not be greater than " + maxNameLength);
            }
            if (superColumnName.remaining() == 0) {
                throw new InvalidRequestException("supercolumn name must not be empty");
            }
            if (metadata.cfType == ColumnFamilyType.Standard) {
                throw new InvalidRequestException("supercolumn specified to ColumnFamily " + metadata.cfName + " containing normal columns");
            }
        }
        AbstractType<?> comparator = SuperColumns.getComparatorFor(metadata, superColumnName);
        CFDefinition cfDef = metadata.getCfDef();
        boolean isCQL3Table = !metadata.isThriftCompatible();
        for (ByteBuffer name : column_names) {
            int minComponents;
            if (name.remaining() > maxNameLength) {
                throw new InvalidRequestException("column name length must not be greater than " + maxNameLength);
            }
            if (name.remaining() == 0) {
                throw new InvalidRequestException("column name must not be empty");
            }
            try {
                comparator.validate(name);
            }
            catch (MarshalException e) {
                throw new InvalidRequestException(e.getMessage());
            }
            if (!isCQL3Table) continue;
            CompositeType composite = (CompositeType)comparator;
            ByteBuffer[] components = composite.split(name);
            if (components.length < (minComponents = composite.types.size() - (cfDef.hasCollections ? 1 : 0))) {
                throw new InvalidRequestException(String.format("Not enough components (found %d but %d expected) for column name since %s is a CQL3 table", components.length, minComponents, metadata.cfName));
            }
            int columnIndex = composite.types.size() - (cfDef.hasCollections ? 2 : 1);
            ByteBuffer CQL3ColumnName = components[columnIndex];
            if (!CQL3ColumnName.hasRemaining()) continue;
            ColumnIdentifier columnId = new ColumnIdentifier(CQL3ColumnName, composite.types.get(columnIndex));
            CFDefinition.Name columnName = cfDef.get(columnId);
            if (columnName == null || columnName.isPrimaryKeyColumn()) {
                throw new InvalidRequestException(String.format("Invalid cell for CQL3 table %s. The CQL3 column component (%s) does not correspond to a defined CQL3 column", metadata.cfName, columnId));
            }
            if (!cfDef.hasCollections || components.length != composite.types.size()) continue;
            assert (components.length >= 2);
            ColumnToCollectionType collectionType = (ColumnToCollectionType)composite.types.get(composite.types.size() - 1);
            if (collectionType.defined.containsKey(CQL3ColumnName)) continue;
            throw new InvalidRequestException(String.format("Invalid collection component, %s is not a collection", UTF8Type.instance.getString(CQL3ColumnName)));
        }
    }

    public static void validateColumnNames(CFMetaData metadata, ColumnParent column_parent, Iterable<ByteBuffer> column_names) throws InvalidRequestException {
        ThriftValidation.validateColumnNames(metadata, column_parent.super_column, column_names);
    }

    public static void validateRange(CFMetaData metadata, ColumnParent column_parent, SliceRange range) throws InvalidRequestException {
        Comparator<ByteBuffer> orderedComparator;
        if (range.count < 0) {
            throw new InvalidRequestException("get_slice requires non-negative count");
        }
        int maxNameLength = 65535;
        if (range.start.remaining() > maxNameLength) {
            throw new InvalidRequestException("range start length cannot be larger than " + maxNameLength);
        }
        if (range.finish.remaining() > maxNameLength) {
            throw new InvalidRequestException("range finish length cannot be larger than " + maxNameLength);
        }
        AbstractType<?> comparator = SuperColumns.getComparatorFor(metadata, column_parent.super_column);
        try {
            comparator.validate(range.start);
            comparator.validate(range.finish);
        }
        catch (MarshalException e) {
            throw new InvalidRequestException(e.getMessage());
        }
        Comparator<ByteBuffer> comparator2 = orderedComparator = range.isReversed() ? comparator.reverseComparator : comparator;
        if (range.start.remaining() > 0 && range.finish.remaining() > 0 && orderedComparator.compare(range.start, range.finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in the order of traversal");
        }
    }

    public static void validateColumnOrSuperColumn(CFMetaData metadata, ByteBuffer key, ColumnOrSuperColumn cosc) throws InvalidRequestException {
        boolean isCommutative = metadata.getDefaultValidator().isCommutative();
        int nulls = 0;
        if (cosc.column == null) {
            ++nulls;
        }
        if (cosc.super_column == null) {
            ++nulls;
        }
        if (cosc.counter_column == null) {
            ++nulls;
        }
        if (cosc.counter_super_column == null) {
            ++nulls;
        }
        if (nulls != 3) {
            throw new InvalidRequestException("ColumnOrSuperColumn must have one (and only one) of column, super_column, counter and counter_super_column");
        }
        if (cosc.column != null) {
            if (isCommutative) {
                throw new InvalidRequestException("invalid operation for commutative columnfamily " + metadata.cfName);
            }
            ThriftValidation.validateTtl(cosc.column);
            ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column((ByteBuffer)null).setColumn(cosc.column.name));
            ThriftValidation.validateColumnData(metadata, key, cosc.column, false);
        }
        if (cosc.super_column != null) {
            if (isCommutative) {
                throw new InvalidRequestException("invalid operation for commutative columnfamily " + metadata.cfName);
            }
            for (Column c : cosc.super_column.columns) {
                ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column(cosc.super_column.name).setColumn(c.name));
                ThriftValidation.validateColumnData(metadata, key, c, true);
            }
        }
        if (cosc.counter_column != null) {
            if (!isCommutative) {
                throw new InvalidRequestException("invalid operation for non commutative columnfamily " + metadata.cfName);
            }
            ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column((ByteBuffer)null).setColumn(cosc.counter_column.name));
        }
        if (cosc.counter_super_column != null) {
            if (!isCommutative) {
                throw new InvalidRequestException("invalid operation for non commutative columnfamily " + metadata.cfName);
            }
            for (Column c : cosc.counter_super_column.columns) {
                ThriftValidation.validateColumnPath(metadata, new ColumnPath(metadata.cfName).setSuper_column(cosc.counter_super_column.name).setColumn(c.name));
            }
        }
    }

    private static void validateTtl(Column column) throws InvalidRequestException {
        if (column.isSetTtl()) {
            if (column.ttl <= 0) {
                throw new InvalidRequestException("ttl must be positive");
            }
            if (column.ttl > 630720000) {
                throw new InvalidRequestException(String.format("ttl is too large. requested (%d) maximum (%d)", column.ttl, 630720000));
            }
        } else assert (column.ttl == 0);
    }

    public static void validateMutation(CFMetaData metadata, ByteBuffer key, Mutation mut) throws InvalidRequestException {
        ColumnOrSuperColumn cosc = mut.column_or_supercolumn;
        Deletion del = mut.deletion;
        int nulls = 0;
        if (cosc == null) {
            ++nulls;
        }
        if (del == null) {
            ++nulls;
        }
        if (nulls != 1) {
            throw new InvalidRequestException("mutation must have one and only one of column_or_supercolumn or deletion");
        }
        if (cosc != null) {
            ThriftValidation.validateColumnOrSuperColumn(metadata, key, cosc);
        } else {
            ThriftValidation.validateDeletion(metadata, del);
        }
    }

    public static void validateDeletion(CFMetaData metadata, Deletion del) throws InvalidRequestException {
        if (del.super_column != null) {
            ThriftValidation.validateColumnNames(metadata, (ByteBuffer)null, Arrays.asList(del.super_column));
        }
        if (del.predicate != null) {
            ThriftValidation.validateSlicePredicate(metadata, del.super_column, del.predicate);
        }
        if (metadata.cfType == ColumnFamilyType.Standard && del.super_column != null) {
            String msg = String.format("Deletion of super columns is not possible on a standard ColumnFamily (KeySpace=%s ColumnFamily=%s Deletion=%s)", metadata.ksName, metadata.cfName, del);
            throw new InvalidRequestException(msg);
        }
        if (metadata.getDefaultValidator().isCommutative()) {
            del.timestamp = System.currentTimeMillis();
        } else if (!del.isSetTimestamp()) {
            throw new InvalidRequestException("Deletion timestamp is not optional for non commutative column family " + metadata.cfName);
        }
    }

    public static void validateSlicePredicate(CFMetaData metadata, ByteBuffer scName, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("A SlicePredicate must be given a list of Columns, a SliceRange, or both");
        }
        if (predicate.slice_range != null) {
            ThriftValidation.validateRange(metadata, new ColumnParent(metadata.cfName).setSuper_column(scName), predicate.slice_range);
        }
        if (predicate.column_names != null) {
            ThriftValidation.validateColumnNames(metadata, scName, (Iterable<ByteBuffer>)predicate.column_names);
        }
    }

    public static void validateColumnData(CFMetaData metadata, ByteBuffer key, Column column, boolean isSubColumn) throws InvalidRequestException {
        ThriftValidation.validateTtl(column);
        if (!column.isSetValue()) {
            throw new InvalidRequestException("Column value is required");
        }
        if (!column.isSetTimestamp()) {
            throw new InvalidRequestException("Column timestamp is required");
        }
        try {
            AbstractType<?> validator = metadata.getValueValidatorFromColumnName(column.name);
            if (validator != null) {
                validator.validate(column.value);
            }
        }
        catch (MarshalException me) {
            if (logger.isDebugEnabled()) {
                logger.debug("rejecting invalid value " + ByteBufferUtil.bytesToHex(ThriftValidation.summarize(column.value)));
            }
            throw new InvalidRequestException(String.format("(%s) [%s][%s][%s] failed validation", me.getMessage(), metadata.ksName, metadata.cfName, SuperColumns.getComparatorFor(metadata, isSubColumn).getString(column.name)));
        }
        SecondaryIndex failedIndex = Keyspace.open((String)metadata.ksName).getColumnFamilyStore((String)metadata.cfName).indexManager.validate(key, ThriftValidation.asDBColumn(column));
        if (failedIndex != null) {
            throw new InvalidRequestException(String.format("Can't index column value of size %d for index %s in CF %s of KS %s", column.value.remaining(), failedIndex.getIndexName(), metadata.cfName, metadata.ksName));
        }
    }

    private static org.apache.cassandra.db.Column asDBColumn(Column column) {
        if (column.ttl <= 0) {
            return new org.apache.cassandra.db.Column(column.name, column.value, column.timestamp);
        }
        return new ExpiringColumn(column.name, column.value, column.timestamp, column.ttl);
    }

    private static ByteBuffer summarize(ByteBuffer buffer) {
        int MAX = Short.MAX_VALUE;
        if (buffer.remaining() <= MAX) {
            return buffer;
        }
        return (ByteBuffer)buffer.slice().limit(buffer.position() + MAX);
    }

    public static void validatePredicate(CFMetaData metadata, ColumnParent column_parent, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be null");
        }
        if (predicate.column_names != null && predicate.slice_range != null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be present");
        }
        if (predicate.getSlice_range() != null) {
            ThriftValidation.validateRange(metadata, column_parent, predicate.slice_range);
        } else {
            ThriftValidation.validateColumnNames(metadata, column_parent, (Iterable<ByteBuffer>)predicate.column_names);
        }
    }

    public static void validateKeyRange(CFMetaData metadata, ByteBuffer superColumn, KeyRange range) throws InvalidRequestException {
        if (range.start_key == null == (range.start_token == null) || range.end_key == null == (range.end_token == null)) {
            throw new InvalidRequestException("exactly one each of {start key, start token} and {end key, end token} must be specified");
        }
        if (range.start_token != null && range.end_key != null) {
            throw new InvalidRequestException("start token + end key is not a supported key range");
        }
        IPartitioner p = StorageService.getPartitioner();
        if (range.start_key != null && range.end_key != null) {
            Object endToken;
            Object startToken = p.getToken(range.start_key);
            if (((Token)startToken).compareTo(endToken = p.getToken(range.end_key)) > 0 && !((Token)endToken).isMinimum(p)) {
                if (p.preservesOrder()) {
                    throw new InvalidRequestException("start key must sort before (or equal to) finish key in your partitioner!");
                }
                throw new InvalidRequestException("start key's token sorts after end key's token.  this is not allowed; you probably should not specify end key at all except with an ordered partitioner");
            }
        } else if (range.start_key != null && range.end_token != null) {
            Token.KeyBound stop = p.getTokenFactory().fromString(range.end_token).maxKeyBound(p);
            if (RowPosition.forKey(range.start_key, p).compareTo(stop) > 0 && !stop.isMinimum()) {
                throw new InvalidRequestException("Start key's token sorts after end token");
            }
        }
        ThriftValidation.validateFilterClauses(metadata, range.row_filter);
        if (!ThriftValidation.isEmpty(range.row_filter) && superColumn != null) {
            throw new InvalidRequestException("super columns are not supported for indexing");
        }
        if (range.count <= 0) {
            throw new InvalidRequestException("maxRows must be positive");
        }
    }

    private static boolean isEmpty(List<IndexExpression> clause) {
        return clause == null || clause.isEmpty();
    }

    public static void validateIndexClauses(CFMetaData metadata, IndexClause index_clause) throws InvalidRequestException {
        if (index_clause.expressions.isEmpty()) {
            throw new InvalidRequestException("index clause list may not be empty");
        }
        if (!ThriftValidation.validateFilterClauses(metadata, index_clause.expressions)) {
            throw new InvalidRequestException("No indexed columns present in index clause with operator EQ");
        }
    }

    public static boolean validateFilterClauses(CFMetaData metadata, List<IndexExpression> index_clause) throws InvalidRequestException {
        if (ThriftValidation.isEmpty(index_clause)) {
            return false;
        }
        SecondaryIndexManager idxManager = Keyspace.open((String)metadata.ksName).getColumnFamilyStore((String)metadata.cfName).indexManager;
        AbstractType<?> nameValidator = SuperColumns.getComparatorFor(metadata, null);
        boolean isIndexed = false;
        for (IndexExpression expression : index_clause) {
            try {
                nameValidator.validate(expression.column_name);
            }
            catch (MarshalException me) {
                throw new InvalidRequestException(String.format("[%s]=[%s] failed name validation (%s)", ByteBufferUtil.bytesToHex(expression.column_name), ByteBufferUtil.bytesToHex(expression.value), me.getMessage()));
            }
            if (expression.value.remaining() > 65535) {
                throw new InvalidRequestException("Index expression values may not be larger than 64K");
            }
            AbstractType<?> valueValidator = metadata.getValueValidatorFromColumnName(expression.column_name);
            try {
                valueValidator.validate(expression.value);
            }
            catch (MarshalException me) {
                throw new InvalidRequestException(String.format("[%s]=[%s] failed value validation (%s)", ByteBufferUtil.bytesToHex(expression.column_name), ByteBufferUtil.bytesToHex(expression.value), me.getMessage()));
            }
            isIndexed |= expression.op == IndexOperator.EQ && idxManager.indexes(expression.column_name);
        }
        return isIndexed;
    }

    public static void validateKeyspaceNotYetExisting(String newKsName) throws InvalidRequestException {
        for (String ksName : Schema.instance.getKeyspaces()) {
            if (!ksName.equalsIgnoreCase(newKsName)) continue;
            throw new InvalidRequestException(String.format("Keyspace names must be case-insensitively unique (\"%s\" conflicts with \"%s\")", newKsName, ksName));
        }
    }

    public static void validateKeyspaceNotSystem(String modifiedKeyspace) throws InvalidRequestException {
        if (modifiedKeyspace.equalsIgnoreCase("system")) {
            throw new InvalidRequestException("system keyspace is not user-modifiable");
        }
    }

    public static IDiskAtomFilter asIFilter(SlicePredicate sp, CFMetaData metadata, ByteBuffer superColumn) {
        IDiskAtomFilter filter;
        SliceRange sr = sp.slice_range;
        if (sr == null) {
            AbstractType<?> comparator = metadata.isSuper() ? ((CompositeType)metadata.comparator).types.get(superColumn == null ? 0 : 1) : metadata.comparator;
            TreeSet<ByteBuffer> ss = new TreeSet<ByteBuffer>(comparator);
            ss.addAll(sp.column_names);
            filter = new NamesQueryFilter(ss);
        } else {
            filter = new SliceQueryFilter(sr.start, sr.finish, sr.reversed, sr.count);
        }
        if (metadata.isSuper()) {
            filter = SuperColumns.fromSCFilter((CompositeType)metadata.comparator, superColumn, filter);
        }
        return filter;
    }
}

