/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.AbstractCommitLogService;
import org.apache.cassandra.db.commitlog.BatchCommitLogService;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogMBean;
import org.apache.cassandra.db.commitlog.CommitLogReplayer;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.CommitLogSegmentManager;
import org.apache.cassandra.db.commitlog.PeriodicCommitLogService;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.DataOutputByteBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.metrics.CommitLogMetrics;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.PureJavaCrc32;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLog
implements CommitLogMBean {
    private static final Logger logger = LoggerFactory.getLogger(CommitLog.class);
    public static final CommitLog instance = new CommitLog();
    private static final long MAX_MUTATION_SIZE = DatabaseDescriptor.getCommitLogSegmentSize() >> 1;
    public final CommitLogSegmentManager allocator;
    public final CommitLogArchiver archiver = new CommitLogArchiver();
    final CommitLogMetrics metrics;
    final AbstractCommitLogService executor;

    private CommitLog() {
        DatabaseDescriptor.createAllDirectories();
        this.allocator = new CommitLogSegmentManager();
        this.executor = DatabaseDescriptor.getCommitLogSync() == Config.CommitLogSync.batch ? new BatchCommitLogService(this) : new PeriodicCommitLogService(this);
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName("org.apache.cassandra.db:type=Commitlog"));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.metrics = new CommitLogMetrics(this.executor, this.allocator);
    }

    public int recover() throws IOException {
        FilenameFilter unmanagedFilesFilter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return CommitLogDescriptor.isValid(name) && !CommitLog.instance.allocator.manages(name);
            }
        };
        for (File file : new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(unmanagedFilesFilter)) {
            this.archiver.maybeArchive(file.getPath(), file.getName());
            this.archiver.maybeWaitForArchiving(file.getName());
        }
        assert (this.archiver.archivePending.isEmpty()) : "Not all commit log archive tasks were completed before restore";
        this.archiver.maybeRestoreArchive();
        Object[] files = new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(unmanagedFilesFilter);
        int replayed = 0;
        if (files.length == 0) {
            logger.info("No commitlog files found; skipping replay");
        } else {
            Arrays.sort(files, new CommitLogSegment.CommitLogSegmentFileComparator());
            logger.info("Replaying {}", (Object)StringUtils.join((Object[])files, (String)", "));
            replayed = this.recover((File[])files);
            logger.info("Log replay complete, {} replayed mutations", (Object)replayed);
            for (Object f : files) {
                CommitLog.instance.allocator.recycleSegment((File)f);
            }
        }
        this.allocator.enableReserveSegmentCreation();
        return replayed;
    }

    public int recover(File ... clogs) throws IOException {
        CommitLogReplayer recovery = new CommitLogReplayer();
        recovery.recover(clogs);
        return recovery.blockForWrites();
    }

    @Override
    public void recover(String path) throws IOException {
        this.recover(new File(path));
    }

    public ReplayPosition getContext() {
        return this.allocator.allocatingFrom().getContext();
    }

    public void forceRecycleAllSegments(Iterable<UUID> droppedCfs) {
        this.allocator.forceRecycleAll(droppedCfs);
    }

    public void forceRecycleAllSegments() {
        this.allocator.forceRecycleAll(Collections.emptyList());
    }

    public void sync(boolean syncAllSegments) {
        CommitLogSegment current = this.allocator.allocatingFrom();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            if (!syncAllSegments && segment.id > current.id) {
                return;
            }
            segment.sync();
        }
    }

    public void requestExtraSync() {
        this.executor.requestExtraSync();
    }

    public ReplayPosition add(Mutation mutation) {
        assert (mutation != null);
        long size = Mutation.serializer.serializedSize(mutation, 8);
        long totalSize = size + 12L;
        if (totalSize > MAX_MUTATION_SIZE) {
            throw new IllegalArgumentException(String.format("Mutation of %s bytes is too large for the maximum size of %s", totalSize, MAX_MUTATION_SIZE));
        }
        CommitLogSegment.Allocation alloc = this.allocator.allocate(mutation, (int)totalSize);
        try {
            PureJavaCrc32 checksum = new PureJavaCrc32();
            ByteBuffer buffer = alloc.getBuffer();
            DataOutputByteBuffer dos = new DataOutputByteBuffer(buffer);
            dos.writeInt((int)size);
            checksum.update(buffer, buffer.position() - 4, 4);
            buffer.putInt(checksum.getCrc());
            int start = buffer.position();
            Mutation.serializer.serialize(mutation, (DataOutputPlus)dos, 8);
            checksum.update(buffer, start, (int)size);
            buffer.putInt(checksum.getCrc());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, alloc.getSegment().getPath());
        }
        finally {
            alloc.markWritten();
        }
        this.executor.finishWriteFor(alloc);
        return alloc.getReplayPosition();
    }

    public void discardCompletedSegments(UUID cfId, ReplayPosition context) {
        logger.debug("discard completed log segments for {}, column family {}", (Object)context, (Object)cfId);
        Iterator<CommitLogSegment> iter = this.allocator.getActiveSegments().iterator();
        while (iter.hasNext()) {
            CommitLogSegment segment = iter.next();
            segment.markClean(cfId, context);
            if (segment.isUnused()) {
                logger.debug("Commit log segment {} is unused", (Object)segment);
                this.allocator.recycleSegment(segment);
            } else {
                logger.debug("Not safe to delete{} commit log segment {}; dirty is {}", new Object[]{iter.hasNext() ? "" : " active", segment, segment.dirtyString()});
            }
            if (!segment.contains(context)) continue;
            break;
        }
    }

    @Override
    public long getCompletedTasks() {
        return (Long)this.metrics.completedTasks.value();
    }

    @Override
    public long getPendingTasks() {
        return (Long)this.metrics.pendingTasks.value();
    }

    @Override
    public long getTotalCommitlogSize() {
        return (Long)this.metrics.totalCommitLogSize.value();
    }

    @Override
    public String getArchiveCommand() {
        return this.archiver.archiveCommand;
    }

    @Override
    public String getRestoreCommand() {
        return this.archiver.restoreCommand;
    }

    @Override
    public String getRestoreDirectories() {
        return this.archiver.restoreDirectories;
    }

    @Override
    public long getRestorePointInTime() {
        return this.archiver.restorePointInTime;
    }

    @Override
    public String getRestorePrecision() {
        return this.archiver.precision.toString();
    }

    @Override
    public List<String> getActiveSegmentNames() {
        ArrayList<String> segmentNames = new ArrayList<String>();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segmentNames.add(segment.getName());
        }
        return segmentNames;
    }

    @Override
    public List<String> getArchivingSegmentNames() {
        return new ArrayList<String>(this.archiver.archivePending.keySet());
    }

    public void shutdownBlocking() throws InterruptedException {
        this.executor.shutdown();
        this.executor.awaitTermination();
        this.allocator.shutdown();
        this.allocator.awaitTermination();
    }

    public void resetUnsafe() {
        this.sync(true);
        this.allocator.resetUnsafe();
    }

    public int activeSegments() {
        return this.allocator.getActiveSegments().size();
    }

    @VisibleForTesting
    public static boolean handleCommitError(String message, Throwable t) {
        JVMStabilityInspector.inspectCommitLogThrowable(t);
        switch (DatabaseDescriptor.getCommitFailurePolicy()) {
            case die: 
            case stop: {
                StorageService.instance.stopTransports();
            }
            case stop_commit: {
                logger.error(String.format("%s. Commit disk failure policy is %s; terminating thread", new Object[]{message, DatabaseDescriptor.getCommitFailurePolicy()}), t);
                return false;
            }
            case ignore: {
                logger.error(message, t);
                return true;
            }
        }
        throw new AssertionError((Object)DatabaseDescriptor.getCommitFailurePolicy());
    }
}

