/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.lifecycle.View;
import org.apache.cassandra.io.sstable.IndexSummaryBuilder;
import org.apache.cassandra.io.sstable.IndexSummaryManagerMBean;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexSummaryManager
implements IndexSummaryManagerMBean {
    private static final Logger logger = LoggerFactory.getLogger(IndexSummaryManager.class);
    public static final String MBEAN_NAME = "org.apache.cassandra.db:type=IndexSummaries";
    public static final IndexSummaryManager instance = new IndexSummaryManager();
    private int resizeIntervalInMinutes = 0;
    private long memoryPoolBytes;
    static final double UPSAMPLE_THRESHOLD = 1.5;
    static final double DOWNSAMPLE_THESHOLD = 0.75;
    private final DebuggableScheduledThreadPoolExecutor executor = new DebuggableScheduledThreadPoolExecutor(1, "IndexSummaryManager", 1);
    private ScheduledFuture future;

    private IndexSummaryManager() {
        long indexSummarySizeInMB = DatabaseDescriptor.getIndexSummaryCapacityInMB();
        int interval = DatabaseDescriptor.getIndexSummaryResizeIntervalInMinutes();
        logger.info("Initializing index summary manager with a memory pool size of {} MB and a resize interval of {} minutes", (Object)indexSummarySizeInMB, (Object)interval);
        this.setMemoryPoolCapacityInMB(DatabaseDescriptor.getIndexSummaryCapacityInMB());
        this.setResizeIntervalInMinutes(DatabaseDescriptor.getIndexSummaryResizeIntervalInMinutes());
    }

    @Override
    public int getResizeIntervalInMinutes() {
        return this.resizeIntervalInMinutes;
    }

    @Override
    public void setResizeIntervalInMinutes(int resizeIntervalInMinutes) {
        long initialDelay;
        int oldInterval = this.resizeIntervalInMinutes;
        this.resizeIntervalInMinutes = resizeIntervalInMinutes;
        if (this.future != null) {
            initialDelay = oldInterval < 0 ? (long)resizeIntervalInMinutes : Math.max(0L, (long)resizeIntervalInMinutes - ((long)oldInterval - this.future.getDelay(TimeUnit.MINUTES)));
            this.future.cancel(false);
        } else {
            initialDelay = resizeIntervalInMinutes;
        }
        if (this.resizeIntervalInMinutes < 0) {
            this.future = null;
            return;
        }
        this.future = this.executor.scheduleWithFixedDelay(new WrappedRunnable(){

            @Override
            protected void runMayThrow() throws Exception {
                IndexSummaryManager.this.redistributeSummaries();
            }
        }, initialDelay, resizeIntervalInMinutes, TimeUnit.MINUTES);
    }

    @VisibleForTesting
    Long getTimeToNextResize(TimeUnit timeUnit) {
        if (this.future == null) {
            return null;
        }
        return this.future.getDelay(timeUnit);
    }

    @Override
    public long getMemoryPoolCapacityInMB() {
        return this.memoryPoolBytes / 1024L / 1024L;
    }

    @Override
    public Map<String, Integer> getIndexIntervals() {
        List<SSTableReader> sstables = this.getAllSSTables();
        HashMap<String, Integer> intervals = new HashMap<String, Integer>(sstables.size());
        for (SSTableReader sstable : sstables) {
            intervals.put(sstable.getFilename(), (int)Math.round(sstable.getEffectiveIndexInterval()));
        }
        return intervals;
    }

    @Override
    public double getAverageIndexInterval() {
        List<SSTableReader> sstables = this.getAllSSTables();
        double total = 0.0;
        for (SSTableReader sstable : sstables) {
            total += sstable.getEffectiveIndexInterval();
        }
        return total / (double)sstables.size();
    }

    @Override
    public void setMemoryPoolCapacityInMB(long memoryPoolCapacityInMB) {
        this.memoryPoolBytes = memoryPoolCapacityInMB * 1024L * 1024L;
    }

    @Override
    public double getMemoryPoolSizeInMB() {
        long total = 0L;
        for (SSTableReader sstable : this.getAllSSTables()) {
            total += sstable.getIndexSummaryOffHeapSize();
        }
        return (double)total / 1024.0 / 1024.0;
    }

    private List<SSTableReader> getAllSSTables() {
        ArrayList<SSTableReader> result = new ArrayList<SSTableReader>();
        for (Keyspace ks : Keyspace.all()) {
            for (ColumnFamilyStore cfStore : ks.getColumnFamilyStores()) {
                result.addAll(cfStore.getSSTables());
            }
        }
        return result;
    }

    private Pair<List<SSTableReader>, Map<UUID, LifecycleTransaction>> getCompactingAndNonCompactingSSTables() {
        ArrayList allCompacting = new ArrayList();
        HashMap<UUID, LifecycleTransaction> allNonCompacting = new HashMap<UUID, LifecycleTransaction>();
        for (Keyspace ks : Keyspace.all()) {
            for (ColumnFamilyStore cfStore : ks.getColumnFamilyStores()) {
                Set<SSTableReader> allSSTables;
                ImmutableSet nonCompacting;
                LifecycleTransaction txn = null;
                do {
                    View view = cfStore.getTracker().getView();
                    allSSTables = view.sstables;
                    nonCompacting = ImmutableSet.copyOf(view.getUncompacting(allSSTables));
                } while (null == (txn = cfStore.getTracker().tryModify((Iterable<SSTableReader>)nonCompacting, OperationType.UNKNOWN)));
                allNonCompacting.put(cfStore.metadata.cfId, txn);
                allCompacting.addAll(Sets.difference(allSSTables, (Set)nonCompacting));
            }
        }
        return Pair.create(allCompacting, allNonCompacting);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void redistributeSummaries() throws IOException {
        Pair<List<SSTableReader>, Map<UUID, LifecycleTransaction>> compactingAndNonCompacting = this.getCompactingAndNonCompactingSSTables();
        try {
            IndexSummaryManager.redistributeSummaries((List)compactingAndNonCompacting.left, (Map)compactingAndNonCompacting.right, this.memoryPoolBytes);
        }
        finally {
            for (LifecycleTransaction modifier : ((Map)compactingAndNonCompacting.right).values()) {
                modifier.close();
            }
        }
    }

    @VisibleForTesting
    public static List<SSTableReader> redistributeSummaries(List<SSTableReader> compacting, Map<UUID, LifecycleTransaction> transactions, long memoryPoolBytes) throws IOException {
        logger.info("Redistributing index summaries");
        ArrayList<SSTableReader> oldFormatSSTables = new ArrayList<SSTableReader>();
        ArrayList<SSTableReader> redistribute = new ArrayList<SSTableReader>();
        for (LifecycleTransaction txn : transactions.values()) {
            for (SSTableReader sstable : ImmutableList.copyOf(txn.originals())) {
                logger.trace("SSTable {} cannot be re-sampled due to old sstable format", (Object)sstable);
                if (sstable.descriptor.version.hasSamplingLevel()) continue;
                oldFormatSSTables.add(sstable);
                txn.cancel(sstable);
            }
            redistribute.addAll(txn.originals());
        }
        long total = 0L;
        for (SSTableReader sstable : Iterables.concat(compacting, redistribute)) {
            total += sstable.getIndexSummaryOffHeapSize();
        }
        logger.trace("Beginning redistribution of index summaries for {} sstables with memory pool size {} MB; current spaced used is {} MB", new Object[]{redistribute.size(), memoryPoolBytes / 1024L / 1024L, (double)total / 1024.0 / 1024.0});
        HashMap<SSTableReader, Double> readRates = new HashMap<SSTableReader, Double>(redistribute.size());
        double totalReadsPerSec = 0.0;
        for (SSTableReader sstable : redistribute) {
            if (sstable.getReadMeter() == null) continue;
            Double readRate = sstable.getReadMeter().fifteenMinuteRate();
            totalReadsPerSec += readRate.doubleValue();
            readRates.put(sstable, readRate);
        }
        logger.trace("Total reads/sec across all sstables in index summary resize process: {}", (Object)totalReadsPerSec);
        ArrayList<SSTableReader> sstablesByHotness = new ArrayList<SSTableReader>(redistribute);
        Collections.sort(sstablesByHotness, new ReadRateComparator(readRates));
        long remainingBytes = memoryPoolBytes;
        for (SSTableReader sstable : Iterables.concat(compacting, oldFormatSSTables)) {
            remainingBytes -= sstable.getIndexSummaryOffHeapSize();
        }
        logger.trace("Index summaries for compacting SSTables are using {} MB of space", (Object)((double)(memoryPoolBytes - remainingBytes) / 1024.0 / 1024.0));
        List<SSTableReader> newSSTables = IndexSummaryManager.adjustSamplingLevels(sstablesByHotness, transactions, totalReadsPerSec, remainingBytes);
        for (LifecycleTransaction txn : transactions.values()) {
            txn.finish();
        }
        total = 0L;
        for (SSTableReader sstable : Iterables.concat(compacting, oldFormatSSTables, newSSTables)) {
            total += sstable.getIndexSummaryOffHeapSize();
        }
        logger.trace("Completed resizing of index summaries; current approximate memory used: {} MB", (Object)((double)total / 1024.0 / 1024.0));
        return newSSTables;
    }

    private static List<SSTableReader> adjustSamplingLevels(List<SSTableReader> sstables, Map<UUID, LifecycleTransaction> transactions, double totalReadsPerSec, long memoryPoolCapacity) throws IOException {
        List<ResampleEntry> toDownsample = new ArrayList<ResampleEntry>(sstables.size() / 4);
        ArrayList<ResampleEntry> toUpsample = new ArrayList<ResampleEntry>(sstables.size() / 4);
        ArrayList<ResampleEntry> forceResample = new ArrayList<ResampleEntry>();
        ArrayList<ResampleEntry> forceUpsample = new ArrayList<ResampleEntry>();
        ArrayList<SSTableReader> newSSTables = new ArrayList<SSTableReader>(sstables.size());
        long remainingSpace = memoryPoolCapacity;
        for (SSTableReader sSTableReader : sstables) {
            long spaceUsed;
            int minIndexInterval = sSTableReader.metadata.getMinIndexInterval();
            int maxIndexInterval = sSTableReader.metadata.getMaxIndexInterval();
            double readsPerSec = sSTableReader.getReadMeter() == null ? 0.0 : sSTableReader.getReadMeter().fifteenMinuteRate();
            long idealSpace = Math.round((double)remainingSpace * (readsPerSec / totalReadsPerSec));
            int currentNumEntries = sSTableReader.getIndexSummarySize();
            double avgEntrySize = (double)sSTableReader.getIndexSummaryOffHeapSize() / (double)currentNumEntries;
            long targetNumEntries = Math.max(1L, Math.round((double)idealSpace / avgEntrySize));
            int currentSamplingLevel = sSTableReader.getIndexSummarySamplingLevel();
            int maxSummarySize = sSTableReader.getMaxIndexSummarySize();
            if (sSTableReader.getMinIndexInterval() != minIndexInterval) {
                int effectiveSamplingLevel = (int)Math.round((double)currentSamplingLevel * ((double)minIndexInterval / (double)sSTableReader.getMinIndexInterval()));
                maxSummarySize = (int)Math.round((double)maxSummarySize * ((double)sSTableReader.getMinIndexInterval() / (double)minIndexInterval));
                logger.trace("min_index_interval changed from {} to {}, so the current sampling level for {} is effectively now {} (was {})", new Object[]{sSTableReader.getMinIndexInterval(), minIndexInterval, sSTableReader, effectiveSamplingLevel, currentSamplingLevel});
                currentSamplingLevel = effectiveSamplingLevel;
            }
            int newSamplingLevel = IndexSummaryBuilder.calculateSamplingLevel(currentSamplingLevel, currentNumEntries, targetNumEntries, minIndexInterval, maxIndexInterval);
            int numEntriesAtNewSamplingLevel = IndexSummaryBuilder.entriesAtSamplingLevel(newSamplingLevel, maxSummarySize);
            double effectiveIndexInterval = sSTableReader.getEffectiveIndexInterval();
            logger.trace("{} has {} reads/sec; ideal space for index summary: {} bytes ({} entries); considering moving from level {} ({} entries, {} bytes) to level {} ({} entries, {} bytes)", new Object[]{sSTableReader.getFilename(), readsPerSec, idealSpace, targetNumEntries, currentSamplingLevel, currentNumEntries, (double)currentNumEntries * avgEntrySize, newSamplingLevel, numEntriesAtNewSamplingLevel, (double)numEntriesAtNewSamplingLevel * avgEntrySize});
            if (effectiveIndexInterval < (double)minIndexInterval) {
                logger.trace("Forcing resample of {} because the current index interval ({}) is below min_index_interval ({})", new Object[]{sSTableReader, effectiveIndexInterval, minIndexInterval});
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                forceResample.add(new ResampleEntry(sSTableReader, spaceUsed, newSamplingLevel));
                remainingSpace -= spaceUsed;
            } else if (effectiveIndexInterval > (double)maxIndexInterval) {
                logger.trace("Forcing upsample of {} because the current index interval ({}) is above max_index_interval ({})", new Object[]{sSTableReader, effectiveIndexInterval, maxIndexInterval});
                newSamplingLevel = Math.max(1, 128 * minIndexInterval / maxIndexInterval);
                numEntriesAtNewSamplingLevel = IndexSummaryBuilder.entriesAtSamplingLevel(newSamplingLevel, sSTableReader.getMaxIndexSummarySize());
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                forceUpsample.add(new ResampleEntry(sSTableReader, spaceUsed, newSamplingLevel));
                remainingSpace = (long)((double)remainingSpace - avgEntrySize * (double)numEntriesAtNewSamplingLevel);
            } else if ((double)targetNumEntries >= (double)currentNumEntries * 1.5 && newSamplingLevel > currentSamplingLevel) {
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                toUpsample.add(new ResampleEntry(sSTableReader, spaceUsed, newSamplingLevel));
                remainingSpace = (long)((double)remainingSpace - avgEntrySize * (double)numEntriesAtNewSamplingLevel);
            } else if ((double)targetNumEntries < (double)currentNumEntries * 0.75 && newSamplingLevel < currentSamplingLevel) {
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                toDownsample.add(new ResampleEntry(sSTableReader, spaceUsed, newSamplingLevel));
                remainingSpace -= spaceUsed;
            } else {
                logger.trace("SSTable {} is within thresholds of ideal sampling", (Object)sSTableReader);
                remainingSpace -= sSTableReader.getIndexSummaryOffHeapSize();
                newSSTables.add(sSTableReader);
                transactions.get(sSTableReader.metadata.cfId).cancel(sSTableReader);
            }
            totalReadsPerSec -= readsPerSec;
        }
        if (remainingSpace > 0L) {
            Pair<List<SSTableReader>, List<ResampleEntry>> result = IndexSummaryManager.distributeRemainingSpace(toDownsample, remainingSpace);
            toDownsample = (List)result.right;
            newSSTables.addAll((Collection)result.left);
            for (SSTableReader sstable : (List)result.left) {
                transactions.get(sstable.metadata.cfId).cancel(sstable);
            }
        }
        toDownsample.addAll(forceResample);
        toDownsample.addAll(toUpsample);
        toDownsample.addAll(forceUpsample);
        for (ResampleEntry resampleEntry : toDownsample) {
            SSTableReader sstable = resampleEntry.sstable;
            logger.trace("Re-sampling index summary for {} from {}/{} to {}/{} of the original number of entries", new Object[]{sstable, sstable.getIndexSummarySamplingLevel(), 128, resampleEntry.newSamplingLevel, 128});
            ColumnFamilyStore cfs = Keyspace.open(sstable.metadata.ksName).getColumnFamilyStore(sstable.metadata.cfId);
            SSTableReader replacement = sstable.cloneWithNewSummarySamplingLevel(cfs, resampleEntry.newSamplingLevel);
            newSSTables.add(replacement);
            transactions.get(sstable.metadata.cfId).update(replacement, true);
        }
        return newSSTables;
    }

    @VisibleForTesting
    static Pair<List<SSTableReader>, List<ResampleEntry>> distributeRemainingSpace(List<ResampleEntry> toDownsample, long remainingSpace) {
        int noDownsampleCutoff;
        long extraSpaceRequired;
        Collections.sort(toDownsample, new Comparator<ResampleEntry>(){

            @Override
            public int compare(ResampleEntry o1, ResampleEntry o2) {
                return Double.compare(o1.sstable.getIndexSummaryOffHeapSize() - o1.newSpaceUsed, o2.sstable.getIndexSummaryOffHeapSize() - o2.newSpaceUsed);
            }
        });
        ArrayList<SSTableReader> willNotDownsample = new ArrayList<SSTableReader>();
        for (noDownsampleCutoff = 0; remainingSpace > 0L && noDownsampleCutoff < toDownsample.size(); remainingSpace -= extraSpaceRequired, ++noDownsampleCutoff) {
            ResampleEntry entry = toDownsample.get(noDownsampleCutoff);
            extraSpaceRequired = entry.sstable.getIndexSummaryOffHeapSize() - entry.newSpaceUsed;
            if (extraSpaceRequired > remainingSpace) break;
            logger.trace("Using leftover space to keep {} at the current sampling level ({})", (Object)entry.sstable, (Object)entry.sstable.getIndexSummarySamplingLevel());
            willNotDownsample.add(entry.sstable);
        }
        return Pair.create(willNotDownsample, toDownsample.subList(noDownsampleCutoff, toDownsample.size()));
    }

    static {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(instance, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static class ReadRateComparator
    implements Comparator<SSTableReader> {
        private final Map<SSTableReader, Double> readRates;

        public ReadRateComparator(Map<SSTableReader, Double> readRates) {
            this.readRates = readRates;
        }

        @Override
        public int compare(SSTableReader o1, SSTableReader o2) {
            Double readRate1 = this.readRates.get(o1);
            Double readRate2 = this.readRates.get(o2);
            if (readRate1 == null && readRate2 == null) {
                return 0;
            }
            if (readRate1 == null) {
                return -1;
            }
            if (readRate2 == null) {
                return 1;
            }
            return Double.compare(readRate1, readRate2);
        }
    }

    private static class ResampleEntry {
        public final SSTableReader sstable;
        public final long newSpaceUsed;
        public final int newSamplingLevel;

        public ResampleEntry(SSTableReader sstable, long newSpaceUsed, int newSamplingLevel) {
            this.sstable = sstable;
            this.newSpaceUsed = newSpaceUsed;
            this.newSamplingLevel = newSamplingLevel;
        }
    }
}

