/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.collect.Iterables;
import java.net.InetAddress;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.ReadRepairDecision;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.SinglePartitionReadCommand;
import org.apache.cassandra.db.partitions.PartitionIterator;
import org.apache.cassandra.exceptions.ReadFailureException;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.metrics.ReadRepairMetrics;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.schema.SpeculativeRetryParam;
import org.apache.cassandra.service.DigestMismatchException;
import org.apache.cassandra.service.DigestResolver;
import org.apache.cassandra.service.ReadCallback;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.tracing.TraceState;
import org.apache.cassandra.tracing.Tracing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractReadExecutor {
    private static final Logger logger = LoggerFactory.getLogger(AbstractReadExecutor.class);
    protected final ReadCommand command;
    protected final List<InetAddress> targetReplicas;
    protected final ReadCallback handler;
    protected final TraceState traceState;

    AbstractReadExecutor(Keyspace keyspace, ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
        this.command = command;
        this.targetReplicas = targetReplicas;
        this.handler = new ReadCallback(new DigestResolver(keyspace, command, consistencyLevel, targetReplicas.size()), consistencyLevel, command, targetReplicas);
        this.traceState = Tracing.instance.get();
        int digestVersion = MessagingService.current_version;
        for (InetAddress replica : targetReplicas) {
            digestVersion = Math.min(digestVersion, MessagingService.instance().getVersion(replica));
        }
        command.setDigestVersion(digestVersion);
    }

    protected void makeDataRequests(Iterable<InetAddress> endpoints) {
        this.makeRequests(this.command, endpoints);
    }

    protected void makeDigestRequests(Iterable<InetAddress> endpoints) {
        this.makeRequests(this.command.copy().setIsDigestQuery(true), endpoints);
    }

    private void makeRequests(ReadCommand readCommand, Iterable<InetAddress> endpoints) {
        boolean hasLocalEndpoint = false;
        for (InetAddress endpoint : endpoints) {
            if (StorageProxy.canDoLocalRequest(endpoint)) {
                hasLocalEndpoint = true;
                continue;
            }
            if (this.traceState != null) {
                this.traceState.trace("reading {} from {}", readCommand.isDigestQuery() ? "digest" : "data", endpoint);
            }
            logger.trace("reading {} from {}", (Object)(readCommand.isDigestQuery() ? "digest" : "data"), (Object)endpoint);
            MessageOut<ReadCommand> message = readCommand.createMessage(MessagingService.instance().getVersion(endpoint));
            MessagingService.instance().sendRRWithFailure(message, endpoint, this.handler);
        }
        if (hasLocalEndpoint) {
            logger.trace("reading {} locally", (Object)(readCommand.isDigestQuery() ? "digest" : "data"));
            StageManager.getStage(Stage.READ).maybeExecuteImmediately(new StorageProxy.LocalReadRunnable(this.command, this.handler));
        }
    }

    public abstract void maybeTryAdditionalReplicas();

    public abstract Collection<InetAddress> getContactedReplicas();

    public abstract void executeAsync();

    public PartitionIterator get() throws ReadFailureException, ReadTimeoutException, DigestMismatchException {
        return this.handler.get();
    }

    public static AbstractReadExecutor getReadExecutor(SinglePartitionReadCommand command, ConsistencyLevel consistencyLevel) throws UnavailableException {
        Keyspace keyspace = Keyspace.open(command.metadata().ksName);
        List<InetAddress> allReplicas = StorageProxy.getLiveSortedEndpoints(keyspace, command.partitionKey());
        ReadRepairDecision repairDecision = consistencyLevel == ConsistencyLevel.EACH_QUORUM ? ReadRepairDecision.NONE : command.metadata().newReadRepairDecision();
        List<InetAddress> targetReplicas = consistencyLevel.filterForQuery(keyspace, allReplicas, repairDecision);
        consistencyLevel.assureSufficientLiveNodes(keyspace, targetReplicas);
        if (repairDecision != ReadRepairDecision.NONE) {
            Tracing.trace("Read-repair {}", (Object)repairDecision);
            ReadRepairMetrics.attempted.mark();
        }
        ColumnFamilyStore cfs = keyspace.getColumnFamilyStore(command.metadata().cfId);
        SpeculativeRetryParam retry = cfs.metadata.params.speculativeRetry;
        if (retry.equals(SpeculativeRetryParam.NONE) || consistencyLevel == ConsistencyLevel.EACH_QUORUM || consistencyLevel.blockFor(keyspace) == allReplicas.size()) {
            return new NeverSpeculatingReadExecutor(keyspace, command, consistencyLevel, targetReplicas);
        }
        if (targetReplicas.size() == allReplicas.size()) {
            return new AlwaysSpeculatingReadExecutor(keyspace, cfs, command, consistencyLevel, targetReplicas);
        }
        InetAddress extraReplica = allReplicas.get(targetReplicas.size());
        if (repairDecision == ReadRepairDecision.DC_LOCAL && targetReplicas.contains(extraReplica)) {
            for (InetAddress address : allReplicas) {
                if (targetReplicas.contains(address)) continue;
                extraReplica = address;
                break;
            }
        }
        targetReplicas.add(extraReplica);
        if (retry.equals(SpeculativeRetryParam.ALWAYS)) {
            return new AlwaysSpeculatingReadExecutor(keyspace, cfs, command, consistencyLevel, targetReplicas);
        }
        return new SpeculatingReadExecutor(keyspace, cfs, command, consistencyLevel, targetReplicas);
    }

    private static class AlwaysSpeculatingReadExecutor
    extends AbstractReadExecutor {
        private final ColumnFamilyStore cfs;

        public AlwaysSpeculatingReadExecutor(Keyspace keyspace, ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
            super(keyspace, command, consistencyLevel, targetReplicas);
            this.cfs = cfs;
        }

        @Override
        public void maybeTryAdditionalReplicas() {
        }

        @Override
        public Collection<InetAddress> getContactedReplicas() {
            return this.targetReplicas;
        }

        @Override
        public void executeAsync() {
            this.makeDataRequests(this.targetReplicas.subList(0, this.targetReplicas.size() > 1 ? 2 : 1));
            if (this.targetReplicas.size() > 2) {
                this.makeDigestRequests(this.targetReplicas.subList(2, this.targetReplicas.size()));
            }
            this.cfs.metric.speculativeRetries.inc();
        }
    }

    private static class SpeculatingReadExecutor
    extends AbstractReadExecutor {
        private final ColumnFamilyStore cfs;
        private volatile boolean speculated = false;

        public SpeculatingReadExecutor(Keyspace keyspace, ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
            super(keyspace, command, consistencyLevel, targetReplicas);
            this.cfs = cfs;
        }

        @Override
        public void executeAsync() {
            List initialReplicas = this.targetReplicas.subList(0, this.targetReplicas.size() - 1);
            if (this.handler.blockfor < initialReplicas.size()) {
                this.makeDataRequests(initialReplicas.subList(0, 2));
                if (initialReplicas.size() > 2) {
                    this.makeDigestRequests(initialReplicas.subList(2, initialReplicas.size()));
                }
            } else {
                this.makeDataRequests(initialReplicas.subList(0, 1));
                if (initialReplicas.size() > 1) {
                    this.makeDigestRequests(initialReplicas.subList(1, initialReplicas.size()));
                }
            }
        }

        @Override
        public void maybeTryAdditionalReplicas() {
            if (this.cfs.sampleLatencyNanos > TimeUnit.MILLISECONDS.toNanos(this.command.getTimeout())) {
                return;
            }
            if (!this.handler.await(this.cfs.sampleLatencyNanos, TimeUnit.NANOSECONDS)) {
                ReadCommand retryCommand = this.command;
                if (this.handler.resolver.isDataPresent()) {
                    retryCommand = this.command.copy().setIsDigestQuery(true);
                }
                InetAddress extraReplica = (InetAddress)Iterables.getLast((Iterable)this.targetReplicas);
                if (this.traceState != null) {
                    this.traceState.trace("speculating read retry on {}", extraReplica);
                }
                logger.trace("speculating read retry on {}", (Object)extraReplica);
                int version = MessagingService.instance().getVersion(extraReplica);
                MessagingService.instance().sendRRWithFailure(retryCommand.createMessage(version), extraReplica, this.handler);
                this.speculated = true;
                this.cfs.metric.speculativeRetries.inc();
            }
        }

        @Override
        public Collection<InetAddress> getContactedReplicas() {
            return this.speculated ? this.targetReplicas : this.targetReplicas.subList(0, this.targetReplicas.size() - 1);
        }
    }

    public static class NeverSpeculatingReadExecutor
    extends AbstractReadExecutor {
        public NeverSpeculatingReadExecutor(Keyspace keyspace, ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
            super(keyspace, command, consistencyLevel, targetReplicas);
        }

        @Override
        public void executeAsync() {
            this.makeDataRequests(this.targetReplicas.subList(0, 1));
            if (this.targetReplicas.size() > 1) {
                this.makeDigestRequests(this.targetReplicas.subList(1, this.targetReplicas.size()));
            }
        }

        @Override
        public void maybeTryAdditionalReplicas() {
        }

        @Override
        public Collection<InetAddress> getContactedReplicas() {
            return this.targetReplicas;
        }
    }
}

